// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
//
//	Motorola 68000, 68010, 68020 Context Switching
//
#ifndef	HardwareContext_h
#define	HardwareContext_h
#pragma once

#include <stream.h>
#include <assert.h>
#include <AwesimeConfig.h>

extern const long MagicStackMarker;

typedef void (*voidFuncP)();

class CpuMultiplexor;
class SingleCpuMux;
class MultiCpuMux;
class SimulationMultiplexor;

typedef long HardwareContextQuad;

//
//	Although the 68881 only has 8 registers, they store 12 bytes each.
//
static const int HardwareContextFloatQuads
	= (6 * 12) / sizeof(HardwareContextQuad);

class HardwareContext {

    void** stackBase;	// bottom of stack
    void** stackEnd;	// maximum depth of stack? (actually, a MIN value)
    unsigned stackMax;	// maximum depth of stack? (actually, a MIN value)
    unsigned stackSize;	// stack size in units of void*
    void **stackMallocAt;
    long *stackCheck;	// point to MagicStackMarker to check for corruption
    unsigned checkStackLimits;
    
    friend class Thread;
    friend class CpuMultiplexor;
    friend class SingleCpuMux;
    friend class MultiCpuMux;
    friend class SimulationMultiplexor;
    
    //
    //	Accessed by friend classes only
    //
    void switchContext(HardwareContext *to);
    void magicSwitchTo(HardwareContext *to);
    void **getSp();
    
    void **mallocAt();
    void buildReturnFrame(void * returnThis, voidFuncP returnAddress);
    void stackOverflow();
    
    //
    // never allocated by anything other than friend classes
    //
    HardwareContext(int checked, unsigned stackSize);
    void reclaimStack();
    
public:
    HardwareContext();
    
    long maxStackDepth();
    void checkStack(int overage = 0);
    void classPrintOn(ostream& strm);
};

inline
HardwareContext::HardwareContext()
{
    int NotReached = 0;
    assert( NotReached );
}

inline ostream&
operator<<(ostream& strm, HardwareContext& ob)
{
    ob.classPrintOn(strm);
    return strm;
}

inline void **
HardwareContext::getSp()
{
    void **foo;
    asm volatile ("movel sp,%0" : "=g" (foo));
    return( foo );
}

inline void
HardwareContext::checkStack(int overage)
{
    unsigned depth = stackBase - getSp() + overage;
    if (stackMax < depth) {
	stackMax = depth;
    }
    if ( stackMax >= stackSize || *stackCheck != MagicStackMarker ) {
	stackOverflow();
    }
}

#endif	HardwareContext_h
