/* SharedQueue.c -- implementation of shared queues

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet: keith@alw.nih.gov
	December, 1985

Function:

Class SharedQueue provides a queue that can be used for communication
between two separate processes.  Semaphores are used to block a process
if next() is called and there are no objects in the queue, or if
nextPut() is called and there is no space in the queue.

$Log:	SharedQueue.c,v $
 * Revision 2.204  89/10/07  23:21:37  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:24:25  keith
 * Pre-release
 * 
 * Revision 2.202.1.2  89/07/01  23:44:06  keith
 * Add copy constructors to fix yacc stack overflows
 * 
 * Revision 2.202.1.1  89/07/01  21:56:07  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:56:07  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.3  89/06/22  10:17:29  keith
 * Remove unnecessary copy constructors.
 * 
 * Revision 2.201.1.2  89/06/21  12:49:46  keith
 * Add Object::desc() to VIRTUAL_BASE_CLASSES.
 * Add explicit base class names to constructor initializer lists.
 * 
 * Revision 2.201.1.1  89/05/19  15:44:12  keith
 * Fix bugs in size() and isEmpty().
 * 
 * Revision 2.201  89/05/12  11:20:28  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.3  89/05/12  11:01:49  keith
 * Revised Object I/O.
 * 
 * Revision 2.200.1.2  89/05/03  23:10:23  keith
 * Utilize abstract classes.
 * 
 * Revision 2.200.1.1  89/04/24  17:18:33  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:31:27  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:09:27  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
*/

#include "SharedQueue.h"
#include "nihclIO.h"

#define	THIS	SharedQueue
#define	BASE	Object
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES ArrayOb::desc(),Semaphore::desc(),Semaphore::desc()
#define VIRTUAL_BASE_CLASSES Object::desc()

DEFINE_CLASS(SharedQueue,2,"$Header: SharedQueue.c,v 2.204 89/10/07 23:21:37 keith Stab $",NULL,NULL);

SharedQueue::SharedQueue(unsigned queueSize)
:
	queue(queueSize),
	valueAvailable(0),
	spaceAvailable(queueSize)
{
	readPosition = writePosition = 0;
}

#ifndef BUG_TOOBIG
// yacc stack overflow

SharedQueue::SharedQueue(const SharedQueue& q)
:
	queue(q.queue),
	valueAvailable(q.valueAvailable),
	spaceAvailable(q.spaceAvailable)
{
	readPosition = q.readPosition;
	writePosition = q.writePosition;
}

#endif

Object* SharedQueue::next()
{
	valueAvailable.wait();
	Object* ob = queue[readPosition];
	queue[readPosition++] = nil;
	if (readPosition >= queue.capacity()) readPosition = 0;
	spaceAvailable.signal();
	return ob;
}

Object* SharedQueue::nextPut(Object& ob)
{
	spaceAvailable.wait();
	queue[writePosition++] = &ob;
	if (writePosition >= queue.capacity()) writePosition = 0;
	valueAvailable.signal();
	return &ob;
}

unsigned SharedQueue::capacity() const	{ return queue.capacity(); }

bool SharedQueue::isEmpty() const	{ return valueAvailable.value() <= 0; }

unsigned SharedQueue::size() const	{ return MAX(valueAvailable.value(),0); }

Object* SharedQueue::copy() const	{ return deepCopy(); }

void SharedQueue::deepenShallowCopy()
{
	queue.deepenShallowCopy();
	valueAvailable.deepenShallowCopy();
	spaceAvailable.deepenShallowCopy();
}

void SharedQueue::dumpOn(ostream& strm) const
{
	strm << className() << "[\n";
	strm << "valueAvailable ";  valueAvailable.dumpOn(strm);
	strm << "spaceAvailable ";  spaceAvailable.dumpOn(strm);
	strm << "queue:\n";
	int i = readPosition;
	for (int n = valueAvailable.value(); n>0; n--) {
		queue[i++]->dumpOn(strm);
		if (i == queue.capacity()) i = 0;
	}
	strm << "]\n";
}

unsigned SharedQueue::hash() const	{ return (unsigned)this; }

bool SharedQueue::isEqual(const Object& ob) const   { return isSame(ob); }

void SharedQueue::printOn(ostream& strm) const
{
	int i = readPosition;
	for (int n = valueAvailable.value(); n>0; n--) {
		if (i != readPosition) strm << '\n';
		queue[i++]->printOn(strm);
		if (i == queue.capacity()) i = 0;
	}
}

static unsigned sharedqueue_capacity;

SharedQueue::SharedQueue(OIOin& strm)
	: BASE(strm),
	queue((strm >> sharedqueue_capacity, sharedqueue_capacity)),
	valueAvailable(strm),
	spaceAvailable(strm)
{
	readPosition = 0;
	writePosition = valueAvailable.value();
	for (register int i =0; i<writePosition; i++) queue[i] = Object::readFrom(strm);
}

void SharedQueue::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << queue.capacity();
	valueAvailable.storeMemberOn(strm);
	spaceAvailable.storeMemberOn(strm);
	register int i = readPosition;
	while (i != writePosition) {
		queue[i++]->storeOn(strm);
		if (i == queue.capacity()) i = 0;
	}
}

SharedQueue::SharedQueue(OIOifd& fd)
	: BASE(fd),
	queue((fd >> sharedqueue_capacity, sharedqueue_capacity)),
	valueAvailable(fd),
	spaceAvailable(fd)
{
	readPosition = 0;
	writePosition = valueAvailable.value();
	for (register int i=0; i < writePosition; i++)
	  queue[i] = Object::readFrom(fd);
}

void SharedQueue::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << queue.capacity();
	valueAvailable.storeMemberOn(fd);
	spaceAvailable.storeMemberOn(fd);
	register int i = readPosition;
	while (i != writePosition) {
	  queue[i++]->storeOn(fd);
	  if (i == queue.capacity()) i = 0;
	  };
}

int SharedQueue::compare(const Object&) const
{
	shouldNotImplement("compare");
	return 0;
}
