(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* hanoi.ml     The Hanoi game                                           *)
(*		Pierre Weis						 *)


(*
  The main procedure when the job of the computer is just to tell the player
  what to do (no display of the movments of the discus).
*)
(*|
Value move : (string -> string -> unit)
|*)
let move From To = message ("De'place un disque de " ^ From ^ " a` " ^ To)
;;

(*|
Value teach_hanoi : (string -> string -> string -> num -> unit)
{move}
|*)
let rec teach_hanoi From Middle To = function
   0 -> ()
 | n -> teach_hanoi From To Middle (n-1);
        move From To;
        teach_hanoi Middle From To (n-1)
;;

(* Example :
#teach_hanoi "A" "B" "C" 3 ;;

De'place un disque de A a C
De'place un disque de A a B
De'place un disque de C a B
De'place un disque de A a C
De'place un disque de B a A
De'place un disque de B a C
De'place un disque de A a C
() : unit
*)


(*
  We define some functions which manipulate lists of (characters) strings to
  get a fancy display of the game.
*)
(*|
Value blank_line : (num -> string list)
 CAML_system{replicate}
|*)
let blank_line n = replicate n " ";;

(* To create a discus with a given size *)
(*|
Value discus : (num -> string list)
 CAML_system{@,replicate}
|*)
let discus n =
    let half_right = replicate n ">"
    and half_left  = replicate n "<" in
        half_left @ ["|"] @ half_right
;;

(* Build a discus with the required number of spaces on both sides *)
(*|
Value discus_number : (num -> num -> string list)
{blank_line,discus},
 CAML_system{@}
|*)
let discus_number n number_of_discus =
 let blanks = blank_line (number_of_discus + 1 - n) in
 blanks @ (discus n) @ blanks
;;

(* Build the basis of a pin (depending of the number of discus) *)
(*|
Value pin_basis : (num -> string list)
 CAML_system{@,replicate}
|*)
let pin_basis n =
 let half = replicate n "_" in
  half @ ("|" :: half)
;;

(* Build a line ready to be printed depending on the discus on the line *)

(*|
Value line : (num -> num -> num -> num -> string list)
{discus_number},
 CAML_system{@}
|*)
let rec line number_of_discus From Middle To =
    discus_number From number_of_discus @ " " ::
    discus_number Middle number_of_discus @ " " ::
    discus_number To number_of_discus
;;

(* To print the last line of the display: the basis of the pins *)
(*|
Value basis_line : (num -> string list)
{pin_basis},
 CAML_system{@}
|*)
let basis_line number_of_discus =
 let pin = pin_basis (number_of_discus + 1) in
 pin @ " " :: pin @ " " :: pin
;;

(*
  We suppose here that From, Middle and To, are the lists of discus which are
  on corresponding pines A, B, C.
*)
(*|
Value print_game : (num -> num list -> num list -> num list -> unit)
 CAML_system{hd,tl}
|*)
let rec print_game number_of_discus From Middle To =
 match From with
  [] -> message (implode (basis_line number_of_discus));print_newline()
 | From ->
     message (implode (line number_of_discus (hd From) (hd Middle) (hd To)));
     print_game number_of_discus (tl From) (tl Middle) (tl To)
;;

(*
  To add a number on top of a pin : add_discus 3 [0;0;4] = [0;3;4]
*)
(*|
Value add_discus : (num -> num list -> num list)
 CAML_system{hd,rev,tl}
|*)
let rec add_discus n pin =
 let rec a_d_rec n p = 
  match hd p with
   0 -> n :: (tl p)
  | _ -> (hd p) :: (a_d_rec n (tl p))
 in
 let p = rev pin in
 rev (a_d_rec n p)
;;

(*
  To move a discus From a pin to another :
  move_discus [1;2;3] [0;0;0] = ([0;2;3],[0;0;1])
*)
(*|
Value move_discus : (num list -> num list -> num list * num list)
{add_discus},
 CAML_system{hd,tl}
|*)
let move_discus From To = 
 let rec remove_discus From =
 match hd From with
  0 -> let f',d = remove_discus (tl From) in (0::f'),d
 | _ -> (0 :: (tl From)),(hd From)
 in
 let From',discus = remove_discus From
 in
 From',(add_discus discus To)
;;

(*
  Do nothing just wait : useful to follow the moves of discus.
*)
(*|
Value wait : (unit -> unit)
 CAML_system{input_line,lookahead_ascii,message,std_in}
|*)
let wait () = 
 message"";
 while (lookahead_ascii std_in) = (-1) do () done;
 input_line std_in;();;



(*
  We define a complete procedure including printing function and hanoi
  function, since we need a references the scope of which is these two last
  functions.
*)
(*|
Value game : (num -> unit)
{move_discus,print_game,wait},
 CAML_system{range,replicate}
|*)
let game n =
 let From = ref (range n)
 and Middle = ref (replicate n 0)
 and To = ref (replicate n 0)

in

let movement (From_name,To_name) =
 match From_name,To_name with
   "A","B" -> let f,m = move_discus !From !Middle in
               From:=f;Middle:=m
 | "B","A" -> let m,f = move_discus !Middle !From in
               Middle:=m;From:= f
 | "A","C" -> let f,t = move_discus !From !To  in
               From:=f;To:=t
 | "C","A" -> let t,f = move_discus !To !From in
               To:=t;From:=f
 | "B","C" -> let m,t = move_discus !Middle !To in
               Middle:=m;To :=t
 | "C","B" -> let t,m = move_discus !To !Middle in
               To:=t;Middle:=m
 | _ -> failwith"invalid pin names"

in

let rec hanoi number_of_discus f m t =
fun 0 -> print_newline ()
  | n -> 
    hanoi number_of_discus f t m (n-1);
    print_string "Je de'place un disque de ";print_string f;
    print_string" a` ";message t;print_newline();
    movement (f,t);
    print_game number_of_discus !From !Middle !To;
    wait();(* To prevent scrolling *)
    hanoi number_of_discus m f t (n-1)

in

 pn_message"J'appelle les tiges A, B et C.";
 pn_message"Position de Depart :";print_newline();
 print_game n !From !Middle !To;wait();

 hanoi n "A" "B" "C" n
;;


(*
Example :

#game 3;;

J'appelle les tiges A, B et C.

Position de Depart :

   <|>        |         |    
  <<|>>       |         |    
 <<<|>>>      |         |    
____|____ ____|____ ____|____



Je de'place un disque de A a C

    |         |         |    
  <<|>>       |         |    
 <<<|>>>      |        <|>   
____|____ ____|____ ____|____



Je de'place un disque de A a B

    |         |         |    
    |         |         |    
 <<<|>>>    <<|>>      <|>   
____|____ ____|____ ____|____



Je de'place un disque de C a B

    |         |         |    
    |        <|>        |    
 <<<|>>>    <<|>>       |    
____|____ ____|____ ____|____



Je de'place un disque de A a C

    |         |         |    
    |        <|>        |    
    |       <<|>>    <<<|>>> 
____|____ ____|____ ____|____



Je de'place un disque de B a A

    |         |         |    
    |         |         |    
   <|>      <<|>>    <<<|>>> 
____|____ ____|____ ____|____



Je de'place un disque de B a C

    |         |         |    
    |         |       <<|>>  
   <|>        |      <<<|>>> 
____|____ ____|____ ____|____



Je de'place un disque de A a C

    |         |        <|>   
    |         |       <<|>>  
    |         |      <<<|>>> 
____|____ ____|____ ____|____



() : unit
*)

