(* $Id: rt.ml,v 8.7 91/08/05 15:19:53 ddr Exp $
 *
 * Rogloglo Toolkit
 *
 * $Log:	rt.ml,v $
 * Revision 8.7  91/08/05  15:19:53  ddr
 * - pb openwin
 * 
 * Revision 8.6  91/08/05  15:16:02  ddr
 * *** empty log message ***
 * 
 * Revision 8.5  91/06/20  11:37:16  ddr
 * - remerge avec zinc pour timeout
 * 
 * Revision 8.4  91/06/19  19:59:03  ddr
 * - merge avec zinc
 * 
 * Revision 8.3  91/06/15  16:00:03  ddr
 * - evolution
 * 
 * Revision 8.2  91/06/15  11:02:59  ddr
 * - merge avec zinc
 * 
 * Revision 8.1  91/06/15  09:34:00  ddr
 * - merge avec zinc
 * 
 * Revision 7.9  91/06/07  20:14:49  ddr
 * - redistrib
 *)

#standard arith false;;
#fast arith false;;

type tparam == unit;;
#directive echo_abbrev "tparam" false;;

type xdata = {
    xdm                       : xdata_m;
    wid_by_win                : (Window * widget) list vect;
    wid_by_name               : (string * widget) list vect
}

and global_info == globinfo -> string and globinfo = C'GI
and local_info == locinfo -> string and locinfo = C'LI
and user_info == usinfo -> string and usinfo = C'UI

and xdata_m = {
    dpy                       : Display;
    scr                       : C_Int;
    vis                       : Visual;
    black                     : C_Long;
    white                     : C_Long;
    rootw                     : Window;
    root_width                : num;
    root_height               : num;
    depth                     : C_Int;
    cmap                      : Colormap;
    connection_number         : C_Int;
    gc 	                      : GC;
    xevent                    : xevent;
    mutable win_but           : window_button_state;
    mutable popped_up         : Window list;
    mutable end_func          : (unit -> unit) list;
    ginfo                     : (string * global_info) list vect
}

and widget = {
  wid_xd            : xdata;
  win               : Window;
  mutable x         : num;
  mutable y         : num;
  mutable width     : num;
  mutable height    : num;
  border            : num;
  mutable is_mapped : bool;
  mutable user_info : user_info;
  wdesc             : widget_desc;
  info              : local_info;
  mutable children  : widget list
}

and xevent = {
  mutable x_win     : num;
  mutable y_win     : num;
  mutable x_root    : num;
  mutable y_root    : num;
  mutable button    : num
}

and widget_desc = {
  wsize         : xdata_m -> num * num * num;
  wcreate       : xdata * Window * widget_desc *
                  num * num * num * num * num -> widget;
  wdestroy      : widget -> unit;
  wdispatch     : widget * XEvent * tparam -> tparam;
  filler        : bool
}

and font_struct = {
  fs      : XFontStruct;
  fid     : Font;
  ascent  : num;
  descent : num;
  fwidth  : num;
  fheight : num
}

and glob_struct = {
  xgcv        : XGCValues;
  xev         : XEvent;
  fds         : fd_set
}

and orientation = C'Horizontal | C'Vertical

and pixmap = {
  pixm_xdm      : xdata_m;
  pixmap        : Pixmap
}

and color = {
  col_xdm       : xdata_m;
  pixel         : C_Long
}

and xargs = {
  mutable xdl           : xdata list;
  mutable fd_list       : (num * (unit -> tparam)) list;
  mutable initial_time  : timeb;
  mutable current_time  : num;
  mutable timeout       : num option;
  mutable timeout_fun   : unit -> tparam;
  mutable running       : bool
}

and background = C'NoneBg | C'PixmapBg of Pixmap | C'ColorBg of C_Long

and drawable = C'WidgetDr of widget | C'PixmapDr of pixmap

and attribute =
  C'BackgroundAtt of background
| C'FillerAtt | C'NameAtt of string
| C'WidthAtt of num | C'HeightAtt of num | C'BorderAtt of num
| C'BorderBackgAtt of background

and window_button_state =
  C'WB_None | C'WB_Win of Window | C'WB_WinBut of Window
| C'WB_WinButExit of Window | C'WB_WinButOther of Window
| C'WB_But | C'WB_ButWin
;;

let add_ms_to_time ms t =
  let ms = ms + t.millitm in {
    time = t.time + (ms quo 1000);
    millitm = ms mod 1000
  }
and timeb_sub t1 t2 =
  (t1.time-t2.time)*1000 + t1.millitm-t2.millitm
;;

let dynamo_tag (a : 'a) b r =
  (fun x (y : 'a) -> r := Some x; b),
  (fun f ->
    r := None;
    let v = f a in
    match !r with
      None -> failwith ("bad type \""^v^"\", should be \""^b^"\"")
    | Some x -> x
  )
;;

let dynamo_local_info = dynamo_tag C'LI
and dynamo_global_info = dynamo_tag C'GI
and dynamo_user_info = dynamo_tag C'UI
;;

let ffail _ = failwith "fail";;
let no_info _ = ""
and any_fun _ = ffail()
;;

let mallocated_var alloc_fun r _ =
(* to allocate at run time (for malloc + save_image do not work) *)
  match !r with
    None -> let v = alloc_fun void in r := Some v; v
  | Some v -> v
;;

let gstr =
  let gstr = ref None in
function () ->
  match !gstr with
    Some gstr -> gstr
  | None ->
      let str = {
        xgcv = alloc_XGCValues void;
        xev = alloc_XEvent void;
        fds = alloc_fd_set void
      }
      in
      gstr := Some str;
      str
;;

let rt_initialize name =
  let dpy = XOpenDisplay name in
  if is_null(dpy) then failwith "Can't open display";
  let scr = XDefaultScreen dpy
  and rootw = XDefaultRootWindow dpy in
  let black = XBlackPixel(dpy, scr)
  and white = XWhitePixel(dpy, scr) in
  let xgcv = (gstr()).xgcv in
  set_XGCValues_background(white, xgcv);
  set_XGCValues_foreground(black, xgcv);
  XInitialize void; (* this sets up the error handlers *)
  let gc = XCreateGC(dpy, rootw, Long_OR(GCBackground, GCForeground), xgcv) in
  let xdm = {
    dpy = dpy;
    scr = scr;
    vis = XDefaultVisual(dpy, scr);
    black = black;
    white = white;
    rootw = rootw;
    root_width = num_of_C_Int(XDisplayWidth(dpy, scr));
    root_height = num_of_C_Int(XDisplayHeight(dpy, scr));
    depth = XDefaultDepth(dpy, scr);
    cmap = XDefaultColormap(dpy, scr);
    connection_number = XConnectionNumber dpy;
    gc = gc;
    xevent = {x_win=0; y_win=0; x_root=0; y_root=0; button=0};
    win_but = C'WB_None;
    popped_up = [];
    end_func = [];
    ginfo = (vector 17 of [])
  } in {
    xdm = xdm;
    wid_by_win = (vector 53 of []);
    wid_by_name = (vector 53 of [])
  }
;;

let rt_end xd =
  let xdm = xd.xdm in
  do_list (function f -> f()) xdm.end_func;
  XCloseDisplay xdm.dpy;
  ()
;;

let rt_display_name dname =
  XDisplayName dname
;;

let rt_args xdl = {
  xdl = xdl;
  fd_list = [];
  initial_time = ftime();
  current_time = 0;
  timeout = None;
  timeout_fun = (fun _ -> failwith "timeout");
  running = false
}
;;

exception bad_wid;;

let treat_next_event xd xev param =
  let xdm = xd.xdm in
  XNextEvent(xdm.dpy, xev);
  let win = XAnyEvent_window(XEvent_xany xev) in
  try
    let wid = try hash_assoc win xd.wid_by_win with _ -> raise bad_wid in
    wid.wdesc.wdispatch(wid, xev, param)
  with bad_wid ->
    ()
;;

let try_pending xdl xev param =
  it_list (fun (param, ok) xd ->
    if not is_null(XPending xd.xdm.dpy) then (
      treat_next_event xd xev param, true
    ) else (param, ok)
  ) (param, false) xdl

and dispatch xa xev fds param =
  let param = it_list (fun param (fd, fd_fun) ->
    if not is_null(FD_ISSET(CINT fd, fds)) then fd_fun()
    else param
  ) param xa.fd_list in
  it_list (fun param xd ->
    let xdm = xd.xdm in
    if not is_null(FD_ISSET(xdm.connection_number, fds)) then
      treat_next_event xd xev param
    else param
  ) param xa.xdl
;;

let rt_treat_one_event xa =

  let param = () in
  let init_fds fds =
    FD_ZERO fds;
    let max_fd = it_list (fun max_fd (fd, _) ->
      FD_SET(CINT fd, fds);
      max max_fd fd
    ) 0 xa.fd_list in
    it_list (fun max_fd xd ->
      let xdm = xd.xdm in
      let fd = xdm.connection_number in
      FD_SET(fd, fds);
      max max_fd (num_of_C_Int fd)
    ) max_fd xa.xdl

  in
  if xa.xdl = [] & xa.fd_list = [] & xa.timeout = None then
    failwith "rt_treat_one_event: no xdata, no fd, no timeout";
  let gstr = gstr() in
  xa.current_time <- timeb_sub (ftime()) xa.initial_time;
  match xa.timeout with
    None ->
      let param, ok = try_pending xa.xdl gstr.xev param in
      if ok then param
      else (
        let max_fd = init_fds gstr.fds in
        fselect(CINT(max_fd+1), gstr.fds, CLONG(-1));
        xa.current_time <- timeb_sub (ftime()) xa.initial_time;
        dispatch xa gstr.xev gstr.fds param
      )
  | Some tmout ->
      if xa.current_time < tmout then (
        let param, ok = try_pending xa.xdl gstr.xev param in
        if ok then param
        else (
          let max_fd = init_fds gstr.fds in
          let tm = tmout - xa.current_time in
          let ns = fselect(CINT(max_fd+1), gstr.fds, CLONG tm) in
          if is_null ns then (
            xa.current_time <- tmout;
            xa.timeout <- None;
            xa.timeout_fun()
          )
          else (
            xa.current_time <- timeb_sub (ftime()) xa.initial_time;
            dispatch xa gstr.xev gstr.fds param
          )
        )
      )
      else (
        xa.current_time <- tmout;
        xa.timeout <- None;
        xa.timeout_fun()
      )
;;

let rt_main_loop xa =
  xa.running <- true;
  while xa.running do rt_treat_one_event xa done

and rt_stop_main_loop xa =
  xa.running <- false;
  ()

and rt_set_timeout_fun(xa, timeout_fun) =
  xa.timeout_fun <- timeout_fun;
  ()

and rt_set_timeout(xa, timeout) =
  xa.timeout <- Some timeout;
  ()

and rt_reset_timeout xa =
  xa.timeout <- None;
  ()

and rt_current_time xa =
  xa.current_time

and rt_select_xdata(xa, xd) =
  xa.xdl <- add_setq xd xa.xdl;
  ()

and rt_unselect_xdata(xa, xd) =
  xa.xdl <- filter_neg (curry eq xd) xa.xdl;
  ()

and rt_select_file(xa, fd, fd_fun) =
  xa.fd_list <- (fd, fd_fun)::(
    filter_neg (fun (fd1, _) -> fd = fd1) xa.fd_list
  );
  ()

and rt_unselect_file(xa, fd) =
  xa.fd_list <- filter_neg (fun (fd1, _) -> fd = fd1) xa.fd_list;
  ()
;;

let set_wm_hints =
  let xwmh = mallocated_var alloc_XWMHints (ref None) in
function(xd, wid) ->
  let xdm = xd.xdm
  and xwmh = xwmh() in
  set_XWMHints_input(One_Int, xwmh);
  set_XWMHints_flags(InputHint, xwmh);
  XSetWMHints(xdm.dpy, wid.win, xwmh);
  ()
;;

let rt_create_widget(xd, wname, iname, wdesc) =
  let xdm = xd.xdm in
  let (width, height, border) = wdesc.wsize xdm in
  let wid = wdesc.wcreate(
    xd, xdm.rootw, wdesc, 0, 0, width, height, border
  ) in
  XStoreName(xdm.dpy, wid.win, wname);
  XSetIconName(xdm.dpy, wid.win, iname);
  set_wm_hints(xd, wid);
  wid

and rt_map_widget wid =
  wid.is_mapped <- true;
  XMapWindow(wid.wid_xd.xdm.dpy, wid.win);
  ()

and rt_unmap_widget wid =
  wid.is_mapped <- false;
  XUnmapWindow(wid.wid_xd.xdm.dpy, wid.win);
  ()

and rt_destroy_widget wid =
  let xdm = wid.wid_xd.xdm
  and win = wid.win in
  let rec destroy wid =
    do_list (fun wid -> destroy wid) wid.children;
    wid.wdesc.wdestroy wid
  in
  destroy wid;
  XDestroyWindow(xdm.dpy, win);
  ()
;;

let rt_create_pixmap(xd, width, height) =
  let xdm = xd.xdm in
  let pixmap = XCreatePixmap(
    xdm.dpy, xdm.rootw, CINT width, CINT height, xdm.depth
  ) in
  {pixm_xdm = xdm; pixmap = pixmap}
;;

let widget_named xd wname =
  try hash_assoc wname xd.wid_by_name
  with _ -> failwith ("widget_named " ^ wname)

and add_widget attr win wid =
  let xd = wid.wid_xd in
  hash_add_assoc (win, wid) xd.wid_by_win;
  do_list (function
    C'NameAtt wname ->
      if try hash_assoc wname xd.wid_by_name; true with _ -> false then
        failwith ("double definition of name \"" ^ wname ^ "\"");
      hash_add_assoc (wname, wid) xd.wid_by_name;
      ()
  | _ -> ()
  ) attr;
  wid

and remove_widget attr win wid =
  let xd = wid.wid_xd in
  do_list (function
    C'NameAtt wname -> hash_remove_assoc wname xd.wid_by_name; ()
  | _ -> ()
  ) attr;
  hash_remove_assoc win xd.wid_by_win;
  ()

and add_ginfo xdm wname global_info ginfo =
  hash_add_assoc (wname, (global_info ginfo)) xdm.ginfo;
  ginfo

and ginfo xdm wname =
  hash_assoc wname xdm.ginfo

and remove_ginfo xdm wname =
  hash_remove_assoc wname xdm.ginfo
;;

let load_query_font(xdm, fname) =
  let fs = XLoadQueryFont(xdm.dpy, fname) in
  if is_null fs then failwith "load_query_font";
  let ascent = num_of_C_Int(XFontStruct_ascent fs)
  and descent = num_of_C_Int(XFontStruct_descent fs) in
  {
    fs = fs;
    fid = XFontStruct_fid fs;
    ascent = ascent;
    descent = descent;
    fwidth = num_of_C_Int(XTextWidth(fs, "m", One_Int));
    fheight = ascent+descent
  }
;;

let create_window(xdm, pwin, x, y, width, height, border, attr, smask) =
  let (bg_att, bd_att) = it_list (fun (bg,bd as att) -> function
    C'BackgroundAtt bg -> (Some bg,bd)
  | C'BorderBackgAtt bd -> (bg,Some bd)
  | _ -> att) (None,None) attr in
  let bg = match bg_att with Some(C'ColorBg c) -> c | _ -> xdm.white
  and bd = match bd_att with Some(C'ColorBg c) -> c | _ -> xdm.black in
  let win = XCreateSimpleWindow(xdm.dpy, pwin,
    CINT x, CINT y,
    CINT(max width 1), CINT(max height 1), CINT(max border 0),
    bd, bg
  ) in
  let bg = match bg_att with
    Some C'NoneBg -> Some XNone
  | Some(C'PixmapBg p) -> Some p
  | _ -> None in
  (match bg with Some bg -> XSetWindowBackgroundPixmap(xdm.dpy, win, bg); ()
  | None -> ());
  XSelectInput(xdm.dpy, win, smask);
  win
;;

let NoneBg = C'NoneBg
and PixmapBg p = C'PixmapBg p.pixmap
and ColorBg c = C'ColorBg c.pixel
;;

let WidgetDr w = C'WidgetDr w
and PixmapDr p = C'PixmapDr p
;;

let BackgroundAtt v = C'BackgroundAtt v
and FillerAtt = C'FillerAtt
and NameAtt v = C'NameAtt v
and WidthAtt v = C'WidthAtt v
and HeightAtt v = C'HeightAtt v
and BorderAtt v = C'BorderAtt v
and BorderBackgAtt v = C'BorderBackgAtt v
;;

let screen_width xd = xd.xdm.root_width
and screen_height xd = xd.xdm.root_height
;;

let widget_x wid = wid.x
and widget_y wid = wid.y
and widget_width wid = wid.width
and widget_height wid = wid.height
and widget_border wid = wid.border
and is_mapped wid = wid.is_mapped
;;

let xevent_x xd = xd.xdm.xevent.x_win
and xevent_y xd = xd.xdm.xevent.y_win
and xevent_x_root xd = xd.xdm.xevent.x_root
and xevent_y_root xd = xd.xdm.xevent.y_root
and xevent_button xd = xd.xdm.xevent.button
;;

let Vertical = C'Vertical
and Horizontal = C'Horizontal
;;

(* to autoload hash at load time: *)
suggested_hash_table_size #1
;;
