(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* hash_table.ml Hash and symbol tables                                  *)
(*               Pierre Weis                                             *)

system module hash;;

#standard arith true;;
#fast arith true;;

(* Hash tables de taille variable, les buckets ne pouvant de'passer une
   certaine longueur *)

type 'a hash_table =
     {!Max_bucket_length : int;
      !Number_of_entries : int;
      !Buckets : 'a bucket vect;
      !Hash_table_length : int;
      !May_grow : bool}

and 'a bucket = {!Bucket_length : int; !Bucket_content : 'a list};;


(* Add an entry in a bucket *)
let add_to_bucket key_val bucket =
    bucket.Bucket_content <- key_val::bucket.Bucket_content;
    bucket.Bucket_length <- succ (bucket.Bucket_length);;

(* Remove an entry in a bucket
   (returns -1 if the item has been removed 0 otherwise) *)
let remove_from_bucket key bucket =
    if not
     bucket.Bucket_content ==
      (bucket.Bucket_content <- except_assoc key bucket.Bucket_content)
    then
     (bucket.Bucket_length <- pred (bucket.Bucket_length); 1)
    else 0;;

forward grow_hash_table : ('a*'b) hash_table -> unit;;

(* To figure out if a table is pretty full or not *)
let table_has_to_grow table =
    2 * table.Number_of_entries >=
    table.Max_bucket_length * table.Hash_table_length;;

(* Store an entry in a hash table *)
let store_in_hash_table table (key,val as key_val) =
    let bucket =
     (table . Buckets) . ((hash_univ key) mod table.Hash_table_length) in
    table.Number_of_entries <- succ table.Number_of_entries;
    (* If the bucket becomes too large we try to rebuilt the table *)
    if add_to_bucket key_val bucket > table.Max_bucket_length &
       table.May_grow &
       table_has_to_grow table
     then grow_hash_table table
;;

let remove_from_hash_table table key =
    let bucket =
     (table . Buckets) . ((hash_univ key) mod table.Hash_table_length) in
    
    table.Number_of_entries <-
     table.Number_of_entries - (remove_from_bucket key bucket);
    ()
;;

(* Find a reasonable size for an hash table
   3727 is the maximum size of a table *)
let new_table_length x = when
    x <= 11 -> 23
  | x <= 23 -> 47
  | x <= 47 -> 101
  | x <= 101 -> 211
  | x <= 211 -> 431
  | x <= 431 -> 863
  | x <= 863 -> 1763
  | _        -> 3727;;

(* Init a vector of buckets of a given size for an hash table *)
let init_hash_table_buckets length =
    let v =
    (vector length of {Bucket_length=0;Bucket_content=[]}) in
     modify_vect (fun _ -> {Bucket_length=0;Bucket_content=[]}) v;
    (unchecked_coercion v : 'a bucket vect)
;;

(* Creates an hash table *)
let init_hash_table max_bucket_length length =
    let new_length =
      if length <= 11 then 11 else new_table_length (length/2) in
     (unchecked_coercion 
     {Max_bucket_length = max_bucket_length;
      Number_of_entries = 0;
      Buckets = (init_hash_table_buckets new_length);
      Hash_table_length = new_length;
      May_grow = true} : 'a hash_table)
;;

(* When an hash table has too long buckets it is spread on a larger table *)
let grow_hash_table table =
    let old_buckets = table.Buckets in
    try
    let new_length = new_table_length table.Hash_table_length in
    table . Buckets <- init_hash_table_buckets new_length;
    table . Hash_table_length <- new_length;
    (* We have not to grow up the table size when spreading it *)
    table.May_grow <- false;
    table.Number_of_entries <- 0;
    do_vect
     (function b ->
       do_list (store_in_hash_table table) b.Bucket_content)
     old_buckets;
    (* No the table size may grow again *)
    table.May_grow <- true;
    ()
    with failure "new_table_length" ->
         table.May_grow <- false;
         table.Max_bucket_length <- table.Max_bucket_length + 10;
         ()
;;

let rec pair_assoc_predicate predicate key = find_rec where rec find_rec =
   function
   [] -> failwith "find"
 | (x,_ as pair)::l ->
   if predicate (x,key) then pair else find_rec l
;;

(* Find an entry in an hash table *)
let find_in_hash_table table predicate key =
    match
(*
     let bucket = table.Buckets . ((hash_univ key) mod table.Hash_table_length)
     in bucket.Bucket_content *)
    table.Buckets.((hash_univ key) mod table.Hash_table_length).Bucket_content
    with
      [] -> failwith "undefined entry"
    | [x,_ as pair] ->
      if predicate(x,key) then pair else failwith "undefined entry"
    | l -> pair_assoc_predicate predicate key l
;;

let getq_in_hash_table table key =
    match
(*
     let bucket = table.Buckets . ((hash_univ key) mod table.Hash_table_length)
     in bucket.Bucket_content *)
    table.Buckets.((hash_univ key) mod table.Hash_table_length).Bucket_content
    with
      [] -> failwith "undefined entry"
    | [x,_ as pair] -> if x==key then pair else failwith "undefined entry"
    | l -> pair_assq key l
;;

let get_in_hash_table table key =
    match
(*
     let bucket = table.Buckets . ((hash_univ key) mod table.Hash_table_length)
     in bucket.Bucket_content *)
    table.Buckets.((hash_univ key) mod table.Hash_table_length).Bucket_content
    with
      [] -> failwith "undefined entry"
    | [x,_ as pair] -> if x=key then pair else failwith "undefined entry"
    | l -> pair_assoc key l
;;

end module with value
    init_hash_table
and store_in_hash_table
and getq_in_hash_table
and get_in_hash_table
and find_in_hash_table
and new_table_length
and remove_from_hash_table;
type hash_table and bucket;;

(*
type vbinding == binding;;

type interface_table =
     { Interface_name : string;
       !Open : bool;
       Types : (string * tbinding) hash_table;
       Exceptions : (string * xbinding) hash_table;
       Values : (string * vbinding) hash_table;
       Labels : (string * lbinding) hash_table;
       Constructors : (string * vbinding) hash_table
     }
;;

let (t:(string*int) hash_table) = init_hash_table 2 5;;
(*
store_*
get_*
eventuellement forget

#pragma let MAX_BUCKET_LENGTH = 2
        and MIN_TABLE_SIZE = 5;;

let (global_interfaces:(string * interface_table) hash_table) =
    init_hash_table #MAX_BUCKET_LENGTH #MIN_TABLE_SIZE;;

let open_interface s = (get_global s global_interfaces).Open <- true;;
*)


*)
