(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* Equal.ml	Interface to search by isomorphism functions             *)
(*		Roberto Di Cosmo and Pierre Weis			 *)

#standard arith true;;

module Equal using
 type Flat_type;
 value unify_Left_Commutative :
        Flat_type -> Flat_type -> bool * (int * int) list
 and SplitTR : int -> gtype -> (int * (int * int) list) * int * gtype list
 and flatten : gtype -> Flat_type
 and full_iso : bool ref
 from (user_lib_directory^"FIND_IN_LIB/LC_unify"),
      (user_lib_directory^"FIND_IN_LIB/TypeRewrite");;

#set default grammar gtype:gtype;;

(*\
\begin{caml_eval}
latex_set_pretty false;;
\end{caml_eval}

Now that we are able to perform  unification up to left commutativity, we  are
almost  done:  we just need a standard quadratic test to check equality of two
lists  of coordinates representing  a type.  This is what is essentially known
as  {\em  bag equality}.   The  code that follows  implements  an  specialized
version  of such  test  by means of the  two functions  {\tt findiso} and {\tt
quadratic\_test}.

The  function  {\tt  findiso} looks in the list  of flat type coordinates {\tt
flat\_list}  for  a  coordinate unifiable to {\tt  flat\_coords}  up  to  left
commutativity.  In case  of success, returns  the  renaming  produced  by  the
unification and the rest of the list.

\*)

let rec findiso flat_list flat_coord = 
  match flat_list with 
    []        ->  (false,[],[])
  | (a::rest) ->  let (yn,ren) = unify_Left_Commutative flat_coord a
                  in if yn  then (true,rest,ren) 
                            else let (yn,rest2,ren2) = findiso rest flat_coord
                                 in (yn,a::rest2,ren2);;
(*\

   {\tt quadratic\_test}  takes  two  list of  coordinates and  performs a
   standard  quadratic test of equality using findiso  to perform a linear
   search  of  an element in  a  list.   This  function is  as curried  as
   possible to take advantage of partial evaluation.

\*)

let quadratic_test a b =
  let renaming = ref [] in
  let rec q_test = 
        function [] -> 
            (function [] -> Some !renaming
                  |   _  -> None)
        |   (a::rest_a) as l ->
            (function [] -> None
             | b::rest_b -> let (yn,rest,ren) = findiso l b 
                            in if yn 
                               then (renaming := append ren !renaming;
                                     q_test rest rest_b)
                               else None)
   in renaming := []; q_test a b;;

(*\

Now, let's put all together to get a predicate that tests for equality  modulo
ML isomorphisms.  We try to avoid as  far  as possible useless work, so in the
code for this predicate we reduce the second argument to full flat normal form
only after we know that the number of coordinates  is the same, and  we cannot
avoid performing unification, while the  first argument is  immediately  fully
reduced in order  to take  advantage of partial evaluation.  The  idea is that
{\tt are\_isos}  will be given  a type to  build  a  predicate  that tests for
equality to the given type.

\*)

let are_isos a =
  let ((nextvar_a,renaming_a),(lgt_a,typ_coords_a)) = SplitTR 1 a in
  let flat_coords_a = map flatten typ_coords_a
  in function b -> 
     let ((_,renaming_b),(lgt_b,typ_coords_b)) = SplitTR nextvar_a b
     in if (lgt_a <> lgt_b)
        then (None,renaming_a,renaming_b)
        else let flat_coords_b = map flatten typ_coords_b
             in  (quadratic_test flat_coords_a flat_coords_b,
                  renaming_a,renaming_b);;
(*\

Using this function we build a filter to be applied  to the CAML  system table
in search for functions satisfying a type query. We use the renamings built by
the  rewriting  step  and  the  unification step  to  rename the type  of  any
identifier matching the type query.  In this way the user can easily visualize
how the retrieved function can be used as he wants.

\*)
let build_renaming ren_unif rena renb = 
  let rename_builder renunif rena (key,val) =
      (key,assoc (assoc val renunif) rena)
  in map (rename_builder ren_unif (map (function (x,y) -> (y,x)) rena)) renb;;

let rec rename_type  = function 
    []       -> (function x -> x)
  | renaming -> (function 
                    <<(^l) ^n>> -> <<^(map (rename_type renaming) l) ^n>>
                 |  <<'^i>>     -> <<'^(assoc i renaming)>>);;

let filter_iso_to a = 
    let is_iso_to_a = are_isos a (* partial evaluation of are_isos *)
    in function x ->
      match is_iso_to_a (type_of x) with
        None,_,_ -> ()
      | Some ren_unif, rena_x, renb_y 
        -> print (x^" : ");
           if null ren_unif then print_gtype (type_of x)
           else print_gtype
                 (rename_type   
                   (build_renaming ren_unif rena_x renb_y) (type_of x)); 
           print_newline()
;;
(*\

We  also  define a simple parser for  strings representing CAML types, so that
the search functions will appear more user friendly.

\*)

let gtype_of_string s =
    match parse_string ("<:gtype<"^s^">>;;") with
    ML E ->
     begin match eval_syntax E with
      dynamic (ty : gtype) -> ty | _ -> failwith "invalid type"
     end
  | _ -> failwith "invalid type";;

(*\

We now use  a simple  iterator on the symbol table of the CAML system to build
our search command: it will check the type of  any defined identifier  against
the seeked one.

\*)

let search_iso_gen x =
    do_on_variables (filter_iso_to (gtype_of_string x));;

let search_isos x = full_iso := true; search_iso_gen x;;
let search_iso  x = full_iso := false; search_iso_gen x;;

(*\

  Finally, we export the  user-level functions:  {\tt search\_isos} takes a
  type and returns  all the defined identifiers with types isomorphic to it
  w.r.t.   the  full theory of isomorphism,  while {\tt  search\_iso}  will
  return only the types that can  be  proved isomorphic without  using  the
  rule $\ARR{\Ta}{Unit} = Unit$, that equates too  much functions with side
  effects.

\*)

end module with value
    search_isos       (* :string -> unit *)
and search_iso        (* :string -> unit *)
;;
