(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* display.ml    The display functions for caml graphics                 *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module display
using
type frame = {xmin:float; xmax:float; ymin:float; ymax:float};
type sketch;
type picture;
type linestyle;
type color;
type ps_channel;
value JPICT : picture * picture -> picture;
value make_default_frame_picture : frame -> picture;
value display_picture : picture -> unit;
value fit_picture_in_frame : picture -> frame -> picture;
value fit_sketch_in_frame : sketch -> frame -> sketch;
value picture_frame : picture -> frame;
value default_display_frame : frame;
value make_default_draw_picture :  sketch -> picture;
value showpage_PS : unit -> unit;
value reset_trps  : unit -> unit;
value send_comment_PS : string -> unit;
value open_PS : string -> ps_channel;
value close_PS : unit -> ps_channel;;

#arith float;;

#pragma infix "JPICT";;

let page_height= ref 700.0;;
let page_width = ref 500.0;;
let page_margin = ref 50.0;;

let compute_display_frame p =
 let {xmin=a; xmax=b; ymin=c; ymax=d} = picture_frame p
 in
  let w=b-a and h=d-c and w'= !page_width - 2.0*!page_margin
                      and h' = !page_height -2.0*!page_margin
                      and X= !page_width/2.0
                      and Y = !page_height/2.0
  in
     if (w'/w) < (h'/h)
               then {xmin=X-w'/2.0; 
                     xmax=X+w'/2.0;
                     ymin=Y-h*w'/(2.0*w);
                     ymax=Y+h*w'/(2.0*w)}
               else {xmin=X-w*h'/(2.0*h); 
                     xmax=X+w*h'/(2.0*h); 
                     ymin=Y-h'/2.0;
                     ymax=Y+h'/2.0};;

let center_picture p =  fit_picture_in_frame p (compute_display_frame p);;


let show_picture p = showpage_PS();
                    display_picture  (center_picture p);;

let show_picture_with_frame p = 
   showpage_PS();
   display_picture 
     ( (center_picture p)
       JPICT
       (make_default_frame_picture (compute_display_frame p)));;

let display_picture_with_frame p = 
   showpage_PS();
   display_picture 
     ( p JPICT  (make_default_frame_picture (picture_frame p)));;

let show p = showpage_PS();
            display_picture 
            (fit_picture_in_frame p default_display_frame);;

let showf p = showpage_PS();
            display_picture 
             ((fit_picture_in_frame p default_display_frame)
              JPICT
              (make_default_frame_picture
                           default_display_frame));;

let show_sketch sk = showpage_PS();
                     display_picture
                        (make_default_draw_picture 
                          (fit_sketch_in_frame sk default_display_frame));;

let show_sketchf sk = showpage_PS();
                     display_picture
                        ((make_default_draw_picture 
                          (fit_sketch_in_frame sk default_display_frame))
                         JPICT
                          (make_default_frame_picture
                              default_display_frame));;

let laser (df:picture -> unit) prf p 
            = comline "rm _laser.ps";
              open_PS "_laser";
              reset_trps();
              df p;
              close_PS ();
              comline (prf ^ " _laser.ps");;


let latex_picture p filename =
   let {xmin=a;xmax=b;ymin=c;ymax=d}= picture_frame p
   in
              open_PS filename;
              reset_trps();
              send_comment_PS ("%BoundingBox:"
                              ^ (string_of_float (  a))
                              ^ " "
                              ^ (string_of_float ( c))
                              ^ " "
                              ^ (string_of_float ( b))
                              ^ " "
                              ^ (string_of_float ( d)));
              display_picture p;
              close_PS();
              comline (user_lib_directory ^ "GRAPHICS/ps2tex " 
                      ^ pwd() ^ "/" ^ filename ^ ".ps");;
                              

end module
with
value display_picture_with_frame
  and show_picture
  and show_picture_with_frame
  and show and showf 
  and show_sketch and show_sketchf 
  and laser
  and latex_picture
  and compute_display_frame
  and center_picture
;;


