(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* frames.ml         basic geometric objects                             *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module frames
using
type point = {xc:float;yc:float};

type geom_element =
  Seg of point list
| Arc of point * float * float * float   
        (* center,radius,start_angle,end_angle*)
| Curve of point * point * point * point;
        (* start,control1,control2,end *)
type transformation = {m11:float;m12:float;m13:float;
                       m21:float;m22:float;m23:float};
value point : float * float -> point;
value sinus : float -> float;
value cosinus : float -> float;
value transform_point : transformation -> point -> point;
value CT : transformation * transformation -> transformation;
value translation : float * float -> transformation;
value scaling: float * float -> transformation;;


#pragma infix "CT";;

#arith float;;

type frame = {xmin:float; xmax:float; ymin:float; ymax:float};;  


let frame_center f = {xc=(f.xmin+f.xmax)/2.0 ; yc=(f.ymin+f.ymax)/2.0};;

type extension = All_ext | Horiz_ext | Vertic_ext | Left_ext
               | Right_ext | Top_ext | Bottom_ext;;

let extend_frame ext k {xmin=a;xmax=b;ymin=c;ymax=d} =
  let hmargin = k*(b-a)
  and vmargin = k*(d-c)
  in 
    match ext 
    with  All_ext   ->   {xmin=a-hmargin;xmax=b+hmargin;
                        ymin=c-vmargin;ymax=d+vmargin} 
     |  Horiz_ext ->   {xmin=a-hmargin;xmax=b+hmargin; ymin=c;ymax=d} 
     |  Vertic_ext->   {xmin=a;xmax=b; ymin=c-vmargin;ymax=d+vmargin} 
     |  Left_ext  ->   {xmin=a-hmargin;xmax=b; ymin=c; ymax=d} 
     |  Right_ext ->   {xmin=a; xmax=b+hmargin; ymin=c;ymax=d} 
     |  Top_ext   ->   {xmin=a;xmax=b; ymin=c;ymax=d+vmargin} 
     |  Bottom_ext->   {xmin=a;xmax=b; ymin=c-vmargin;ymax=d} 
;;
  

let point_frame {xc=x;yc=y} = {xmin=x ; xmax =x ; ymin =y; ymax =y};;

let seg_frame = function
   [] -> failwith"seg_frame"
 | (pt::ptl) ->
  let frame = point_frame pt

  in it_list  (fun b {xc=x;yc=y} ->
                  {xmin=min (b.xmin) x;
                   xmax= max (b.xmax) x;
                   ymin= min (b.ymin) y;
                   ymax= max (b.ymax) y})
               frame
               ptl;;


let ordered_angles(a1,x,a2) =
         a1<=x & x <= a2
     or  a1<= x+360.0 & x+360.0 <=a2;;

let arc_frame (c,r,a1,a2) =  
  {xmin = (if ordered_angles(a1,180.0,a2) 
             then c.xc - r
             else  let m = min (cosinus a1) (cosinus a2)
                   in c.xc + m*r);
   xmax = (if ordered_angles(a1,0.0,a2) 
             then c.xc + r
             else  let m = max (cosinus a1) (cosinus a2)
                   in c.xc + m*r);
   ymin = (if ordered_angles(a1,270.0,a2) 
             then c.yc - r
             else  let m = min (sinus a1) (sinus a2)
                   in c.yc + m*r);
   ymax = (if ordered_angles(a1,90.0,a2) 
             then c.yc + r
             else  let m = max (sinus a1) (sinus a2)
                   in c.yc + m*r)};;

let curve_frame (pt1,pt2,pt3,pt4) =
  let x = pt1.xc
  and xl = [pt2.xc;pt3.xc;pt4.xc]
  and y = pt1.yc
  and yl = [pt2.yc;pt3.yc;pt4.yc]
  in
    {xmin = it_list min x xl;
     xmax = it_list max x xl;
     ymin = it_list min y yl;
     ymax = it_list max y yl};;


let merge_frames {xmin=xmin1;xmax=xmax1;ymin=ymin1;ymax=ymax1}
                 {xmin=xmin2;xmax=xmax2;ymin=ymin2;ymax=ymax2}
=
      {xmin= min xmin1 xmin2;
       xmax= max xmax1 xmax2;
       ymin= min ymin1 ymin2;
       ymax= max ymax1 ymax2};;

let compose_frames ({xmin=xmin1;xmax=xmax1;ymin=ymin1;ymax=ymax1} as b)
                   {xmin=xmin2;xmax=xmax2;ymin=ymin2;ymax=ymax2}
=
      {xmin = min xmin1 xmin2;
       xmax = max xmax1 xmax2;
       ymin = min ymin1 ymin2;
       ymax = max ymax1 ymax2};;

let compute_geom_elem_frame =
      fun  (Seg pl) -> seg_frame pl
        |  (Arc a) ->  arc_frame a
        |  (Curve pts) -> curve_frame pts;;

let compute_frame = function
  [] -> failwith "compute_frame"
| ge::gel ->
  let frame = compute_geom_elem_frame ge
  in
  it_list  (fun b ge ->
              merge_frames b (compute_geom_elem_frame ge))
           frame
           gel;;


let transform_frame t  {xmin=xmin;xmax=xmax;ymin=ymin;ymax=ymax} =
 let ptl = map (transform_point t) 
               [point(xmin,ymin);point(xmin,ymax);
                point(xmax,ymin);point(xmax,ymax)]
 in seg_frame ptl;;

let frame_to_frame_transform {xmin=a; xmax=b; ymin=c; ymax=d}
                             {xmin=a'; xmax=b'; ymin=c'; ymax=d'} =

  let hscale = (b'-a')/(b-a)
  and vscale = (d'-c')/(d-c)
  in
    let T1 = translation(-a,-c)
    and  S = scaling (hscale,vscale)
    and	T2 = translation (a',c')

    in  T2 CT S CT T1;;


end module
with
type frame;
type extension;
value compute_geom_elem_frame
  and compute_frame
  and frame_center
  and extend_frame
  and merge_frames
  and compose_frames
  and transform_frame
  and frame_to_frame_transform;;


