(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* texts.ml
              Emmanuel Chailloux & Guy Cousineau
              creation : 22 March 1990                 
	      modification : 26 September 1990                             *)

module text 
using 
type frame = {xmin:float; xmax:float; ymin:float; ymax:float} ;
type point = {xc:float; yc:float};
type font = { Name : string ;  Size : float };
type font_description =
     { Name : string ; Height : float ; Width : float ;
       Descr : float vect;
       Descr_bbox : ((float * float ) * (float * float)) vect };

value  Courier:font_description;
value Courier_Oblique:font_description;
value Courier_Bold:font_description;
value Courier_BoldOblique:font_description;
value Times_Roman:font_description;
value Times_Bold:font_description;
value Times_Italic:font_description;
value Times_BoldItalic:font_description;
value Helvetica:font_description;
value Helvetica_Bold:font_description;
value Helvetica_Oblique:font_description;
value Helvetica_BoldOblique:font_description;
value Symbol:font_description
;;

#arith float;;

(* basic types *)

type text = { t_string : string ;  t_font : font }
;;

(* somes variables of font description *)

let font_list =
 ["Default",Courier; "Courier",Courier; 
  "Courier-Oblique",Courier_Oblique; "Courier-Bold",Courier_Bold; 
  "Courier-BoldOblique",Courier_BoldOblique; "Times-Roman",Times_Roman; 
  "Times-Bold",Times_Bold; "Times-Italic",Times_Italic; 
  "Times-BoldItalic",Times_BoldItalic; "Helvetica",Helvetica; 
  "Helvetica-Bold",Helvetica_Bold; 
  "Helvetica-Oblique",Helvetica_Oblique; 
  "Helvetica-BoldOblique",Helvetica_BoldOblique; "Symbol",Symbol]
;;

let print_font_list () = 
    print_string "Default font is ";
    print_string (assoc "Default" font_list).Name;
    print_newline();
    print_string "Available fonts are: ";
    do_list (fun s -> message (fst s)) font_list;;

let print_info_font f =
 print_string ("Name : "^f.Name);
 print_newline ();print_string ("Max height : "^string_of_float f.Height);
 print_newline ();print_string ("Max width : "^string_of_float f.Width);
 print_newline ();
 if f.Descr = [||]
  then (print_string "Width fixed : true";print_newline ())
  else (print_string "Width fixed : false";print_newline ())
;;

let print_info_all_fonts () =
 let pf (s,f) =
  print_string s;print_newline ();print_info_font f;print_newline () in
  print_newline ();map pf font_list;()
;;
                  
let find_font_description (f:font) =
 try assoc f.Name font_list with
    find -> failwith ("Text: find_font_description: bad name "^f.Name)
;;

(* create a text structure *)



let value_default_font = {Name="Default"; Size=12.0}
;;

let default_font = ref value_default_font
;;

let change_default_font f = default_font := f
;;

let reset_default_font () = default_font := value_default_font

;;

let make_font str sz =
let f = {Name=str; Size=sz} in find_font_description f;f
;;


let make_text str font  = {t_string=str; t_font=font}
;;

let make_default_text  str = { t_string=str; t_font=!default_font}
;;

let change_size_text {t_string=str; t_font={Name=n;Size=_}} sz =
                         {t_string=str; t_font={Name=n;Size=sz}}
;;

let change_font_text { t_string=str; t_font=_} fnt  =
                              { t_string=str; t_font=fnt}     
;;


(* Compute text dimensions *)

let text_frame { t_string =s ; t_font =fnt } =
  let sl = explode_ascii s
  and f = find_font_description fnt
  and sca = fnt.Size / 12.0
  in
    let constant_width = (f.Descr=[||])
    in
      let (ymin,ymax,width) =
         it_list (fun(y1,y2,w) c 
                     -> let descr_bbox= f.Descr_bbox.(c)
                        and descr= if constant_width 
                                      then f.Width
                                      else f.Descr.(c)
                        in
                          let y1' = snd(fst descr_bbox)            
                          and y2' = snd(snd descr_bbox)            
                          in
                           (min y1 y1' , max y2 y2' , w+descr))
                 (f.Height , -f.Height , 0.0)
                 sl
    in
     let xmin= fst(fst f.Descr_bbox.(hd sl))
     in
      let xmax =width- (if constant_width then f.Width else f.Descr.(last sl))
                            + fst( snd f.Descr_bbox.(last sl)) 
      in 
        {xmin=(xmin-0.5)*sca; xmax=(xmax+0.5)*sca; 
         ymin=(ymin-0.5)*sca; ymax=(ymax+0.5)*sca}
;;

let text_width fnt str=
  let frame = text_frame {t_string=str; t_font=fnt}
  in frame.xmax - frame.xmin;;

end module
with value text_frame 
 and text_width
 and make_text
 and make_default_text
 and change_font_text
 and change_size_text
 and find_font_description
 and make_font
 and print_info_all_fonts
 and print_info_font
 and print_font_list
 ; type text
;;



