\begin{caml_primitive}
base_digit_of_char : char * int -> int
sys_string_of_digit : nat * int -> string
string_of_digit : nat -> string
sys_string_list_of_nat : int * nat * int * int -> string list
sys_string_of_nat  : int * string * nat * int * int * string -> string
string_of_nat : nat -> string
string_for_read_of_nat : nat -> string
sys_print_nat : int * string * nat * int * int * string -> unit
print_nat : nat -> unit
print_nat_for_read : nat -> unit
debug_string_vect_nat : nat -> string vect
debug_string_nat : nat -> string
debug_print_nat : nat -> unit
\end{caml_primitive}
\conceptindex{coercion between types \verb"nat" and \verb"string"}
\conceptindex{\verb"print"}
\conceptindex{printing}

\verb"base_digit_of_char (c, base)" verifies \verb"`c`" is a correct 
\verb"base"-digit (as regards to usual conventions only uppercase letters are 
authorized in case of \verb"base" greater than 10) and give its decimal value.

\verb"sys_string_of_digit (nat, off)" map digit $nat_{off}$ onto the 
corresponding string in 10-radix representation.

\verb"string_of_digit nat" map first digit $nat_0$ onto the 
corresponding string in 10-radix representation.

\verb"sys_string_list_of_nat (base, nat, off, len)" maps \verb"base"-radix 
representation of subnat $nat_{off, len}$ onto a list of strings, the first 
string corresponds to the beginning of the number, and so on. Usually one 
string is enough for representing nats but for example the longest
\verb"nat" needs 32 strings for printing its 2-radix representation. 

\verb"sys_string_list_of_nat" is the most general printing function. 
It takes more time than the other printing functions but it can print any 
natural number even when all other printing functions fail.
So the CAML system when printing values of type \verb"nat", \verb"big_int", 
\verb"ratio" or \verb"num" uses this function.

\verb"sys_string_of_nat (base, before, nat, off, len, after)" writes into the 
resulting string first the string \verb"before", then the representation of 
subnat $nat_{off, len}$ in base \verb"base", and last the string \verb"after".
This function is optimized in base 10.

\verb"string_of_nat nat" is semantically (but not practically for efficiency 
reasons) equivalent to:
\begin{small}
\begin{verbatim}
let string_of_nat nat = 
  sys_string_of_nat (#10, "", nat, #0, length_nat nat, "")
;;
\end{verbatim}
\end{small}
\verb"string_for_read_of_nat nat" is analogous to \verb"string_of_nat nat" 
except that the number is written between \verb"#<" and \verb">", according to 
conventions to write natural numbers. 

This function is equivalent to:
\begin{small}
\begin{verbatim}
let string_for_read_of_nat nat = 
  sys_string_of_nat (#10, "#<", nat, #0, length_nat nat, ">")
;;
\end{verbatim}
\end{small}

\begin{beware}
\item This convention is not properly speaking a lexical convention since 
no natural number can be read directly, even though between \verb"#<" and 
\verb">".
\end{beware}

\verb"sys_print_of_nat (base, before, nat, off, len, after)" print the 
successive strings of \verb"sys_string_list_of_nat (base, nat, off, len)", 
between string \verb"before" and \verb"after".

\verb"print_nat" and \verb"print_nat_for_read nat" are equivalent to:
\begin{small}
\begin{verbatim}
let print_nat nat =
  sys_print_nat (#10, "", nat, #0, 
                 num_digits_nat (nat, #0, length_nat nat), "")
;;

let print_nat_for_read nat = 
  sys_print_nat (#10, "#<", nat, #0, 
                 num_digits_nat (nat, #0, length_nat nat), ">")
;;
\end{verbatim}
\end{small}

\verb"debug_string_vect_nat nat" and \verb"debug_string_nat" map $nat$ onto 
its 16-radix representation, without normalizing the number according to its 
real length. If the result cannot fit in only one string,
\verb"debug_string_nat nat" fails but \verb"debug_string_vect_nat nat" is a 
vector of more than one string, the first string correspond to the end 
of the number, and so on.

\verb"debug_print_nat nat" prints the successive strings of
\verb"debug_string_vect_nat nat".

\exple
\begin{small}
\begin{verbatim}
#base_digit_of_char (`9`, #10);;
9 : int

#base_digit_of_char (`A`, #10);;

Evaluation Failed: failure "base_digit_of_char"

#base_digit_of_char (`A`, #16);;
10 : int

#let nat = create_nat #2;;
Value nat is #<0> : nat

#set_digit_nat (nat, #0, #1);;
() : unit

#set_digit_nat (nat, #1, #2);;
() : unit

#string_of_digit nat;;
"1" : string

#sys_string_of_digit (nat, #1);;
"2" : string

#(* A good example for this function would have a really long result.
#   The resulting string should have more than biggest_int digits so
#   takes more than six pages, too long for doing it here but it works well *) 
#sys_string_list_of_nat (#10, nat, #0, #1);;  
#sys_string_of_nat (#16, "nat 16-radix representation = ", nat, #0, #2, "");;
["1"] : string list

#sys_string_of_nat (#2, "nat 2-radix representation = ", nat, #0, #2, "");;
"nat 2-radix representation = 1000000000000000000000000000000001" : string

#sys_string_of_nat (#8, "nat 8-radix representation = ", nat, #0, #2, "");;
"nat 8-radix representation = 100000000001" : string

#sys_string_of_nat (#10, "nat 10-radix representation = ", nat, #0, #2, "");;
"nat 10-radix representation = 8589934593" : string

#sys_string_of_nat (#10, "nat first digit 10-radix representation = ", 
#                   nat, #0, #1, "");;
"nat first digit 10-radix representation = 1" : string

#string_of_nat nat;;
"8589934593" : string

#string_for_read_of_nat nat;;
"#<8589934593>" : string

#sys_print_nat (#16, "nat 16-radix representation = ", nat, #0, #2, "");;
nat 16-radix representation = 200000001() : unit

#print_nat nat;;
8589934593() : unit

#print_nat_for_read nat;;
#<8589934593>() : unit

#(* Same remark as for sys_string_list_of_nat *)
#debug_string_vect_nat nat;;
[|"|00000002|00000001|"|] : string vect

#debug_string_nat nat;;
"|00000002|00000001|" : string

#debug_print_nat nat;;
|00000002|00000001|() : unit
\end{verbatim}
\end{small}
