#! /usr/bin/env false

use v6.d;

use Base64;
use JSON::Fast;
use LibCurl::Easy;
use URL;

unit class HTTP::API::Pingdom;

#| The base URL to use for constructing requests to Pingdom.
has URL:D $.base-url = URL.new("https://api.pingdom.com");

#| The email address to use for authentication to the API. This is the same
#| email address as you use to log in to the web interface.
has Str:D $.email is required;

#| The password to use for authentication to the API. This is the same password
#| as you use to log in to the web interface.
has Str:D $.password is required;

#| The application key to use for this application. You need to generate one in
#| the web interface, in I<The Pingdom API>, found under the I<Integrations>
#| section.
has Str:D $.app-key is required;

#| The email address to perform actions with. This is different from your
#| authentication email address, and is used by Pingdom Team Accounts.
has Str $.account-email;

#| Get the base64 representation of the credentials. These are needed for the
#| Authentication header.
method !http-credentials (--> Str)
{
	"$!email:$!password".&encode-base64(:str);
}

#| Make a request to the API
method !request (
	#| The endpoint to make a request to.
	@endpoint-parts,

	#| Additional parameters to include in the request.
	%params = {},

	#| The HTTP method to use.
	Str:D :$method = "GET",
) {
	my $url = $!base-url.add-path(|@endpoint-parts);

	# Add parameters, if any were supplied

	# Create a new request
	my $request = LibCurl::Easy.new;

	given $method.fc {
		when "get" { $url .= add-query(|%params) if %params }
		when "delete"|"post"|"put" {
			$request.setopt(postfields => %params.keys.sort.map({ "$_={%params{$_}}" }).join("&"))
		}
	}

	$request.URL(~$url);
	$request.customrequest($method.uc);

	# Add headers
	$request.set-header(Authorization => "Basic {self!http-credentials}");
	$request.set-header(App-Key => $!app-key);
	$request.set-header(Account-Email => $!account-email) if $!account-email;

	# Make the request
	note "> Authorization: Basic {self!http-credentials}" if %*ENV<DEBUG>;
	note "> App-Key: $!app-key" if %*ENV<DEBUG>;
	note "> Account-Email: $!account-email" if %*ENV<DEBUG>;
	note "> {$method.uc} $url" if %*ENV<DEBUG>;

	$request.perform;

	if ($request.response-code ≠ 200) {
		say $request.content if %*ENV<DEBUG>;
		die "HTTP request failed ($request.getinfo("effective-url")): {$request.statusline}";
	}

	note "< Req-Limit-Short: {$request.get-header("Req-Limit-Short")}" if %*ENV<DEBUG>;
	note "< Req-Limit-Long: {$request.get-header("Req-Limit-Long")}" if %*ENV<DEBUG>;

	$request.content.&from-json;
}

#
# Here be the user-facing interface.
#

#| Returns a list of actions (alerts) that have been generated for your
#| account.
method get-actions-list (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 actions >, %params)
}

#| Returns a list of the latest root cause analysis results for a specified
#| check.
method get-root-cause-analysis-results-list (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 analysis $checkid », %params)
}

#| Returns the raw result for a specified error analysis. This data is
#| primarily intended for internal use, but you might be interested in it as
#| well. However, there is no real documentation for this data at the moment.
method get-raw-analysis-results (
	Int:D $checkid,
	Int:D $analysisid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 analysis $checkid $analysisid », %params)
}

#| Returns a list overview of all checks.
method get-check-list (
	*%params,

--> Hash
) {
	self!request(< api 2.1 checks >, %params)
}

#| Returns a detailed description of a specified check.
method get-detailed-check-information (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 checks $checkid », %params)
}

#| Creates a new check with settings specified by provided parameters.
method create-new-check (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 checks >, %params, :method<post>)
}

#| Modify settings for a check. The provided settings will overwrite previous
#| values. Settings not provided will stay the same as before the update. To
#| clear an existing value, provide an empty value. Please note that you cannot
#| change the type of a check once it has been created.
method modify-check (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 checks $checkid », %params, :method<put>)
}

#| Pause or change resolution for multiple checks in one bulk call.
method modify-multiple-checks (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 checks >, %params, :method<put>)
}

#| Deletes a check. THIS METHOD IS IRREVERSIBLE! You will lose all collected
#| data. Be careful!
method delete-check (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(< api 2.1 checks >, %params, :method<delete>)
}

#| Deletes a list of checks. THIS METHOD IS IRREVERSIBLE! You will lose all
#| collected data. Be careful!
method delete-multiple-checks (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 checks >, %params, :method<delete>)
}

#| Returns information about remaining checks, SMS credits and SMS auto-refill
#| status.
method get-credits-list (
	--> Hash
) {
	self!request(< api 2.1 credits >)
}

#| Returns a list of user's maintenance windows.
method get-maintenance-window-list (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 maintenance >, %params)
}

#| Returns the maintenance window specified by its id.
method get-maintenance-window-detail (
	Int:D $id,

	--> Hash
) {
	self!request(« api 2.1 maintenance $id »)
}

#| Create new maintenance window.
method create-new-maintenance-window (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 maintenance >, %params, :method<post>)
}

#| Modify the maintenance window.
method modify-the-maintenance-window (
	Int:D $id,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 maintenance $id », %params, :method<put>)
}

#| Delete the maintenance window. Note that only future maintenance window can
#| be deleted.
method delete-the-maintenance-window (
	Int:D $id,

	--> Hash
) {
	self!request(« api 2.1 maintenance $id », :method<delete>)
}

#| Delete multiple maintenance windows. Note that only future maintenance
#| windows can be deleted.
method delete-multiple-maintenance-windows (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 maintenance >, %params, :method<delete>)
}

#| Gets a maintenance occurrence details specified by its identifier.
method get-maintenance-occurrence (
	Int:D $id,

	--> Hash
) {
	self!request(« api 2.1 maintenance.occurrences $id »)
}

#| Returns a list of maintenance occurrences.
method get-maintenance-occurrences-list (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 maintenance.occurrences >, %params)
}

#| Modifies a maintenance occurrence specified by its identifier.
method modify-maintenance-occurence (
	Int:D $id,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 maintenance.occurrences $id », %params, :method<put>)
}

#| Deletes the maintenance occurrence specified by its identifier. Note that
#| only future occurrences can be deleted.
method delete-maintenance-occurrence (
	Int:D $id,

	--> Hash
) {
	self!request(« api 2.1 maintenance.occurrences $id », :method<delete>)
}

#| Deletes multiple maintenance occurrences specified by their identifiers.
#| Note that only future occurrences can be deleted.
method delete-multiple-maintenance-occurrences (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 maintenance.occurrences >, %params, :method<delete>)
}

#| Returns a list of all Pingdom probe servers for Uptime and Transaction
#| checks.
method get-probe-server-list (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 probes >, %params)
}

#| Get a reference of regions, timezones and date/time/number formats and their
#| identifiers.
method get-reference (
	--> Hash
) {
	self!request(< api 2.1 reference >)
}

#| Returns a list of email report subscriptions.
method get-email-report-subscription-list (
	--> Hash
) {
	self!request(< api 2.1 reports.email >)
}

#| Creates a new email report.
method create-email-report (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 reports.email >, %params, :method<post>)
}

#| Modify an email report.
method modify-email-report (
	Int:D $reportid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 reports.email $reportid », %params, :method<put>)
}

#| Delete an email report.
method delete-email-report (
	Int:D $reportid,

	--> Hash
) {
	self!request(« api 2.1 reports.email $reportid », :method<delete>)
}

#| Returns a list of public (web-based) reports.
method get-public-report-list (
	--> Hash
) {
	self!request(< api 2.1 reports.public >)
}

#| Activate public report for a specified check.
method publish-public-report (
	Int:D $checkid,

	--> Hash
) {
	self!request(« api 2.1 reports.public $checkid », :method<put>)
}

#| Deactivate public report for a specified check.
method withdraw-public-report (
	Int:D $checkid,

	--> Hash
) {
	self!request(« api 2.1 reports.public $checkid », :method<delete>)
}

#| Returns a list of shared reports (banners).
method get-shared-reports-list (
	--> Hash
) {
	self!request(< api 2.1 reports.shared >)
}

#| Create a shared report (banner).
method create-shared-report (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 reports.shared >, %params, :method<post>)
}

#| Delete a shared report (banner).
method delete-shared-report (
	Int:D $reportid,

	--> Hash
) {
	self!request(« api 2.1 reports.shared $reportid », :method<delete>)
}

#| Return a list of raw test results for a specified check
method get-raw-check-results (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 results $checkid », %params)
}

#| Get the current time of the API server.
method get-current-server-time (
	--> Hash
) {
	self!request(< api 2.1 servertime >)
}

#| Returns all account-specific settings.
method get-account-settings (
	--> Hash
) {
	self!request(< api 2.1 settings >)
}

#| Modify account-specific settings.
method modify-account-settings (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 settings >, %params, :method<put>)
}

#| Performs a single test using a specified Pingdom probe against a specified
#| target. Please note that this method is meant to be used sparingly, not to
#| set up your own monitoring solution.
method make-a-single-test (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 single >, %params)
}

#| Get the average time / uptime value for a specified check and time period.
method get-a-response-time (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 summary.average $checkid », %params)
}

#| Get the average time / uptime value for a specified check and time period.
method uptime-average (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self.get-a-response-time($checkid, |%params)
}

#| Returns the average response time for each hour of the day (0-23) for a
#| specific check over a selected time period. I.e. it shows you what an
#| average day looks like during that time period.
method get-response-time-averages-for-each-hour-of-the-day (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 summary.hoursofday $checkid », %params)
}

#| Get a list of status changes for a specified check and time period. If order
#| is speficied to descending, the list is ordered by newest first. (Default is
#| ordered by oldest first.)
method get-list-of-outages (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 summary.outage $checkid », %params)
}

#| For a given interval in time, return a list of sub intervals with the given
#| resolution. Useful for generating graphs. A sub interval may be a week, a
#| day or an hour depending on the choosen resolution.
method get-intervals-of-average-response-time-and-uptime-during-a-given-interval (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 summary.performance $checkid », %params)
}

#| Get a list of probes that performed tests for a specified check during a
#| specified period.
method get-active-probes-for-a-period (
	Int:D $checkid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 summary.probes $checkid », %params)
}

#| Returns a list of all teams.
method get-teams (
	--> Hash
) {
	self!request(< api 2.1 teams >)
}

#| Returns team information.
method get-team (
	Int:D $teamid,

	--> Hash
) {
	self!request(« api 2.1 teams $teamid »)
}

#| Creates a new team.
method create-team (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 teams >, %params, :method<post>)
}

#| Update a team.
method update-team (
	Int:D $teamid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 teams $teamid », %params, :method<put>)
}

#| Remove one or more team.
method delete-team (
	Int:D $teamid,

	--> Hash
) {
	self!request(« api 2.1 teams $teamid », :method<delete>)
}

#| Perform a traceroute to a specified target from a specified Pingdom probe.
method make-a-traceroute (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 traceroute >, %params)
}

#| Returns a list of all users and contact targets
method get-list-of-users-and-contact-targets (
	--> Hash
) {
	self!request(< api 2.1 users >)
}

#| Creates a new user.
method create-user (
	*%params,

	--> Hash
) {
	self!request(< api 2.1 users >, %params, :method<post>)
}

#| Modify a User.
method modify-user (
	Int:D $userid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 users $userid », %params, :method<put>)
}

#| Modify a Contact Target.
method modify-contact-target (
	Int:D $userid,
	Int:D $targetid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 users $userid $targetid », %params, :method<put>)
}

#| Deletes a user.
method delete-user (
	Int:D $userid,

	--> Hash
) {
	self!request(« api 2.1 users $userid », :method<delete>);
}

#| Deletes a contacts target.
method delete-contact-target (
	Int:D $userid,
	Int:D $targetid,

	--> Hash
) {
	self!request(« api 2.1 users $userid $targetid », :method<delete>)
}

#| Creates a new contact target.
method create-a-new-contact-target (
	Int:D $userid,
	*%params,

	--> Hash
) {
	self!request(« api 2.1 users $userid », %params, :method<post>)
}

=begin pod

=NAME    HTTP::API::Pingdom
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.1.0

=head1 Synopsis

=begin code

use HTTP::API::Pingdom;

my $api = HTTP::API::Pingdom.new(
	email => "whomever@example.org",
	password => "donthackme",
	app-key => "again,pleasedonthackme,thatsrude",
);

$api.get-check-list;

=end code

=head1 Description

A Perl 6 library to interface with the L<Pingdom
API|https://www.pingdom.com/api/2.1/>.

The method names are similar to the names used in the Pingdom API
documentation, with spaces changed to C<->, lowercase and text in parenthesis
ignored. For the methods that have alternate names (denoted by a C</> in their
names), both method names are available in this library, and both function in
the same way.

=end pod

# vim: ft=perl6 noet
