=head1 NAME

Sys::Statistics::Linux::SockStats - Collect linux socket statistics.

=head1 SYNOPSIS

    use Sys::Statistics::Linux::SockStats;

    my $lxs  = Sys::Statistics::Linux::SockStats->new;
    my $stat = $lxs->get;

=head1 DESCRIPTION

Sys::Statistics::Linux::SockStats gathers socket statistics from the virtual F</proc> filesystem (procfs).

For more information read the documentation of the front-end module L<Sys::Statistics::Linux>.

=head1 SOCKET STATISTICS

Generated by F</proc/net/sockstat>.

    used    -  Total number of used sockets.
    tcp     -  Number of tcp sockets in use.
    udp     -  Number of udp sockets in use.
    raw     -  Number of raw sockets in use.
    ipfrag  -  Number of ip fragments in use (only available by kernels > 2.2).

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

    my $lxs = Sys::Statistics::Linux::SockStats->new;

It's possible to set the path to the proc filesystem.

     Sys::Statistics::Linux::SockStats->new(
        files => {
            # This is the default
            path => '/proc',
            sockstat => 'net/sockstat',
        }
    );

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

    my $stat = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::SockStats;

use strict;
use warnings;
use Carp qw(croak);

our $VERSION = '0.08';

sub new {
    my ($class, %opts) = @_;

    my %self = (
        files => {
            path => '/proc',
            sockstat => 'net/sockstat',
        }
    );

    foreach my $file (keys %{ $opts{files} }) {
        $self{files}{$file} = $opts{files}{$file};
    }

    return bless \%self, $class;
}

sub get {
    my $self = shift;
    my $class = ref $self;
    my $file  = $self->{files};
    my %socks = ();

    my $filename = $file->{path} ? "$file->{path}/$file->{sockstat}" : $file->{sockstat};
    open my $fh, '<', $filename or croak "$class: unable to open $filename ($!)";

    while (my $line = <$fh>) {
        if ($line =~ /sockets: used (\d+)/) {
            $socks{used} = $1;
        } elsif ($line =~ /TCP: inuse (\d+)/) {
            $socks{tcp} = $1;
        } elsif ($line =~ /UDP: inuse (\d+)/) {
            $socks{udp} = $1;
        } elsif ($line =~ /RAW: inuse (\d+)/) {
            $socks{raw} = $1;
        } elsif ($line =~ /FRAG: inuse (\d+)/) {
            $socks{ipfrag} = $1;
        }
    }

    close($fh);
    return \%socks;
}

1;
