use 5.14.0;
use strict;
use warnings;

package OpenGbg::Service::StyrOchStall::GetBikeStation;

our $VERSION = '0.1301'; # VERSION
# ABSTRACT: Get data about a bike station

use OpenGbg::Types -types;
use Kavorka;
use XML::Rabbit::Root;
use DateTime::Format::HTTP;
use MooseX::AttributeShortcuts;

has xml => (
    is => 'ro',
    isa => Str,
    required => 1,
);

add_xpath_namespace 'x' => 'TK.DevServer.Services.RentalBikeService';

has_xpath_value _timestamp => '/x:RentalBikes/x:TimeStamp';

has_xpath_object stations => '/x:RentalBikes/x:Stations' => 'OpenGbg::Service::StyrOchStall::Stations';

has timestamp => (
    is => 'ro',
    isa => DateTime,
    lazy => 1,
    builder => 1,
);

method _build_timestamp {
    return DateTime::Format::HTTP->parse_datetime($self->_timestamp);
}

method station {
    return $self->stations->get_by_index(0);
}

finalize_class();

1;

__END__

=pod

=encoding utf-8

=head1 NAME

OpenGbg::Service::StyrOchStall::GetBikeStation - Get data about a bike station

=head1 VERSION

Version 0.1301, released 2015-01-16.

=head1 SYNOPSIS

    my $service = OpenGbg->new->styr_och_stall;
    my $response = $service->get_bike_station(1);

    printf 'Time: %s', $response->timestamp;
    print $response->station->to_text;

=head1 METHODS

=head2 timestamp

Returns the timestamp given in the response as a L<DateTime> object.

=head2 station

Returns the L<OpenGbg::Service::StyrOchStall::Station> object given in the response.

=head1 SOURCE

L<https://github.com/Csson/p5-OpenGbg>

=head1 HOMEPAGE

L<https://metacpan.org/release/OpenGbg>

=head1 AUTHOR

Erik Carlsson <info@code301.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Erik Carlsson.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
