#----------------------------------------------------------------------
package DBIx::DataModel::ConnectedSource;
#----------------------------------------------------------------------
# see POD doc at end of file

use warnings;
use strict;
use DBIx::DataModel::Meta::Utils qw/define_readonly_accessors/;
use Carp::Clan                   qw[^(DBIx::DataModel::|SQL::Abstract)];
use namespace::clean;

our $AUTOLOAD;


sub new {
  my ($class, $meta_source, $schema) = @_;
  my $self = bless {meta_source => $meta_source, schema => $schema}, $class;
}


# accessors
define_readonly_accessors(__PACKAGE__, qw/meta_source schema/);

# additional accessor; here, 'metadm' is a synonym for 'meta_source'
sub metadm {
  my $self = shift;
  return $self->{meta_source};
}


sub DESTROY {} # to prevent being handled by AUTOLOAD

sub AUTOLOAD { # need AUTOLOAD because each source may have path methods
               # that are not known statically
  my $self   = shift;

  no strict 'refs';
  my $method = $AUTOLOAD;
  $method =~ s/^.*:://;

  # create a fake instance of the source classe, containing the schema
  my $obj = bless {__schema => $self->{schema}}, $self->{meta_source}->class;

  # call that instance with all remaining args
  $obj->$method(@_);
}



1;


__END__

=encoding ISO8859-1

=head1 NAME

DBIx::DataModel::ConnectedSource - metasource and schema paired together

=head1 DESCRIPTION

A I<connected source> is a pair of a C<$meta_source> and  and a C<$schema>.
The meta_source holds information about the data structure, and the schema
holds a connection to the database.

Connected sources are used mainly in multi-schema mode, for implementing
calls such as 

  $schema->table($table_name)->select(...);
  $schema->join(qw/Table path1 path2 .../)->select(...);


=head1 METHODS

Methods are documented in 
L<DBIx::DataModel::Doc::Reference/"CONNECTED SOURCES">
Most methods are just delegated calls to C<< $meta_source->class >>,
passing along a reference to C<$schema> through a fake object

=head2 Constructor

=head3 new

  my $connected_source 
    = DBIx::DataModel::ConnectedSource->new($meta_source, $schema);


=head2 Accessors

=head3 meta_source

=head3 schema

=head3 metadm


=head2 Data retrieval

=head3 select

=head3 fetch

=head3 fetch_cached

=head3 join


=head2 Data manipulation

=head3 insert

=head3 update

=head3 delete


=cut


