#include <X11/Xlib.h>


#include "tkgsXlib.h"
#include "tkgsXlibInt.h"

/* FIXME: include our own? We need this for strcasecmp */
#include <tkPort.h>

/*
 * Global data.
 */

static Tcl_ThreadDataKey dataKey;

typedef struct ThreadSpecificData {
    Tcl_HashTable fontFamilyTable;
				/* The list of font families that are 
				 * currently loaded.  As screen fonts
				 * are loaded, this list grows to hold 
				 * information about what characters
				 * exist in each font family. */
    XlibFontFamily controlFamily;   
				/* Font family used to handle control 
				 * character expansions.  The encoding
				 * of this font family converts UTF-8 to 
				 * backslashed escape sequences. */
} ThreadSpecificData;

/*
 * Information cached about the system at startup time.
 */

Tcl_Encoding defaultEncoding, unicodeEncoding;

/*
 * The preferred font encodings.
 */

static CONST char *encodingList[] = {
    "ucs-2be", "iso8859-1", "jis0208", "jis0212", NULL
};

/*
 * The following structure and definition is used to keep track of the
 * alternative names for various encodings.  Asking for an encoding that
 * matches one of the alias patterns will result in actually getting the
 * encoding by its real name.
 */
 
typedef struct EncodingAlias {
    char *realName;		/* The real name of the encoding to load if
				 * the provided name matched the pattern. */
    char *aliasPattern;		/* Pattern for encoding name, of the form
				 * that is acceptable to Tcl_StringMatch. */
} EncodingAlias;

/*
 * The set of builtin encoding alises to convert the XLFD names for the
 * encodings into the names expected by the Tcl encoding package.
 */

static EncodingAlias encodingAliases[] = {
    {"gb2312",		"gb2312*"},
    {"big5",		"big5*"},
    {"cns11643-1",	"cns11643*-1"},
    {"cns11643-1",	"cns11643*.1-0"},
    {"cns11643-2",	"cns11643*-2"},
    {"cns11643-2",	"cns11643*.2-0"},
    {"jis0201",		"jisx0201*"},
    {"jis0201",		"jisx0202*"},
    {"jis0208",		"jisc6226*"},
    {"jis0208",		"jisx0208*"},
    {"jis0212",		"jisx0212*"},
    {"tis620",		"tis620*"},
    {"ksc5601",		"ksc5601*"},
    {"dingbats",	"*dingbats"},
    {"ucs-2be",		"iso10646-1"},
    {NULL,		NULL}
};

/*
 * The following defines specify the meaning of the fields in a fully
 * qualified XLFD.
 */

#define XLFD_FOUNDRY		0
#define XLFD_FAMILY_NAME	1
#define XLFD_WEIGHT_NAME	2
#define XLFD_SLANT		3
#define XLFD_SETWIDTH_NAME	4
#define XLFD_ADD_STYLE_NAME	5
#define XLFD_PIXEL_SIZE		6
#define XLFD_POINT_SIZE		7
#define XLFD_RESOLUTION_X	8
#define XLFD_RESOLUTION_Y	9
#define XLFD_SPACING		10
#define XLFD_AVERAGE_WIDTH	11
#define XLFD_CHARSET		12

#define XLFD_NUMFIELDS		13 /* Number of fields in XLFD. */

/*
 * XLFD parsing structure.
 */

typedef struct XLFDFields {
    char buf[256];		/* FIXME: maximum size of XLFD. */
    CONST char *fields[XLFD_NUMFIELDS];
} XLFDFields;

typedef struct StateMap {
    unsigned int numKey;	/* Integer representation of a value. */
    char *strKey;		/* String representation of a value. */
} StateMap;

static StateMap xlfdWeightMap[] = {
    {TKGS_FW_NORMAL,	"normal"},
    {TKGS_FW_NORMAL,	"medium"},
    {TKGS_FW_NORMAL,	"book"},
    {TKGS_FW_NORMAL,	"light"},
    {TKGS_FW_BOLD,	"bold"},
    {TKGS_FW_BOLD,	"demi"},
    {TKGS_FW_BOLD,	"demibold"},
    {TKGS_FW_NORMAL,	NULL}		/* Assume anything else is "normal". */
}; 

static StateMap xlfdSlantMap[] = {
    {TKGS_FS_ROMAN,	"r"},
    {TKGS_FS_ITALIC,	"i"},
    {TKGS_FS_OBLIQUE,	"o"},
    {TKGS_FS_ROMAN,	NULL}		/* Assume anything else is "roman". */
};


/*
 * Prototypes for procedures defined later in this file:
 */

static TkGS_FreeInternalRepProc		FreeXlibFontIntRep;
static TkGS_SetFromAnyProc		SetXlibFontFromAny;

static TkGSFreeSubFontProc		FreeXlibSubFont;
static TkGSFreeFontFamilyProc		FreeXlibFontFamily;


static TkGSFontFamily   AllocFontFamily _ANSI_ARGS_((Display *display,
			    XFontStruct *fontStructPtr));
static int		ControlUtfProc _ANSI_ARGS_((ClientData clientData,
			    CONST char *src, int srcLen, int flags,
			    Tcl_EncodingState *statePtr, char *dst,
			    int dstLen, int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
static XFontStruct *	CreateClosestFont _ANSI_ARGS_((TkGS_Drawable d,
			    CONST TkGS_FontAttributes *faPtr,
			    Tcl_Encoding encoding));
static TkGSSubFont	GetDrawableSubFont _ANSI_ARGS_((TkGS_Drawable d,
			    TkGS_InternalRep *intRepPtr));
static CONST char *	GetEncodingAlias _ANSI_ARGS_((CONST char *name));
static XLFDFields *	GetFontAttributes _ANSI_ARGS_((Display *display,
			    XFontStruct *fontStructPtr,
			    XLFDFields *xlfdPtr));
static XFontStruct *	GetScreenFont _ANSI_ARGS_((Display *display,
			    CONST TkGS_FontAttributes *wantPtr,
			    char **nameList, int bestIdx[], 
			    unsigned int bestScore[]));
static XFontStruct *	GetSystemFont _ANSI_ARGS_((Display *display));
static unsigned int	FindStateNum _ANSI_ARGS_((CONST StateMap * mapPtr, 
			    CONST char * strKey));
static char *		FindStateString _ANSI_ARGS_((
			    CONST StateMap * mapPtr, unsigned int numKey));
static void		InitFont _ANSI_ARGS_((TkGS_Drawable d, 
			    TkGS_Font font, XlibFont *xlibFontPtr, 
			    Display *display, XFontStruct *fontStructPtr));
static void		InitSubFont _ANSI_ARGS_((TkGS_Drawable d, 
			    TkGS_Font font, XlibSubFont *xlibSubFontPtr, 
			    Display *display, XFontStruct *fontStructPtr));
static int		IdentifySymbolEncodings _ANSI_ARGS_((
			    XLFDFields *xlfdPtr));
static char **		ListFonts _ANSI_ARGS_((Display *display,
			    CONST char *faceName, int *numNamesPtr));
static char **		ListFontOrAlias _ANSI_ARGS_((Display *display,
			    CONST char *faceName, int *numNamesPtr));
static XLFDFields *	ParseXLFD _ANSI_ARGS_((CONST char *string,
			    XLFDFields *xlfdPtr));
static unsigned int	RankAttributes _ANSI_ARGS_((
			    CONST TkGS_FontAttributes *wantPtr,
			    CONST char *wantCharset, XLFDFields *gotPtr));
static int		Ucs2beToUtfProc _ANSI_ARGS_((ClientData clientData,
			    CONST char *src, int srcLen, int flags,
			    Tcl_EncodingState *statePtr, char *dst, int dstLen,
			    int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
static int		UtfToUcs2beProc _ANSI_ARGS_((ClientData clientData,
			    CONST char *src, int srcLen, int flags,
			    Tcl_EncodingState *statePtr, char *dst, int dstLen,
			    int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));



static XLFDFields *
ParseXLFD(string, xlfdPtr) 
    CONST char *string;
    XLFDFields *xlfdPtr;
{
    char *b;
    CONST char *s;
    int f;

    /*
     * Quick check.
     */

    if (string[0] != '-') {
	return NULL;
    }

    /*
     * Copy and split string into fields.
     */

    for (f=0, b=xlfdPtr->buf, s=string+1; f!=XLFD_NUMFIELDS; f++) {
	xlfdPtr->fields[f] = b;
	for (;;) {
	    if (*s == '\0' || (*s == '-' && f != XLFD_CHARSET)) {
		*b++ = '\0';
		break;
	    }
	    *b++ = *s++;
	}
	if (!*s++) break;
    }

    /*
     * Check nb fields.
     */

    if (f!= XLFD_NUMFIELDS-1) {
	return NULL;
    }

    return xlfdPtr;
}

static int
IsXLFDScalable(xlfdPtr)
    XLFDFields *xlfdPtr;
{
    if (xlfdPtr && strcmp(xlfdPtr->fields[XLFD_PIXEL_SIZE], "0") == 0) {
	return 1;
    } else {
	return 0;
    }
}

#define GetXLFDFoundry(xlfdPtr) \
    ((xlfdPtr) ? (xlfdPtr)->fields[XLFD_FOUNDRY] : NULL)
#define GetXLFDFamily(xlfdPtr) \
    ((xlfdPtr) ? (xlfdPtr)->fields[XLFD_FAMILY_NAME] : NULL)
#define GetXLFDWeight(xlfdPtr) \
    ((xlfdPtr) ? (FindStateNum(xlfdWeightMap, (xlfdPtr)->fields[XLFD_WEIGHT_NAME])) : 0)
#define GetXLFDSlant(xlfdPtr) \
    ((xlfdPtr) ? (FindStateNum(xlfdSlantMap, (xlfdPtr)->fields[XLFD_SLANT])) : 0)

/* TODO: better pixel <-> point conversion */
#define GetXLFDPixelSize(xlfdPtr) \
    ((xlfdPtr) ? (atoi((xlfdPtr)->fields[XLFD_PIXEL_SIZE])) : 12)
#define GetXLFDPointSize(xlfdPtr) \
    ((xlfdPtr) ? (atoi((xlfdPtr)->fields[XLFD_POINT_SIZE])/10) : 12)

#define GetXLFDCharset(xlfdPtr) \
    ((xlfdPtr) ? (xlfdPtr)->fields[XLFD_CHARSET] : NULL)

/*
 * Xlib Font Obj type
 */

/* Object type */
TkGS_ObjType XlibFontType = {
    "Xlib",
    NULL,				/* Base type, initialized at runtime */

    FreeXlibFontIntRep,			/* freeIntRepProc */
    SetXlibFontFromAny			/* setFromAnyProc */
};
TkGS_ObjType *XlibFontTypePtr = &XlibFontType;


/* Object type procs */

static void
FreeXlibFontFamily(fontFamily)
    TkGSFontFamily fontFamily;
{
    register XlibFontFamily *familyPtr = (XlibFontFamily *) fontFamily;

    /* 
     * Delete from table. 
     */

    Tcl_DeleteHashEntry(familyPtr->hashPtr);
}

static void
FreeXlibSubFont(font, subFont)
    TkGSMultiFont font;
    TkGSSubFont subFont;
{
    TkGSFreeSubFont(subFont, FreeXlibFontFamily);

    XFreeFont(((XlibFont *) font)->display, XlibSubFont_XFontStruct(subFont));
}

static void
FreeXlibFontIntRep(intRepPtr)
    TkGS_InternalRep *intRepPtr;
{
    TkGSFreeMultiFont((TkGSMultiFont) XlibFont_InternalRep(intRepPtr), 
	sizeof(XlibSubFont), FreeXlibSubFont);

    /* Free the allocated memory */
    ckfree((char *) XlibFont_InternalRep(intRepPtr));
}

static int
SetXlibFontFromAny(interp, objPtr, intRepPtr)
    Tcl_Interp *interp;
    TkGS_Obj *objPtr;
    TkGS_InternalRep *intRepPtr;
{
    register XlibFont *xlibFontPtr;

    /* 
     * Free old internal rep
     */

    if (   intRepPtr->typePtr != NULL
	&& intRepPtr->typePtr->freeIntRepProc != NULL) {
	intRepPtr->typePtr->freeIntRepProc(intRepPtr);
    }

    /* 
     * Initialize internal rep.
     */

    xlibFontPtr = (XlibFont *) ckalloc(sizeof(XlibFont));
    memset(xlibFontPtr, 0, sizeof(XlibFont));
    XlibFont_InternalRep(intRepPtr) = xlibFontPtr;

    /* Generic multi-font initialization. */
    TkGSInitMultiFont((TkGSMultiFont) xlibFontPtr);

    /* XFontStruct depends on display. */
    XlibFont_Display(intRepPtr) = None;
    XlibFont_XFontStruct(intRepPtr) = NULL;

    /* Change type */
    intRepPtr->typePtr = XlibFontTypePtr;

    return TKGS_OK;
}

XlibFont *
XlibTkGS_GetXlibFontFromFont(interp, d, font)
    Tcl_Interp *interp;
    TkGS_Drawable d;
    TkGS_Font font;
{
    DECLAREDRAWINGVARS(d, dIntRepPtr, display, xd, xgc)
    register TkGS_InternalRep 
	*intRepPtr = TkGS_FindInternalRep((TkGS_Obj *) font, &XlibFontType);
    XFontStruct *fontStructPtr;
    TkGS_FontAttributes *faPtr;

    /*
     * Get a XlibFont from TkGS Font's internal rep
     */

    if (intRepPtr == NULL) {
	intRepPtr = TkGS_AddNewInternalRep(interp, (TkGS_Obj *) font, 
					   &XlibFontType);
	if (!intRepPtr) {
	    return NULL;
	}
    }

    if (XlibFont_Display(intRepPtr) != display) {

	/* 
	 * Fonts are display-dependent.
	 */

	TkGSFreeMultiFont((TkGSMultiFont) XlibFont_InternalRep(intRepPtr), 
	    sizeof(XlibSubFont), FreeXlibSubFont);
	XlibFont_XFontStruct(intRepPtr) = NULL;
    }

    if (XlibFont_XFontStruct(intRepPtr) == NULL) {

	/*
	 * Get the closest font to the name requested.
	 */

	faPtr = &TkGSFont_Attributes(font);
	fontStructPtr = CreateClosestFont(d, faPtr, NULL);
	InitFont(d, font, XlibFont_InternalRep(intRepPtr), display, fontStructPtr);

	XlibFont_Display(intRepPtr) = display;
	XlibFont_XFontStruct(intRepPtr) = fontStructPtr;
    }

    return XlibFont_InternalRep(intRepPtr);
}

/*
 *---------------------------------------------------------------------------
 *
 * GetScreenFont --
 *
 *	Given the names for the best scalable and best bitmapped font,
 *	actually construct an XFontStruct based on the best XLFD.
 *	This is where all the alias and fallback substitution bottoms
 *	out.
 *
 * Results:
 *	The screen font that best corresponds to the set of attributes.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static XFontStruct *
GetScreenFont(display, wantPtr, nameList, bestIdx, bestScore)
    Display *display;		/* Display for new XFontStruct. */
    CONST TkGS_FontAttributes *wantPtr;
				/* Contains desired actual pixel-size if the
				 * best font was scalable. */
    char **nameList;		/* Array of XLFDs. */
    int bestIdx[2];		/* Indices into above array for XLFD of
				 * best bitmapped and best scalable font. */
    unsigned int bestScore[2];	/* Scores of best bitmapped and best
				 * scalable font.  XLFD corresponding to
				 * lowest score will be constructed. */
{
    XFontStruct *fontStructPtr;

    if ((bestIdx[0] < 0) && (bestIdx[1] < 0)) {
	return NULL;
    }

    /*
     * Now we know which is the closest matching scalable font and the
     * closest matching bitmapped font.  If the scalable font was a
     * better match, try getting the scalable font; however, if the
     * scalable font was not actually available in the desired
     * pointsize, fall back to the closest bitmapped font.
     */

    fontStructPtr = NULL;

    if (bestScore[1] < bestScore[0]) {
	char buf[256];
	char *ptSize, *pxSize;
	char size[16];
	XLFDFields xlfd;
	
	/*
	 * Fill in the desired pixel size for this font.
	 */

	tryscale:

	/* TODO: better pixel <-> point conversion */
	if (wantPtr->size < 0) {
	    /* Pixel size */
	    sprintf(size, "%d", -wantPtr->size);
	    pxSize = size;
	    ptSize = "*";
	} else {
	    /* Point size */
	    sprintf(size, "%d", wantPtr->size*10);
	    pxSize = "*";
	    ptSize = size;
	}
	ParseXLFD(nameList[bestIdx[1]], &xlfd);
	sprintf(buf, "-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s", 
		    xlfd.fields[XLFD_FOUNDRY],
		    xlfd.fields[XLFD_FAMILY_NAME],
		    xlfd.fields[XLFD_WEIGHT_NAME],
		    xlfd.fields[XLFD_SLANT],
		    xlfd.fields[XLFD_SETWIDTH_NAME],
		    xlfd.fields[XLFD_ADD_STYLE_NAME],
		    pxSize,
		    ptSize,
		    xlfd.fields[XLFD_RESOLUTION_X],
		    xlfd.fields[XLFD_RESOLUTION_Y],
		    xlfd.fields[XLFD_SPACING],
		    xlfd.fields[XLFD_AVERAGE_WIDTH],
		    xlfd.fields[XLFD_CHARSET]);

	fontStructPtr = XLoadQueryFont(display, buf);
	bestScore[1] = INT_MAX;
    }
    if (fontStructPtr == NULL) {
	fontStructPtr = XLoadQueryFont(display, nameList[bestIdx[0]]);
	if (fontStructPtr == NULL) {
	    /*
	     * This shouldn't happen because the font name is one of the
	     * names that X gave us to use, but it does anyhow.
	     */

	    if (bestScore[1] < INT_MAX) {
		goto tryscale;
	    }
	    return GetSystemFont(display);
	}
    }
    return fontStructPtr;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetSystemFont --
 *
 *	Absolute fallback mechanism, called when we need a font and no
 *	other font can be found and/or instantiated.
 *
 * Results:
 *	A pointer to a font.  Never NULL.
 *
 * Side effects:
 *	If there are NO fonts installed on the system, this call will
 *	panic, but how did you get X running in that case?
 *
 *---------------------------------------------------------------------------
 */

static XFontStruct *
GetSystemFont(display)
    Display *display;		/* Display for new XFontStruct. */
{
    XFontStruct *fontStructPtr;

    fontStructPtr = XLoadQueryFont(display, "fixed");
    if (fontStructPtr == NULL) {
	fontStructPtr = XLoadQueryFont(display, "*");
	if (fontStructPtr == NULL) {
	    panic("TkpGetFontFromAttributes: cannot get any font");
	}
    }
    return fontStructPtr;
}

/*
 *-------------------------------------------------------------------------
 *
 * CreateClosestFont --
 *
 *	Given a set of font attributes, construct a close XFontStruct.
 *	If requested face name is not available, automatically
 *	substitutes an alias for requested face name.  If encoding is
 *	not specified (or the requested one is not available),
 *	automatically chooses another encoding from the list of
 *	preferred encodings.  If the foundry is not specified (or
 *	is not available) automatically prefers "adobe" foundry.
 *	For all other attributes, if the requested value was not
 *	available, the appropriate "close" value will be used.
 *
 * Results:
 *	Return value is the XFontStruct that best matched the
 *	requested attributes.  The return value is never NULL; some
 *	font will always be returned.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static XFontStruct *
CreateClosestFont(d, faPtr, encoding)
    TkGS_Drawable d;
    CONST TkGS_FontAttributes *faPtr;
				/* Set of generic attributes to match. */
    Tcl_Encoding encoding;	/* Desired charset encoding. */
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    char *faceName, *actualName;
    char **nameList;
    int numNames, nameIdx;
    XFontStruct *fontStructPtr;
    int bestIdx[2];
    unsigned int bestScore[2];
    char *wantCharset;

    /*
     * Get the closest font to the name requested.
     */

    if (encoding == NULL) {
	encoding = defaultEncoding;
    }
    wantCharset = Tcl_GetEncodingName(encoding);

    /* 
     * TODO: manage multiple font family specifiers. For now only take 
     * the first.
     */

    faceName = TkGSFontFaces_Names(faPtr->faces)[0];
    actualName = TkGSGetClosestFaceName(d, faceName);
    if (actualName != NULL) {
	faceName = actualName;
    }

    nameList = ListFontOrAlias(display, faceName, &numNames);
    if (numNames == 0) {
	nameList = ListFonts(display, "fixed", &numNames);
	if (numNames == 0) {
	    nameList = ListFonts(display, "*", &numNames);
	}
    }
    if (numNames == 0) {
	return GetSystemFont(display);
    }

    bestIdx[0] = -1;
    bestIdx[1] = -1;
    bestScore[0] = (unsigned int) -1;
    bestScore[1] = (unsigned int) -1;
    for (nameIdx = 0; nameIdx < numNames; nameIdx++) {
	XLFDFields xlfd;
	int scalable;
	unsigned int score;
	
	if (!ParseXLFD(nameList[nameIdx], &xlfd)) {
	    continue;
	}
	IdentifySymbolEncodings(&xlfd);
	scalable = IsXLFDScalable(&xlfd);
	score = RankAttributes(faPtr, wantCharset, &xlfd);
	if (score <= bestScore[scalable]) {
	    bestIdx[scalable] = nameIdx;
	    bestScore[scalable] = score;
	}
	if (score == 0) {
	    break;
	}
    }

    fontStructPtr = GetScreenFont(display, faPtr, nameList, bestIdx, bestScore);
    XFreeFontNames(nameList);
    
    if (fontStructPtr == NULL) {
	return GetSystemFont(display);
    }
    return fontStructPtr;
}

/*
 *-------------------------------------------------------------------------
 *
 * ControlUtfProc --
 *
 *	Convert from UTF-8 into the ASCII expansion of a control
 *	character.
 *
 * Results:
 *	Returns TCL_OK if conversion was successful.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static int 
ControlUtfProc(clientData, src, srcLen, flags, statePtr, dst, dstLen,
	srcReadPtr, dstWrotePtr, dstCharsPtr)
    ClientData clientData;	/* Not used. */
    CONST char *src;		/* Source string in UTF-8. */
    int srcLen;			/* Source string length in bytes. */
    int flags;			/* Conversion control flags. */
    Tcl_EncodingState *statePtr;/* Place for conversion routine to store
				 * state information used during a piecewise
				 * conversion.  Contents of statePtr are
				 * initialized and/or reset by conversion
				 * routine under control of flags argument. */
    char *dst;			/* Output buffer in which converted string
				 * is stored. */
    int dstLen;			/* The maximum length of output buffer in
				 * bytes. */
    int *srcReadPtr;		/* Filled with the number of bytes from the
				 * source string that were converted.  This
				 * may be less than the original source length
				 * if there was a problem converting some
				 * source characters. */
    int *dstWrotePtr;		/* Filled with the number of bytes that were
				 * stored in the output buffer as a result of
				 * the conversion. */
    int *dstCharsPtr;		/* Filled with the number of characters that
				 * correspond to the bytes stored in the
				 * output buffer. */
{
    CONST char *srcEnd;
    char *dstStart, *dstEnd;
    Tcl_UniChar ch;
    int result;
    static char hexChars[] = "0123456789abcdef";
    static char mapChars[] = {
	0, 0, 0, 0, 0, 0, 0,
	'a', 'b', 't', 'n', 'v', 'f', 'r'
    };

    result = TCL_OK;    

    srcEnd = src + srcLen;

    dstStart = dst;
    dstEnd = dst + dstLen - 6;

    for ( ; src < srcEnd; ) {
	if (dst > dstEnd) {
	    result = TCL_CONVERT_NOSPACE;
	    break;
	}
	src += Tcl_UtfToUniChar(src, &ch);
	dst[0] = '\\';
	if ((ch < sizeof(mapChars)) && (mapChars[ch] != 0)) {
	    dst[1] = mapChars[ch];
	    dst += 2;
	} else if (ch < 256) {
	    dst[1] = 'x';
	    dst[2] = hexChars[(ch >> 4) & 0xf];
	    dst[3] = hexChars[ch & 0xf];
	    dst += 4;
	} else {
	    dst[1] = 'u';
	    dst[2] = hexChars[(ch >> 12) & 0xf];
	    dst[3] = hexChars[(ch >> 8) & 0xf];
	    dst[4] = hexChars[(ch >> 4) & 0xf];
	    dst[5] = hexChars[ch & 0xf];
	    dst += 6;
	}
    }
    *srcReadPtr = src - srcEnd;
    *dstWrotePtr = dst - dstStart;
    *dstCharsPtr = dst - dstStart;
    return result;
}

/*
 *-------------------------------------------------------------------------
 *
 * Ucs2beToUtfProc --
 *
 *	Convert from UCS-2BE (big-endian 16-bit Unicode) to UTF-8.
 *
 * Results:
 *	Returns TCL_OK if conversion was successful.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static int 
Ucs2beToUtfProc(clientData, src, srcLen, flags, statePtr, dst, dstLen,
	srcReadPtr, dstWrotePtr, dstCharsPtr)
    ClientData clientData;	/* Not used. */
    CONST char *src;		/* Source string in Unicode. */
    int srcLen;			/* Source string length in bytes. */
    int flags;			/* Conversion control flags. */
    Tcl_EncodingState *statePtr;/* Place for conversion routine to store
				 * state information used during a piecewise
				 * conversion.  Contents of statePtr are
				 * initialized and/or reset by conversion
				 * routine under control of flags argument. */
    char *dst;			/* Output buffer in which converted string
				 * is stored. */
    int dstLen;			/* The maximum length of output buffer in
				 * bytes. */
    int *srcReadPtr;		/* Filled with the number of bytes from the
				 * source string that were converted.  This
				 * may be less than the original source length
				 * if there was a problem converting some
				 * source characters. */
    int *dstWrotePtr;		/* Filled with the number of bytes that were
				 * stored in the output buffer as a result of
				 * the conversion. */
    int *dstCharsPtr;		/* Filled with the number of characters that
				 * correspond to the bytes stored in the
				 * output buffer. */
{
    CONST Tcl_UniChar *wSrc, *wSrcStart, *wSrcEnd;
    char *dstEnd, *dstStart;
    int result, numChars;
    
    result = TCL_OK;
    if ((srcLen % sizeof(Tcl_UniChar)) != 0) {
	result = TCL_CONVERT_MULTIBYTE;
	srcLen /= sizeof(Tcl_UniChar);
	srcLen *= sizeof(Tcl_UniChar);
    }

    wSrc = (Tcl_UniChar *) src;

    wSrcStart = (Tcl_UniChar *) src;
    wSrcEnd = (Tcl_UniChar *) (src + srcLen);

    dstStart = dst;
    dstEnd = dst + dstLen - TCL_UTF_MAX;

    for (numChars = 0; wSrc < wSrcEnd; numChars++) {
	if (dst > dstEnd) {
	    result = TCL_CONVERT_NOSPACE;
	    break;
	}
	/* 
	 * On a little-endian machine (Intel) the UCS-2BE is in the
	 * wrong byte-order in comparison to "unicode", which is
	 * in native host order.
	 */
	dst += Tcl_UniCharToUtf(htons(*wSrc), dst);
	wSrc++;
    }

    *srcReadPtr = (char *) wSrc - (char *) wSrcStart;
    *dstWrotePtr = dst - dstStart;
    *dstCharsPtr = numChars;
    return result;
}

/*
 *-------------------------------------------------------------------------
 *
 * UtfToUcs2beProc --
 *
 *	Convert from UTF-8 to UCS-2BE.
 *
 * Results:
 *	Returns TCL_OK if conversion was successful.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static int 
UtfToUcs2beProc(clientData, src, srcLen, flags, statePtr, dst, dstLen,
	srcReadPtr, dstWrotePtr, dstCharsPtr)
    ClientData clientData;	/* TableEncodingData that specifies encoding. */
    CONST char *src;		/* Source string in UTF-8. */
    int srcLen;			/* Source string length in bytes. */
    int flags;			/* Conversion control flags. */
    Tcl_EncodingState *statePtr;/* Place for conversion routine to store
				 * state information used during a piecewise
				 * conversion.  Contents of statePtr are
				 * initialized and/or reset by conversion
				 * routine under control of flags argument. */
    char *dst;			/* Output buffer in which converted string
				 * is stored. */
    int dstLen;			/* The maximum length of output buffer in
				 * bytes. */
    int *srcReadPtr;		/* Filled with the number of bytes from the
				 * source string that were converted.  This
				 * may be less than the original source length
				 * if there was a problem converting some
				 * source characters. */
    int *dstWrotePtr;		/* Filled with the number of bytes that were
				 * stored in the output buffer as a result of
				 * the conversion. */
    int *dstCharsPtr;		/* Filled with the number of characters that
				 * correspond to the bytes stored in the
				 * output buffer. */
{
    CONST char *srcStart, *srcEnd, *srcClose;
    Tcl_UniChar *wDst, *wDstStart, *wDstEnd;
    int result, numChars;
    
    srcStart = src;
    srcEnd = src + srcLen;
    srcClose = srcEnd;
    if ((flags & TCL_ENCODING_END) == 0) {
	srcClose -= TCL_UTF_MAX;
    }

    wDst = (Tcl_UniChar *) dst;
    wDstStart = (Tcl_UniChar *) dst;
    wDstEnd = (Tcl_UniChar *) (dst + dstLen - sizeof(Tcl_UniChar));

    result = TCL_OK;
    for (numChars = 0; src < srcEnd; numChars++) {
	if ((src > srcClose) && (!Tcl_UtfCharComplete(src, srcEnd - src))) {
	    /*
	     * If there is more string to follow, this will ensure that the
	     * last UTF-8 character in the source buffer hasn't been cut off.
	     */

	    result = TCL_CONVERT_MULTIBYTE;
	    break;
	}
	if (wDst > wDstEnd) {
	    result = TCL_CONVERT_NOSPACE;
	    break;
        }
	src += Tcl_UtfToUniChar(src, wDst);
	/*
	 * Byte swap for little-endian machines.
	 */
	*wDst = htons(*wDst);
	wDst++;
    }
    *srcReadPtr = src - srcStart;
    *dstWrotePtr = (char *) wDst - (char *) wDstStart;
    *dstCharsPtr = numChars;
    return result;
}

/*
 *---------------------------------------------------------------------------
 *
 * RankAttributes --
 *
 *	Determine how close the attributes of the font in question match
 *	the attributes that we want.
 *
 * Results:
 *	The return value is the score; lower numbers are better.
 *	*scalablePtr is set to 0 if the font was not scalable, 1 otherwise.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static unsigned int
RankAttributes(wantPtr, wantCharset, gotPtr)
    CONST TkGS_FontAttributes *wantPtr;	/* The desired attributes. */
    CONST char *wantCharset;		/* The desired charset. */
    XLFDFields *gotPtr;			/* The attributes we have to live with. */
{
    unsigned int penalty;
    CONST char *gotCharset;

    penalty = 0;

    /* FIXME */

/* 
    if (gotPtr->xa.foundry != wantPtr->xa.foundry) {
	penalty += 4500;
    }
*/

/*
    if (gotPtr->fa.family != wantPtr->fa.family) {
	penalty += 9000;
    }
*/

/*
    if (gotPtr->fa.weight != wantPtr->fa.weight) {
	penalty += 90;
    }
    */

    if (   wantPtr->weight != GetXLFDWeight(gotPtr)) {
	penalty += 90;
    }
/*
    if (gotPtr->fa.slant != wantPtr->fa.slant) {
	penalty += 60;
    }*/
    if (   wantPtr->slant != GetXLFDSlant(gotPtr)) {
	penalty += 60;
    }
/*
    if (gotPtr->xa.slant != wantPtr->xa.slant) {
	penalty += 10;
    }
*/

/*
    if (gotPtr->xa.setwidth != wantPtr->xa.setwidth) {
	penalty += 1000;
    }
*/

/*    if (gotPtr->fa.size == 0) {*/
    if (IsXLFDScalable(gotPtr)) {
	/*
	 * A scalable font is almost always acceptable, but the
	 * corresponding bitmapped font would be better.
	 */

	penalty += 10;
    } else {
	int diff, size;

	/*
	 * It's worse to be too large than to be too small.
	 */
	 
	if (wantPtr->size < 0) {
	    /* Pixel size */
	    size = GetXLFDPixelSize(gotPtr);
	    diff = (size - (-wantPtr->size));
	} else {
	    /* Point size */
	    size = GetXLFDPointSize(gotPtr);
	    diff = (size - wantPtr->size);
	}
	if (diff > 0) {
	    penalty += 600;
	} else if (diff < 0) {
	    penalty += 150;
	    diff = -diff;
	}
	penalty += 150 * diff;
    }

/*
    if (gotPtr->xa.charset != wantPtr->xa.charset) {
	int i;
	CONST char *gotAlias, *wantAlias;

	penalty += 65000;
	gotAlias = GetEncodingAlias(gotPtr->xa.charset);
	wantAlias = GetEncodingAlias(wantPtr->xa.charset); 
	if (strcmp(gotAlias, wantAlias) != 0) {
	    penalty += 30000;
	    for (i = 0; encodingList[i] != NULL; i++) {
		if (strcmp(gotAlias, encodingList[i]) == 0) {
		    penalty -= 30000;
		    break;
		}
		penalty += 20000;
	    }
	}
    }
*/

    gotCharset = GetXLFDCharset(gotPtr);
    if (strcasecmp(gotCharset, wantCharset) != 0) {
	int i;
	CONST char *gotAlias, *wantAlias;

	penalty += 65000;
	gotAlias = GetEncodingAlias(gotCharset);
	wantAlias = GetEncodingAlias(wantCharset); 
	if (strcmp(gotAlias, wantAlias) != 0) {
	    penalty += 30000;
	    for (i = 0; encodingList[i] != NULL; i++) {
		if (strcmp(gotAlias, encodingList[i]) == 0) {
		    penalty -= 30000;
		    break;
		}
		penalty += 20000;
	    }
	}
    }

    return penalty;
}

/*
 *---------------------------------------------------------------------------
 *
 * InitFont --
 *
 *	Helper for XlibTkGS_GetWinFontFromFont().
 *	Initializes the memory for a new font.
 *
 * Results:
 *	Fills the font structure.
 *
 * Side effects:
 *	Memory allocated.
 *
 *---------------------------------------------------------------------------
 */ 

static void
InitFont(d, font, xlibFontPtr, display, fontStructPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    XlibFont *xlibFontPtr;
    Display *display;
    XFontStruct *fontStructPtr;
{
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    unsigned long value;
    int minHi, maxHi, minLo, maxLo, fixed, width, limit, i, n;
    XLFDFields xlfd, *xlfdPtr;
    TkGS_FontAttributes *faPtr;
    TkGS_FontMetrics *fmPtr;
    XlibSubFont *controlPtr, *subFontPtr;
    TkGSSubFontId subFontId;
    char *pageMap;

    /*
     * Initialize base font (first subfont).
     */

    xlibFontPtr->header.numSubFonts++;
    xlibFontPtr->header.subFontArray = 
	(TkGSSubFont_ *) ckrealloc((char *) xlibFontPtr->header.subFontArray,
				   sizeof(XlibSubFont));
    subFontPtr = XlibTkGSGetSubFontFromIndex(xlibFontPtr, 0);
    InitSubFont(d, font, subFontPtr, display, fontStructPtr);

    /*
     * Initialize control subfont.
     */

    xlibFontPtr->controlSubFont	= *subFontPtr;
    subFontId = TkGSFindSubFontForChar(d, font, (TkGSMultiFont) xlibFontPtr, '0');
    subFontPtr			= XlibTkGSGetSubFontFromId(xlibFontPtr, subFontId);
    controlPtr			= &xlibFontPtr->controlSubFont;
    controlPtr->header.family	= (TkGSFontFamily) &tsdPtr->controlFamily;
    controlPtr->header.fontMap	= tsdPtr->controlFamily.header.fontMap;
    controlPtr->fontStructPtr	= subFontPtr->fontStructPtr;

    /*
     * Get all font attributes and metrics.
     */
     
    xlfdPtr = GetFontAttributes(display, fontStructPtr, &xlfd);
    
    minHi = fontStructPtr->min_byte1;
    maxHi = fontStructPtr->max_byte1;
    minLo = fontStructPtr->min_char_or_byte2;
    maxLo = fontStructPtr->max_char_or_byte2;
	
    fixed = 1;
    if (fontStructPtr->per_char != NULL) {
	width = 0;
	limit = (maxHi - minHi + 1) * (maxLo - minLo + 1);
	for (i = 0; i < limit; i++) {
	    n = fontStructPtr->per_char[i].width;
	    if (n != 0) {
		if (width == 0) {
		    width = n;
		} else if (width != n) {
		    fixed = 0;
		    break;
		}
	    }
	}
    }

    xlibFontPtr->display	= display;
    xlibFontPtr->fontStructPtr	= fontStructPtr;

    /*
     * Compute font attributes.
     */

    faPtr		= &xlibFontPtr->fa;
    faPtr->faces	= NULL;
    faPtr->size		= GetXLFDPointSize(xlfdPtr);
    faPtr->weight	= GetXLFDWeight(xlfdPtr);
    faPtr->slant	= GetXLFDSlant(xlfdPtr);
    faPtr->underline	= font->attributes.underline;
    faPtr->overstrike	= font->attributes.overstrike;
    
    /*
     * Compute font metrics.
     */

    fmPtr		= &xlibFontPtr->fm;
    fmPtr->ascent	= fontStructPtr->ascent;
    fmPtr->descent	= fontStructPtr->descent;
    fmPtr->maxWidth	= fontStructPtr->max_bounds.width;
    fmPtr->fixed	= fixed;

    /*
     * Measure chars.
     */

    subFontPtr = XlibTkGSGetSubFontFromIndex(xlibFontPtr, 0);
    pageMap = subFontPtr->header.fontMap[0];
    for (i = 0; i < XLIB_BASE_CHARS; i++) {
	if ((minHi > 0) || (i < minLo) || (i > maxLo) ||
		(((pageMap[i >> 3] >> (i & 7)) & 1) == 0)) {
	    n = 0;
	} else if (fontStructPtr->per_char == NULL) {
	    n = fontStructPtr->max_bounds.width;
	} else {
	    n = fontStructPtr->per_char[i - minLo].width;
	}
	xlibFontPtr->widths[i] = n;
    }

    /* TODO: measure tabs. */
    
    /*
     * Make sure the tab width isn't zero (some fonts may not have enough
     * information to set a reasonable tab width).
     */

    if (fmPtr->tabWidth == 0) {
	fmPtr->tabWidth = 1;
    }

    /*
     * Get information used for drawing underlines in generic code on a
     * non-underlined font.
     */
    
    if (XGetFontProperty(fontStructPtr, XA_UNDERLINE_POSITION, &value)) {
	fmPtr->underlinePos = value;
    } else {
	/*
	 * If the XA_UNDERLINE_POSITION property does not exist, the X
	 * manual recommends using the following value:
	 */

	fmPtr->underlinePos = fontStructPtr->descent / 2;
    }
    fmPtr->underlineHeight = 0;
    if (XGetFontProperty(fontStructPtr, XA_UNDERLINE_THICKNESS, &value)) {
	fmPtr->underlineHeight = value;
    }
    if (fmPtr->underlineHeight == 0) {
	/*
	 * If the XA_UNDERLINE_THICKNESS property does not exist, the X
	 * manual recommends using the width of the stem on a capital
	 * letter.  I don't know of a way to get the stem width of a letter,
	 * so guess and use 1/3 the width of a capital I.
	 */

	fmPtr->underlineHeight = xlibFontPtr->widths['I'] / 3;
	if (fmPtr->underlineHeight == 0) {
	    fmPtr->underlineHeight = 1;
	}
    }
    if (fmPtr->underlinePos + fmPtr->underlineHeight > fmPtr->descent) {
	/*
	 * If this set of values would cause the bottom of the underline
	 * bar to stick below the descent of the font, jack the underline
	 * up a bit higher.
	 */

	fmPtr->underlineHeight = fmPtr->descent - fmPtr->underlinePos;
	if (fmPtr->underlineHeight == 0) {
	    fmPtr->underlinePos--;
	    fmPtr->underlineHeight = 1;
	}
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * InitSubFont --
 *
 *	Wrap a screen font and load the FontFamily that represents
 *	it.  Used to prepare a SubFont so that characters can be mapped
 *	from UTF-8 to the charset of the font.
 *
 * Results:
 *	The subFontPtr is filled with information about the font.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static void

InitSubFont(d, font, xlibSubFontPtr, display, fontStructPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    XlibSubFont *xlibSubFontPtr;
    Display *display;
    XFontStruct *fontStructPtr;
{
    /* Generic subfont initialization. */
    TkGSInitSubFont((TkGSSubFont) xlibSubFontPtr, 
	AllocFontFamily(display, fontStructPtr));

    /* Xlib-specific subfont initialization. */
    xlibSubFontPtr->fontStructPtr = fontStructPtr;
}

/*
 *-------------------------------------------------------------------------
 *
 * AllocFontFamily --
 *
 *	Find the font family structure associated with the given font
 *	name.  The information should be stored by the caller in a 
 *	subfont and used when determining if that SubFont supports a 
 *	character.
 *
 *	Cannot use the string name used to construct the font as the 
 *	key, because the capitalization may not be canonical.  Therefore
 *	use the face name actually retrieved from the font metrics as
 *	the key.
 *
 * Results:
 *	A pointer to a font family.  The reference count in the font family
 *	is automatically incremented.  When the subfont is released, the
 *	reference count is decremented.  When no subfont is using this
 *	font family, it may be deleted.
 *
 * Side effects:
 *	A new font family structure will be allocated if this font family
 *	has not been seen.
 *
 *-------------------------------------------------------------------------
 */

static TkGSFontFamily
AllocFontFamily(display, fontStructPtr)
    Display *display;
    XFontStruct *fontStructPtr;
{
    XlibFontFamily *familyPtr;
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    XLFDFields xlfd, *xlfdPtr;
    char key[256];
    Tcl_Encoding encoding;
    int bNew;
    Tcl_HashEntry *entryPtr;
    
    xlfdPtr = GetFontAttributes(display, fontStructPtr, &xlfd);
    encoding = Tcl_GetEncoding(NULL, GetEncodingAlias(GetXLFDCharset(xlfdPtr)));

    /* 
     * Set key for this FontFamily. 
     */
     
    sprintf(key, "%s-%s-%s\n", GetXLFDFoundry(xlfdPtr), 
	GetXLFDFamily(xlfdPtr), Tcl_GetEncodingName(encoding));

    /*
     * Find corresponding entry in family table or create one.
     */

    entryPtr = Tcl_CreateHashEntry(&tsdPtr->fontFamilyTable, key, &bNew);
    if (!bNew) {
	/* 
	 * Already existing font family.
	 */

	Tcl_FreeEncoding(encoding);
	familyPtr = (XlibFontFamily *) Tcl_GetHashValue(entryPtr);
	familyPtr->header.refCount++;
	return (TkGSFontFamily) familyPtr;
    }

    /*
     * Create new font family.
     */

    familyPtr = (XlibFontFamily *) ckalloc(sizeof(XlibFontFamily));

    /* Generic font family initialization. */
    TkGSInitFontFamily((TkGSFontFamily) familyPtr, encoding);
    familyPtr->header.refCount++;

    /* Xlib-specific font family initialization. */
    Tcl_SetHashValue(entryPtr, (ClientData) familyPtr);
    familyPtr->hashPtr      = entryPtr;

    /*
     * One byte/character fonts have both min_byte1 and max_byte1 0,
     * and max_char_or_byte2 <= 255.
     * Anything else specifies a two byte/character font.
     */

    familyPtr->isTwoByteFont = !(
	    (fontStructPtr->min_byte1 == 0) &&
	    (fontStructPtr->max_byte1 == 0) &&
	    (fontStructPtr->max_char_or_byte2 < 256));

    return (TkGSFontFamily) familyPtr;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetFontAttributes --
 *
 *	Given a screen font, determine its actual attributes, which are
 *	not necessarily the attributes that were used to construct it.
 *
 * Results:
 *	The given xlfdPtr, or NULL in case of failure.
 *
 * Side effects:
 *	The returned values may change between calls.
 *
 *---------------------------------------------------------------------------
 */

static XLFDFields *
GetFontAttributes(display, fontStructPtr, xlfdPtr)
    Display *display;		/* Display that owns the screen font. */
    XFontStruct *fontStructPtr;	/* Screen font to query. */
    XLFDFields *xlfdPtr;	/* For storing attributes of screen font. */
{
    unsigned long value;
    char *name;
    
    if ((XGetFontProperty(fontStructPtr, XA_FONT, &value) != False) &&
	    (value != 0)) {
	name = XGetAtomName(display, (Atom) value);
	xlfdPtr = ParseXLFD(name, xlfdPtr);
	XFree(name);
	IdentifySymbolEncodings(xlfdPtr);
	return xlfdPtr;
    }
    return NULL;
}

/*
 *---------------------------------------------------------------------------
 *
 * ListFonts, ListFontOrAlias --
 *
 *	Utility functions to return the array of all XLFDs on the system
 *	with the specified face name or alias.
 *
 * Results:
 *	The return value is an array of XLFDs, which should be freed with
 *	XFreeFontNames(), or NULL if no XLFDs matched the requested name.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
 
static char **
ListFonts(display, faceName, numNamesPtr)
    Display *display;		/* Display to query. */
    CONST char *faceName;	/* Desired face name, or "*" for all. */
    int *numNamesPtr;		/* Filled with length of returned array, or
				 * 0 if no names were found. */
{
    char buf[256];

    sprintf(buf, "-*-%.80s-*-*-*-*-*-*-*-*-*-*-*-*", faceName);
    return XListFonts(display, buf, 10000, numNamesPtr);
}

static char **
ListFontOrAlias(display, faceName, numNamesPtr)
    Display *display;		/* Display to query. */
    CONST char *faceName;	/* Desired face name, or "*" for all. */
    int *numNamesPtr;		/* Filled with length of returned array, or
				 * 0 if no names were found. */
{
    char **nameList, **aliases;
    int i;
    
    nameList = ListFonts(display, faceName, numNamesPtr);
    if (nameList != NULL) {
	return nameList;
    }
    aliases = TkGSFontGetAliasList(faceName);
    if (aliases != NULL) {
	for (i = 0; aliases[i] != NULL; i++) {
	    nameList = ListFonts(display, aliases[i], numNamesPtr);
	    if (nameList != NULL) {
		return nameList;
	    }
	}
    }
    *numNamesPtr = 0;
    return NULL;
}

/*
 *---------------------------------------------------------------------------
 *
 * IdentifySymbolEncodings --
 *
 *	If the font attributes refer to a symbol font, update the
 *	charset field of the font attributes so that it reflects the
 *	encoding of that symbol font.  In general, the raw value for
 *	the charset field parsed from an XLFD is meaningless for symbol
 *	fonts.
 *
 *	Symbol fonts are all fonts whose name appears in the symbolClass.
 *
 * Results:
 *	The return value is non-zero if the font attributes specify a
 *	symbol font, or 0 otherwise.  If a non-zero value is returned
 *	the charset field of the font attributes will be changed to
 *	the string that represents the actual encoding for the symbol font.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
IdentifySymbolEncodings(xlfdPtr)
    XLFDFields *xlfdPtr;
{
    int i, j;
    char **aliases, **symbolClass;

    if (!xlfdPtr) return 0;

    symbolClass = TkGSFontGetSymbolClass();
    for (i = 0; symbolClass[i] != NULL; i++) {
	if (strcasecmp(xlfdPtr->fields[XLFD_FAMILY_NAME], symbolClass[i]) == 0) {
	    xlfdPtr->fields[XLFD_CHARSET] = GetEncodingAlias(symbolClass[i]);
	    return 1;
	}
	aliases = TkGSFontGetAliasList(symbolClass[i]);
	for (j = 0; (aliases != NULL) && (aliases[j] != NULL); j++) {
	    if (strcasecmp(xlfdPtr->fields[XLFD_FAMILY_NAME], aliases[j]) == 0) {
		xlfdPtr->fields[XLFD_CHARSET] = GetEncodingAlias(aliases[j]);
		return 1;
	    }
	}
    }
    return 0;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetEncodingAlias --
 *
 *	Map the name of an encoding to another name that should be used
 *	when actually loading the encoding.  For instance, the encodings
 *	"jisc6226.1978", "jisx0208.1983", "jisx0208.1990", and
 *	"jisx0208.1996" are well-known names for the same encoding and
 *	are represented by one encoding table: "jis0208".
 *
 * Results:
 *	As above.  If the name has no alias, the original name is returned.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static CONST char *
GetEncodingAlias(name)
    CONST char *name;		/* The name to look up. */
{
    EncodingAlias *aliasPtr;
    
    for (aliasPtr = encodingAliases; aliasPtr->aliasPattern != NULL; ) {
	if (Tcl_StringMatch((char *) name, aliasPtr->aliasPattern)) {
	    return aliasPtr->realName;
	}
	aliasPtr++;
    }
    return name;
}

/*
 *---------------------------------------------------------------------------
 *
 * FindStateString --
 *	(Code taken from tkUtil.c)
 *	Given a lookup table, map a number to a string in the table.
 *
 * Results:
 *	If numKey was equal to the numeric key of one of the elements
 *	in the table, returns the string key of that element.
 *	Returns NULL if numKey was not equal to any of the numeric keys
 *	in the table.
 *
 * Side effects.
 *	None.
 *
 *---------------------------------------------------------------------------
 */

char *
FindStateString(mapPtr, numKey)
    CONST StateMap *mapPtr;	/* The state table. */
    unsigned int numKey;			/* The key to try to find in the table. */
{
    for ( ; mapPtr->strKey != NULL; mapPtr++) {
	if (numKey == mapPtr->numKey) {
	    return mapPtr->strKey;
	}
    }
    return NULL;
}

/*
 *---------------------------------------------------------------------------
 *
 * FindStateNum --
 *
 *	(Code taken from tkUtil.c)
 *	Given a lookup table, map a string to a number in the table.
 *
 * Results:
 *	If strKey was equal to the string keys of one of the elements
 *	in the table, returns the numeric key of that element.
 *	Returns the numKey associated with the last element (the NULL
 *	string one) in the table if strKey was not equal to any of the
 *	string keys in the table.  In that case, an error message is
 *	also left in the interp's result (if interp is not NULL).
 *
 * Side effects.
 *	None.
 *
 *---------------------------------------------------------------------------
 */

unsigned int
FindStateNum(mapPtr, strKey)
    CONST StateMap *mapPtr;	/* Lookup table. */
    CONST char *strKey;		/* String to try to find in lookup table. */
{
    CONST StateMap *mPtr;

    for (mPtr = mapPtr; mPtr->strKey != NULL; mPtr++) {
	if (strcmp(strKey, mPtr->strKey) == 0) {
	    return mPtr->numKey;
	}
    }
    return mPtr->numKey;
}

/*
 *-------------------------------------------------------------------------
 *
 * GetDrawableSubFont --
 *
 *	Helper proc for getting the current subfont.
 *
 * Results:
 *	Return value is the TkGSSubFont that is currently selected in the
 *	drawable. It is never null.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static TkGSSubFont
GetDrawableSubFont(d, intRepPtr)
    TkGS_Drawable d;			/* Drawable to get the subfont from. */
    TkGS_InternalRep *intRepPtr;	/* Xlib-specific internal rep of the
					 * above drawable. */
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    XlibFont *xlibFontPtr = XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);

    return (TkGSSubFont) XlibTkGSGetSubFontFromId(xlibFontPtr, XlibDrawable_SubFontId(intRepPtr));
}



/*
 * Driver font primitives.
 */


void
XlibTkGS_EnumerateFontFamilies(d, name, enumProc, clientData)
    TkGS_Drawable		d;
    CONST char			*name;
    TkGS_FontFamiliesEnumProc	*enumProc;
    ClientData			clientData;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)

    /* Code adapted from Tk's TkpGetFontFamilies */
    int i, bnew, numNames;
    char *family, *string;
    Tcl_HashTable familyTable;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch search;
    char **nameList;

    /* TODO: better font name matching. For now we load all the fonts then 
     * compare their name with the given name. This is inefficient, as we 
     * should directly list the fonts with the given name. */

    Tcl_InitHashTable(&familyTable, TCL_STRING_KEYS);
    nameList = XListFonts(display, "-*-*-*-*-*-*-*-*-*-*-*-*-*-*", 10000, &numNames);
    for (i = 0; i < numNames; i++) {
	family = strchr(nameList[i] + 1, '-') + 1;
	strchr(family, '-')[0] = '\0';
	if (name == NULL || strcasecmp(family, name) == 0) {
	    Tcl_CreateHashEntry(&familyTable, family, &bnew);
	}
    }
    XFreeFontNames(nameList);

    hPtr = Tcl_FirstHashEntry(&familyTable, &search);
    while (hPtr != NULL) {
	/* FIXME: system to UTF-8 conversion? */
	string = Tcl_GetHashKey(&familyTable, hPtr);
	if (!(enumProc)(d, clientData, string)) {
	    break;
	}
	hPtr = Tcl_NextHashEntry(&search);
    }
    Tcl_DeleteHashTable(&familyTable);
}

int 
XlibTkGS_FontFamilyExists(d, name)
    TkGS_Drawable		d;
    CONST char			*name;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    Tcl_DString buf;
    char pattern[256];
    char **nameList;
    int numNames;

    Tcl_DStringInit(&buf);
    Tcl_UtfToExternalDString(NULL /* system-specific encoding */, 
	name, -1, &buf);
    sprintf(pattern, "-*-%s-*-*-*-*-*-*-*-*-*-*-*-*", buf);
    Tcl_DStringFree(&buf);

    nameList = XListFonts(display, pattern, 1, &numNames);
    
    return (numNames > 0);
}

void 
XlibTkGS_GetFontMetrics(d, font, fmPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGS_FontMetrics *fmPtr;
{
    XlibFont *xlibFontPtr = XlibTkGS_GetXlibFontFromFont(NULL, d, font);
    if (!xlibFontPtr) {
	/* Failed! */
	memset(fmPtr, 0, sizeof(TkGS_FontMetrics));
    	return;
    }

    *fmPtr = xlibFontPtr->fm;
}


void 
XlibTkGS_GetActualFontAttributes(d, font, faPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGS_FontAttributes *faPtr;
{
    XlibFont *xlibFontPtr = XlibTkGS_GetXlibFontFromFont(NULL, d, font);
    if (!xlibFontPtr) {
	/* Failed! */
	memset(faPtr, 0, sizeof(TkGS_FontAttributes));
	return;
    }

    *faPtr = xlibFontPtr->fa;
}


/*
 * Unicode font primitives.
 */

int
XlibTkGS_MeasureCharsUni(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);

    return TkGSMultiFontMeasureCharsUni(d, gcValues->font, multiFont, 
	string, length, maxPixels, flags, lengthPtr);
}

void
XlibTkGS_DrawCharsUni(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);

    TkGSMultiFontDrawCharsUni(d, gcValues->font, multiFont, 
	string, length, x, y, widthPtr);
}


/*
 * UTF-8 font primitives.
 */

int
XlibTkGS_MeasureCharsUtf(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);

    return TkGSMultiFontMeasureCharsUtf(d, gcValues->font, multiFont, 
	string, length, maxPixels, flags, lengthPtr);
}

void
XlibTkGS_DrawCharsUtf(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);

    TkGSMultiFontDrawCharsUtf(d, gcValues->font, multiFont, 
	string, length, x, y, widthPtr);
}


/*
 * System-specific font primitives.
 */

int
XlibTkGS_TextWidthSys(d, string, length)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    register XlibSubFont *subFontPtr;
    register XlibFontFamily *familyPtr;

    TkGSSetDrawableSubFont(d, TKGS_BASE_SUBFONT);

    /* FIXME: how to handle system-specific strings? */
    subFontPtr = (XlibSubFont *) GetDrawableSubFont(d, intRepPtr);
    familyPtr = (XlibFontFamily *) subFontPtr->header.family;
    if (familyPtr->isTwoByteFont) {
	return XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) string, length);
    } else {
	return XTextWidth(subFontPtr->fontStructPtr,
		    string, length);
    }
}


void
XlibTkGS_DrawCharsSys(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length; 
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    register XlibSubFont *subFontPtr;
    register XlibFontFamily *familyPtr;

    TkGSSetDrawableSubFont(d, TKGS_BASE_SUBFONT);

    /* FIXME: how to handle system-specific strings? */
    subFontPtr = (XlibSubFont *) GetDrawableSubFont(d, intRepPtr);
    familyPtr = (XlibFontFamily *) subFontPtr->header.family;
    if (familyPtr->isTwoByteFont) {
	XDrawString16(display, xd, xgc, x, y, 
		(XChar2b *) string, length);
	if (widthPtr) {
	    *widthPtr = XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) string, length);
	}
    } else {
	XDrawString(display, xd, xgc, x, y, string, length);
	if (widthPtr) {
	    *widthPtr = XTextWidth(subFontPtr->fontStructPtr,
		    string, length);
	}
    }
}

/*
 * Multi-font Unicode primitives.
 */

TkGSSubFont 
XlibTkGSGetSubFont(d, multiFont, subFontId)
    TkGS_Drawable d;
    TkGSMultiFont multiFont;
    TkGSSubFontId subFontId;
{
    return (TkGSSubFont) XlibTkGSGetSubFontFromId(multiFont, subFontId);
}


void 
XlibTkGSSetDrawableSubFont(d, subFontId)
    TkGS_Drawable d;
    TkGSSubFontId subFontId;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    XlibSubFont *xlibSubFontPtr;

    if (XlibDrawable_SubFontId(intRepPtr) == subFontId)
	return;

    XlibDrawable_SubFontId(intRepPtr) = subFontId;

    /* Select subfont's FID. */
    xlibSubFontPtr = (XlibSubFont *) GetDrawableSubFont(d, intRepPtr);
    XSetFont(display, xgc, xlibSubFontPtr->fontStructPtr->fid);
}

TkGSSubFontId 
XlibTkGSDefaultSubFont(d, font, multiFont) 
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;
{
    return XlibTkGSGetControlSubFontId();
}

void 
XlibTkGSFontMapLoadPage(d, subFont, row)
    TkGS_Drawable d;
    TkGSSubFont subFont;
    int row;
{
    char src[TCL_UTF_MAX], buf[16];
    int minHi, maxHi, minLo, maxLo, scale, checkLo;
    int i, end, bitOffset, isTwoByteFont, n;
    Tcl_Encoding encoding;
    XFontStruct *fontStructPtr;
    XCharStruct *widths;
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    XlibSubFont *xlibSubFontPtr = (XlibSubFont *) subFont;
    XlibFontFamily *xlibFamilyPtr = (XlibFontFamily *) subFont->family;

    if (xlibFamilyPtr == &tsdPtr->controlFamily) {
	return;
    }

    fontStructPtr   = xlibSubFontPtr->fontStructPtr;
    encoding	    = subFont->family->encoding;
    isTwoByteFont   = xlibFamilyPtr->isTwoByteFont;

    widths	= fontStructPtr->per_char;
    minHi	= fontStructPtr->min_byte1;
    maxHi	= fontStructPtr->max_byte1;
    minLo	= fontStructPtr->min_char_or_byte2;
    maxLo	= fontStructPtr->max_char_or_byte2;
    scale	= maxLo - minLo + 1;
    checkLo	= minLo;

    if (! isTwoByteFont) {
	if (minLo < 32) {
	    checkLo = 32;
	}
    }

    end = (row + 1) << FONTMAP_SHIFT;
    for (i = row << FONTMAP_SHIFT; i < end; i++) {
	int hi, lo;
	
	if (Tcl_UtfToExternal(NULL, encoding, src, Tcl_UniCharToUtf(i, src),
        	TCL_ENCODING_STOPONERROR, NULL, buf, sizeof(buf), NULL, 
		NULL, NULL) != TCL_OK) {
	    continue;
	}
	if (isTwoByteFont) {
	    hi = ((unsigned char *) buf)[0];
	    lo = ((unsigned char *) buf)[1];
	} else {
	    hi = 0;
	    lo = ((unsigned char *) buf)[0];
	}
	if ((hi < minHi) || (hi > maxHi) || (lo < checkLo) || (lo > maxLo)) {
	    continue;
	}
	n = (hi - minHi) * scale + lo - minLo;
	if ((widths == NULL) || ((widths[n].width + widths[n].rbearing) != 0)) {
	    bitOffset = i & (FONTMAP_BITSPERPAGE - 1);
	    subFont->fontMap[row][bitOffset >> 3] |= 1 << (bitOffset & 7);
	}
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * XlibTkGSCanUseFallback --
 *
 *	If the specified screen font has not already been loaded into 
 *	the font object, determine if it can display the given character.
 *
 * Results:
 *	The return value is the id of a newly allocated subfont, owned
 *	by the font object.  This subfont can be used to display the given
 *	character.  The subfont represents the screen font with the base set 
 *	of font attributes from the font object, but using the specified 
 *	font name.  NULL is returned if the font object already holds
 *	a reference to the specified physical font or if the specified 
 *	font doesn't exist or cannot display the given character.
 *
 * Side effects:				       
 *	The font object's subFontArray is updated to contain a reference
 *	to the newly allocated SubFont.
 *
 *-------------------------------------------------------------------------
 */

TkGSSubFontId
XlibTkGSCanUseFallback(d, font, multiFont, faceName, ch)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;	/* The font object that will own the new
				 * screen font. */
    CONST char *faceName;	/* Desired face name for new screen font. */
    TkGS_UniChar ch;		/* The Unicode character that the new
				 * screen font must be able to display. */
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    XlibFont *fontPtr = (XlibFont *) multiFont;
    int i, nameIdx, numNames, srcLen;
    char hate[2][32];
    const char *hateFoundry;
    int bestIdx[2];
    CONST char *charset, *hateCharset;
    unsigned int bestScore[2];
    char **nameList, **nameListOrig;
    XLFDFields xlfd;
    char src[TCL_UTF_MAX];
    XlibSubFont subFont, *subFontPtr;
    XFontStruct *fontStructPtr;
    Tcl_DString dsEncodings;
    int numEncodings;
    Tcl_Encoding *encodingCachePtr;

    /*
     * Assume: the face name is times.
     * Assume: adobe:times:iso8859-1 has already been used.
     *
     * Are there any versions of times that can display this
     *    character (e.g., perhaps linotype:times:iso8859-2)?
     *	  a. Get list of all times fonts.
     *	  b1. Cross out all names whose encodings we've already used.
     *	  b2. Cross out all names whose foundry & encoding we've already seen.
     *	  c. Cross out all names whose encoding cannot handle the character.
     *	  d. Rank each name and pick the best match.
     *	  e. If that font cannot actually display the character, cross
     *	     out all names with the same foundry and encoding and go
     *	     back to (c).
     */

    nameList = ListFonts(display, faceName, &numNames);
    if (numNames == 0) {
	return 0;
    }
    nameListOrig = nameList;

    srcLen = Tcl_UniCharToUtf(ch, src);

    hateFoundry = NULL;
    hateCharset = NULL;
    numEncodings = 0;
    Tcl_DStringInit(&dsEncodings);

    charset = NULL;	/* lint, since numNames must be > 0 to get here. */

    retry:
    bestIdx[0] = -1;
    bestIdx[1] = -1;
    bestScore[0] = (unsigned int) -1;
    bestScore[1] = (unsigned int) -1;
    for (nameIdx = 0; nameIdx < numNames; nameIdx++) {
	Tcl_Encoding encoding;
	char dst[16];
	int scalable, srcRead, dstWrote;
	unsigned int score;
	
	if (nameList[nameIdx] == NULL) {
	    continue;
	}
	if (!ParseXLFD(nameList[nameIdx], &xlfd)) {
	    goto crossout;
	}
	IdentifySymbolEncodings(&xlfd);
	charset = GetEncodingAlias(GetXLFDCharset(&xlfd));
	if (hateFoundry != NULL) {
	    /*
	     * E. If the font we picked cannot actually display the
	     * character, cross out all names with the same foundry and
	     * encoding. 
	     */

	    if ( (strcasecmp(hateFoundry, GetXLFDFoundry(&xlfd)) == 0)
		    && (strcasecmp(hateCharset, charset) == 0)) {
		goto crossout;
	    }
	} else {
	    /*
	     * B. Cross out all names whose encodings we've already used.
	     */
	     
	    for (i = 0; i < multiFont->numSubFonts; i++) {
		subFontPtr = XlibTkGSGetSubFontFromIndex(multiFont, i);
		encoding = subFontPtr->header.family->encoding;
		if (strcasecmp(charset, Tcl_GetEncodingName(encoding)) == 0) {
		    goto crossout;
		}
	    }
	}
	
	/*
	 * C. Cross out all names whose encoding cannot handle the character.
	 */
	 
	encodingCachePtr = (Tcl_Encoding *) Tcl_DStringValue(&dsEncodings);
	for (i = numEncodings; --i >= 0; encodingCachePtr++) {
	    encoding = *encodingCachePtr;
	    if (strcasecmp(Tcl_GetEncodingName(encoding), charset) == 0) {
		break;
	    }
	}
	if (i < 0) {
	    encoding = Tcl_GetEncoding(NULL, charset);
	    if (encoding == NULL) {
		goto crossout;
	    }

	    Tcl_DStringAppend(&dsEncodings, (char *) &encoding,
		    sizeof(encoding));
	    numEncodings++;
	}
	Tcl_UtfToExternal(NULL, encoding, src, srcLen, 
		TCL_ENCODING_STOPONERROR, NULL, dst, sizeof(dst), &srcRead, 
		&dstWrote, NULL);
	if (dstWrote == 0) {
	    goto crossout;
	}

	/*
	 * D. Rank each name and pick the best match.
	 */

	scalable = IsXLFDScalable(&xlfd);
	score = RankAttributes(&fontPtr->fa, charset, &xlfd);
	if (score <= bestScore[scalable]) {
	    bestIdx[scalable] = nameIdx;
	    bestScore[scalable] = score;
	}
	if (score == 0) {
	    break;
	}
	continue;

	crossout:
	if (nameList == nameListOrig) {
	    /*
	     * Not allowed to change pointers to memory that X gives you,
	     * so make a copy.
	     */

	    nameList = (char **) ckalloc(numNames * sizeof(char *));
	    memcpy(nameList, nameListOrig, numNames * sizeof(char *));
	}
	nameList[nameIdx] = NULL;
    }

    fontStructPtr = GetScreenFont(display, &fontPtr->fa, nameList, bestIdx, bestScore);

    encodingCachePtr = (Tcl_Encoding *) Tcl_DStringValue(&dsEncodings);
    for (i = numEncodings; --i >= 0; encodingCachePtr++) {
	Tcl_FreeEncoding(*encodingCachePtr);
    }
    Tcl_DStringFree(&dsEncodings);
    numEncodings = 0;

    if (fontStructPtr == NULL) {
	if (nameList != nameListOrig) {
	    ckfree((char *) nameList);
	}
	XFreeFontNames(nameListOrig);
	return 0;
    }

    InitSubFont(d, font, &subFont, display, fontStructPtr);
    if (TkGSFontMapLookup(d, (TkGSSubFont) &subFont, ch) == 0) {
	/*
	 * E. If the font we picked cannot actually display the character,
	 * cross out all names with the same foundry and encoding and pick
	 * another font.
	 */

	strcpy(hate[0], GetXLFDFoundry(&xlfd));
	hateFoundry = hate[0];
	strcpy(hate[1], charset);
	hateCharset = hate[1];
	FreeXlibSubFont(multiFont, (TkGSSubFont) &subFont);
	goto retry;
    }
    if (nameList != nameListOrig) {
	ckfree((char *) nameList);
    }
    XFreeFontNames(nameListOrig);

    /*
     * Add subfont.
     */

    fontPtr->header.subFontArray = (TkGSSubFont) ckrealloc(
				(char *) fontPtr->header.subFontArray, 
				sizeof(XlibSubFont) 
				* (fontPtr->header.numSubFonts + 1));

    *XlibTkGSGetSubFontFromIndex(fontPtr, fontPtr->header.numSubFonts) = subFont;
    fontPtr->header.numSubFonts++;
    return XlibTkGSGetSubFontId(fontPtr->header.numSubFonts - 1);
}

TkGSSubFontId
XlibTkGSCanUseFont(d, font, multiFont, ch) 
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;
    TkGS_UniChar ch;
{
    int i;
    
    for (i = 0; i < multiFont->numSubFonts; i++) {
	if (TkGSFontMapLookup(d, (TkGSSubFont) XlibTkGSGetSubFontFromIndex(multiFont, i), ch)) {
	    return XlibTkGSGetSubFontId(i);
	}
    }
    if (TkGSFontMapLookup(d, (TkGSSubFont) XlibTkGSGetSubFontFromId(multiFont, XlibTkGSGetControlSubFontId()), ch)) {
	return XlibTkGSGetControlSubFontId();
    }
    return 0;
}

/*  - Unicode version */

int
XlibTkGSSubFontMeasureCharsUni(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    XlibFont *xlibFontPtr = XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);
    Tcl_UniChar ch;
    CONST Tcl_UniChar *p, *end, *next;
    CONST Tcl_UniChar *term;
    int newX, termX, sawNonSpace;
    int curX, curByte;

    /*
     * This first version may be inefficient because it measures
     * every character individually.
     */

    ch = *string;
    next = string + 1;
    newX = curX = termX = 0;
    
    term = string;
    end = string + length;

    sawNonSpace = (ch > 255) || !isspace(ch);
    for (p = string; ; ) {
	if (ch < XLIB_BASE_CHARS) {
	    newX += xlibFontPtr->widths[ch];
	} else {
	    newX += TkGSSubFontTextWidthUni(d, p, next - p);
	}
	if (newX > maxPixels) {
	    break;
	}
	curX = newX;
	p = next;
	if (p >= end) {
	    term = end;
	    termX = curX;
	    break;
	}

	ch = *next;
	next++;
	if ((ch < 256) && isspace(ch)) {
	    if (sawNonSpace) {
		term = p;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}
    }

    /*
     * P points to the first character that doesn't fit in the desired
     * span.  Use the flags to figure out what to return.
     */

    if ((flags & TK_PARTIAL_OK) && (p < end) && (curX < maxPixels)) {
	/*
	 * Include the first character that didn't quite fit in the desired
	 * span.  The width returned will include the width of that extra
	 * character.
	 */

	curX = newX;
	p++;
    }
    if ((flags & TK_AT_LEAST_ONE) && (term == string) && (p < end)) {
	term = p;
	termX = curX;
	if (term == string) {
	    term++;
	    termX = newX;
	}
    } else if ((p >= end) || !(flags & TK_WHOLE_WORDS)) {
	term = p;
	termX = curX;
    }

    curX = termX;
    curByte = term - string;

    *lengthPtr = curX;
    if (term >= end) {
	return length;
    } else {
	return term-string;
    }
}

int
XlibTkGSSubFontTextWidthUni(d, string, length)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    XlibSubFont *subFontPtr = (XlibSubFont *) subFont;
    XlibFontFamily *familyPtr = (XlibFontFamily *) subFont->family;
    Tcl_DString dsUtf, ds;
    char *str;		/* String converted to subfont's encoding. */
    int len;		/* Length of converted string in bytes. */
    int width;

    if (familyPtr->header.encoding != unicodeEncoding) {
	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&ds);
	Tcl_UniCharToUtfDString(string, length, &dsUtf);
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		Tcl_DStringValue(&dsUtf), Tcl_DStringLength(&dsUtf), &ds);
	str = Tcl_DStringValue(&ds);
	len = Tcl_DStringLength(&ds);
    } else {
	str = (char *) string;
	len = length << 1;
    }

    if (familyPtr->isTwoByteFont) {
	width = XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) str, len >> 1);
    } else {
	width = XTextWidth(subFontPtr->fontStructPtr,
		    str, len);
    }

    if (familyPtr->header.encoding != unicodeEncoding) {
	Tcl_DStringFree(&ds);
	Tcl_DStringFree(&dsUtf);
    }
    return width;
}

void
XlibTkGSSubFontDrawCharsUni(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    XlibSubFont *subFontPtr = (XlibSubFont *) subFont;
    XlibFontFamily *familyPtr = (XlibFontFamily *) subFont->family;
    Tcl_DString dsUtf, ds;
    char *str;		/* String converted to subfont's encoding. */
    int len;		/* Length of converted string in bytes. */

    if (familyPtr->header.encoding != unicodeEncoding) {
	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&ds);
	Tcl_UniCharToUtfDString(string, length, &dsUtf);
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		Tcl_DStringValue(&dsUtf), Tcl_DStringLength(&dsUtf), &ds);
	str = Tcl_DStringValue(&ds);
	len = Tcl_DStringLength(&ds);
    } else {
	str = (char *) string;
	len = length << 1;
    }

    if (familyPtr->isTwoByteFont) {
	XDrawString16(display, xd, xgc, x, y, 
		(XChar2b *) str, len >> 1);
	if (widthPtr) {
	    *widthPtr = XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) str, len >> 1);
	}
    } else {
	XDrawString(display, xd, xgc, x, y, 
		str, len);
	if (widthPtr) {
	    *widthPtr = XTextWidth(subFontPtr->fontStructPtr,
		    str, len);
	}
    }

    if (familyPtr->header.encoding != unicodeEncoding) {
	Tcl_DStringFree(&ds);
	Tcl_DStringFree(&dsUtf);
    }
}


/*  - UTF-8 version */

int
XlibTkGSSubFontMeasureCharsUtf(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    XlibFont *xlibFontPtr = XlibTkGS_GetXlibFontFromFont(NULL, d, gcValues->font);
    Tcl_UniChar ch;
    CONST char *p, *end, *next;
    CONST char *term;
    int newX, termX, sawNonSpace;
    int curX, curByte;

    /*
     * This first version may be inefficient because it measures
     * every character individually. 
     */

    next = string + Tcl_UtfToUniChar(string, &ch);
    newX = curX = termX = 0;
    
    term = string;
    end = string + length;

    sawNonSpace = (ch > 255) || !isspace(ch);
    for (p = string; ; ) {
	if (ch < XLIB_BASE_CHARS) {
	    newX += xlibFontPtr->widths[ch];
	} else {
	    newX += TkGSSubFontTextWidthUtf(d, p, next - p);
	}
	if (newX > maxPixels) {
	    break;
	}
	curX = newX;
	p = next;
	if (p >= end) {
	    term = end;
	    termX = curX;
	    break;
	}

	next += Tcl_UtfToUniChar(next, &ch);
	if ((ch < 256) && isspace(ch)) {
	    if (sawNonSpace) {
		term = p;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}
    }

    /*
     * P points to the first character that doesn't fit in the desired
     * span.  Use the flags to figure out what to return.
     */

    if ((flags & TK_PARTIAL_OK) && (p < end) && (curX < maxPixels)) {
	/*
	 * Include the first character that didn't quite fit in the desired
	 * span.  The width returned will include the width of that extra
	 * character.
	 */

	curX = newX;
	p += Tcl_UtfToUniChar(p, &ch);
    }
    if ((flags & TK_AT_LEAST_ONE) && (term == string) && (p < end)) {
	term = p;
	termX = curX;
	if (term == string) {
	    term += Tcl_UtfToUniChar(term, &ch);
	    termX = newX;
	}
    } else if ((p >= end) || !(flags & TK_WHOLE_WORDS)) {
	term = p;
	termX = curX;
    }

    curX = termX;
    curByte = term - string;

    *lengthPtr = curX;
    if (term >= end) {
	return length;
    } else {
	return term-string;
    }
}

int
XlibTkGSSubFontTextWidthUtf(d, string, length)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    XlibSubFont *subFontPtr = (XlibSubFont *) subFont;
    XlibFontFamily *familyPtr = (XlibFontFamily *) subFont->family;
    Tcl_DString ds;
    char *str;		/* String converted to subfont's encoding. */
    int len;		/* Length of converted string in bytes. */
    int width;

    Tcl_DStringInit(&ds);
    Tcl_UtfToExternalDString(familyPtr->header.encoding, 
	    string, length, &ds);
    str = Tcl_DStringValue(&ds);
    len = Tcl_DStringLength(&ds);

    if (familyPtr->isTwoByteFont) {
	width = XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) str, len >> 1);
    } else {
	width = XTextWidth(subFontPtr->fontStructPtr,
		    str, len);
    }

    Tcl_DStringFree(&ds);
    return width;
}

void
XlibTkGSSubFontDrawCharsUtf(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, display, xd, xgc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    XlibSubFont *subFontPtr = (XlibSubFont *) subFont;
    XlibFontFamily *familyPtr = (XlibFontFamily *) subFont->family;
    Tcl_DString ds;
    char *str;		/* String converted to subfont's encoding. */
    int len;		/* Length of converted string in bytes. */

    Tcl_DStringInit(&ds);
    Tcl_UtfToExternalDString(familyPtr->header.encoding, 
	    string, length, &ds);
    str = Tcl_DStringValue(&ds);
    len = Tcl_DStringLength(&ds);

    if (familyPtr->isTwoByteFont) {
	XDrawString16(display, xd, xgc, x, y, 
		(XChar2b *) str, len >> 1);
	if (widthPtr) {
	    *widthPtr = XTextWidth16(subFontPtr->fontStructPtr,
		    (XChar2b *) str, len >> 1);
	}
    } else {
	XDrawString(display, xd, xgc, x, y, 
		str, len);
	if (widthPtr) {
	    *widthPtr = XTextWidth(subFontPtr->fontStructPtr,
		    str, len);
	}
    }

    Tcl_DStringFree(&ds);
}

/*
 *-------------------------------------------------------------------------
 * 
 * XlibTkGSInitFontPackage --
 *
 *	This procedure is called when an application loads the device 
 *	driver.  It initializes all the structures that are used by the 
 *	platform-dependent code on a per application basis.
 *
 * Results:
 *	None.  
 *
 * Side effects:
 *	See comments below.
 *
 *-------------------------------------------------------------------------
 */

void
XlibTkGSInitFontPackage() {
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    Tcl_EncodingType type;
    TkGSSubFont_ dummy;
    int i;

    Tcl_InitHashTable(&tsdPtr->fontFamilyTable, TCL_STRING_KEYS);

    /*
     * Initialize font family for control chars.
     */

    if (tsdPtr->controlFamily.header.encoding == NULL) {
	/* Encoding for control chars. */
	type.encodingName	= "X11ControlChars";
	type.toUtfProc		= ControlUtfProc;
	type.fromUtfProc	= ControlUtfProc;
	type.freeProc		= NULL;
	type.clientData		= NULL;
	type.nullSize		= 0;
	
	/* Generic font family initialization. */
	TkGSInitFontFamily((TkGSFontFamily) &tsdPtr->controlFamily, Tcl_CreateEncoding(&type));
	tsdPtr->controlFamily.header.refCount++;

	/* Xlib-specific font family initialization. */
	tsdPtr->controlFamily.hashPtr = NULL;
	tsdPtr->controlFamily.isTwoByteFont = 0;

	/* Load control characters. */
	dummy.family = (TkGSFontFamily) &tsdPtr->controlFamily;
	dummy.fontMap = dummy.family->fontMap;
	TkGSFontMapInitPage(&dummy, 0);
	for (i = 0x00; i < 0x20; i++) {
	    /* 
	     * First parameter (TkGS_Drawable) shouldn't be needed since page 
	     * is already loaded. 
	     */

	    TkGSFontMapInsert(NULL, &dummy, i);
	    TkGSFontMapInsert(NULL, &dummy, i + 0x80);
	}

	/*
	 * UCS-2BE is unicode in big-endian format.
	 * It is used in iso10646 fonts.
	 */

	type.encodingName	= "ucs-2be";
	type.toUtfProc		= Ucs2beToUtfProc;
	type.fromUtfProc	= UtfToUcs2beProc;
	type.freeProc		= NULL;
	type.clientData		= NULL;
	type.nullSize		= 2;
	Tcl_CreateEncoding(&type);
    }

    defaultEncoding = Tcl_GetEncoding(NULL, "iso8859-1");	/* locale. */
    unicodeEncoding = Tcl_GetEncoding(NULL, "unicode");
}

