/* tkBargraph.c --
 *
 *	This module implements a bargraph widget for the Tk toolkit.
 *
 * Copyright 1992 Victoria University of Wellington, NZ.
 * This code is derived from the tkScale widget.
 * Copyright 1990 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.	 The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "pkg.h"

/*
 *  Default bargraph configuration values
 */

#define DEF_BAR_MODE			"bargraph"
#define DEF_BAR_SLIDER_LENGTH		"10"

#define DEF_BAR_BG_COLOR		NORMAL_BG
#define DEF_BAR_BAR_BG_COLOR		NORMAL_BG
#define DEF_BAR_BAR_FG_COLOR		"red"
#define DEF_BAR_TEXT_COLOR		"black"
#define DEF_BAR_TICK_COLOR		"blue"

/* for monochrome displays */

#define DEF_BAR_BG_MONO			"white"
#define DEF_BAR_BAR_MONO		"black"
#define DEF_BAR_TEXT_MONO		"black"
#define DEF_BAR_TICK_MONO		"black"

#define DEF_BAR_BORDER_WIDTH		"0"
#define DEF_BAR_CURSOR			""
#define DEF_BAR_HIGHLIGHT_WIDTH		"0"
#define DEF_BAR_RELIEF			"flat"

#define DEF_BAR_FONT			DEF_FONT
#define DEF_BAR_DIGITS			"6"
#define DEF_BAR_FROM			"0.0"
#define DEF_BAR_TO			"100.0"
#define DEF_BAR_BASEVALUE		"0.0"
#define DEF_BAR_LENGTH			"100"
#define DEF_BAR_ORIENT			"vertical"
#define DEF_BAR_SHOW_VALUE		"1"
#define DEF_BAR_SHOW_RANGE		"1"
#define DEF_BAR_TAKE_FOCUS		(char *) NULL
#define DEF_BAR_TICK_INTERVAL		"20.0"
#define DEF_BAR_TICK_LENGTH		"4"
#define DEF_BAR_LABEL			""
#define DEF_BAR_WIDTH			"20"
#define DEF_BAR_BAR_BORDER_WIDTH	"2"
#define DEF_BAR_BAR_RELIEF		"groove"
#define DEF_BAR_PADX			"2"
#define DEF_BAR_PADY			"2"

/*
 * A data structure of the following type is kept for each barchart
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the entry. NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with entry. */
    Tcl_Command widgetCmd;	/* Token for entry's widget command. */
    Tk_OptionTable optionTable;	/* Table that defines configuration options
				 * available for this widget. */

    /* Border stuff */
    Tk_3DBorder border;		/* Structure used to draw 3-D border */
    int borderWidth;		/* Width of 3-D border around window */
    Tk_Cursor cursor;		/* Current cursor for window. */
    int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
    XColor *highlightBgColorPtr;	/* Color for drawing traversal highlight
					 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */

    /* bargraph stuff */
    int orient;			/* Orientation for window ("vertical" or
				 * "horizontal"). */
    double value;		/* Current value of Bargraph. */
    double baseVal;		/* Value that is the root of the bar (ignored
				 * if making a slider).
				 */
    double toVal_u;		/* as entered by user, before validation */
    double toVal;		/* Value corresponding to right or top of
				 * Bargraph. */
    double fromVal_u;		/* as entered by user, before validation */
    double fromVal;		/* Value corresponding to left or bottom
				 * of Bargraph. */
    double tickInterval_u;	/* as entered by user, before validation */
    double tickInterval;	/* Distance between tick marks;	 0 means
				 * don't display any tick marks. */
    int mode;			/* MODE_BARGRAPH or MODE_SLIDER */
    int sliderlength;		/* length of bar, in pixels: if mode is slider,
				 * we draw a slider of this size.
				 */
    char *label;		/* Label to display above or to right of
				 * Bargraph;  NULL means don't display a
				 * label.  Malloc'ed. */

    /* Information used when displaying widget: */
    int barBorderWidth;		/* Width of bar border */
    int barRelief;		/* Bar relief, e.g., TK_RELIEF_RAISED */
    Tk_3DBorder barBorder;	/* Used for drawing bar background. */
    Tk_3DBorder barColor;	/* Used for drawing bar. */

    Tk_Font tkfont;		/* Information about text font, or NULL. */
    int digits;			/* Precision for formatting values */
    XColor *textColorPtr;	/* Color for drawing text. */
    GC textGC;			/* GC for drawing text. */

    XColor *tickColorPtr;	/* Color for drawing ticks and their labels */
    GC tickGC;			/* GC for drawing ticks */

    int width;			/* Desired narrow dimension of Bargraph,
				 * in pixels. */
    int max_height;		/* Desired long dimension of Bargraph,
				 * in pixels. */
    int tickLength;		/* Size of ticks */
    int padX;			/* Horiz padding around strings */
    int padY;			/* Vertical padding around strings */
    int showValue;		/* 1 means to display the Bargraph value */
    int showRange;		/* 1 means to display min and max values */
    int minmaxPixels;		/* Number of pixels required for widest text
				 * labelling tickmark.	0 means don't display
				 * ticks. */
    int numTicks;		/* The number of tick marks. */
    XSegment *ticks;		/* The tick marks.  Malloc'd. */

    char *alabelstr_u;		/* The "a" complex labels, as a string,
				 * as entered by user */
    Tk_Font Atkfont;		/* alabel text font, or NULL. */
    GC alabGC;			/* GC for drawing alabels. */
    char *alabelstr;		/* The "a" complex labels, as a string */
    int n_alabels;		/* number of "a" complex labels */
    char **alabels;		/* The "a" complex labels, as a list. */
    int alabelPixels;		/* Number of pixels required for
				 * widest alabel  */

    char *blabelstr_u;		/* The "b" complex labels, as a string, as
				 * entered by user  */
    Tk_Font Btkfont;		/* blabel text font, or NULL. */
    GC blabGC;			/* GC for drawing blabels. */
    char *blabelstr;		/* The "b" complex labels, as a string */
    int n_blabels;		/* number of "b" complex labels */
    char **blabels;		/* The "b" complex labels, as a list. */
    int blabelPixels;		/* Number of pixels required for
				 * widest blabel */

    int valuePixels;		/* Number of pixels required for value text */

    struct {			/* Pixel offsets of various components */
	int bot_edge;		/* left/bottom edge of bar */
	int top_edge;		/* top/right edge of bar */
	int base;		/* base value in bar */
	int label_y;
	int minmax0_x, minmax0_y;
	int minmax1_x, minmax1_y;
	int value_x, value_y;
	int lab_a, lab_b;
	int tick_a, tick_b;
	int bar_x, bar_y;
    } offset;

    /* Miscellaneous information: */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    long int flags;		/* Various flags; see below for definitions */
} Bargraph;

/* Handy shorthands */
#define SHOW_TICKS(bgPtr)	((bgPtr)->tickInterval != 0.)
#define SHOW_RANGE(bgPtr)	(SHOW_TICKS(bgPtr) && (bgPtr)->showRange)
#define SHOW_RANGE_OR_BASEV(bgPtr) \
     ((((bgPtr)->baseVal != (bgPtr)->fromVal) && ((bgPtr)->baseVal != (bgPtr)->toVal)) || SHOW_RANGE(bgPtr))
#define HAS_ALAB(bgPtr)		((bgPtr)->n_alabels != 0)
#define HAS_BLAB(bgPtr)		((bgPtr)->n_blabels != 0)


/*
 *  flags:
 *  DISPLAY_TICKS - draw the tick points and min/max vals
 *  DISPLAY_BAR - draw the bar
 *  DISPLAY_BORDER - draw the border
 *  DISPLAY_LABELS - draw the alabels and blabels
 *  DISPLAY_LABEL - draw the label
 *  DISPLAY_VALUE - draw the textual value
 */

#define	DISPLAY_TICKS		01
#define	DISPLAY_BAR		02
#define	DISPLAY_BORDER		04
#define	DISPLAY_LABELS		010
#define	DISPLAY_VALUE		020
#define	DISPLAY_LABEL		040
#define	REDRAW_PENDING		0100
#define	CLEAR_NEEDED		0200
#define GOT_FOCUS		0400

#define	DISPLAY_ALL	(DISPLAY_TICKS | DISPLAY_BAR | DISPLAY_BORDER | \
			 DISPLAY_LABELS | DISPLAY_VALUE | DISPLAY_LABEL)

/*
 * Information used for argv parsing.
 */

enum modeOpt {
    MODE_BARGRAPH, MODE_SLIDER
};
static CONST84 char *modeStrings[]= {
    "bargraph", "slider", (char*) NULL
};

enum orientOpt {
    ORIENT_HORIZONTAL, ORIENT_VERTICAL
};
static CONST84 char *orientStrings[]= {
    "horizontal", "vertical", (char*) NULL
};

static Tk_OptionSpec optionSpecs[] = {
    {TK_OPTION_STRING, "-alabels", "alabels", "Alabels",
     DEF_BAR_LABEL, -1, Tk_Offset(Bargraph, alabelstr_u), 0, 0, 0},
    {TK_OPTION_FONT, "-alabfont", "alabFont", "AlabFont",
     DEF_BAR_FONT, -1, Tk_Offset(Bargraph, Atkfont), 0, 0, 0},

    {TK_OPTION_BORDER, "-background", "background", "Background",
     DEF_BAR_BG_COLOR, -1, Tk_Offset(Bargraph, border),
     0, (ClientData) DEF_BAR_BG_MONO, 0},
    {TK_OPTION_BORDER, "-barbackground", "barBackground", "Background",
     DEF_BAR_BAR_BG_COLOR, -1, Tk_Offset(Bargraph, barBorder),
     0, (ClientData) DEF_BAR_BG_MONO, 0},
    {TK_OPTION_INT, "-barborderwidth", "barBorderWidth", "barBorderWidth",
     DEF_BAR_BAR_BORDER_WIDTH, -1, Tk_Offset(Bargraph, barBorderWidth), 0, 0, 0},
    {TK_OPTION_BORDER, "-barcolor", "barColor", "Foreground",
     DEF_BAR_BAR_FG_COLOR, -1, Tk_Offset(Bargraph, barColor),
     0, (ClientData) DEF_BAR_BAR_MONO, 0},
    {TK_OPTION_RELIEF, "-barrelief", "barRelief", "BarRelief",
     DEF_BAR_BAR_RELIEF, -1, Tk_Offset(Bargraph, barRelief), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-base", "base", "Base",
     DEF_BAR_BASEVALUE, -1, Tk_Offset(Bargraph, baseVal), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-bd", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-borderwidth", 0},
    {TK_OPTION_SYNONYM, "-bg", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-background", 0},
    {TK_OPTION_STRING, "-blabels", "blabels", "Blabels",
     DEF_BAR_LABEL, -1, Tk_Offset(Bargraph, blabelstr_u), 0, 0, 0},
    {TK_OPTION_FONT, "-blabfont", "blabFont", "BlabFont",
     DEF_BAR_FONT, -1, Tk_Offset(Bargraph, Btkfont), 0, 0, 0},
    {TK_OPTION_INT, "-borderwidth", "borderwidth", "Borderwidth",
     DEF_BAR_BORDER_WIDTH, -1, Tk_Offset(Bargraph, borderWidth), 0, 0, 0},
    {TK_OPTION_CURSOR, "-cursor", "cursor", "Cursor",
     DEF_BAR_CURSOR, -1, Tk_Offset(Bargraph, cursor), TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_INT, "-digits", "digits", "Digits",
     DEF_BAR_DIGITS, -1, Tk_Offset(Bargraph, digits), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-fg", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
    {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
     DEF_BAR_TEXT_COLOR, -1, Tk_Offset(Bargraph, textColorPtr),
     0, (ClientData) DEF_BAR_TEXT_MONO, 0},
    {TK_OPTION_FONT, "-font", "font", "Font",
     DEF_BAR_FONT, -1, Tk_Offset(Bargraph, tkfont), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-from", "from", "From",
     DEF_BAR_FROM, -1, Tk_Offset(Bargraph, fromVal_u), 0, 0, 0},
    {TK_OPTION_PIXELS, "-height", "height", "Height",
     DEF_BAR_LENGTH, -1, Tk_Offset(Bargraph, max_height), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightbackground", "highlightBackground",
     "HighlightBackground", HIGHLIGHT_BG, -1,
     Tk_Offset(Bargraph, highlightBgColorPtr), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
     HIGHLIGHT, -1, Tk_Offset(Bargraph, highlightColorPtr), 0, 0, 0},
    {TK_OPTION_PIXELS, "-highlightthickness", "highlightThickness",
     "HighlightThickness",
     HIGHLIGHT_WIDTH, -1, Tk_Offset(Bargraph, highlightWidth), 0, 0, 0},
    {TK_OPTION_STRING, "-label", "label", "Label",
     DEF_BAR_LABEL, -1, Tk_Offset(Bargraph, label), TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_STRING_TABLE, "-mode", "mode", "Mode",
     DEF_BAR_MODE, -1, Tk_Offset(Bargraph, mode),
     0, (ClientData) modeStrings, 0},
    {TK_OPTION_STRING_TABLE, "-orient", "orient", "Orient",
     DEF_BAR_ORIENT, -1, Tk_Offset(Bargraph, orient),
     0, (ClientData) orientStrings, 0},
    {TK_OPTION_PIXELS, "-padx", "padX", "Pad",
     DEF_BAR_PADX, -1, Tk_Offset(Bargraph, padX), 0, 0, 0},
    {TK_OPTION_PIXELS, "-pady", "padY", "Pad",
     DEF_BAR_PADY, -1, Tk_Offset(Bargraph, padY), 0, 0, 0},
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
     DEF_BAR_RELIEF, -1, Tk_Offset(Bargraph, relief), 0, 0, 0},
    {TK_OPTION_BOOLEAN, "-showrange", "showRange", "ShowRange",
     DEF_BAR_SHOW_RANGE, -1, Tk_Offset(Bargraph, showRange), 0, 0, 0},
    {TK_OPTION_BOOLEAN, "-showvalue", "showValue", "ShowValue",
     DEF_BAR_SHOW_VALUE, -1, Tk_Offset(Bargraph, showValue), 0, 0, 0},
    {TK_OPTION_PIXELS, "-sliderlength", "sliderlength", "Sliderlength",
     DEF_BAR_SLIDER_LENGTH, -1, Tk_Offset(Bargraph, sliderlength), 0, 0, 0},
    {TK_OPTION_STRING, "-takefocus", "takeFocus", "TakeFocus",
     DEF_BAR_TAKE_FOCUS, -1, Tk_Offset(Bargraph, takeFocus),
     TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_COLOR, "-tickcolor", "tickcolor", "Tickcolor",
     DEF_BAR_TICK_COLOR, -1, Tk_Offset(Bargraph, tickColorPtr),
     0, (ClientData) DEF_BAR_TICK_MONO, 0},
    {TK_OPTION_DOUBLE, "-tickinterval", "tickInterval", "TickInterval",
     DEF_BAR_TICK_INTERVAL, -1, Tk_Offset(Bargraph, tickInterval_u), 0, 0, 0},
    {TK_OPTION_PIXELS, "-ticklength", "tickLength", "TickLength",
     DEF_BAR_TICK_LENGTH, -1, Tk_Offset(Bargraph, tickLength), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-to", "to", "To",
     DEF_BAR_TO, -1, Tk_Offset(Bargraph, toVal_u), 0, 0, 0},
    {TK_OPTION_PIXELS, "-width", "width", "Width",
     DEF_BAR_WIDTH, -1, Tk_Offset(Bargraph, width), 0, 0, 0},

    /* These are maintained purely for 100% compatibility with scale
     * widget options, but are not guaranteed to translate 100% correctly */
    {TK_OPTION_SYNONYM, "-length", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-height", 0},

    {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
     (char *) NULL, 0, 0}
};

/*
 * The following tables define the widget commands (and sub-
 * commands) and map the indexes into the string tables into 
 * enumerated types used to dispatch the widget command.
 */

static CONST84 char *commandNames[] = {
    "cget", "configure", "get", "set", (char *) NULL
};

enum command {
    CMD_CGET, CMD_CONFIGURE, CMD_GET, CMD_SET
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int	BargraphWidgetObjCmd _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[]));
static void	DestroyBargraph _ANSI_ARGS_((char *clientData));
static int	ConfigureBargraph _ANSI_ARGS_((Tcl_Interp *interp,
	Bargraph *bgPtr, int argc, Tcl_Obj *CONST objv[], int flags));
static void	ComputeBargraphGeometry _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayLabel _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayValue _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayAlabels _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBlabels _ANSI_ARGS_((Bargraph *bgp));
static void	ComputeTicks _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayMinmax _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBaseV _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBar _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBargraph _ANSI_ARGS_((ClientData clientData));
static void	BargraphCmdDeletedProc _ANSI_ARGS_((ClientData clientData));
static void	EventuallyRedrawBargraph _ANSI_ARGS_((Bargraph *bgp,
	int what));
static void	BargraphEventProc _ANSI_ARGS_((ClientData clientData,
	XEvent *eventPtr));
static void	SetBargraphValue _ANSI_ARGS_((Bargraph *bgp, double value));
static int	ValueToPixel _ANSI_ARGS_((Bargraph *bgp, double val, int lim));
static void	format_double _ANSI_ARGS_((Bargraph *bgp,
	char *result, double v));
static double	limit_dvalue _ANSI_ARGS_((double minv,double maxv,double val));
static int	limit_ivalue _ANSI_ARGS_((int minv, int maxv, int val));

/*
 *--------------------------------------------------------------
 *
 * Tk_BargraphCmd --
 *
 *	This procedure is invoked to process the "Bargraph" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
int
Tk_BargraphObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* do not use */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    register Bargraph *bgPtr;
    Tk_OptionTable optionTable;
    Tk_Window tkwin;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pathName ?options?");
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp),
            Tcl_GetString(objv[1]), (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    /*
     * Create the option table for this widget class.  If it has already
     * been created, Tk will return the cached value.
     */

    optionTable = Tk_CreateOptionTable(interp, optionSpecs);

    /*
     * Initialize fields that won't be initialized by ConfigureBargraph,
     * or which ConfigureBargraph expects to have reasonable values
     * (e.g. resource pointers).
     */
    bgPtr = (Bargraph *) ckalloc(sizeof(Bargraph));
    memset((VOID *) bgPtr, 0, sizeof(Bargraph));

    bgPtr->tkwin	= tkwin;
    bgPtr->display	= Tk_Display(tkwin);
    bgPtr->interp	= interp;
    bgPtr->widgetCmd	= Tcl_CreateObjCommand(interp,
	    Tk_PathName(bgPtr->tkwin), BargraphWidgetObjCmd,
	    (ClientData) bgPtr, BargraphCmdDeletedProc);
    bgPtr->optionTable	= optionTable;

    bgPtr->cursor	= None;
    bgPtr->relief	= TK_RELIEF_RAISED;
    bgPtr->textGC	= None;
    bgPtr->tickGC	= None;
    bgPtr->alabGC	= None;
    bgPtr->blabGC	= None;
    bgPtr->mode		= MODE_BARGRAPH;

    Tk_SetClass(bgPtr->tkwin, "Bargraph");
    Tk_CreateEventHandler(bgPtr->tkwin,
	    ExposureMask|StructureNotifyMask|FocusChangeMask,
	    BargraphEventProc, (ClientData) bgPtr);

    if ((Tk_InitOptions(interp, (char *) bgPtr, optionTable, tkwin)
	    != TCL_OK) ||
	    (ConfigureBargraph(interp, bgPtr, objc-2, objv+2, 0) != TCL_OK)) {
	Tk_DestroyWindow(bgPtr->tkwin);
	return TCL_ERROR;
    }

    Tcl_SetStringObj(Tcl_GetObjResult(interp), Tk_PathName(bgPtr->tkwin), -1);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BargraphWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static int
BargraphWidgetObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Information about the widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    register Bargraph *bgPtr = (Bargraph *) clientData;
    int cmdIndex, result = TCL_OK;
    Tcl_Obj *objPtr, *resultPtr;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) bgPtr);

    /* parse the first parameter */
    result = Tcl_GetIndexFromObj(interp, objv[1], commandNames,
				 "option", 0, &cmdIndex);
    if (result != TCL_OK) {
	return result;
    }
    resultPtr = Tcl_GetObjResult(interp);

    switch (cmdIndex) {
	case CMD_CGET:
	    if (objc != 3) {
		Tcl_WrongNumArgs(interp, 2, objv, "option");
		result = TCL_ERROR;
	    } else {
		objPtr = Tk_GetOptionValue(interp, (char *) bgPtr,
			bgPtr->optionTable, objv[2],
			bgPtr->tkwin);
		if (objPtr == NULL) {
		    result = TCL_ERROR;
		} else {
		    Tcl_SetObjResult(interp, objPtr);
		}
	    }
	    break;

	case CMD_CONFIGURE:
	    if (objc <= 3) {
		objPtr = Tk_GetOptionInfo(interp, (char *) bgPtr,
			bgPtr->optionTable,
			(objc == 3) ? objv[2] : (Tcl_Obj *) NULL,
			bgPtr->tkwin);
		if (objPtr == NULL) {
		    result = TCL_ERROR;
		} else {
		    Tcl_SetObjResult(interp, objPtr);
		}
	    } else {
		result = ConfigureBargraph(interp, bgPtr, objc-2, objv+2, 0);
	    }
	    break;

	case CMD_GET:
	    if (objc != 2) {
		Tcl_WrongNumArgs(interp, 2, objv, (char *) NULL);
		result = TCL_ERROR;
	    } else {
		Tcl_SetDoubleObj(resultPtr, bgPtr->value);
	    }
	    break;

	case CMD_SET: {
	    double value;

	    if (objc != 3 && objc != 2) {
		Tcl_WrongNumArgs(interp, 1, objv, "set ?value?");
		result = TCL_ERROR;
		break;
	    } else if (objc == 3) {
		if (Tcl_GetDoubleFromObj(interp, objv[2], &value) != TCL_OK) {
		    result = TCL_ERROR;
		    break;
		}
		SetBargraphValue(bgPtr, value);
	    }
	    Tcl_SetDoubleObj(resultPtr, bgPtr->value);
	    break;
	}
    }

    Tcl_Release((ClientData) bgPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyBargraph --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a button at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the Bargraph is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyBargraph(clientData)
    char *clientData;	/* Info about Bargraph widget. */
{
    register Bargraph *bgPtr = (Bargraph *) clientData;

    if (bgPtr->textGC != None)
	Tk_FreeGC(bgPtr->display, bgPtr->textGC);

    if (bgPtr->alabGC != None)
	Tk_FreeGC(bgPtr->display, bgPtr->alabGC);

    if (bgPtr->blabGC != None)
	Tk_FreeGC(bgPtr->display, bgPtr->blabGC);

    if (bgPtr->tickGC != None)
	Tk_FreeGC(bgPtr->display, bgPtr->tickGC);

    if (bgPtr->ticks != NULL)
	ckfree((char *)bgPtr->ticks);

    if (bgPtr->alabelstr != NULL)
	ckfree(bgPtr->alabelstr);

    if (bgPtr->blabelstr != NULL)
	ckfree(bgPtr->blabelstr);

    if (bgPtr->alabels != NULL)
	Tcl_Free((char *) bgPtr->alabels);

    if (bgPtr->blabels != NULL)
	Tcl_Free((char *) bgPtr->blabels);

    Tk_FreeConfigOptions((char *) bgPtr, bgPtr->optionTable, bgPtr->tkwin);
    bgPtr->tkwin = NULL;
    Tcl_EventuallyFree((ClientData) bgPtr, TCL_DYNAMIC);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureBargraph --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a Bargraph widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for bgPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureBargraph(interp, bgPtr, objc, objv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    register Bargraph *bgPtr;	/* Information about widget;  may or may
				 * not already have values for some fields */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    Tk_SavedOptions savedOptions;
    XGCValues gcValues;
    GC newGC;
    int error;
    unsigned int length;
    Tcl_Obj *errorResult = NULL;

    for (error = 0; error <= 1; error++) {
	if (!error) {
	    /*
	     * First pass: set options to new values.
	     */

	    if (Tk_SetOptions(interp, (char *) bgPtr,
		    bgPtr->optionTable, objc, objv,
		    bgPtr->tkwin, &savedOptions, (int *) NULL) != TCL_OK) {
		continue;
	    }
	} else {
	    /*
	     * Second pass: restore options to old values.
	     */

	    errorResult = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(errorResult);
	    Tk_RestoreSavedOptions(&savedOptions);
	}
	break;
    }
    if (!error) {
	Tk_FreeSavedOptions(&savedOptions);
    }
    if (error) {
	Tcl_SetObjResult(interp, errorResult);
	Tcl_DecrRefCount(errorResult);
	return TCL_ERROR;
    }

    /*
     * A few options need special processing, such as parsing the
     * orientation or setting the background from a 3-D border.
     */

    /*
     * Check for user-entered complex labels.
     * This is a list of values; each element is {value text ...}
     */
    if (!(bgPtr->alabelstr) || strcmp(bgPtr->alabelstr, bgPtr->alabelstr_u)) {
	/* There is a new value */
	length = strlen(bgPtr->alabelstr_u);
	if (bgPtr->alabelstr != NULL) {
	    ckfree(bgPtr->alabelstr);
	    bgPtr->alabelstr = NULL;
	}
	bgPtr->alabelstr = (char *) ckalloc(length+1);
	strcpy(bgPtr->alabelstr, bgPtr->alabelstr_u);
	if (bgPtr->alabels != NULL) {
	    Tcl_Free((char *) bgPtr->alabels);
	    bgPtr->alabels = NULL;
	}
	if (Tcl_SplitList(interp, bgPtr->alabelstr, &bgPtr->n_alabels,
		(CONST84 char ***) &bgPtr->alabels) == TCL_ERROR) {
	    Tcl_AppendResult(interp, "-alabels argument is not a proper list",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (!(bgPtr->blabelstr) || strcmp(bgPtr->blabelstr, bgPtr->blabelstr_u)) {
	/* There is a new value */
	length = strlen(bgPtr->blabelstr_u);
	if (bgPtr->blabelstr != NULL) {
	    ckfree(bgPtr->blabelstr);
	    bgPtr->blabelstr = NULL;
	}
	bgPtr->blabelstr = (char *) ckalloc(length+1);
	strcpy(bgPtr->blabelstr, bgPtr->blabelstr_u);
	if (bgPtr->blabels != NULL) {
	    Tcl_Free((char *) bgPtr->blabels);
	    bgPtr->blabels = NULL;
	}
	if (Tcl_SplitList(interp, bgPtr->blabelstr, &bgPtr->n_blabels,
		(CONST84 char ***) &bgPtr->blabels) == TCL_ERROR) {
	    Tcl_AppendResult(interp, "-blabels argument is not a proper list",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    }


    /*
     * Make sure that the tick interval has the right sign so that
     * addition moves from fromVal to toVal.
     */
    bgPtr->tickInterval = fabs(bgPtr->tickInterval);

    /*
     * Make sure that the number of ticks isn't excessive: not more
     * than height/2 ticks.
     */
    if (bgPtr->tickInterval_u > 0.) {
	if ( 2.*(fabs(bgPtr->toVal_u - bgPtr->fromVal_u) /
		bgPtr->tickInterval_u) > bgPtr->max_height) {
	    Tcl_AppendResult(interp,
		    "Bad parameter combination: ",
		    "must have ((max-min)/tickInterval) < height/2",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    }
    bgPtr->toVal = bgPtr->toVal_u;
    bgPtr->fromVal = bgPtr->fromVal_u;
    /* Ensure that fromVal and toVal are never equal */
    if (bgPtr->toVal == bgPtr->fromVal) {
	bgPtr->toVal = 1.01 * bgPtr->fromVal;
	if (bgPtr->toVal == bgPtr->fromVal)
	    bgPtr->toVal = 1.;
    }
    bgPtr->tickInterval = bgPtr->tickInterval_u;

    /*
     * Set the Bargraph value to itself;  all this does is to make sure
     * that the Bargraph's value is within the new acceptable range for
     * the Bargraph.
     */
    SetBargraphValue(bgPtr, bgPtr->value);

    Tk_SetBackgroundFromBorder(bgPtr->tkwin, bgPtr->border);

    gcValues.font = Tk_FontId(bgPtr->tkfont);
    gcValues.foreground = bgPtr->textColorPtr->pixel;
    newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
    if (bgPtr->textGC != None) {
	Tk_FreeGC(bgPtr->display, bgPtr->textGC);
    }
    bgPtr->textGC = newGC;

    gcValues.font = Tk_FontId(bgPtr->Atkfont);
    gcValues.foreground = bgPtr->textColorPtr->pixel;
    newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
    if (bgPtr->alabGC != None) {
	Tk_FreeGC(bgPtr->display, bgPtr->alabGC);
    }
    bgPtr->alabGC = newGC;

    gcValues.font = Tk_FontId(bgPtr->Btkfont);
    gcValues.foreground = bgPtr->textColorPtr->pixel;
    newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
    if (bgPtr->blabGC != None) {
	Tk_FreeGC(bgPtr->display, bgPtr->blabGC);
    }
    bgPtr->blabGC = newGC;

    gcValues.foreground = bgPtr->tickColorPtr->pixel;
    newGC = Tk_GetGC(bgPtr->tkwin, GCForeground, &gcValues);
    if (bgPtr->tickGC != None) {
	Tk_FreeGC(bgPtr->display, bgPtr->tickGC);
    }
    bgPtr->tickGC = newGC;
    /*
     * Recompute display-related information, and let the geometry
     * manager know how much space is needed now.
     */
    ComputeBargraphGeometry(bgPtr);

    EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeBargraphGeometry --
 *
 *	This procedure is called to compute various geometrical
 *	information for a Bargraph, such as where various things get
 *	displayed.  It's called when the window is reconfigured.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Display-related numbers get changed in *scrollPtr.  The
 *	geometry manager gets told about the window's preferred size.
 *
 *----------------------------------------------------------------------
 */
static void
ComputeBargraphGeometry(bgPtr)
    register Bargraph *bgPtr;		/* Information about widget. */
{
    char valueString[30];
    int bd, i, l, boolv;
    int minvpixels, maxvpixels;
    int win_width, win_height;
    int a_has_ticks=0;
    int has_ticks=0;
    Tk_FontMetrics fm, afm, bfm;
    char *s;
    CONST84 char **elt;
    int nelt;
    Tcl_Interp *interp = bgPtr->interp;

    Tk_GetFontMetrics(bgPtr->tkfont, &fm);
    Tk_GetFontMetrics(bgPtr->Atkfont, &afm);
    Tk_GetFontMetrics(bgPtr->Btkfont, &bfm);

    /* Ensure that the base value is between the minimum and maximum values */
    bgPtr->baseVal = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, bgPtr->baseVal);

    /*
     * Calculate text extents for the value.
     * This value is used when bargraph is horizontal or vertical
     */
    format_double(bgPtr, valueString, bgPtr->fromVal);
    minvpixels = Tk_TextWidth(bgPtr->tkfont, valueString,
	    (int) strlen(valueString));
    format_double(bgPtr, valueString, bgPtr->toVal);
    maxvpixels = Tk_TextWidth(bgPtr->tkfont, valueString,
	    (int) strlen(valueString));
    if (bgPtr->showValue) {
	bgPtr->valuePixels = MAX(minvpixels, maxvpixels);
    } else {
	bgPtr->valuePixels = 0;
    }

    /* Check for space required by complex labels */
    bgPtr->alabelPixels = 0;
    for (l=0, i=0; i<bgPtr->n_alabels; i++) {
	s = bgPtr->alabels[i];
	if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
	    /* ignore this value */
	} else if (nelt >= 2) {
	    l = Tk_TextWidth(bgPtr->Atkfont, elt[1],
		    (int) strlen(elt[1]));
	    Tcl_Free((char *) elt);
	    if (!a_has_ticks && nelt > 2) {
		a_has_ticks = ((Tcl_GetBoolean(interp, elt[2], &boolv)
			== TCL_OK) && (boolv));
	    }
	}
	if (bgPtr->alabelPixels < l) {
	    bgPtr->alabelPixels = l;
	}
    }
    bgPtr->blabelPixels = 0;
    for (l=0,i=0; i<bgPtr->n_blabels; i++) {
	s = bgPtr->blabels[i];
	if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
	    /* ignore this value */
	} else if (nelt >= 2) {
	    l = Tk_TextWidth(bgPtr->Btkfont, elt[1],
		    (int) strlen(elt[1]));
	    Tcl_Free((char *) elt);
	    if (!has_ticks && nelt > 2) {
		has_ticks = ((Tcl_GetBoolean(interp, elt[2], &boolv) == TCL_OK)
			&& boolv);
	    }
	}
	if (bgPtr->blabelPixels < l)
	    bgPtr->blabelPixels = l;
    }
    bd = bgPtr->borderWidth;

    if (bgPtr->orient != ORIENT_VERTICAL) {
	/* HORIZONTAL */
	/*
	 *	 -------------------------------------------
	 *	 |\_______________________________________/|
	 *	 | |	    Label  (optional)		 | |
	 *	 | |-------------------------------------| |
	 *	 | |	   min	 (optional)	max	 | |
	 *	 | |-------------------------------------| |
	 *	 | |	    alabels(optional)		 | |
	 *	 | |-------------------------------------| |
	 *	 | |	   |	  Ticks (opt)		 | |
	 *	 | | Value | #####################	 | |
	 *	 | | (opt) | #####################	 | |
	 *	 | |-------------------------------------| |
	 *	 | |	    blabels(optional)		 | |
	 *	 |/---------------------------------------\|
	 *	  ------------------------------------------
	 *			^-- BorderWidth
	 */

	if (bgPtr->tickInterval != 0.)
	    bgPtr->minmaxPixels = fm.linespace + bgPtr->padY;
	else
	    bgPtr->minmaxPixels = 0;

	if (fm.linespace > bgPtr->width )
	    bgPtr->width = fm.linespace;

	/* Left edge of bar area. */
	bgPtr->offset.bot_edge = bd + bgPtr->padX +
	    MAX(bgPtr->showValue*(bgPtr->valuePixels + bgPtr->padX),
		    SHOW_RANGE(bgPtr)*minvpixels/2);

	/* Right edge of bar area */
	bgPtr->offset.top_edge = bgPtr->offset.bot_edge + bgPtr->max_height;

	/* Base value position */
	bgPtr->offset.base = ValueToPixel(bgPtr, bgPtr->baseVal, 1);

	/* Top of label string */
	bgPtr->offset.label_y = bd;

	/* Right edge of minmax "max" value. */
	bgPtr->offset.minmax1_x = bgPtr->offset.top_edge + maxvpixels/2;

	/* Left edge of minmax "min" value. */
	bgPtr->offset.minmax0_x = bgPtr->offset.bot_edge - minvpixels/2;

	/* Top edge of minmax "max" and "min" values. */
	bgPtr->offset.minmax1_y = 
	    bgPtr->offset.minmax0_y = bgPtr->offset.label_y +
	    (bgPtr->label != NULL)*(fm.linespace + 2*bgPtr->padY);

	/* Top edge of alabels.  Note that we assume the minmax values
	 * will be printed in a box that doesn't include space for
	 * descenders, since numbers have none. */
	bgPtr->offset.lab_a = bgPtr->offset.minmax0_y + 
	    SHOW_RANGE_OR_BASEV(bgPtr)*(fm.ascent + bgPtr->padY);

	/* Top edge of upper ticks */
	bgPtr->offset.tick_a = bgPtr->offset.lab_a +
	    HAS_ALAB(bgPtr)*(afm.linespace+bgPtr->padY +
		    a_has_ticks*bgPtr->tickLength);

	/* Top edge of bar */
	bgPtr->offset.bar_y = bgPtr->offset.tick_a +
	    (SHOW_TICKS(bgPtr) || HAS_ALAB(bgPtr))*bgPtr->tickLength;

	/* Top edge of value string */
	bgPtr->offset.value_y = bgPtr->offset.bar_y +
	    (bgPtr->width - fm.linespace)/2;

	/* Top edge of lower ticks */
	bgPtr->offset.tick_b = bgPtr->offset.bar_y + bgPtr->width;

	/* Top edge of blabels */
	bgPtr->offset.lab_b = bgPtr->offset.tick_b +
	    HAS_BLAB(bgPtr)*(bgPtr->tickLength + bgPtr->padY +
		    has_ticks*bgPtr->tickLength);

	win_width = (SHOW_RANGE(bgPtr) ?
		bgPtr->offset.minmax1_x : bgPtr->offset.top_edge) +
	    bgPtr->borderWidth;

	win_height = bgPtr->offset.lab_b +
	    HAS_BLAB(bgPtr)*(bfm.linespace + bgPtr->padY) +
	    bgPtr->borderWidth;
    } else {
	/* the bargraph is vertical */
	/*
	 *	________________________________________
	 *	|\____________________________________/|
	 *	| |	       Label (opt)	     | |
	 *	| |----------------------------------| |
	 *	| |		      |		     | |
	 *	| | MAX(opt)	      |	  ###	     | |
	 *	| |		      |	  ###	     | |
	 *	| |	 Alabs	 Ticks|	  ###  Blabs | |
	 *	| |	 (opt)	 (opt)|	  ###  (opt) | |
	 *	| |		      |	  ###	     | |
	 *	| |		      |	  ###	     | |
	 *	| | MIN(opt)	      |	  ###	     | |
	 *	| |----------------------------------| |
	 *	| |	       Value (opt)	     | |
	 *	|/------------------------------------\|
	 *	----------------------------------------
	 *	     ^-- borderWidth
	 */

	/* Calculate the extents for the ticks text */
	if ( SHOW_RANGE_OR_BASEV(bgPtr) ) {
	    format_double(bgPtr, valueString, bgPtr->fromVal);
	    bgPtr->minmaxPixels = Tk_TextWidth(bgPtr->tkfont,
		    valueString, (int) strlen(valueString));

	    format_double(bgPtr, valueString, bgPtr->toVal);
	    l = Tk_TextWidth(bgPtr->tkfont, valueString,
		    (int) strlen(valueString));
	    if (bgPtr->minmaxPixels < l)
		bgPtr->minmaxPixels = l;

	    bgPtr->minmaxPixels += bgPtr->padX;
	} else {
	    bgPtr->minmaxPixels = 0;
	}

	/*
	 * Value is always displayed directly below the bar, so the bar needs
	 * to be wide enough so that the value text isn't clipped.
	 */
	bgPtr->width = MAX(bgPtr->width, bgPtr->valuePixels);

	/* Top of label string */
	bgPtr->offset.label_y = bd;

	/*
	 * Top of bar area
	 * If we have a label, make space for that.  If we have range values
	 * showing, make space for the top half of it.
	 */
	bgPtr->offset.top_edge = bgPtr->offset.label_y;
	if (bgPtr->label) {
	    bgPtr->offset.top_edge += fm.linespace + 2*bgPtr->padY;
	} else if (SHOW_RANGE_OR_BASEV(bgPtr)) {
	    bgPtr->offset.top_edge += fm.linespace/2 + bgPtr->padY;
	}

	/* Bottom of bar area */
	bgPtr->offset.bot_edge = bgPtr->offset.top_edge + bgPtr->max_height;

	/* base value */
	bgPtr->offset.base = ValueToPixel(bgPtr, bgPtr->baseVal, 1);

	/* Left edge of minmax "max" and "min" values. */
	bgPtr->offset.minmax1_x =
	    bgPtr->offset.minmax0_x = bd + bgPtr->padX;

	/* Baseline of minmax "max" value */
	bgPtr->offset.minmax1_y = bgPtr->offset.top_edge + fm.ascent/2;

	/* Baseline of minmax "min" value */
	bgPtr->offset.minmax0_y = bgPtr->offset.bot_edge + fm.ascent/2;

	/* Left edge of alabels */
	bgPtr->offset.lab_a = bd + bgPtr->padX +
	    SHOW_RANGE_OR_BASEV(bgPtr)*(bgPtr->minmaxPixels + bgPtr->padX);

	/* Left edge of left-side ticks */
	bgPtr->offset.tick_a = bgPtr->offset.lab_a +
	    HAS_ALAB(bgPtr)*(bgPtr->alabelPixels + bgPtr->padX);

	/* Left edge of bar */
	bgPtr->offset.bar_x = bgPtr->offset.tick_a +
	    (SHOW_TICKS(bgPtr) || a_has_ticks)*bgPtr->tickLength;

	/* left/right center of the value string */
	bgPtr->offset.value_x = bgPtr->offset.bar_x + bgPtr->width/2;

	/* Top of the value string */
	bgPtr->offset.value_y = bgPtr->offset.bot_edge + bgPtr->padY/2;

	/* Left edge of the right-side ticks */
	bgPtr->offset.tick_b = bgPtr->offset.bar_x + bgPtr->width;

	/* Left edge of the blabel text */
	bgPtr->offset.lab_b = bgPtr->offset.tick_b +
	    has_ticks*(bgPtr->tickLength + bgPtr->padX);

	win_width = bgPtr->offset.lab_b +
	    HAS_BLAB(bgPtr)*(bgPtr->blabelPixels + bgPtr->padX) + bd;

	win_height = (bgPtr->showValue ? 
		(bgPtr->offset.value_y + fm.linespace) :
		(SHOW_RANGE_OR_BASEV(bgPtr) ?
			(bgPtr->offset.bot_edge + fm.linespace/2) :
			bgPtr->offset.bot_edge))
	    + bgPtr->padY/2 + bd;
    }

    Tk_GeometryRequest(bgPtr->tkwin, win_width, win_height);
    Tk_SetInternalBorder(bgPtr->tkwin, bgPtr->borderWidth);
}

/*
 * Display the window's label.
 */
static void
DisplayLabel(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    int bd = bgPtr->borderWidth;
    int win_width = Tk_Width(tkwin);
    int x, width;
    Tk_FontMetrics fm;

    if (bgPtr->label == NULL) return;

    Tk_GetFontMetrics(bgPtr->tkfont, &fm);

    width = Tk_TextWidth(bgPtr->tkfont, bgPtr->label,
	    (int) strlen(bgPtr->label));
    if (width < win_width - 2*bd)
	x = (win_width - width)/2;
    else
	x = bd + bgPtr->padX;

    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
	    bd, bgPtr->offset.label_y + bgPtr->padY,
	    win_width - 2*bd, fm.linespace + bgPtr->padY,
	    0, TK_RELIEF_FLAT);
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
	    bgPtr->textGC, bgPtr->tkfont,
	    bgPtr->label, (int) strlen(bgPtr->label), x,
	    bgPtr->offset.label_y + fm.ascent);
}

/*
 * Display the value
 */
static void
DisplayValue(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    int twidth, bd = bgPtr->borderWidth;
    char valueString[40];
    Tk_FontMetrics fm;

    Tk_GetFontMetrics(bgPtr->tkfont, &fm);

    format_double(bgPtr, valueString, bgPtr->value);
    twidth = Tk_TextWidth(bgPtr->tkfont, valueString,
	    (int) strlen(valueString));

    if (bgPtr->orient == ORIENT_VERTICAL) {
	/* Display the value, centered below the bar.  */
	/* Note: if you change (padY/2) + ... + (padY/2) to be
	 * (padY + ...), then you'll overshoot by 1 pixel in the downward
	 * direction when padY is odd.
	 */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		bgPtr->offset.value_x - bgPtr->valuePixels/2,
		bgPtr->offset.value_y - bgPtr->padY/2,
		bgPtr->valuePixels,
		bgPtr->padY/2 + fm.ascent + bgPtr->padY/2,
		0, TK_RELIEF_FLAT);
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
		bgPtr->tkfont, valueString, (int) strlen(valueString),
		bgPtr->offset.value_x - twidth/2,
		bgPtr->offset.value_y + fm.linespace);
    } else {
	/* HORIZONTAL */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		bd + bgPtr->padX, bgPtr->offset.value_y,
		bgPtr->valuePixels,
		fm.linespace + bgPtr->padY,
		0, TK_RELIEF_FLAT);
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
		bgPtr->tkfont, valueString, (int) strlen(valueString),
		bd + bgPtr->padX + bgPtr->valuePixels - twidth,
		bgPtr->offset.value_y + fm.linespace);
    }
}

/*
 * Display the alabels
 * Note that ticks are "scheduled" for drawing, but not yet drawn.
 */
static void
DisplayAlabels(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    int boolv, awidth;

    CONST84 char *s, **elt;
    int i, nelt, tickoff;
    double d;
    Tcl_Interp *interp = bgPtr->interp;
    Tk_FontMetrics afm;

    Tk_GetFontMetrics(bgPtr->Atkfont, &afm);

    for (i = 0; i < bgPtr->n_alabels; i++) {
	s = bgPtr->alabels[i];
	tickoff = 0;
	if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
	    /* ignore this value */
	} else if (nelt >= 2) {
	    d = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, strtod(elt[0], NULL));
	    s = elt[1];
	    awidth = Tk_TextWidth(bgPtr->Atkfont, s, (int) strlen(s));
	    if (bgPtr->orient == ORIENT_VERTICAL) {
		Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
			bgPtr->alabGC, bgPtr->Atkfont, s, (int) strlen(s),
			bgPtr->offset.lab_a + bgPtr->alabelPixels - awidth,
			ValueToPixel(bgPtr, d, 1) + afm.descent);
		if (nelt > 2) {
		    /* Draw a tick if flag is true. (Ignore errors in value). */ 
		    boolv = 0;
		    Tcl_GetBoolean(interp, elt[2], &boolv);
		    if (boolv) {
			/* Draw a tick */
			bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
			bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a +
			    bgPtr->tickLength;
			bgPtr->ticks[bgPtr->numTicks].y1 =
			    bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, d, 1);
			bgPtr->numTicks++;
		    }
		}
	    } else {
		/* HORIZONTAL */
		Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
			bgPtr->alabGC, bgPtr->Atkfont, s, (int) strlen(s),
			ValueToPixel(bgPtr, d, 1) - awidth/2,
			bgPtr->offset.lab_a + afm.ascent);
		if (nelt > 2) {
		    /* Draw a tick if flag is true. (Ignore errors in value). */ 
		    boolv = 0;
		    Tcl_GetBoolean(interp, elt[2], &boolv);
		    if (boolv) {
			bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
			bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + 
			    bgPtr->tickLength;
			bgPtr->ticks[bgPtr->numTicks].x1 =
			    bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, d, 1);
			bgPtr->numTicks++;
		    }
		}
	    }
	    Tcl_Free((char *) elt);
	}
    }
}

/*
 * Display the blabels
 * Note that ticks are "scheduled" for drawing, but not yet drawn.
 */
static void
DisplayBlabels(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    int boolv, bwidth;
    CONST84 char *s, **elt;
    int i, nelt, tickoff;
    double d;
    Tcl_Interp *interp = bgPtr->interp;
    Tk_FontMetrics bfm;

    Tk_GetFontMetrics(bgPtr->Btkfont, &bfm);

    for (i = 0; i < bgPtr->n_blabels; i++) {
	s = bgPtr->blabels[i];
	tickoff = 0;
	if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
	    /* ignore this value */
	} else if (nelt >= 2) {
	    d = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, strtod(elt[0], NULL));
	    s = elt[1];
	    bwidth = Tk_TextWidth(bgPtr->Btkfont, s, (int) strlen(s));
	    if (bgPtr->orient == ORIENT_VERTICAL) {
		Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
			bgPtr->blabGC, bgPtr->Btkfont, s, (int) strlen(s),
			bgPtr->offset.lab_b,
			ValueToPixel(bgPtr, d, 1) + bfm.descent);
		if (nelt > 2) {
		    /* Draw a tick if flag is true. (Ignore errors in value). */ 
		    boolv = 0;
		    Tcl_GetBoolean(interp, elt[2], &boolv);
		    if (boolv) {
			/* Draw a tick */
			bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_b;
			bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_b +
			    bgPtr->tickLength;
			bgPtr->ticks[bgPtr->numTicks].y1 =
			    bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, d, 1);
			bgPtr->numTicks++;
		    }
		}
	    } else {
		/* HORIZONTAL */
		Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
			bgPtr->blabGC, bgPtr->Btkfont, s, (int) strlen(s),
			ValueToPixel(bgPtr, d, 1) - bwidth/2,
			bgPtr->offset.lab_b + bfm.ascent);
		if (nelt > 2) {
		    /* Draw a tick if flag is true. (Ignore errors in value). */ 
		    boolv = 0;
		    Tcl_GetBoolean(interp, elt[2], &boolv);
		    if (boolv) {
			bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_b;
			bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_b + 
			    bgPtr->tickLength;
			bgPtr->ticks[bgPtr->numTicks].x1 =
			    bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, d, 1);
			bgPtr->numTicks++;
		    }
		}
	    }
	    Tcl_Free((char *) elt);
	}
    }
}

/*
 * compute the ticks
 */
static void
ComputeTicks(bgPtr)
    Bargraph *bgPtr;
{
    double v, start, end;

    start = MIN(bgPtr->fromVal, bgPtr->toVal);
    end = MAX(bgPtr->fromVal, bgPtr->toVal);

    if (bgPtr->orient == ORIENT_VERTICAL) {
	for ( v = bgPtr->baseVal; v >= start;
	      v-=bgPtr->tickInterval, bgPtr->numTicks++ ) {
	    bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a + bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].y1 =
		bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, v, 1);
	}

	for ( v = bgPtr->baseVal+bgPtr->tickInterval; v <= end;
	      v+=bgPtr->tickInterval, bgPtr->numTicks++ ) {
	    bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a + bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].y1 =
		bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, v, 1);
	}
    } else {
	for ( v = bgPtr->baseVal; v >= start;
	      v-=bgPtr->tickInterval, bgPtr->numTicks++ ) {
	    bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].x1 =
		bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, v, 1);
	}

	for ( v = bgPtr->baseVal+bgPtr->tickInterval; v <= end;
	      v+=bgPtr->tickInterval, bgPtr->numTicks++ ) {
	    bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].x1 =
		bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, v, 1);
	}
    }
}

/*
 * Display the min and max values
 */
static void
DisplayMinmax(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    char valueString[40];
    int width, len, x, y;
    Tk_FontMetrics fm;
    Tk_GetFontMetrics(bgPtr->tkfont, &fm);

    format_double(bgPtr, valueString, bgPtr->fromVal);
    len = strlen(valueString);
    width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

    if (bgPtr->orient == ORIENT_VERTICAL) {
	x = bgPtr->offset.minmax0_x +
	    SHOW_RANGE_OR_BASEV(bgPtr)*bgPtr->minmaxPixels - width;
	y = bgPtr->offset.minmax0_y;
    } else {
	x = bgPtr->offset.minmax0_x;
	y = bgPtr->offset.minmax0_y + fm.ascent;
    }
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	    bgPtr->tkfont, valueString, len, x, y);

    format_double(bgPtr, valueString, bgPtr->toVal);
    len = strlen(valueString);
    width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

    if (bgPtr->orient == ORIENT_VERTICAL) {
	x = bgPtr->offset.minmax1_x +
	    SHOW_RANGE_OR_BASEV(bgPtr)*bgPtr->minmaxPixels - width;
	y = bgPtr->offset.minmax1_y;
    } else {
	x = bgPtr->offset.minmax1_x - width;
	y = bgPtr->offset.minmax1_y + fm.ascent;
    }
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	    bgPtr->tkfont, valueString, len, x, y);
}

/*
 * Display the base value
 */
static void
DisplayBaseV(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    double basev = bgPtr->baseVal;
    char valueString[40];
    int width, len, x, y;

    format_double(bgPtr, valueString, basev);
    len = strlen(valueString);
    width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

    if (bgPtr->orient == ORIENT_VERTICAL) {
	Tk_FontMetrics fm;
	Tk_GetFontMetrics(bgPtr->tkfont, &fm);
	x = bgPtr->offset.lab_a + HAS_ALAB(bgPtr)*bgPtr->alabelPixels - width;
	y = ValueToPixel(bgPtr, basev, 1) + fm.descent;
    } else {
	x = ValueToPixel(bgPtr, basev, 1) - width/2;
	y = bgPtr->offset.lab_a;
    }
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	    bgPtr->tkfont, valueString, len, x, y);
}

/*
 * Display the bar.
 */
static void
DisplayBar(bgPtr)
    Bargraph *bgPtr;
{
    register Tk_Window tkwin = bgPtr->tkwin;
    int length, y, a, b;
    int barBorderWidth = bgPtr->barBorderWidth;

    if (bgPtr->mode == MODE_SLIDER) {
	/* In order to get a central line, draw this as two half-bars
	 * around the value. Ensure the bar has a !0 borderwidth so that
	 * the value is visible.
	 */
	if (barBorderWidth == 0)
	    barBorderWidth++;
	length = bgPtr->sliderlength/2;
	y = ValueToPixel(bgPtr, bgPtr->value, 0);
	a = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y - length);
	b = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y + length);
	y = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y);
    } else {
	/* Compute ValueToPixel for endpoints, then difference these.
	 * This will ensure that the length matches to the tick marks,
	 * instead of sometimes generating off-by-one errors.
	 */
	a = ValueToPixel( bgPtr, bgPtr->baseVal, 1);
	b = ValueToPixel( bgPtr, bgPtr->value, 1);
	if (a < b) {
	    y = a;
	    length = b + 1 - a;
	} else {
	    y = b;
	    length = a + 1 - b;
	}
    }

    if (bgPtr->orient == ORIENT_VERTICAL) {
	/* Erase entire bar region */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		bgPtr->offset.bar_x,				/* x */
		bgPtr->offset.bot_edge - bgPtr->max_height,	/* y */
		bgPtr->width,					/* width */
		bgPtr->max_height,				/* height */
		0, TK_RELIEF_FLAT);

	/* Draw bar background */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		bgPtr->barBorder, 
		bgPtr->offset.bar_x,				/* x */
		bgPtr->offset.bot_edge - bgPtr->max_height,	/* y */
		bgPtr->width,					/* width */
		bgPtr->max_height,				/* height */
		bgPtr->barBorderWidth, bgPtr->barRelief );

	/* Draw bar */
	if (bgPtr->mode == MODE_SLIDER) {
	    /*
	     * Two bars: a->y, y->b
	     */
	    if (y > a) {
		Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
			bgPtr->barColor,
			bgPtr->offset.bar_x,			/* x */
			a,					/* y */
			bgPtr->width,				/* width */
			y + 1 - a,				/* height */
			barBorderWidth, bgPtr->barRelief );
	    }
	    if (b > y) {
		Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
			bgPtr->barColor, 
			bgPtr->offset.bar_x,			/* x */
			y,					/* y */
			bgPtr->width,				/* width */
			b + 1 - y,				/* height */
			barBorderWidth, bgPtr->barRelief );
	    }
	} else {
	    /*
	     * Bar from y, length "length".
	     */
	    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		    bgPtr->barColor, 
		    bgPtr->offset.bar_x,			/* x */
		    y,						/* y */
		    bgPtr->width,				/* width */
		    length,					/* height */
		    barBorderWidth, bgPtr->barRelief );
	}

    } else {
	/* HORIZONTAL */

	/* Erase entire bar region */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		bgPtr->offset.bot_edge,				/* x */
		bgPtr->offset.bar_y,				/* y */
		bgPtr->max_height,				/* width */
		bgPtr->width,					/* height */
		0, TK_RELIEF_FLAT);

	/* Draw bar background */
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		bgPtr->barBorder, 
		bgPtr->offset.bot_edge,				/* x */
		bgPtr->offset.bar_y,				/* y */
		bgPtr->max_height,				/* width */
		bgPtr->width,					/* height */
		bgPtr->barBorderWidth, bgPtr->barRelief );

	/* Draw bar */
	if (bgPtr->mode == MODE_SLIDER) {
	    /*
	     * Two bars: a->y, y->b
	     */
	    if (y > a) {
		Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
			bgPtr->barColor, 
			a,					/* x */
			bgPtr->offset.bar_y,			/* y */
			y + 1 - a,				/* width */
			bgPtr->width,				/* height */
			barBorderWidth, bgPtr->barRelief );
	    }
	    if (b > y) {
		Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
			bgPtr->barColor, 
			y,					/* x */
			bgPtr->offset.bar_y,			/* y */
			b + 1 - y,				/* width */
			bgPtr->width,				/* height */
			barBorderWidth, bgPtr->barRelief );
	    }

	} else {
	    /*
	     * Bar from y, length "length".
	     */
	    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		    bgPtr->barColor, 
		    y,						/* x */
		    bgPtr->offset.bar_y,			/* y */
		    length,					/* width */
		    bgPtr->width,				/* height */
		    barBorderWidth, bgPtr->barRelief );
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * DisplayBargraph --
 *
 *	This procedure redraws the contents of a vert. or hor. Bargraph
 *	window.	 It is invoked as a do-when-idle handler, so it only
 *	runs when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */
static void
DisplayBargraph(clientData)
    ClientData clientData;	/* Information about widget. */
{
    register Bargraph *bgPtr = (Bargraph *) clientData;
    register Tk_Window tkwin = bgPtr->tkwin;
    double basev = bgPtr->baseVal;
    double minv = bgPtr->fromVal;
    double maxv = bgPtr->toVal;
    int max_ticks;
    double pixel_interval;

    /* When we are done, there will be no redraw pending; clear that bit. */
    bgPtr->flags &= ~REDRAW_PENDING;
    if ((bgPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return;
    }

    if (bgPtr->flags & CLEAR_NEEDED) {
	XFillRectangle(bgPtr->display, Tk_WindowId(bgPtr->tkwin),
		Tk_3DBorderGC(tkwin, bgPtr->border, TK_3D_FLAT_GC), 0, 0,
		(unsigned int)Tk_Width(tkwin), (unsigned int)Tk_Height(tkwin));
    }

    /*
     * Display the label, centered in the window if there is enough space.
     * Otherwise left justified and clipped on the right.
     */
    if (bgPtr->label && (bgPtr->flags & DISPLAY_LABEL))
	DisplayLabel(bgPtr);

    /*
     *  Display the value, right justified to the left of the bar and centered
     *  within the bar.
     */
    if (bgPtr->showValue && (bgPtr->flags & DISPLAY_VALUE))
	DisplayValue(bgPtr);

    bgPtr->numTicks = 0;
    if ( bgPtr->tickInterval != 0.) {
	pixel_interval = bgPtr->max_height *
	    bgPtr->tickInterval /
	    ((minv < maxv) ? (maxv-minv) : (minv-maxv));
	max_ticks = bgPtr->max_height / (int)pixel_interval + 1 +
	    bgPtr->n_alabels +	/* in case there are -tick'd -alabels */
	    bgPtr->n_blabels;	/* in case there are -tick'd -blabels */
    } else {
	max_ticks = 1 +
	    bgPtr->n_alabels +	/* in case there are -tick'd -alabels */
	    bgPtr->n_blabels;	/* in case there are -tick'd -blabels */
    }
    if ((bgPtr->flags & DISPLAY_LABELS)) {
	if (bgPtr->ticks != NULL) ckfree((char *)bgPtr->ticks);
	bgPtr->ticks = (XSegment *) ckalloc( (max_ticks+2) * sizeof(XSegment));
    }

    /*
     * draw alabels, blabels.  Ticks for same are computed but not yet drawn.
     */
    if (HAS_ALAB(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
	DisplayAlabels(bgPtr);

    if (HAS_BLAB(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
	DisplayBlabels(bgPtr);

    /* Compute the ticks */
    if (SHOW_TICKS(bgPtr) && (bgPtr->flags & DISPLAY_TICKS))
	ComputeTicks(bgPtr);

#ifndef WIN32
    /* If any ticks were created, draw them.  */
    if (bgPtr->numTicks > 0) {
	XDrawSegments(bgPtr->display, Tk_WindowId(tkwin), bgPtr->tickGC,
		bgPtr->ticks, bgPtr->numTicks);
    }
#endif

    /* draw fromVal and toVal */
    if (SHOW_RANGE(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
	DisplayMinmax(bgPtr);

    /* draw base value if required */
    if (basev != minv && basev != maxv && (bgPtr->flags & DISPLAY_LABELS))
	DisplayBaseV(bgPtr);

    /* * Display the bar */

    if ((bgPtr->flags & DISPLAY_BAR))
	DisplayBar(bgPtr);

    /* Display the border */
    if ((bgPtr->flags & DISPLAY_BORDER))
	Tk_Draw3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		0, 0, Tk_Width(tkwin), Tk_Height(tkwin),
		bgPtr->borderWidth, bgPtr->relief);

    /* All done with the updating */
    bgPtr->flags &= ~DISPLAY_ALL;
}

/*
 *----------------------------------------------------------------------
 *
 * ValueToPixel --
 *
 *	Convert a value to the x- or y-coordinate corresponding to that value,
 *	depending on whether the Bargraph is vertical or horizontal,
 *	respectively.
 *
 * Results:
 *	An integer value giving the pixel location corresponding
 *	to the value.  If the "limit" flag is set, then the returned
 *	value is limited to the fromVal..toVal range; otherwise,
 *	it can have any value.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
ValueToPixel(bgPtr, value, limit)
    register Bargraph *bgPtr;	/* Information about widget. */
    double value;
    int limit;
{
    double minv = bgPtr->fromVal;
    double maxv = bgPtr->toVal;
    int length;

    if (limit)
	value = limit_dvalue(minv, maxv, value);

    /* Distance in pixels from minv; can be positive or negative. */
    length = (int) (bgPtr->max_height * (value - minv) / (maxv - minv));

    if (bgPtr->orient == ORIENT_VERTICAL) {
	return bgPtr->offset.bot_edge - length;
    } else {
	return bgPtr->offset.bot_edge + length;
    }
}

/*
 *--------------------------------------------------------------
 *
 * BargraphEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on Bargraphs.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
BargraphEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Bargraph *bgPtr = (Bargraph *) clientData;

    switch (eventPtr->type) {
	case Expose:
	    if (eventPtr->xexpose.count == 0)
		EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
	    break;
	case DestroyNotify:
	    if (bgPtr->tkwin != NULL) {
		bgPtr->tkwin = NULL;
		Tcl_DeleteCommandFromToken(bgPtr->interp, bgPtr->widgetCmd);
	    }

	    Tcl_CancelIdleCall(DisplayBargraph, (ClientData) bgPtr);
	    Tcl_EventuallyFree((ClientData) bgPtr, DestroyBargraph);
	    break;
	case ConfigureNotify:
	    ComputeBargraphGeometry(bgPtr);
	    EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
	    break;
	case FocusIn:
	    if (eventPtr->xfocus.detail != NotifyInferior) {
		bgPtr->flags |= GOT_FOCUS;
		if (bgPtr->highlightWidth > 0) {
		    EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL);
		}
	    }
	    break;
	case FocusOut:
	    if (eventPtr->xfocus.detail != NotifyInferior) {
		bgPtr->flags &= ~GOT_FOCUS;
		if (bgPtr->highlightWidth > 0) {
		    EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL);
		}
	    }
	    break;
    }
}

/*
 *--------------------------------------------------------------
 *
 * SetBargraphValue --
 *
 *	This procedure changes the value of a Bargraph and invokes
 *	a Tcl command to reflect the current position of a Bargraph
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional error-processing
 *	command may also be invoked.  The Bargraph's slider is redrawn.
 *
 *--------------------------------------------------------------
 */
static void
SetBargraphValue(bgPtr, value)
    register Bargraph *bgPtr;	/* Info about widget. */
    double value;		/* New value for Bargraph.  Gets
				 * adjusted if it's off the Bargraph. */
{
    if (value == bgPtr->value)
	return;

    bgPtr->value = value;
    EventuallyRedrawBargraph(bgPtr, DISPLAY_BAR | DISPLAY_VALUE);
}

/*
 *----------------------------------------------------------------------
 *
 * BargraphCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
BargraphCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    Bargraph *bgPtr = (Bargraph *) clientData;
    Tk_Window tkwin = bgPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	bgPtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedrawBargraph --
 *
 *	Arrange for part or all of a Bargraph widget to redrawn at
 *	the next convenient time in the future.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If "what" is DISPLAY_BAR then just the slider and the
 *	value readout will be redrawn;	if "what" is DISPLAY_ALL
 *	then the entire widget will be redrawn.
 *
 *--------------------------------------------------------------
 */
static void
EventuallyRedrawBargraph(bgPtr, what)
    register Bargraph *bgPtr;	/* Information about widget. */
    int what;			/* What to redraw */
{

    /* Make sure it's gonna redraw everything we want */
    bgPtr->flags |= what;
    if ( (bgPtr->tkwin == NULL) || !Tk_IsMapped(bgPtr->tkwin) ||
	    (bgPtr->flags & REDRAW_PENDING) ) {
	return;
    }

    bgPtr->flags |= REDRAW_PENDING;
    Tcl_DoWhenIdle(DisplayBargraph, (ClientData) bgPtr);

}

/*
 * Format a double value for printing.
 * If fractional part is .0..., trim it off
 * so that it looks like an integer value.
 */

static void
format_double(bgPtr, result, v)
    Bargraph *bgPtr;
    char *result;	/* result buffer; assumed large enough */
    double v;		/* value to format */
{
    char *p, *s;

#if 0
    Tcl_PrintDouble(bgPtr->interp, v, result);
#else
    sprintf(result, "%.*f", bgPtr->digits, v);
#endif

    p = strchr(result, '.');
    if (p) {
	/* There is a decimal point; trim trailing 0's */
	for (s = p + strlen(p); s > p && *(s-1) == '0'; s--)
	    *(s-1) = '\0';
	/* Assertion: s pts to terminating null.
	 * If prev char is '.', remove it.
	 */
	if ((s-1) == p)
	    *p = '\0';
    }
}

/* Keep a value between minv and maxv.	*/
static double
limit_dvalue(minv, maxv, value)
    double minv, maxv, value;
{
    if ( maxv > minv ) {
	if ( value > maxv )
	    value = maxv;
	else if ( value < minv )
	    value = minv;
    } else {
	/* maxv <= minv */
	if ( value < maxv )
	    value = maxv;
	else if ( value > minv )
	    value = minv;
    }
    return value;
}

static int
limit_ivalue(minv, maxv, value)
    int minv, maxv, value;
{
    if ( maxv > minv ) {
	if ( value > maxv )
	    value = maxv;
	else if ( value < minv )
	    value = minv;
    } else {
	/* maxv <= minv */
	if ( value < maxv )
	    value = maxv;
	else if ( value > minv )
	    value = minv;
    }
    return value;
}
