/*
@(#)  FILE: ncl_timer.c  RELEASE: 1.1  DATE: 3/11/94, 18:00:01
*/
/*******************************************************************************

File:

    ncl_timer.c

    "ncl_timer" Command Functions.


Author:    Alex Measday, ISI


Purpose:

    This file contains the functions that process the NICL-specific, Tcl
    command "ncl_timer".


Public Procedures:

    NCL_TIMER - interprets the "ncl_timer" Tcl extension.

Private Procedures:

    NCL_TIMER_CALLBACK - responds to a timeout event.
    NCL_TIMER_OBJECT - interprets a Tcl timer object command.

*******************************************************************************/


#include  <errno.h>			/* System error definitions. */
#include  <stdio.h>			/* Standard I/O definitions. */
#include  <stdlib.h>			/* Standard C library definitions. */
#include  "libncl.h"			/* NICL library definitions. */
#include  "libutilgen.h"		/* LIBUTILGEN definitions. */
#include  "nix_util.h"			/* Network I/O Handler definitions. */
#include  "opt_util.h"			/* Option scanning definitions. */


/*******************************************************************************
    Timer Objects - contain information about a timer.
*******************************************************************************/

typedef  struct  timer_object {
    double  interval ;			/* Timeout interval in seconds. */
    int  periodic ;			/* 0 = one-shot timer; 1 = periodic. */
    Tcl_Interp  *interpreter ;		/* Tcl interpreter. */
    char  *command ;			/* Tcl command to execute upon timeout. */
    char  *name ;			/* Name of timer object and its Tcl command. */
    NxAppContext  *appl_context ;	/* NIX application context. */
    NxIntervalId  timer_ID ;		/* NIX ID assigned to timer. */
}  timer_object ;


/*******************************************************************************
    Private Functions
*******************************************************************************/

static  int  ncl_timer_callback (
#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data
#    endif
    ) ;

static  int  ncl_timer_object (
#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[]
#    endif
    ) ;

/*******************************************************************************

Procedure:

    ncl_timer ()

    Create a Timer Object.


Purpose:

    Function NCL_TIMER interprets the "ncl_timer" Tcl command.  This
    command creates a one-shot or cyclic timer that, on each firing,
    executes a Tcl command.  The "ncl_timer" command is entered as
    follows:

        ncl_timer <interval> [-command <command>] [-name <name>] [-periodic]

    where:

        "<interval>"
            is the interval in seconds of the timer.  For a one-shot timer,
            this is the amount of time that elapses until the timer times out.
            For a periodic timer, this is the amount of time between timer
            firings.
        "-command <command>"
            is a Tcl command to be executed when the timer fires.  For a
            one-shot timer, this command will be executed once after the
            timer times out.  For a periodic timer, this command will be
            executed on each timer firing.
        "-name <name>"
            specifies the name of the timer.  This should be specified if
            you wish to be able to cancel the timer.  This name is used as
            the keyword for a dynamically-defined Tcl command that refers
            to the timer.
        "-periodic"
            specifies that the timer should fire periodically.  The timer
            can be cancelled with a "<name> cancel" command, where "<name>"
            is the name specified by the "-name" option.  By default, a
            timer is a one-shot timer that goes away after its first firing.


    Invocation:

        status = ncl_timer (&appl_context, interpreter, arg_count, arg_list) ;

    where:

        <appl_context>	- I/O
            is the address of a variable containing (an opaque pointer to) the
            NIX application context.
        <interpreter>	- I
            is the handle for the Tcl interpreter.
        <arg_count>	- I
            is the number of entries in the argument list.
        <arg_list>	- I
            is the list of arguments (stored in an array) for the command.
            The zero-th element in the list is always the command keyword.
        <status>	- O
            returns the status of interpreting the command, TCL_OK if no
            errors occurred and TCL_ERROR otherwise.

*******************************************************************************/


int  ncl_timer (

#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[])
#    else
        clientData, interp, argc, argv)

        ClientData  clientData ;
        Tcl_Interp  *interp ;
        int  argc ;
        char  *argv[] ;
#    endif

{    /* Local variables. */
    char  *argument, *command, *name ;
    double  interval ;
    int  errflg, option, periodic ;
    timer_object  *tob ;
    static  char  *option_list[] = {
        "{command:}", "{name:}", "{periodic}", NULL
    } ;
    static  void  *context = NULL ;




/*******************************************************************************
    Scan the command's arguments.
*******************************************************************************/

    interval = -1.0 ;  name = NULL ;  periodic = 0 ;

    if (context == NULL)  opt_init (argc, argv, 1, option_list, &context) ;
    opt_reset (context, argc, argv) ;
    errflg = 0 ;
    while (option = opt_get (context, &argument)) {
        switch (option) {
        case 1:			/* "-command <command>" */
            command = argument ;  break ;
        case 2:			/* "-name <name>" */
            name = argument ;  break ;
        case 3:			/* "-periodic" */
            periodic = 1 ;  break ;
        case NONOPT:
            interval = atof (argument) ;  break ;
        case OPTERR:
            errflg++ ;  break ;
        default:
            break ;
        }
    }

    if (errflg || (interval < 0.0)) {
        Tcl_SetResult (interp, "missing or invalid argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }

/*******************************************************************************
    Create a timer object for the timer and register a timeout timer for the
    specified interval with the NIX I/O handler.  When the timeout period
    expires, NCL_TIMER_CALLBACK() will execute the Tcl command defined for
    the timer and, if the timer is periodic, re-register the timeout interval
    with the NIX I/O handler.
*******************************************************************************/

/* Create a timer object structure for the timer. */

    tob = (timer_object *) malloc (sizeof (timer_object)) ;
    if (tob == NULL) {
        vperror ("(ncl_timer) Error allocating timer object for %g-second interval.\nmalloc: ",
                 interval) ;
        Tcl_SetResult (interp, "error allocating timer object", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }

    tob->interval = interval ;
    tob->periodic = periodic ;
    tob->interpreter = interp ;
    tob->command = (command == NULL) ? NULL : str_dupl (command, -1) ;
    tob->appl_context = (NxAppContext *) clientData ;

/* Register the timer with the NIX I/O handler. */

    tob->timer_ID = NxAddTimeOut (tob->appl_context, interval,
                                  ncl_timer_callback, (void *) tob) ;

/* Create a new Tcl command for the timer.  The dynamically-defined Tcl
   command can be used to cancel the timer. */

    if (name != NULL) {
        Tcl_CreateCommand (interp, name, (Tcl_CmdProc *) ncl_timer_object,
                           (ClientData) tob, NULL) ;
    }

    return (TCL_OK) ;

}

/*******************************************************************************

Procedure:

    ncl_timer_callback ()

    Respond to a Timer Firing.


Purpose:

    Function NCL_TIMER_CALLBACK is automatically invoked by the NIX I/O handler
    when a timer's timeout interval has elapsed.  NCL_TIMER_CALLBACK() first
    executes the Tcl command defined (by the "-command <command>" option) for
    the timer.  Then, if the timer is a one-shot timer, its timer object and
    its dynamically-defined Tcl command (used to cancel the timer) are deleted.
    If the timer is periodic, the timer object and its command are not deleted.
    Instead, another timeout timer for the specified interval is registered
    with the NIX I/O handler.


    Invocation:

        status = ncl_timer_callback (appl_context, timer_ID, client_data) ;

    where:

       <appl_context>	- I
            is the application context used when the timeout timer was
            registered with the NIX I/O handler.
        <timer_ID>	- I
            is the identifier returned by NXADDTIMEOUT() when the timeout
            timer was registered with the NIX I/O handler.  A NULL timer ID
            indicates a periodic timer that is to be cancelled.
        <client_data>	- I
            is the address of the timer object structure created by NCL_TIMER().
        <status>	- O
            returns the status of processing the timeout event, zero if there
            were no errors and ERRNO otherwise.  The status value is ignored
            by the NIX I/O handler.

*******************************************************************************/


static  int  ncl_timer_callback (

#    if __STDC__ || defined(vaxc)
        NxAppContext  appl_context,
        NxIntervalId  timer_ID,
        void  *client_data)
#    else
        appl_context, timer_ID, client_data)

        NxAppContext  appl_context ;
        NxIntervalId  timer_ID ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    timer_object  *tob ;




    tob = (timer_object *) client_data ;

/* If the timer was not cancelled, execute the Tcl command defined for the
   timer firing. */

    if (timer_ID != NULL) {
        if (libncl_debug)
            printf ("(ncl_timer_callback) Timer command: %s\n",
                    (tob->command == NULL) ? "<none>" : tob->command) ;
        if (tob->command != NULL)
            Tcl_Eval (tob->interpreter, tob->command) ;
    }

/* If the timer is a periodic timer and it was not cancelled, then register
   another timeout timer with the NIX I/O handler. */

    if (tob->periodic && (timer_ID != NULL)) {
        tob->timer_ID = NxAddTimeOut (tob->appl_context, tob->interval,
                                      ncl_timer_callback, (void *) tob) ;
    }

/* If the timer was a one-shot timer or a cancelled periodic timer, then
   delete its timer object as well as its dynamically-defined Tcl command. */

    else {
        if (tob->periodic)  NxRemoveTimeOut (tob->appl_context, tob->timer_ID) ;
        if (tob->name != NULL) {
            Tcl_DeleteCommand (tob->interpreter, tob->name) ;
            str_free (&tob->name, -1) ;
        }
        if (tob->command != NULL)  str_free (&tob->command, -1) ;
        free (tob) ;
    }

    return (0) ;

}

/*******************************************************************************

Procedure:

    ncl_timer_object ()

    Respond to Commands for a Timer Object.


Purpose:

    Function NCL_TIMER_OBJECT interprets the Tcl command defined for a timer.
    This command, whose keyword is the name assigned to the timer, is
    registered with the Tcl interpreter when the timer is created (see the
    "ncl_timer" command).  Timer objects respond to the following Tcl commands:

        <name> cancel

    where:

        "<name>"
            is the name assigned to the timer when the timer was created by
            an "ncl_timer" command.
        "cancel"
            cancels the timer.


    Invocation:

        status = ncl_timer_object (timer, interpreter, arg_count, arg_list) ;

    where:

        <timer>		- I
            is the address of the timer object structure created for the timer
            by NCL_TIMER().
        <interpreter>	- I
            is the handle for the Tcl interpreter.
        <arg_count>	- I
            is the number of entries in the argument list.
        <arg_list>	- I
            is the list of arguments (stored in an array) for the command.
            The zero-th element in the list is always the command keyword.
        <status>	- O
            returns the status of interpreting the command, TCL_OK if no
            errors occurred and TCL_ERROR otherwise.

*******************************************************************************/


static  int  ncl_timer_object (

#    if __STDC__ || defined(vaxc)
        ClientData  clientData,
        Tcl_Interp  *interp,
        int  argc,
        char  *argv[])
#    else
        clientData, interp, argc, argv)

        ClientData  clientData ;
        Tcl_Interp  *interp ;
        int  argc ;
        char  *argv[] ;
#    endif

{    /* Local variables. */
    char  *action ;
    timer_object  *tob ;




/*******************************************************************************
    Scan the command's arguments.
*******************************************************************************/

    tob = (timer_object *) clientData ;

    action = (argc > 1) ? argv[1] : NULL ;

    if (action == NULL) {
        Tcl_SetResult (interp, "missing argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }


/*******************************************************************************
    CANCEL - cancel the timer.  This is done by invoking the timer's callback
        function, NCL_TIMER_CALLBACK(), with a NULL timer ID.  The NULL timer
        ID tells NCL_TIMER_CALLBACK() to delete the timer's object structure.
*******************************************************************************/

    if (strcmp (action, "cancel") == 0) {
        ncl_timer_callback (NULL, NULL, (void *) tob) ;
    }


/*******************************************************************************
    Invalid keyword.
*******************************************************************************/

    else {
        Tcl_SetResult (interp, "invalid argument", TCL_STATIC) ;
        return (TCL_ERROR) ;
    }


    return (TCL_OK) ;

}
