static char rcsid[]="$Id: imgeom.c,v 1.5 94/02/06 14:33:11 mangin Exp $";

/**  This files contains bilinear routines for
 **    image rotation and rescaling.
 **  These routines are slooooow...
 **/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <math.h>

#define ZEROF 0.000001
#define FNULL(A) (((A) > -ZEROF) && ((A) < ZEROF))
#define FEQUAL(A,B) FNULL((A)-(B))

/**
 **  Prototypes for functions defined later in this file
 **/

static void Zoom
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst,
	       int dstw, int dsth, int dstlinew, int normalize));
     
static void ZoomCopy
  _ANSI_ARGS_((float *src, int w, int h, float *dst, int dstlinew));

static void ZoomResize
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst,
	       int dstw, int dsth, int dstlinew, int normalize));

static void Rotate
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst,
	       double angle, double csn, double sn,
	       Region region, int dstlinew));

static void RotatePos90
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst, int dstlinew));

static void RotateNeg90
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst, int dstlinew));


static void Rotate180
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst, int dstlinew));

static void RotateAny
  _ANSI_ARGS_((float *src, int srcw, int srch, float *dst,
	       double csn, sn, Region region, int dstlinew));

/***************************************************************
 ****	    The externally callable routine                 ****
 ***************************************************************/

void ZoomTurn (src, srcw, srch,
	       dst, dstw, dsth, dstlinew,
	       zflag, xfact, yfact,
	       rflag, angle, csn, sn, xregion,
	       a, b, pixels, numpixels)
     float *src;			/* source data buffer		*/
     int srcw, srch;			/* source data dims		*/
     unsigned char *dst;		/* destination data buffer	*/
     int dstw, dsth, dstlinew;		/* destination buffer dims	*/
     int zflag;				/* whether to rescale or not	*/
     float xfact, yfact;		/* rescale ratios		*/
     int rflag;				/* whether to rotate or not	*/
     float angle, csn, sn;		/* cos and sine of rotation	*/
     Region xregion;			/* X Region defined by final image */
     float a, b;			/* ax+b gives pixel indice	*/
     int *pixels, numpixels;		/* pixel values lookup table	*/
{
  float *zsrc, *rzsrc;
  float *psrc, *srcend, *srclineend;
  unsigned char *pdst;
  unsigned char minpix, maxpix;
  int val, line_garb;

  /** step by step ... **/
  /**  zoomed image -> zsrc  **/
  /**  rotated zoomed image -> rzsrc  **/

  if (!zflag && !rflag) {
    /*  No rotation, no scaling  */
    rzsrc = zsrc = src;
  } else if (!zflag && rflag) {
    /*  just rotate  */
    zsrc = src;
    rzsrc = (float *)malloc(dstw*dsth * sizeof(float));

    Rotate(zsrc, srcw, srch,
           rzsrc, angle, csn, sn, xregion, dstw);
  } else if (zflag && !rflag) {
    /* just scale */
    zsrc = (float *) malloc(dstw*dsth* sizeof(float));
    
    Zoom(src, srcw, srch, zsrc, dstw, dsth, dstw, 1);
    
    rzsrc = zsrc;
  } else {
    /* first scale: overestimate the int dims, to make sure to */
    /* provide enough data for rotate */
    int iw, ih;

    iw = (int)(1.0 + srcw/xfact);
    ih = (int)(1.0 + srch/yfact);

    zsrc = (float *)malloc(iw*ih * sizeof(float));
    Zoom(src, srcw, srch, zsrc, iw, ih, iw, 1);

    /*  then rotate */
    rzsrc = (float *)malloc(dstw*dsth * sizeof(float));

    Rotate(zsrc, iw, ih, rzsrc, angle, csn, sn, xregion, dstw);
  }

  /****  Map values into pixels  ****/

  minpix = pixels[0];
  maxpix = pixels[numpixels - 1];

  psrc = rzsrc;
  srclineend = rzsrc + dstw;
  srcend = rzsrc + dstw*dsth;
      
  pdst = dst;
  line_garb = dstlinew - dstw;
      
  while (psrc < srcend) {
    while (psrc < srclineend) {
      val = (int)(a * (*psrc++) + b);
      if (val < 0)
	*pdst++ = minpix;
      else if (val >= numpixels)
	*pdst++ = maxpix;
      else
	*pdst++ = pixels[val];
    }
    pdst += line_garb;

    srclineend += dstw;;
  }
  
  if (rzsrc != zsrc) free(rzsrc);
  if (zsrc != src) free(zsrc);
}

/***************************************************************
 ***************************************************************/

/***************************************************************
 ****                 ImageZooming                          ****
 ***************************************************************/

/***************************************************************
 ***************************************************************/

/********  Resizing of pixel arrays.  ********/
/********  Expansion uses linear extrapolation.  ********/
/********  Reduction uses exact averaging.  ********/

static void Zoom(src, srcw, srch,
		 dst, dstw, dsth, dstlinew,
		 normalize)
     float *src;                /** source data buffer          **/
     int srcw, srch;            /** source dimensions           **/
     float *dst;                /** destination data buffer     **/
     int dstw, dsth;            /** destination dimensions      **/
     int dstlinew;              /** destination line length     **/
     int normalize;             /** flag : normalize when shrinking **/
{
  if ((dstw == srcw) && (dsth == srch))
    ZoomCopy(src, srcw, srch, dst, dstlinew);
  else {
    ZoomResize(src, srcw, srch,
               dst, dstw, dsth, dstlinew, normalize);
  }
}

/***************************************************************
 ****     Copying: just take care of dstlinew               ****
 ***************************************************************/

static void ZoomCopy(src, w, h, dst, dstlinew)
     float *src;                /** source data buffer          **/
     int w, h;                  /** dimensions                  **/
     float *dst;                /** destination data buffer     **/
     int dstlinew;              /** destination line length     **/
{
  register float *psrc, *pdst, *srcend, *srclineend;
  int dst_garbage;              /** offset between the end of a dst line **/
  /** and start of following line          **/
  
  psrc = src;
  srcend = src + w*h;
  srclineend = src + w;
  
  pdst = dst;
  dst_garbage = dstlinew - w;
  
  while (psrc < srcend) {
    while (psrc < srclineend)
      *pdst++ = *psrc++;
    pdst += dst_garbage;
    srclineend += w;
  }
}

/***************************************************************
 ****	   General case: slow but general and bug-free ...  ****
 ****      first compute x-resized image, then y-resize     ****
 ***************************************************************/
static void ZoomResize(src, srcw, srch,
		       dst, dstw, dsth, dstlinew,
		       normalize)
     float *src;		/** source data buffer 		**/
     int srcw, srch;		/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     int dstw, dsth;		/** destination dimensions	**/
     int dstlinew;		/** destination line length	**/
     int normalize;
{
  float *xrdst, *pxrdst, *xrlineend;
  float *psrc, *pdst, *dstend;
  int line_garbage, i, j;
  
  
  line_garbage = dstlinew - dstw;
  
  /********  resizing in x dimension  ********/
  
  if (dstw > srcw) {
    /****  x dilatation  ****/
    float x, deltax;
    int isrc;
    
    xrdst = (float *)malloc(dstw*srch*sizeof(float));
    deltax = srcw / ((float)dstw);
    psrc = src;
    pxrdst = xrdst;
    xrlineend = xrdst + dstw;
    
    for (j = 0; j < srch; j++) {
      x = 0.0;
      isrc = 0;
      while (isrc < srcw-1) {
	*pxrdst++ = (1.0 - (x - isrc)) * psrc[isrc] +
	  (x - isrc) * psrc[isrc+1];
	x += deltax;
	isrc = (int)x;
      }
      while (pxrdst < xrlineend)
	*pxrdst++ = psrc[isrc];
      xrlineend += dstw;
      psrc += srcw;
    }
  } else if (dstw < srcw) {
    /****  x shrinking  ****/
    float deltax, nextx;
    int *dstColumns;
    float *sharedRatio, fact;
    int *sharedFlag;
    int nextSharedi, dstcol;
    
    dstColumns = (int *)malloc(srcw*sizeof(int));
    sharedFlag = (int *)calloc(srcw, sizeof(int));
    sharedRatio = (float *)malloc(srcw*sizeof(float));
    
    deltax = srcw / ((float)dstw);
    nextx = deltax;
    nextSharedi = (int)nextx;
    dstcol = 0;
    i = 0;
    
    /**  fill in hash tables  **/
    while (i < srcw-1) { 
      while ((i < nextSharedi) && (i < srcw-1)) /* pixels entirely included in dst col. */
	dstColumns[i++] = dstcol;
      /* now pixel i==nextSharedi is shared between dstcol and dstcol+1 */
      dstColumns[i] = dstcol;
      sharedFlag[i] = 1;
      sharedRatio[i] = nextx - i;
      i++;
      /*  advance dstcolumn  */
      nextx += deltax;
      nextSharedi = (int)nextx;
      dstcol++;
    }
    
    /* last src pixel goes in last column */
    dstColumns[srcw - 1] = dstw - 1;
    sharedFlag[srcw - 1] = 0;
    
    /**  Perform line by line resize  **/
    /* initialixe xrdst to zero */
    xrdst = (float *)calloc(dstw*srch, sizeof(float));
    pxrdst = xrdst;
    psrc = src;
    
    for (j = 0; j < srch; j++) {
      /* add each src pixel in its dst column */
      for (i = 0; i < srcw; i++, psrc++)
	if (sharedFlag[i]) {
	  dstcol = dstColumns[i];
	  fact = sharedRatio[i];
	  pxrdst[dstcol] += fact*(*psrc);
	  pxrdst[dstcol+1] += (1.0-fact)*(*psrc);
	} else {
	  pxrdst[dstColumns[i]] += *psrc;
	}
      
      /**  advance line in dst  **/
      pxrdst += dstw;
    }
    
    free(dstColumns);
    free(sharedFlag);
    free(sharedRatio);
    
    /**  Normalization  **/
    if (normalize) {
      float fact, *xrdstend;
      
      fact = 1.0 / deltax; /** to handle rounding errors **/
      pxrdst = xrdst;
      xrdstend = xrdst + dstw * srch;
      
      while (pxrdst < xrdstend) {
	*pxrdst *= fact;
	if (*pxrdst < 0.0) *pxrdst = 0.0;
	if (*pxrdst > 1.0) *pxrdst = 1.0;
	pxrdst++;
      }
    }
  } else {
    /**  No modification in x dimension  **/
    xrdst = src;
  }
  
  /********  resizing in y dimension  ********/
  
  if (dsth == srch) {
    pdst = dst;
    pxrdst = xrdst;
    
    for (i = 0; i < dstw*dsth; i++)
      *pdst++ = *pxrdst++;
  } else
    if (dsth > srch) {
      /****  y dilatation  ****/
      float y, deltay;
      float *srcline1, *srcline2;
      int jsrc;
      
      deltay = srch / ((float)dsth);
      y = 0.0;
      jsrc = 0;
      pdst = dst;
      pxrdst = xrdst;
      
      while (jsrc < srch - 1) {
	srcline1 = xrdst + dstw * jsrc;
	srcline2 = srcline1 + dstw;
	for (i = 0; i < dstw; i++)
	  *pdst++ = (1 - (y-jsrc)) * srcline1[i] +
	    (y-jsrc) * srcline2[i];
	y += deltay;
	jsrc = (int)y;
	pdst += line_garbage;
      }
      
      dstend = dst + dstw*dsth;
      srcline1 = xrdst + dstw * (srch - 1);
      while (pdst < dstend) {
	for (i = 0; i < dstw; i++)
	  *pdst++ = srcline1[i];
	pdst += line_garbage;
      }    
    } else {
      /****  y shrinking  ****/
      float deltay, nexty;
      float **dstLines;
      float *sharedRatio, fact;
      int *sharedFlag;
      int nextSharedj;
      float *dstline, *dstnextline;
      
      dstLines = (float **)malloc(srch*sizeof(float *));
      sharedFlag = (int *)calloc(srch, sizeof(int));
      sharedRatio = (float *)malloc(srch*sizeof(float));
      
      /**  fill in hash tables  **/
      /** dst lines are stored as pointers on line starts in dst **/
      
      deltay = srch / ((float)dsth);
      nexty = deltay;
      nextSharedj = (int)nexty;
      dstline = dst;
      j = 0;
      
      while (j < srch-1) { /* last src line treated apart */
	while ((j < nextSharedj) && (j < srch-1)) /* src lines entirely included in dst line */
	  dstLines[j++] = dstline;
	/* now src line j is shared between dstline and dstline+1 */
	dstLines[j] = dstline;
	sharedFlag[j] = 1;
	sharedRatio[j] = nexty - j;
	j++;
	/*  advance dstline  */
	nexty += deltay;
	nextSharedj = (int)nexty;
	dstline += dstlinew;
      }
      
      /* last src line entirely goes in last dst line */
      dstLines[srch - 1] = dst + (dsth-1)*dstlinew;
      sharedFlag[srch - 1] = 0;
      
      /**  Perform resize  **/
      pxrdst = xrdst;
      bzero(dst, dstlinew*dsth*sizeof(float));
      
      for (j = 0; j < srch; j++) {
	dstline = dstLines[j];
	if (sharedFlag[j]) {
	  dstnextline = dstline + dstlinew;
	  fact = sharedRatio[j];
	  for (i = 0; i < dstw; i++, pxrdst++) {
	    dstline[i] += fact*(*pxrdst);
	    dstnextline[i] += (1.0-fact)*(*pxrdst);
	  }
	} else {
	  for (i = 0; i < dstw; i++, pxrdst++)
	    dstline[i] += *pxrdst;
	}
      }	  
      
      free(dstLines);
      free(sharedFlag);
      free(sharedRatio);

      /**  Normalization  **/
      if (normalize) {
	float fact;
	
	fact = 1.0 / deltay; /** to handle rounding errors **/
	pdst = dst;
	dstend = dst + dstlinew * dsth;
	
	while (pdst < dstend) {
	  *pdst *= fact;
	  if (*pdst < 0.0) *pdst = 0.0;
	  if (*pdst > 1.0) *pdst = 1.0;
	  pdst++;
	}
      }
    }
  
  if (xrdst != src)
    free(xrdst);
}

/***************************************************************
 ***************************************************************/

/***************************************************************
 ****		     Image Rotation                         ****
 ***************************************************************/

/***************************************************************
 ***************************************************************/

static void Rotate(src, srcw, srch,
		   dst, angle, csn, sn, region, dstlinew)
     float *src;		/** source data buffer 		**/
     int srcw, srch; 		/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     double angle, csn, sn;	/** destination dimensions	**/
     Region region;
     int dstlinew;		/** destination line length	**/
{
  if (FNULL(angle)) {
    ZoomCopy(src, srcw, srch, dst, dstlinew);
  } else if (FEQUAL(angle, 90.0)) {
    RotatePos90(src, srcw, srch, dst, dstlinew);
  } else if (FEQUAL(angle, -90.0)) {
    RotateNeg90(src, srcw, srch, dst, dstlinew);
  } else if (FEQUAL(fabs(angle), 180.0)) {
    Rotate180(src, srcw, srch, dst, dstlinew);
  } else {
    RotateAny(src, srcw, srch, dst, csn, sn, region, dstlinew);
  }
}

/***************************************************************
 ****		  90 degrees rotation                       ****
 ***************************************************************/

static void RotatePos90(src, srcw, srch, dst, dstlinew)
     float *src;		/** source data buffer 		**/
     int srcw, srch;		/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     int dstlinew;		/** destination line length	**/
{
  float *pdst, *psrc, *srcline, *srclineend, *srcend, *dstcol;
  
  srcend = src + srcw*srch;
  srclineend = src + srcw;
  
  /**  First src pixel goes in last line of first dst column  **/
  dstcol = dst + dstlinew*(srcw-1);
  
  for (srcline = src;
       srcline < srcend;
       srcline += srcw, srclineend += srcw) {
    psrc = srcline;
    pdst = dstcol;
    while (psrc < srclineend) {
      *pdst = *psrc++;
      pdst -= dstlinew;
    }
    dstcol++;
  }
}

/***************************************************************
 ****		  -90 degrees rotation                       ****
 ***************************************************************/

static void RotateNeg90(src, srcw, srch, dst, dstlinew)
     float *src;		/** source data buffer 		**/
     int srcw, srch;		/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     int dstlinew;		/** destination line length	**/
{
  float *pdst, *psrc, *srcline, *srclineend, *srcend, *dstcol;
  
  srcend = src + srcw*srch;
  srclineend = src + srcw;
  /**  First src pixel goes in first line of last dst column  **/
  dstcol = dst + srch - 1;
  
  for (srcline = src;
       srcline < srcend;
       srcline += srcw, srclineend += srcw) {
    psrc = srcline;
    pdst = dstcol;
    while (psrc < srclineend) {
      *pdst = *psrc++;
      pdst += dstlinew;
    }
    dstcol--;
  }
}

/***************************************************************
 ****		  180 degrees rotation                      ****
 ***************************************************************/

static void Rotate180(src, srcw, srch, dst, dstlinew)
     float *src;		/** source data buffer 		**/
     int srcw, srch;		/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     int dstlinew;		/** destination line length	**/
{
  float *pdst, *psrc, *srcline, *srclineend, *srcend, *dstline;
  
  srcend = src + srcw*srch;
  srclineend = src + srcw;
  dstline = dst + srch*dstlinew - 1;
  
  for (srcline = src;
       srcline < srcend;
       srcline += srcw, srclineend += srcw) {
    psrc = srcline;
    pdst = dstline;
    while (psrc < srclineend) {
      *pdst-- = *psrc++;
    }
    dstline -= dstlinew;
  }
}

/***************************************************************
 ****		      General case                          ****
 ***************************************************************/

static void RotateAny(src, srcw, srch, dst, csn, sn, region, dstlinew)
     float *src;		/** source data buffer 		**/
     int srcw, srch;	/** source dimensions		**/
     float *dst;		/** destination data buffer	**/
     double csn, sn;		/** cos and sin of rotation angle **/
     Region region;		/** Xregion of dst image **/
     int dstlinew;		/** destination line length	**/
{
  Region cregion;
  XRectangle rect;
  double srcx, srcy, dsty;
  double srcx0, srcy0, dstx0;
  float **srcline, *pdst, *psrc;
  int I, J, i, j, dstgarb;
  
  /*  translate region so that ul-corner of its bbox is in (0,0)  */
  cregion = XCreateRegion();
  XUnionRegion(cregion, region, cregion);
  XClipBox(cregion, &rect);
  XOffsetRegion(cregion, -rect.x, -rect.y);
  
  srcline = (float **)malloc(srch * sizeof(float *));
  psrc = src;
  for (j = 0; j < srch; j++, psrc += srcw)
    srcline[j] = psrc;
  
  dstx0 = -0.5*rect.width;
  srcx0 = 0.5*srcw + csn*dstx0;
  srcy0 = 0.5*srch + sn*dstx0;
  
  pdst = dst;
  dstgarb = dstlinew - rect.width;
  
  for (J = 0, dsty = -0.5*rect.height;
       J < rect.height;
       J++, dsty += 1.0) {
    
    srcx = srcx0 - sn*dsty;
    srcy = srcy0 + csn*dsty;
    
    for (I = 0;
	 I < rect.width;
	 I++, pdst++) {
      *pdst = 0.0;
      srcx += csn;
      srcy += sn;
      
      if (XPointInRegion(cregion, I, J) != True)
	continue;
      
      i = 1 + (int)srcx;
      j = 1 + (int)srcy;
      if (i < 1) {
	/*  first column  */
	if (j < 1)
	  *pdst = *src;
	else if (j > srch-1)
	  *pdst = *srcline[srch-1];
	else {
	  *pdst = (j - srcy) * (*srcline[j-1]) +
	    (1 - j + srcy) * (*srcline[j]);
	}
      } else if (i > srcw-1) {
	/*  last column  */
	if (j < 1)
	  *pdst = src[srcw-1];
	else if (j > srch-1)
	  *pdst = srcline[srch-1][srcw-1];
	else {
	  *pdst = (j - srcy) * srcline[j-1][srcw-1] +
	    (1 - j + srcy) * srcline[j][srcw-1];
	}
      } else {
	/* any column */
	if (j < 1) {
	  *pdst = (i - srcx)*src[i-1] +
	    (1 - i + srcx)*src[i];
	} else if (j > srch-1) {
	  *pdst = (i - srcx)*srcline[srch-1][i-1] +
	    (1 - i + srcx)*srcline[srch-1][i];
	} else {
	  psrc = srcline[j-1] + i - 1;
	  *pdst = (i - srcx)*(j - srcy)*(*psrc++);
	  *pdst += (1 - i + srcx)*(j - srcy)*(*psrc);
	  psrc += -1 + srcw;
	  *pdst += (i - srcx)*(1 - j + srcy)*(*psrc++);
	  *pdst += (1 - i + srcx)*(1 - j + srcy)*(*psrc);
	}
      }
    }
    pdst += dstgarb;
  }
  XDestroyRegion(cregion);
  free(srcline);
}
