#
# This module implements several dialog types.
#
# Michael Moore
# March 1993
#

#
# This just clears the respond help text
#
proc respond_help_clear {} {
    .respond.help.text delete 1.0 end
    pack forget .respond.help.text 
    pack configure .respond.help.label -side left -fill both -expand 1
}

#
# This procedure shows help for options in the respond_dialog.
#
proc respond_help {path} {
    pack forget .respond.help.label 
    pack configure .respond.help.text -side left -fill both -expand 1
    case $path in {
	{.respond.info} {
	    set title "Reference Article Information"
	    set s "This section displays information about the article to"
	    append s "which you are replying."
	} 
	{.respond.type.media.email} {
	    set title "Electronic Mail"
	    set s "Use this media to restrict your reply to the author of the article."
	}
	{.respond.type.media.news} {
	    set title "Newsgroup"
	    set s "Use this media if you want everyone to see your reply."
	}
	{.respond.type.context.followup} {
	    set title "Followup to Article"
	    set s "Use this context if your message is related to the article."
	} 
	{.respond.type.context.original} {
	    set title "Unrelated Topic"
	    set s "Use this context if your message is not related to the article."
	}
	{.respond.options.include} {
	    set title "Include Article Text" 
	    set s "This option will include the article's text for you, using"
	    append s " the Followup Prefix to begin quoted lines."
	}
	{.respond.options.signature} {
	    set title "Append Signature"
	    set s "Select this if you wish to have your signature file appended"
	    append s " to the end of your article."
	}
	{.respond.options.prefix} {
	    set title "Included Text Prefix"
	    append s "The string here is placed before text included from"
 	    append s "the reference article."
	}
	{default} {return}
    }
    .respond.help.text delete 1.0 end
    .respond.help.text insert 1.0 $title
    .respond.help.text insert end "\n  $s"
    .respond.help.text tag add sel 1.0 1.[string length $title]
}
    
    
proc respond_help_bind {w class} {
    set old [bind $w <Any-Enter>]
    append old [bind $class <Any-Enter>]
    bind $w <Enter> "$old ; respond_help $w"
    set old [bind $w <Any-Leave>]
    append old [bind $class <Any-Leave>]
    bind $w <Leave> "$old ; respond_help_clear"
}

#
# Set the bitmap according to Respond options.
#
proc respond_set_bitmap {} {
    global Respond LIBRARY_DIR

    set a [expr {$Respond(media) == "email"}]
    set b [expr {$Respond(type) == "followup"}]
    set c $Respond(include)
    if {$a && $b && $c} { set name letter_zap.xbm }
    if {$a && $b && !$c} {set name letter2.xbm}
    if {$a && !$b} {set name letter.xbm}
    if {!$a && $b && $c} {set name article_include.xbm}
    if {!$a && $b && !$c} {set name article_followup.xbm}
    if {!$a && !$b} {set name write.xbm}
    .respond.help.picture configure \
	-bitmap @$LIBRARY_DIR/bitmaps/$name
}

#
# Quit and call mail/post article.
#
proc respond_ok {} {
    global Respond

    grab release .respond
    destroy .respond
    set a [expr {$Respond(media) == "email"}]
    set b [expr {$Respond(type) == "followup"}]
    set c $Respond(include)
    if {$a && $b && $c} { mail_article Reply }
    if {$a && $b && !$c} { mail_article reply}
    if {$a && !$b} { mail_article compose }
    if {!$a && $b && $c} { post_article Followup}
    if {!$a && $b && !$c} { post_article followup}
    if {!$a && !$b} { post_article original}
}
    
#
# Quit and don't mail or post anything.
#
proc respond_cancel {} {
    grab release .respond
    destroy .respond
}


#
# This is a specific dialog for posting/replying to articles.
#
proc respond_dialog {group subject author} {
	global Resources Respond LIBRARY_DIR

	set Respond(media) "email"
	set Respond(type) "followup"
	set Respond(include) 0
	set Respond(signature) 1

	toplevel .respond
	wm positionfrom .respond user
	wm title .respond "Respond"
	set pointer [QueryPointer]
	set xpos [lindex $pointer 0]
	set ypos [lindex $pointer 1]
	incr xpos -10
	incr ypos -10

	# adjust for the location of the virtual window if necessary

	# put the window off in never-never land while we construct it.

	wm geometry .respond "-9999-9999"

	### Information about the article we are replying
	### to.
	frame .respond.info -relief flat -borderwidth 2
	respond_help_bind .respond.info Frame
	frame .respond.info.group -relief flat -borderwidth 0
	label .respond.info.group.label -width 10 -relief flat \
	    -text "Newsgroup:" -anchor w -borderwidth 0
	label .respond.info.group.val -relief flat \
	    -text $group -anchor w -borderwidth 0 
	pack configure .respond.info.group.label -side left 
	pack configure .respond.info.group.val -side left
	
	frame .respond.info.subject -relief flat -borderwidth 0
	label .respond.info.subject.label -width 10 -relief flat \
	    -text "Subject:" -anchor w -borderwidth 0
	label .respond.info.subject.val -relief flat \
	    -text $subject -anchor w -borderwidth 0
	pack configure .respond.info.subject.label -side left
	pack configure .respond.info.subject.val -side left
	
	frame .respond.info.author -relief flat -borderwidth 0
	label .respond.info.author.label -width 10 -relief flat \
	    -text "Author:" -anchor w -borderwidth 0
	label .respond.info.author.val -relief flat \
	    -text $author -anchor w -borderwidth 0
	pack configure .respond.info.author.label -side left
	pack configure .respond.info.author.val -side left

	pack configure .respond.info.group -side top -anchor nw
	pack configure .respond.info.subject -side top -anchor nw
	pack configure .respond.info.author -side top -anchor nw


	### information about settings
	frame .respond.help -borderwidth 2 -relief raised
	text .respond.help.text -width 1 -height 4 -wrap word
	label .respond.help.label -anchor center \
	    -bitmap @$LIBRARY_DIR/bitmaps/info.xbm
	label .respond.help.picture -relief groove -borderwidth 3 \
	    -padx 15 -pady 10
	respond_set_bitmap
	pack .respond.help.picture -side right -fill both
	### determining type of post.
	frame .respond.type -relief raised -borderwidth 2
	## what type of media?
	frame .respond.type.media -borderwidth 2 -relief ridge
	label .respond.type.media.label -text "Media :" -relief flat \
	    -anchor center \
	    -foreground [lindex [.respond.help.text config -background] 4] \
	    -background [lindex [.respond.help.text config -foreground] 4]
	radiobutton .respond.type.media.email -text "Electronic Mail (Email)" \
	    -variable Respond(media) -value "email" \
	    -anchor nw -relief flat -padx 10 -command respond_set_bitmap
	respond_help_bind .respond.type.media.email Radiobutton
	radiobutton .respond.type.media.news -text "Newsgroup (Usenet)" \
	    -variable Respond(media) -value "news" \
	    -anchor nw -relief flat -padx 10 -command respond_set_bitmap
	respond_help_bind .respond.type.media.news Radiobutton

	pack .respond.type.media.label -side top -fill x -expand 1
	pack .respond.type.media.email -side top -fill x -expand 1
	pack .respond.type.media.news -side top -fill x -expand 1

	## context of the message
	frame .respond.type.context -relief ridge -borderwidth 2
	label .respond.type.context.label -text "Context of Message :" \
	    -anchor center \
	    -foreground [lindex [.respond.help.text config -background] 4] \
	    -background [lindex [.respond.help.text config -foreground] 4]
	radiobutton .respond.type.context.followup -text "Followup to article" \
	    -variable Respond(type) -value "followup" \
	    -anchor nw -relief flat -padx 10 -command respond_set_bitmap
	respond_help_bind .respond.type.context.followup Radiobutton
	radiobutton .respond.type.context.original -text "Unrelated Topic" \
	    -variable Respond(type) -value "original" \
	    -anchor nw -relief flat -padx 10 -command respond_set_bitmap
	respond_help_bind .respond.type.context.original Radiobutton

	pack .respond.type.context.label -side top -fill x -expand 1
	pack .respond.type.context.followup -side top -fill x -expand 1
	pack .respond.type.context.original -side top -fill x -expand 1

	pack .respond.type.media -side left -fill both -expand 1
	pack .respond.type.context -side right -fill both -expand 1

	### options
	frame .respond.options -relief raised -borderwidth 2
	label .respond.options.label -text "Options:" \
	    -foreground [lindex [.respond.help.text config -background] 4] \
	    -background [lindex [.respond.help.text config -foreground] 4]
	checkbutton .respond.options.include -text "Include Article Text" \
	    -variable Respond(include) -anchor nw -relief flat \
	    -command respond_set_bitmap
	respond_help_bind .respond.options.include Checkbutton
	checkbutton .respond.options.signature -text "Include Signature" \
	    -variable Respond(signature) -anchor nw -relief flat
	respond_help_bind .respond.options.signature Checkbutton
	label .respond.options.prefixlabel -text "Prefix for quoted text:" \
	    -relief flat
	entry .respond.options.prefix -textvariable Resources(quotePrefix) \
	    -relief ridge -borderwidth 2
	respond_help_bind .respond.options.prefix Entry

	pack .respond.options.label -side top -fill x -expand 1
	pack .respond.options.include -side top -anchor nw -fill x
	pack .respond.options.signature -side top -anchor nw -fill x 
	pack .respond.options.prefixlabel -side top -anchor nw
	pack .respond.options.prefix -side top -fill x -anchor nw
	### ok/cancel
	frame .respond.action -relief flat 
	button .respond.action.ok -relief raised -text "OK" \
	    -command respond_ok
	button .respond.action.cancel -relief raised -text "Cancel" \
	    -command "destroy .respond"
	pack .respond.action.ok -side left -padx 5 -pady 3 -fill both \
	    -expand 1
	pack .respond.action.cancel -side right -padx 5 -pady 3 -fill both \
	    -expand 1

	pack .respond.help -side top -fill x
	if {$group != ""} {
	    pack .respond.info -side top -fill x
	}
	pack .respond.type -side top -fill x 
	pack .respond.options -side top -fill x
	pack .respond.action -side bottom -fill x
	respond_help_clear
	grab set .respond

	# update the window, get its geometry, make sure it will fit where
	# we want it, and move it on screen.
	
	update
	set width [winfo width .respond]
	set height [winfo height .respond]
		
	incr xpos -[expr $width/2]
	incr ypos -[expr $width/2]
	
	# adjust if too far to right of screen
	set dif [expr [winfo screenwidth .]-[expr $width+$xpos]]
	if {$dif < 5} {
	    incr xpos [expr $dif-5]
	}

	# adjust if too far to bottom of screen
	set dif [expr [winfo screenheight .]-[expr $height+$ypos]]
	if {$dif < 20} {
	    incr ypos [expr $dif-20]
	}
	
	if {$ypos < 5} {set ypos 5}
	if {$xpos < 5} {set xpos 5}

	incr ypos [expr [winfo vrooty .]*-1]
	incr xpos [expr [winfo vrootx .]*-1]

	wm geometry .respond +$xpos+$ypos
}


proc prompt_delete {w char} {
    set i [catch {set f [$w index sel.first]}]
    set j [catch {set t [$w index sel.last]}]
    set string [$w get]
    if {$i || $j} {
	set t -1
	set f -1
    }
    if {$t != $f} {
	set new [string range $string 0 [expr $f-1]]
	append new [string range $string [expr $t+1] end]
    } else {
	set new [string range $string 0 [expr [string length $string]-2]]
    }
    $w delete 0 end
    $w insert 0 "$new$char"
    bind $w <Any-Key> "$w insert insert %A"
    bind $w <Key-Return> "set PromptReturn 1"
    bind $w <Key-Delete> "prompt_delete $w \"\""
    bind $w <Key-BackSpace> "prompt_delete $w \"\""
}


#
# The prompt dialog is used to query the user for an answer to
# a question.  It may be centered on the pointer, or in a 
# specific position on the screen, as specified in "place".
# The title string is placed in the dialog's title bar, and the
# prompt string is written to its message widget.  If default is
# provided it is placed inside the entry widget.
#
    
proc prompt_dialog {root aspect title prompt default} {
    global PromptReturn

    toplevel $root
    wm title $root $title

    # Determine the position of the pointer through our special
    # call.
    
    set pointer [QueryPointer]
    set xpos [lindex $pointer 0]
    set ypos [lindex $pointer 1]
    incr xpos -10
    incr ypos -10
    
    # adjust for the location of the virtual window if necessary
    
    # put window off in never never land until we know how big it is.
    wm geometry $root "-9999-9999"
    
    # Now place the message widget and the prompt inside the toplevel
    # window.

    frame $root.mess -borderwidth 2 -relief groove
    label $root.mess.bitmap -padx 40 -pady 40 -relief flat \
	-bitmap questhead 
    message $root.mess.message -padx 15 -pady 15 -aspect 200 \
	-text $prompt -relief flat
    pack configure $root.mess.bitmap -side left -fill y
    pack configure $root.mess.message -side right -fill both -expand 1

    entry $root.entry -relief sunken
    $root.entry insert 0 $default
    $root.entry select from 0 
    $root.entry select to end
    bind $root.entry <Any-Key> "prompt_delete $root.entry %A"
    bind $root.entry <Key-Return> "set PromptReturn 1"
    bind $root.entry <Key-Delete> "prompt_delete $root.entry \"\""
    bind $root.entry <Key-BackSpace> "prompt_delete $root.entry \"\""
    frame $root.buttons -relief flat
    button $root.buttons.okay -text "OK" -command {set PromptReturn 1}
    button $root.buttons.cancel -text "Cancel" -command {set PromptReturn 0}
    pack configure $root.buttons.okay -side left -fill both -expand 1
    pack configure $root.buttons.cancel -side right -fill both -expand 1

    pack configure $root.mess -side top -fill both -expand 1
    pack configure $root.entry -side top -fill x -expand 1
    pack configure $root.buttons -side bottom -fill x -expand 1

    # update the window, get its geometry, make sure it will fit where we
    # want it, and move it on screen.
    update
    set width [winfo width $root]
    set height [winfo height $root]
    
    incr xpos -[expr $width/2]
    incr ypos -[expr $height/2]
    
    # adjust if too far to right of screen
    set dif [expr [winfo screenwidth .]-[expr $width+$xpos]]
    if {$dif < 5} {
	incr xpos [expr $dif-5]
    }
    
    # adjust if too far to bottom of screen
    set dif [expr [winfo screenheight .]-[expr $height+$ypos]]
    if {$dif < 20} {
	incr ypos [expr $dif-20]
    }
    
    if {$ypos < 5} {set ypos 5}
    if {$xpos < 5} {set xpos 5}

    incr ypos [expr [winfo vrooty .]*-1]
    incr xpos [expr [winfo vrootx .]*-1]

    # map window to xpos,ypos
    wm geometry $root "+$xpos+$ypos"

    focus $root.entry 
    grab set $root
    set PromptReturn 0
    tkwait variable PromptReturn
    grab release $root
    if {$PromptReturn == 1} {
	set ret [$root.entry get]
    } else {
	set ret ""
    }
    destroy $root
    return $ret
}


#
# This procedure presents the user with a set of options,
# one for each of the arguments in the arg list.
#

proc option_dialog {root aspect title message args} {
    global OptionReturn

    toplevel $root
    wm title $root $title
    
    # Determine the position of the pointer through our special call.
    
    set pointer [QueryPointer]
    set xpos [lindex $pointer 0]
    set ypos [lindex $pointer 1]
    incr xpos -10
    incr ypos -10

    # adjust for the location of the virtual window if necessary
    
    wm geometry $root "+$xpos+$ypos"
    
    message $root.message -text $message -aspect $aspect -relief flat -borderwidth 8
    frame $root.buttons -relief flat -borderwidth 8
    set count 0
    foreach item $args {
	button $root.buttons.$count -text $item \
	    -command "set OptionReturn $count" -padx 4 -pady 2
	pack append $root.buttons \
	    $root.buttons.$count {left fillx expand}
	incr count
    }
    pack append $root \
	$root.message {top frame center} \
	$root.buttons {top frame center fillx expand}
 
    update
    set sw [winfo screenwidth $root]
    set sh [winfo screenheight $root]
    set x [expr $xpos%$sw]
    set y [expr $ypos%$sh]
    set w [winfo reqwidth $root]
    set h [winfo reqheight $root]
    set xpos [winfo rootx $root]
    set ypos [winfo rooty $root]
    set xadj [expr $sw-[expr $w+$x]]
    set yadj [expr $sh-[expr $h+$y]]
    if {$xadj < 0} {
	incr xpos $xadj
    }
    set yadj -$yadj
    if {$yadj > 0} {
	incr ypos $yadj
    }
    incr ypos [expr [winfo vrooty .]*-1]
    incr xpos [expr [winfo vrootx .]*-1]

    wm geometry $root "${w}x${h}+$xpos+$ypos"




    grab set $root
    set OptionReturn -1
    tkwait variable OptionReturn
    grab release $root
    set ret [lindex $args $OptionReturn]
    destroy $root
    return $ret
}
    
proc free_dialog {root aspect title message args} {
    global OptionReturn

    toplevel $root
    wm title $root $title
    
    # Determine the position of the pointer through our special call.
    
    set pointer [QueryPointer]
    set xpos [lindex $pointer 0]
    set ypos [lindex $pointer 1]
    incr xpos -10
    incr ypos -10

    # adjust for the location of the virtual window if necessary
    
    incr ypos [expr [winfo vrooty .]*-1]
    incr xpos [expr [winfo vrootx .]*-1]
    
    # now adjust to keep it on the screen
    
    wm geometry $root "+$xpos+$ypos"
    
    message $root.message -text $message -aspect $aspect -relief flat
    frame $root.buttons -relief flat
    set count 0
    foreach item $args {
	button $root.buttons.$count -text $item \
	    -command "destroy $root"
	pack append $root.buttons \
	    $root.buttons.$count {left fillx expand}
	incr count
    }
    pack append $root \
	$root.message {top frame center} \
	$root.buttons {top frame center fillx expand}
    update
    set sw [winfo screenwidth $root]
    set sh [winfo screenheight $root]
    set x [expr $xpos%$sw]
    set y [expr $ypos%$sh]
    set w [winfo reqwidth $root]
    set h [winfo reqheight $root]
    set xpos [winfo rootx $root]
    set ypos [winfo rooty $root]
    set xadj [expr $sw-[expr $w+$x]]
    set yadj [expr $sh-[expr $h+$y]]
    if {$xadj < 0} {
	incr xpos $xadj
    }
    set yadj -$yadj
    if {$yadj > 0} {
	incr ypos $yadj
    }
    wm geometry $root "${w}x${h}+$xpos+$ypos"
}
    
#
# This procedure displays an error message for the user, and forces them
# to acknowledge it before they can continue.
#
proc error_dialog {mess} {

    set w .errordialog

    puts stdout "Part 1 : $mess."
    toplevel $w -borderwidth 0
    wm title $w "Error Report"
    
    # where is the pointer
    set pointer [QueryPointer]
    set xpos [expr [lindex $pointer 0] - 10]
    set ypos [expr [lindex $pointer 1] - 10]
    
    # adjust our location for the virtual screen if necessary	
    
    # map the window outside of our view so we can get its geometry and
    # make sure it will fit entirely on the screen if placed at our coordinates
    wm geometry $w "-9999-9999"
    
    # build the widget
    frame $w.mess -relief raised -borderwidth 2
    label $w.mess.bitmap -relief flat -padx 40 -pady 40 \
	-bitmap error
    message $w.mess.message -relief flat -padx 15 -pady 5 \
	-text $mess -aspect 200 
    pack configure $w.mess.bitmap -side left -fill y 
    pack configure $w.mess.message -side right -fill both -expand 1
    frame $w.buts -borderwidth 0 -relief raised
    button $w.buts.continue -text "Ok" -padx 10 -pady 5 -relief raised \
	-command "grab release $w ; destroy $w"
    button $w.buts.exit -text "Kill Tknews" -padx 10 -pady 5 -relief raised \
	-command "grab release $w ; destroy $w ; exit"
    pack $w.mess -side top -fill both -expand 1
    pack configure $w.buts.continue -side left -padx 25 -pady 15 -anchor center
    pack configure $w.buts.exit -side right -padx 25 -pady 15 -anchor center
    pack configure $w.buts -side bottom -anchor center
    update
    #grab the window
    grab set $w
    
    # update the window, get its geometry, make sure it will fit where we
    # want it, and move it on screen.
    set width [winfo width $w]
    set height [winfo height $w]
    
    incr xpos -[expr $width/2]
    incr ypos -[expr $height/2]
    
    # adjust if too far to right of screen
    set dif [expr [winfo screenwidth .]-[expr $width+$xpos]]
    if {$dif < 5} {
	incr xpos [expr $dif-5]
    }
    
    # adjust if too far to bottom of screen
    set dif [expr [winfo screenheight .]-[expr $height+$ypos]]
    if {$dif < 20} {
	incr ypos [expr $dif-20]
    }
    
    incr ypos [expr [winfo vrooty .]*-1]
    incr xpos [expr [winfo vrootx .]*-1]

    # map window to xpos,ypos
    wm geometry $w "+$xpos+$ypos"
    
    tkwait window $w
}

