
# atexit -- command to install a Tcl callback to be invoked when
#	 -- the exit command is evalutated.
#
#   exit -- command to exit process, after all callbacks installed by
#	 -- the atexit command have been invoked.
#

#######################################################################
#
# atexit -- manages atexit callbacks.
#

proc atexit {{option list} args} {

  # The option may be append, prepend, insert, delete, clear, set, or list.
  # The args depends on the option specified.
  #

  # The atexit_callbacks list is where we store the 
  # installed atexit callbacks.
  #
  global atexit_callbacks;
  if {[catch {set atexit_callbacks}]} {
    set atexit_callbacks {};
  }

  case $option in {
    set {
      #
      # set callbacks list.
      #
      set atexit_callbacks $args;
    }
    append {
      #
      # append callback to end of the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args : should be "atexit append callback"};
      }
      set callback [lindex $args 0];
      lappend atexit_callbacks $callback;
    }
    prepend {
      #
      # prepend callback to front of the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args : should be "atexit prepend callback"};
      }
      set callback [lindex $args 0];
      set atexit_callbacks "\{$callback\} $atexit_callbacks";
    }
    insert {
      #
      # insert callback before the "before" callback in the callbacks list.
      #
      if {[llength $args] != 2} {
	error {wrong # args : should be "atexit insert before callback"};
      }
      set before   [lindex $args 0];
      set callback [lindex $args 1];
      set l {};
      foreach c $atexit_callbacks {
	if {[string compare $before $c] == 0} {
	  lappend l $callback;
	}
	lappend l $c;
      }
      set atexit_callbacks $l;
    }
    delete {
      #
      # delete callback from the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args : should be "atexit delete callback"};
      }
      set callback [lindex $args 0];
      set l {};
      foreach c $atexit_callbacks {
	if {[string compare $callback $c] != 0} {
	  lappend l $c;
	}
      }
      set atexit_callbacks $l;
    }
    clear {
      #
      # clear callbacks list.
      #
      if {[llength $args] != 0} {
	error {wrong # args : should be "atexit clear"};
      }
      set atexit_callbacks {};
    }
    list {
      #
      # list currently installed callbacks.
      #
    }
    default {
      error {options, append, prepend, insert, delete, clear, set, or list};
    }
  }
  return $atexit_callbacks;
}

#######################################################################
#
# Hide real exit command.
#

rename exit atexit_exit;

#######################################################################
#
# exit -- Wrapper exit command that first invokes all callbacks installed
#      -- by the atexit command before doing real exit.
#

proc exit {{code 0}} {
  global atexit_callbacks;

  while {1} {

    # Every iteration, we rescan atexit_callbacks, in case
    # some callback modifies it.
    #
    if {[catch {set atexit_callbacks} callbacks]} {
      break;
    }
    if {[llength $callbacks] <= 0} {
      break;
    }
    set        callback  [lindex $callbacks 0];
    set atexit_callbacks [lrange $callbacks 1 end];

    catch {uplevel #0 $callback};
  }

  catch {unset atexit_callbacks};
  catch {atexit_exit $code};
}

