
# $Id: fileBrowserC.tcl,v 1.3 1993/06/08 06:10:15 david Exp $
# fileBrowserC.tcl - File Browser class definition. 
#
# AUTHOR: David Herron <david@davids.mmdf.com (home)>, <david@twg.com (work)>
#
# Copyright 1993 David Herron.
# Permission to use, copy, modify and distribute this software
# and its documentation for any purpose and without fee is
# hereby granted, provided that the above copyright notice
# appear in all copies.  David Herron makes no representations
# about the suitability of this software for any purpose save
# printing it out and using the paper as bird cage lining.
#
# $Log: fileBrowserC.tcl,v $
# Revision 1.3  1993/06/08  06:10:15  david
# Add copyright notice.  Change `thisInterpretor' --> [interp this].
#
# Revision 1.2  1993/06/06  06:31:24  david
# Change to new format required by `require'/etc.
#
# Revision 1.1.1.1  1993/06/03  06:23:46  david
# Initial revision of `minterp' as its own package.
#
# Revision 1.3  1993/06/01  04:55:04  david
# Round of testing & fixing & improvement on file browser.  Have streamlined
# creation of a filebrowser object & it's interaction with the outside world.
#
# Revision 1.2  1993/02/02  04:29:11  david
# A bit of cleaning.  Correct some command names to the new and improved ones.
#
# Revision 1.1  1993/01/25  06:32:14  david
# Initial revisions of the interp module, documentation, and file browser.
#
#
#
# The file browser continually presents the contents of a particular
# directory, with the goal of selecting a file.  The user is able to
# change the current directory at will.  The current list of files can
# be limited with a pattern, and the pattern can be modified at any
# time by the user.  Once a file is selected the browser goes away,
# and calls the okCommand.  The cancel button calls cancelCommand, and
# the help button calls helpCommand.
#
# Each place where a path name is shown there are two entry
# boxes.  One for the path component, and the other for
# the file component.  Two such places are shown, one for
# the current directory and file pattern.  The other for
# the last selected file.
#
# METHODS:
#
# new
#
#	Create a new fileBrowser instance.
#
# delete
#
#	Delete a fileBrowser.  Sets MainInterp's DONEvar to "DONE!!" first.
#
# MakeWidgets
#
#	Create the visual components.
#
# setDirectory dirString
#
#	Change directory to the named one.  If dirString ends in ".."
#	then go to the parent.
#
# changeDirectory
#
#	Changes directory to the one stored in $dirEntry.  Finds
#	the files matching the pattern in $patEntry.  Displays
#	all directories there in the directory list, and all matching
#	files in the file list.
#
# setPattern newpat
#
#	Sets the text in $patEntry.
#
# setFile file
#
#	Sets the selected file to be the path from the current
#	directory, and the file name passed in.
#
# create args
#
#	More streamlined way to create a filebrowser dialog.  It encapsulates
#	the needed steps into one easy call into FileBrowserClass::create.
#	The args are two-element lists, the 0'th being a tag to say which
#	option is being set, the 1'th being the value to which it is set.
#
#	The options are:
#
#	-top	The name of the `toplevel' widget to create.  No default,
#		and an error if not specified.
#	-dir	The starting directory.  Defaults to [pwd].
#	-pattern The initial value for the pattern field.  Defaults to "*".
#	-OKvar	The variable which is set to the /path/file when OK is pressed.
#		This variable is set in MainInterp.
#	-DONEvar The variable set to "DONE!!" when either OK or CANCEL
#		is pressed.  This lets you `tkwait' on the variable.
#		This variable is set in MainInterp.
#
# okCommand
#
#	Method to be called when OK button is pressed.  By default
#	this retrieves the /path/file from the dialog and sets
#	MainInterp's OKvar to this value.  Then it calls delete.
#
# cancelCommand
#
#	Method to be called when OK button is pressed.  By default
#	this just calls delete.
#
# helpCommand
#
#	Method called when HELP button is pressed.  Does nothing currently.
#
# OKvar, DONEvar
#
#	Exported variables as described for -OKvar and -DONEvar above.
#

Define_Module FileBrowserClass {


InterpBase -import FileBrowserClass InterpBase [InterpBase -get EXPORTS]
-chain MainInterp [InterpBase -get TkChain]

Method new {} {
	set name [gen_interp_name "fileb"]
	interp new $name

	-import $name InterpBase [InterpBase -get EXPORTS]
	-import $name FileBrowserClass [FileBrowserClass -get EXPORTS]
	$name -exec [list -chain MainInterp [InterpBase -get TkChain]]

	$name -exec {
		-export OKvar
		-export DONEvar
		set OKvar ""
		set DONEvar ""
	}

	return $name
}

Method delete {} {
	global DONEvar
	if {[info exists DONEvar]} {
		MainInterp -set $DONEvar "DONE!!"
	}
	-exit
}


Method MakeWidgets top {
	global topFrame patFrame lstFrame filFrame cmdFrame \
		patLabel dirEntry slashLabel patEntry \
		dirList dirScroll filList filScroll \
		filLabel pathEntry filslashLabel filEntry \
		okBtn canBtn travBtn hlpBtn

	-destroyHook "catch {destroy ${top}}"

	set topFrame ${top}
	set patFrame ${top}.pat
	set lstFrame ${top}.lst
	set filFrame ${top}.fil
	set cmdFrame ${top}.cmd

	frame $topFrame
	frame $patFrame
	frame $lstFrame
	frame $filFrame
	frame $cmdFrame
	pack append  $topFrame \
				$patFrame {top fillx} \
				$lstFrame {top fill expand} \
				$filFrame {top fillx} \
				$cmdFrame {top fillx}
	-chain MainInterp [list $topFrame $patFrame $lstFrame $filFrame $cmdFrame]

	set patLabel   ${patFrame}.l
	set dirEntry   ${patFrame}.dir
	set slashLabel ${patFrame}.slash
	set patEntry   ${patFrame}.pat

	label $patLabel -text Pattern
	entry $dirEntry
	label $slashLabel -text /
	entry $patEntry
	pack append  $patFrame \
				$patLabel   {left fillx} \
				$dirEntry   {left fillx expand} \
				$slashLabel {left fillx} \
				$patEntry   {left fillx expand}
	-chain MainInterp [list $patLabel $dirEntry $slashLabel $patEntry]

	set dirList	${lstFrame}.dl
	set dirScroll	${lstFrame}.ds
	set filList	${lstFrame}.fl
	set filScroll	${lstFrame}.fs

	scrollbar $dirScroll -command        "$dirList   yview"
	listbox   $dirList 	-yscrollcommand "$dirScroll set"
	scrollbar $filScroll -command        "$filList   yview"
	listbox   $filList 	-yscrollcommand "$filScroll set"

	pack append  $lstFrame \
				$dirList   {left fill expand} \
				$dirScroll {left filly} \
				$filList   {left fill expand} \
				$filScroll {left filly}
	-chain MainInterp [list $dirList $dirScroll $filList $filScroll]

	set filLabel ${filFrame}.l
	set pathEntry ${filFrame}.p
	set filslashLabel ${filFrame}.sl
	set filEntry ${filFrame}.e

	label $filLabel -text File
	entry $pathEntry
	label $filslashLabel -text /
	entry $filEntry
	pack append  $filFrame \
				$filLabel {left fillx} \
				$pathEntry {left fillx expand} \
				$filslashLabel {left fillx} \
				$filEntry {left fillx expand}
	-chain MainInterp [list $filLabel $pathEntry $filslashLabel $filEntry]

	set okBtn   ${cmdFrame}.ok
	set canBtn  ${cmdFrame}.can
	set travBtn ${cmdFrame}.trav
	set hlpBtn  ${cmdFrame}.hlp

	button $okBtn   -text OK \
			   -command "[interp this] okCommand"
	button $canBtn  -text Cancel \
			   -command "[interp this] cancelCommand"
	button $travBtn -text {Change Directory} \
			   -command "[interp this] changeDirectory"
	button $hlpBtn  -text Help \
			   -command "[interp this] helpCommand"

	pack append  $cmdFrame \
				$okBtn   {left fillx expand} \
				$canBtn  {left fillx expand} \
				$travBtn {left fillx expand} \
				$hlpBtn  {left fillx expand}
	-chain MainInterp [list $okBtn $canBtn $travBtn $hlpBtn]

	-destroyHook "destroy $top"

	bind $dirEntry <Return> "[interp this] -exec {
				setDirectory \[$dirEntry get\]
				changeDirectory
			}
			$travBtn flash
	"
	bind $patEntry <Return> \
			"[interp this] changeDirectory; $travBtn flash"

	bind $filEntry <Return> \
			"$okBtn flash; update; [interp this] okCommand"


	# Override the unaddorned <1> bindings so that we get
	# notified of any clicks.  This unfortunately means that
	# if the default binding were to change we'd have to be
	# aware of that and change it here.

	bind $dirList <1> "
		%W select from \[%W nearest %y\]
		[interp this] setDirectory \
			\[%W get \[lindex \[%W curselection\] 0\]\]
	"
	bind $dirList <Double-Button-1> "
		%W select from \[%W nearest %y\]
		[interp this] setDirectory \
			\[%W get \[lindex \[%W curselection\] 0\]\]
		[interp this] changeDirectory
		$travBtn flash
	"
	bind $filList <1> "
		%W select from \[%W nearest %y\]
		[interp this] setFile \
			\[%W get \[lindex \[%W curselection\] 0\]\]
	"
	bind $filList <Double-Button-1> "
		%W select from \[%W nearest %y\]
		[interp this] setFile \
			\[%W get \[lindex \[%W curselection\] 0\]\]
		[interp this] okCommand
	"

	return $topFrame
}

# setDirectory - Set the given directory into $dirEntry.  If the last
# component is ".." then strip it & its parent off.  If the length of
# the whole thing is too short when stripping away the ".."  then assume
# we've gone to/through the root and change to `/'.
#
# If the first component is "." then we expand that to be [pwd].
#
# BUG(let): If the string is something weird (like `a/..') then
# the result is `/'.  


Method setDirectory dir {
	global dirEntry
	$dirEntry delete 0 end
	set dl [split $dir "/"]
	if {[lindex $dl 0] == "."} {
		set s [split [pwd] "/"]
		foreach d [lrange $dl 1 end] {lappend s $d}
		set dl $s
		set dlen [llength $dl]
		set dir "/[join [lrange $dl 1 [expr $dlen-1]] /]"
	} else {
		set dlen [llength $dl]
	}
	if {[lindex $dl [expr $dlen-1]] == ".."} {
		if {$dlen <= 3} {
			set dir "/"
		} else {
			set dir "/[join [lrange $dl 1 [expr $dlen-3]] /]"
		}
	}
	$dirEntry insert end $dir
}

Method changeDirectory {} {
	global dirEntry patEntry dirList filList

	set newDir  [$dirEntry get]
	set pattern [$patEntry get]

	if {[catch {set list [glob "${newDir}/${pattern}"]}] != 0} {
		    set list ""
	}
	if {$newDir == "/"} {
		set dirs [list "/.."]
	} else {
		set dirs [list "$newDir/.."]
	}
	set files ""
	foreach f $list {
		if {[file isdirectory $f]}      {
			lappend dirs  $f
			continue
		}
		if {[string match $pattern $f]} {
			set fl [split $f "/"]
			# This should've been just [lindex $fl end]
			set end [expr [llength $fl]-1]
			lappend files [lindex $fl $end]
		}
	}

	catch		 {$dirList delete 0 end}
	foreach d $dirs  {$dirList insert end $d}
	catch 		 {$filList delete 0 end}
	foreach f $files {$filList insert end $f}
}

Method setPattern newpat {
	global patEntry
	$patEntry delete 0 end
	$patEntry insert end $newpat
	changeDirectory
}

Method setFile file {
	global filEntry pathEntry dirEntry
	$filEntry delete 0 end
	$filEntry insert end $file
	$pathEntry delete 0 end
	$pathEntry insert end [$dirEntry get]
}

Method okCommand {} {
	global OKvar
	global pathEntry filEntry
	if [info exists OKvar] {
		MainInterp -set $OKvar "[$pathEntry get]/[$filEntry get]"
	}
	delete
}
Method cancelCommand {} { delete }
Method helpCommand   {} {}

Method create args {

	set pattern "*"
	set dir     [pwd]

	foreach arg $args {
		case [lindex $arg 0] in {
		-top	 {set top	[lindex $arg 1]}
		-dir	 {set dir	[lindex $arg 1]}
		-pattern {set pattern	[lindex $arg 1]}
		-OKvar	 {set OKvar	[lindex $arg 1]}
		-DONEvar {set DONEvar	[lindex $arg 1]}
		}
	}

	if ![info exists top] {
		error "INTERNAL ERROR: A top level widget must be specified."
	}

	set fb [new]
	toplevel $top -class Dialog
	$fb MakeWidgets $top.f
	pack append $top $top.f { top fill expand }
	$fb -exec "-destroyHook \"destroy $top\""
	$fb setDirectory $dir
	$fb setPattern $pattern
	$fb changeDirectory
	$fb -set OKvar $OKvar
	$fb -set DONEvar $DONEvar

	return $fb
}


set EXPORTS {
	delete MakeWidgets
	setDirectory changeDirectory
	setPattern setFile
	okCommand cancelCommand helpCommand
	new create
	}

-export EXPORTS
-export OKvar
-export DONEvar


}
