/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute 
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: Receiver.c
 *      This file contains the low level creation and deletion of Receivers.
 *      These functions are based on George Drapeau's MAEstro Receiver.c.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)Receiver.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include "tcl.h"
#include "tclIPC.h"            /* RPC stuff */
#include "tclipcP.h"           /* Private structures */
#include "tclipc.h"            /* Public interface */

/* In file prototypes */
static int CreateSocketOnPort _ANSI_ARGS_((Tcl_Interp*, int*));

/*
 *--------------------------------------------------------------
 *
 * CreateSocketOnPort
 *
 *	This function attempts to create a socket, bind an address
 *	to that socket, and return that bound socket to the caller.
 *	The function takes as argument a pointer to an integer.
 *	The integer pointed to should contain the port number
 *	being requested (this function allows the caller to
 *	request a specific port number).  If the port number
 *	passed in as argument is set to the defined constant
 *	"AnyPort", this function will let the kernel pick a port
 *	number.
 *	The function returns the socket descriptor, and will also
 *	replace the portNumber passed in as argument with the
 *	port number to which the socket is bound.
 *	If the function fails, a zero (0) will be returned, and
 *	the port number passed in as argument will be set to 0.
 *
 * Results:
 *      Returns a port or 0 if it didn't work.
 *--------------------------------------------------------------
 */
static int CreateSocketOnPort(interp, portNumber)
Tcl_Interp *interp;
int *portNumber;
{
  int newSocket;
  struct sockaddr_in address;
  int addressLength = sizeof(struct sockaddr_in);
  int result;
  
  /* Try to create a new socket */
  if ((newSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
    Tcl_AppendResult(interp, 
		     "IPC Protocol: Unable to create a socket.",
		     (char *)NULL);
    *portNumber = 0;
    return(0);
  }
  /* Clear out the address to which to bind the socket */
  bzero((char *)&address, sizeof (address));
  address.sin_family = AF_INET;
  address.sin_port = *portNumber;
  /* Attempt to make the socket use the port number passed in */
  result = bind(newSocket, (struct sockaddr *)&address,	addressLength);
  if (result == -1) {
    Tcl_AppendResult(interp, 
        "IPC Protocol: Unable to reserve the port the PortManager requests.",
		       (char *)NULL);
    *portNumber = 0;
    return(0);
  }
  *portNumber = ntohs(address.sin_port);
  return newSocket;
}

/*
 *--------------------------------------------------------------
 *
 * NewReceiver
 *
 *      Create a new Receiver structure and set up the RPC transport.
 *
 * Results:
 *      Returns a new receiver or NULL if it fails.
 *--------------------------------------------------------------
 */
Receiver NewReceiver(portMgrSender, appName, receivingPort)
Sender portMgrSender;
char* appName;
int receivingPort;
{
  Receiver receiver;
  int transportSocket;
  int tempPortNumber = 0;

  /* The dispatcher is located in tclIPC_ksvc.c */
  extern void tclipc_4 _ANSI_ARGS_((struct svc_req *rqstp, 
				    SVCXPRT *transp));
  
  /* Create a new receiver */
  tempPortNumber = receivingPort;
  receiver = (Receiver)ckalloc(sizeof(ReceiverR));
  receiver->receiveSocket = 0;		    /* Initialize member variables  */
  receiver->transport = (SVCXPRT*) NULL;
  receiver->receivePort.portNumber = 0;
  receiver->receivePort.appName = NULL;
  receiver->receivePort.hostName = NULL;
  /* The received commands will execute in the passed interpreter */
  receiver->interp = portMgrSender->interp;

  /* Unregister this protocol with the PortMapper every time */
  pmap_unset(TCLIPC, FIRSTTESTVERSION);

  /* Attempt to create a socket bound to the receivingPort */
  transportSocket = CreateSocketOnPort(portMgrSender->interp, &tempPortNumber);
  if (transportSocket == 0) {
    /* failure */
    Tcl_AppendResult(portMgrSender->interp,
            "Can not create a new socket.\n",
            (char *)NULL);
    ckfree((char *)receiver);
    return((Receiver)NULL);
  }

  /* Attempt to create port on which to listen for messages*/
  /* Creates a TCP RPC service, zeros give default send/receive buffer sizes */
  receiver->transport = svctcp_create(transportSocket,0,0);
  if (receiver->transport == NULL) {
    /* Failure */
    Tcl_AppendResult(portMgrSender->interp, 
                     "Can not create the tcp service.",
                     (char *)NULL);
    close(transportSocket);
    ckfree((char *)receiver);
    return((Receiver)NULL);
  }

  /* Attempt to associate this version of the Link...*/
  /* The final 0 means don't register this pair with the PortMapper */
  if (!svc_register(receiver->transport, TCLIPC,
		    FIRSTTESTVERSION,
		    tclipc_4, 0)) {
    /* Failure -- report the error and return an error code*/
    Tcl_AppendResult(portMgrSender->interp,
		    "Unable to register (TCLIPC, FIRSTTESTVERSION, tcp).",
                    (char *)NULL);
    close(transportSocket);
    ckfree((char *)receiver);
    return((Receiver)NULL);
  }
  /* Fill in the name of this application*/
  receiver->receivePort.appName = strdup(appName);

  /* Fill Port structure with info from RPC service creation*/
  receiver->receivePort.portNumber = receiver->transport->xp_port;
  receiver->receivePort.hostName = strdup(ipcThisHostName);
  receiver->receiveSocket = receiver->transport->xp_sock; 
 
  /* If the Port Manager Sender is valid (i.e. not the Portmanager) ... */
  /* This registers with port manager in PortManUser.c */
  if (SenderPortNumber(portMgrSender) != 0) {
    if (pmConnectWithPortMgr(portMgrSender, 
                             &(receiver->receivePort)) != TCL_OK) {
      /* Destroy the RPC service data structures */
      svc_destroy(receiver->transport);
      /* Free space taken by Port fields.  Cannot use... */
      free(receiver->receivePort.appName);
      free(receiver->receivePort.hostName);
      /* Free space taken by the receiver */
      ckfree(receiver);
      receiver = (Receiver)NULL;
    }
  }
  return receiver;
}

/*
 *--------------------------------------------------------------
 *
 * DestroyReceiver
 *
 *      Low level routine to disconnect an entry in the PortManager
 *      and to free the receiver resources.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void DestroyReceiver(portMgrSender, receiver)
Sender portMgrSender;
Receiver receiver;
{
  int result = 0;
  
  /* Was the Sender passed in valid? */
  if (portMgrSender != NULL && SenderPortNumber(portMgrSender) != 0) {
    /* Yes, tell the Port Manager "i'm outtahere" */
    result = pmDisconnectFromPortMgr(portMgrSender,
				 &(receiver->receivePort));
    if (result != TCL_OK) {
      Tcl_AppendResult(portMgrSender->interp,
		 "IPC Protocol: Could not disconnect from the Port Manager.",
		       (char *)NULL);
    }
  }
  /* Destroy the RPC service data structures */
  svc_destroy(receiver->transport);
  /* Free space taken by Port fields.  Cannot use... */
  if (receiver->receivePort.appName != (char*)NULL) {
    /* ...DestroyPort() here, since receiver->receivePort is... */
    free(receiver->receivePort.appName);
  }
  /* allocated at compile time (i.e., not by malloc()) */
  if (receiver->receivePort.hostName != (char*)NULL) {
    free(receiver->receivePort.hostName);
  }
  /* Free space taken by the receiver */
  ckfree(receiver);
}

/*
 *--------------------------------------------------------------
 *
 * ipcGetSenderReceiver
 *
 *      Return a valid Receiver's socket.
 *
 * Results:
 *      an integer socket
 *--------------------------------------------------------------
 */
int ipcGetReceiverSocket(receiver)
Receiver receiver;
{
  return(receiver->receiveSocket);
}
