/*
 * tclVogle.c --
 *
 * Set of tcl functions to work in concert with the graphic tk widget.
 */
#include <stdio.h>
#include <unistd.h> /* access(2) */
#include <errno.h> /* strerror(errno) */
#include <tk.h>
#include <vogle.h>
#include <assert.h>
#include <vogle_inlines.h>

#define FLOAT_STR_SIZE 64
#define INT_STR_SIZE 32
#define MAX_DEV_NAME_LEN 100

/*
 *macros 
 */
#define strEQ(a,b) (*(a)==*(b)&&!strcmp(a,b))

#define Set_Float(interp, scratch, varname, f) \
	sprintf(scratch, "%g", f), Tcl_SetVar(interp, varname, scratch, 0)


g_dblRetDblCmd(cdata, interp, argc, argv)
    double (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    char buf[FLOAT_STR_SIZE];
    double d;
    if (argc != 2 || (Tcl_GetDouble(interp, argv[1], &d) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "floatArg");
    }
    sprintf(buf, "%g", (*cdata)(d));
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

/*
 * e.g.
 *	g_init
 *	g_newdev
 */
g_deviceStrCmd(cdata, interp, argc, argv)
    int (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    if (argc != 2)
	return args_should_be(interp, argv[0], "device_name");
    (void) (*cdata)(argv[1]);
    return TCL_OK;
}

g_newdevCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    if (argc != 2)
	return args_should_be(interp, argv[0], "device_name");
    /* 
     * if no current device, init vogle package to the postscript device 
     */
    if (!CurrentVogleDevice()) {
	voutput("/dev/null");
	vinit("postscript"); 
    }
    (void) vnewdev(argv[1]);
    return TCL_OK;
}

/* 
 * Redirect output if any to file argv[1] for current device.
 *
 * See vogle voutput() documentation
 */
g_outputCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    extern FILE *_voutfile();

    if (argc != 2)
	return args_should_be(interp, argv[0], "file_name");

    /* 
     * Do the open, then check it. Awkward, but the only way to do it 
     * with vogle as-is
     */
    voutput(argv[1]);
    if (_voutfile() == (FILE *)NULL) {
	Tcl_AppendResult(interp,  argv[0], 
		": could not open file '", argv[1], "': ", 
		strerror(errno),  (char *)NULL);
	return TCL_ERROR;
    } else {
	return TCL_OK;
    }
}

g_getdevCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    Tcl_AppendResult(interp, CurrentVogleDevice(), (char *)NULL);
    return TCL_OK;
}

g_getdepthCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    char buf[8];
    if (argc != 1)
	return no_args_allowed(interp, argv[0]);

    sprintf(buf, "%d", getdepth());
    Tcl_AppendResult(interp, buf, (char *)NULL);
    return TCL_OK;
}

g_viewportCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double left, right, top, bottom;
    if (argc!=5) {
	return args_should_be(interp, argv[0], "left right bottom top");
    }
    if ((Tcl_GetDouble(interp, argv[1], &left) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &right) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &bottom) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &top) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    viewport((float)left, (float)right, (float)bottom, (float)top);
    return TCL_OK;
}


g_getviewportCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    float left,right,bottom,top;
    char buf[FLOAT_STR_SIZE*4];
    if (argc != 1) 
	return no_args_allowed(interp, argv[0]);
    getviewport(&left, &right, &bottom, &top);
    sprintf(buf, "{ %g %g %g %g }", left, right, bottom, top);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

g_getaspectCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    float aspect;
    char buf[FLOAT_STR_SIZE];
    if (argc != 1) 
	return no_args_allowed(interp, argv[0]);
    getaspect(&aspect);
    sprintf(buf, "%g", aspect);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

g_orthoCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double left,right,bottom,top,near_d,far_d;
    if (argc != 7 && argc != 5) {
	return args_should_be(interp, argv[0], "left right bottom top ?near_d? ?far_d?");
    }
    if ((Tcl_GetDouble(interp, argv[1], &left) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &right) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &bottom) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &top) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    if (argc == 5) {
	/* 
	 * 2d 
	 */
	ortho2((float)left, (float)right, (float)bottom, (float)top);
    } else {
	/* 
	 * 3d 
	 */
	if ((Tcl_GetDouble(interp, argv[5], &near_d) == TCL_ERROR) ||
	    (Tcl_GetDouble(interp, argv[6], &far_d) == TCL_ERROR)) {
	    return TCL_ERROR;
	}
	ortho((float)left, (float)right, (float)bottom, (float)top,
	  (float)near_d, (float)far_d);
    }
    return TCL_OK;
}

g_perspectiveCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double fov,aspect,near,far;
    if (argc != 5) {
	return args_should_be(interp, argv[0], "fov aspect near far");
    }
    if ((Tcl_GetDouble(interp, argv[1], &fov) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &aspect) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &near) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &far) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    perspective((float)fov, (float)aspect, (float)near, (float)far);
    return TCL_OK;
}

g_windowCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double left,right,bottom,top,near_d,far_d;
    if (argc != 7)
	return args_should_be(interp, argv[0], 
		"left right bottom top near_d far_d");

    if ((Tcl_GetDouble(interp, argv[1], &left) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &right) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &bottom) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &top) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[5], &near_d) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[6], &far_d) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    window((float)left, (float)right, (float)bottom, (float)top,
	  (float)near_d, (float)far_d);
    return TCL_OK;
}

g_polarviewCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double dist,azim,inc,twist;
    if (argc != 5)
	return args_should_be(interp, argv[0], "dist azim inc twist");
    if ((Tcl_GetDouble(interp, argv[1], &dist) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &azim) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &inc) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &twist) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    polarview((float)dist, (float)azim, (float)inc, (float)twist);
    return TCL_OK;
}

g_lookatCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double vx,vy,vz,px,py,pz,twist;
    if (argc != 8)
	return args_should_be(interp, argv[0], "vx vy vz px py pz twist");
    if ((Tcl_GetDouble(interp, argv[1], &vx) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &vy) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &vz) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &px) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[5], &py) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[6], &pz) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[7], &twist) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    lookat(
	(float)vx,(float)vy,(float)vz,
	(float)px,(float)py,(float)pz,
	(float)twist);
    return TCL_OK;
}


g_hatchpitchCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double pitch;
    if ((argc != 2) || (Tcl_GetDouble(interp, argv[1], &pitch) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "pitch");
    }
    hatchpitch((float)pitch);
    return TCL_OK;
}

static float Pbuf[MAXVERTS][3];

g_polyCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    register int i;
    int dimension = (int)cdata;
    char buf[32];

    argc--,argv++;
    if (!(argc%dimension)) {
	if (argc/dimension > MAXVERTS) {
	    sprintf(buf, "%d", argc);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    argc=MAXVERTS*dimension;
	}
	argc /= dimension;
	for(i=0; i < argc; i++, argv+=dimension) {
	    switch(dimension) {
	    case 2:
		if ((Tcl_GetFloat(interp, argv[0], &(Pbuf[i][0])) != TCL_OK) ||
		    (Tcl_GetFloat(interp, argv[1], &(Pbuf[i][1])) != TCL_OK)) {
		    return(TCL_ERROR);
		}
		Pbuf[i][2] = 0.0;
		break;
	    case 3:
		if ((Tcl_GetFloat(interp, argv[0], &(Pbuf[i][0])) != TCL_OK) ||
		    (Tcl_GetFloat(interp, argv[1], &(Pbuf[i][1])) != TCL_OK) ||
		    (Tcl_GetFloat(interp, argv[2], &(Pbuf[i][2])) != TCL_OK)) {
		    return(TCL_ERROR);
		}
		break;
	    default:
		assert(0);
	    }
	}
	poly(argc, Pbuf);
	return(TCL_OK);
    } else {
	char *should;
	argc++;argv--;
	switch(dimension) {
	    case 2: should = "x0 y0...xN yN";		break;
	    case 3: should = "x0 y0 z0...xN yN zN";	break;
	    default: assert(0);
	}
	return args_should_be(interp, argv[0], should);
    }
}

g_fontCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    if (argc != 2)
	return args_should_be(interp, argv[0], "font_name"); 
    font(argv[1]);
    return TCL_OK;
}

g_strlengthCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    char buf[FLOAT_STR_SIZE];
    if (argc != 2)
	return args_should_be(interp, argv[0], "string");
    sprintf(buf, "%g", strlength(argv[1]));
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

g_boxtextCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,w,h;
    if ((argc != 6) ||
	(Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &w) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &h) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "x y width height string");
    }
    boxtext((float)x,(float)y,(float)w,(float)h, argv[5]);
    return TCL_OK;
}

g_boxfitCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double w,h;
    int nchars;
    if ((argc != 4) ||
	(Tcl_GetDouble(interp, argv[1], &w) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &h) == TCL_ERROR) ||
        (Tcl_GetInt(interp, argv[3], &nchars) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "width height numberOfChars");
    }
    boxfit((float)w,(float)h,nchars);
    return TCL_OK;
}

g_rotateCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double angle;
    if ((argc != 3) ||
	(Tcl_GetDouble(interp, argv[1], &angle) == TCL_ERROR) ||
	((*argv[2] != 'x') && (*argv[2] != 'y') && (*argv[2] != 'z'))) {
	return args_should_be(interp, argv[0], "angle axis (x|y|z)");
    }
    rotate((float)angle,*argv[2]);
    return TCL_OK;
}

g_isobjCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    int objectId;
    if ((argc != 2) ||
	(Tcl_GetInt(interp, argv[1], &objectId) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "object_identifier");
    }
    Tcl_SetResult(interp, isobj(objectId) ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

g_loadobjCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    /* possible enhancements: */
    /* -- put in tilde expansion here */
    /* -- look in special directory is no leading directories are specified */
    int objectId, r;
    if ((argc != 3) ||
	(Tcl_GetInt(interp, argv[1], &objectId) == TCL_ERROR) ||
	(r = access(argv[2], R_OK) != 0)) {
	if (r != 0) {
	    Tcl_AppendResult(interp, 
		"cannot read file ", "'",argv[2],"'", strerror(errno),  " ",
		(char *)NULL);
	}
	return args_should_be(interp, argv[0], "object_identifier filename");
    }
    loadobj(objectId, argv[2]);
    return TCL_OK;
}

g_saveobjCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    int objectId;
    if ((argc != 3) ||
	(Tcl_GetInt(interp, argv[1], &objectId) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "object_identifier filename");
    }
    saveobj(objectId, argv[2]);
    return TCL_OK;
}

g_getgpCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    float x,y,z;
    char buf[(FLOAT_STR_SIZE*3)+16];
    x = y = z = 0.0;
    getgp(&x,&y,&z);
    switch(argc) {
    case 4:
	Set_Float(interp, buf, argv[3], z);
    case 3:
	Set_Float(interp, buf, argv[2], y);
    case 2:
	Set_Float(interp, buf, argv[1], x);
    case 1:
	sprintf(buf, "{%g %g %g}", x, y, z);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;
    default:
	return args_should_be(interp, 
	    argv[0], "?x_varname? ?y_varname? ?z_varname?");
    }
}

/* 
 * Convert locatorX & locatorY to desired coordinate system
 * the conversion type is passed via ClientData
 */

g_cvtcoordsCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    int locatorX,locatorY;
    Vector res, v;
    Matrix de_transform;

    char buf[(FLOAT_STR_SIZE*3)+16];

    if ((argc < 3) || (argc > 6) ||
        (Tcl_GetInt(interp, argv[1], &locatorX) == TCL_ERROR) ||
        (Tcl_GetInt(interp, argv[2], &locatorY) == TCL_ERROR)) {
	return args_should_be(interp, 
	    argv[0], "locatorX locatorY ?x_varname? ?y_varname? ?z_varname?");
    }
    if (vdevice.initialised) {
	switch((int)cdata) {

	/* 
	 * convert the supplied pair to screen coordinates
	 */
	case 's':

	    VtoWxy((float)locatorX, (float)(((int)vdevice.sizeSy)-locatorY), 
		&(v[0]), 
		&(v[1]));
	    v[0] += 1.0;
	    v[1] += 1.0;
	    v[0] *= vdevice.sizeX/2.0;
	    v[1] *= vdevice.sizeY/2.0;
	    break;

	/* 
	 * convert the supplied pair to world coordinates
	 */
	case 'w':

	    /*
	     * get the de-transform matrix 
	     */
	    copymatrix(de_transform, vdevice.transmat->m);

	    /*
	     * build a vector holding the point to be de-transformed
	     */
	    v[0] = (float)locatorX;
	    v[1] = (float) (((int)vdevice.sizeSy)-locatorY), 
	    v[2] = .0;
	    v[2] = 1.0;

	    /*
	     * de-transform it
	     */
	    multvector(res, v, de_transform);

	    VtoWxy(res[0], res[1],
		&(v[0]), 
		&(v[1]));
	    break;

	default:
	    assert(0);

	}
    } else {
	Tcl_AppendResult(interp, 
	    " ", argv[0], ": no vogle device currently exists", 
	    (char *)NULL);
	return TCL_ERROR;
    }
    /*
     * set optional variables if supplied
     */
    switch(argc) {
    case 6:
	Set_Float(interp, buf, argv[5], 0.0);
    case 5:
	Set_Float(interp, buf, argv[4], v[1]);
    case 4:
	Set_Float(interp, buf, argv[3], v[0]);
    }
    sprintf(buf, "{%g %g 0.0}", v[0], v[1]);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}


/*
 * generic command for function of form
 *
 * command
 *
 * Which returns an integer value upon successful completion
 */
g_noArgsReturnIntCmd(cdata, interp, argc, argv)
    int (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    char buf[INT_STR_SIZE];
    if (argc != 1)
	return no_args_allowed(interp, argv[0]);
    sprintf(buf, "%d", (*cdata)());
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}


/*
 * generic command for function of form
 *
 * command object_number
 *
 * Which returns nothing (but the empty string) upon success
 */
g_objargCmd(cdata, interp, argc, argv)
    int (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    int o;
    if ((argc != 2) ||
	(Tcl_GetInt(interp, argv[1], &o) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "object_number");
    }
    (void) (*cdata)(o);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command string
 *
 * Which returns nothing (but the empty string) upon success
 */
g_drawstrCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    /* 
     * hack -- turn off attempts to clip hardware fonts 
     */
    /* int x = vdevice.clipoff; vdevice.clipoff = 1;
    */

    if (argc != 2)
	return args_should_be(interp, argv[0], "string");
    drawstr(argv[1]);

    /* vdevice.clipoff = x;
    */
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command ?wVar? ?hVar?
 *
 * Which returns a 2 element list of the form {width height}
 * if wVar is supplied, then width is also placed into wVar
 * if hVar is supplied, then height is also placed into wVar
 *
 */
g_noArgsReturnWHCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    float w,h;
    char buf[(FLOAT_STR_SIZE*2) + 4];
    (void) (*cdata)(&w, &h);

    switch(argc) {
    case 3:
	Set_Float(interp, buf, argv[1], h);
    case 2:
	Set_Float(interp, buf, argv[2], w);
    case 1:
	sprintf(buf, "{%g %g}", w, h);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	break;
    default:
	return args_should_be(interp, argv[0], "?wVar? ?hVar?");
    }
    return TCL_OK;
}

/* g_getcharsize c ?wVar ?hVar */
g_getcharsizeCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    float w,h;
    char buf[(FLOAT_STR_SIZE*2) + 4];
    if (argc>1) {
	(void) getcharsize(*(argv[1]), &w, &h);
    }
    switch(argc) {
    case 4:
	Set_Float(interp, buf, argv[2], h);
    case 3:
	Set_Float(interp, buf, argv[3], w);
    case 2:
	sprintf(buf, "{%g %g}", w, h);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	break;
    default:
	return args_should_be(interp, argv[0], "char ?wVar? ?hVar?");
    }
    return TCL_OK;
}


/*
 * generic command for function of form
 *
 * command angle
 *
 * Which returns nothing (but the empty string) upon success
 *
 */

g_angleCmd(cdata, interp, argc, argv)
    int (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double angle;
    if ((argc != 2) || (Tcl_GetDouble(interp, argv[1], &angle) == TCL_ERROR))
	return args_should_be(interp, argv[0], "angle");
    (void) (*cdata)((float)angle);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command x y x2 y2
 *
 * Which returns nothing (but the empty string) upon success
 *
 */

g_rectargsCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,x2,y2;
    if (argc != 5)
	return args_should_be(interp, argv[0], "x y x2 y2");
    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &x2) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &y2) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    (void) (*cdata)((float)x,(float)y,(float)x2,(float)y2);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command x y radius start_angle end_angle
 *
 * Which returns nothing (but the empty string) upon success
 *
 */

g_arcargsCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,radius,startangle,endangle;
    if (argc != 6)
	return args_should_be(interp, argv[0], 
			"x y radius startangle endangle");

    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &radius) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &startangle) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[5], &endangle) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    (void) (*cdata)((float)x,(float)y,(float)radius,
		(float)startangle,(float)endangle);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command positive_integer
 *
 * Which returns nothing (but the empty string) upon success
 *
 *  
 */
g_posintCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    int i;
    if ((argc != 2) || (Tcl_GetInt(interp, argv[1], &i) == TCL_ERROR) || i<0) {
	return args_should_be(interp, argv[0], "number");
    }
    (void) (*cdata)(i);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command
 *
 * Which returns nothing (but the empty string) upon success
 *
 * e.g.
 *	g_clear
 */
g_noArgsCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    if (argc != 1) 
	return no_args_allowed(interp, argv[0]);
    (void) (*cdata)();
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command boolean
 *
 * Which returns nothing (but the empty string) upon success
 *
 * e.g.
 *	g_clipping
 *	g_fixedwidth
 */
g_booleanCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    int b;
    if (argc != 2)
	return args_should_be(interp, argv[0], "boolean");
    if (Tcl_GetBoolean(interp, argv[1], &b) != TCL_OK)
	return TCL_ERROR;
    (void) (*cdata)(b);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command x y
 * e.g.
 *	g_smove
 *	g_rsmove
 */
g_XYCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double x,y;
    if ((argc != 3) ||
	(Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "x y");
    }
    (void) (*cdata)((float)x,(float)y);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command x y radius
 *
 */
g_XYRadiusCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,z;
    if ((argc != 4) ||
	(Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
	(Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &z) == TCL_ERROR)) {
	return args_should_be(interp, argv[0], "x y radius");
    }
    (void) (*cdata)((float)x,(float)y, (float)z);
    return TCL_OK;
}

/*
 * generic command for function of form
 *
 * command x y ?z?
 * e.g.
 *	g_up
 * 	g_move
 *	g_rmove
 *
 */
g_XYoptZCmd(cdata, interp, argc, argv)
    int (*cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,z;
    if (argc != 3 && argc != 4) {
	return args_should_be(interp, argv[0], "x y ?z?");
    }
    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    if (argc == 3) {
	z = 0.0;
    } else if (Tcl_GetDouble(interp, argv[3], &z) == TCL_ERROR) {
	return TCL_ERROR;
    }
    (void) (*cdata)((float)x,(float)y,(float)z);
    return TCL_OK;
}

g_scaleCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,z;
    if (argc != 3 && argc != 4) {
	return args_should_be(interp, argv[0], "x y ?z?");
    }
    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    if (argc == 3) {
	z = 1.0;
    } else if (Tcl_GetDouble(interp, argv[3], &z) == TCL_ERROR) {
	return TCL_ERROR;
    }
    scale((float)x,(float)y,(float)z);
    return TCL_OK;
}

g_colorCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    int index = -1;
    char *dev;

    if (argc != 2) {
	return args_should_be(interp, argv[0], "colorname");
    }

    dev = CurrentVogleDevice();
    if (*dev == '.') {
	index = TkGraphic_AllocColor(NULL, argv[1]);
	if (index < 0) {
	    Tcl_AppendResult(interp, 
		argv[0], ": could not allocate color '", argv[1], "'", NULL);
	    return TCL_ERROR;
	}
    } else {
	/* some other device */
	/*- 
	 * XXX needs work right now just assumes canned black and white is 
	 * all that is available 
	 */
	if (!strncmp(argv[1], "black")) {
	    index = 0; /* black */
	} else {
	    index = 7; /* else white */
	}
    }
    color(index);
    return TCL_OK;
}

g_pointCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,z;
    if (argc != 3 && argc != 4) {
	return args_should_be(interp, argv[0], "x y ?z?");
    }
    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    if (argc == 3) {
	z = 0.0;
    } else if (Tcl_GetDouble(interp, argv[3], &z) == TCL_ERROR) {
	return TCL_ERROR;
    }
    Point((float)x,(float)y,(float)z);
    return TCL_OK;
}

/* 
 * g_rectangle, a 		--> x y w h 
 * version of g_rect which is 	--> x y x' y' 
 */
g_rectangleCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    double x,y,x2,y2;
    if (argc != 5)
	return args_should_be(interp, argv[0], "x y w h");
    if ((Tcl_GetDouble(interp, argv[1], &x) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[2], &y) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[3], &x2) == TCL_ERROR) ||
        (Tcl_GetDouble(interp, argv[4], &y2) == TCL_ERROR)) {
        return TCL_ERROR;
    }
    rect((float)x,(float)y,(float)(x+x2),(float)(y+y2));
    return TCL_OK;
}

/*
 * only applies to graphic widget
 * XXX need to update to use vogle
 */
g_flushCmd(cdata, interp, argc, argv)
    ClientData cdata; Tcl_Interp *interp; int argc; char **argv;
{
    return FlushGraphic(CurrentVogleDevice());
}


/*
 * Set the current transformation to be the identity matrix
 */
g_identtransCmd(cdata, interp, argc, argv)
    int (* cdata)(); Tcl_Interp *interp; int argc; char **argv;
{
    Matrix ident_m;
    if (argc != 1)
	return no_args_allowed(interp, argv[0]);
    identmatrix(ident_m);
    loadmatrix(ident_m);
    return TCL_OK;
}
