#---------------------------------------------------------------------------
#
#	File:	the-methods.tcl
#	Title:	Module Pathname Construction and Module Loading Functions
#	Author:	Juergen Wagner (J_Wagner@iao.fhg.de)
#	Date:	Aug 1992
#
# 93-01-19 J_Wagner@iao.fhg.de
#	Modified to use load object.
#
#---------------------------------------------------------------------------

global a b
catch "unset a b"

proc default {var index value} {

  global $var

  if { ! [info exists ${var}($index)] } {
    set ${var}($index) $value
  }
  return $value
}

#---------------------------------------------------------------------------
#
#	The module loader knows about two locations of directories:
#	o  the library directory,
#	o  the system directory
#
#	The library directory is given by "system(loc,library)". If the
#	environment variable TCLTK is set, it overrides that value. This
#	directory contains the toolkit itself and useful utilities.
#
#	The system directory contains all files necessary to implement
#	a family of applications. It is defined by "system(loc,system)"
#	which is the concatenation of "/home/" and the lowercase version
#	of "system(system)". It may be overridden by an environment variable
#	whose name if the uppercase version of "system(system)".
#
#	Under each directory root, a subset of the following hierarchy
#	is assumed:
#		./
#		  Software/
#		    $system(library):Tk/
#		    $system(library):Bin/
#		    $system(system):Tk/
#		    $system(system):Bin/
#		    $system(application):Tk/
#		    $system(application):Bin/
#		  Images/
#		    xbm/
#		    ppm/
#		  Help/
#		  Projects/
#		    $system(application):$system(project)/
#
#	Enable "system(autoload)" if you want to use the autoloader. The
#	main library directory will be added to the autoload path. In that
#	case, "the-toolkit.tcl" won't load all the files. Note: This has not
#	been tested thoroughly!
#
#	The variable "system(verbose)" controls the verbosity of the system
#	with respect to loading and object creation: no messages (0), both
#	types of message (1), loading messages only (2).
#

global system

default system	verbose		1

default system	autoload	0

default system	library		Library
default system	system		Demo

set system(systemL)	[string tolower $system(system)]
set system(systemU)	[string toupper $system(system)]

default system	loc,library	"/usr/local/lib/theObjects"
default system	loc,system	"/home/$system(systemL)"

default system	path		[list $env(HOME)]

#---------------------------------------------------------------------------
#
#--	Check for presence of extensions in current Tcl/Tk
#
proc check(extension) {args} {
  global system

  foreach name $args {
    if { [info command $name] == {} } {
      set system(has_$name) 0
      proc $name {args} "puts stdout {** $name not available.}; return 0"
    } {
      set system(has_$name) 1
      if { $system(verbose) } {
	puts stdout "** Found extension $name"
      }
    }
  }
  return
}

check(extension) photo xygraph htext opencons warp defobject

#---------------------------------------------------------------------------
#
#--	Location of library files
#

if { [info exists env(TCLTK)] } then {
  set system(loc,library) $env(TCLTK)
  if { $system(verbose) } {
    puts stdout "** Override system(loc,library): $system(loc,library)"
  }
}

if { [info exists env($system(systemU))] } {
  set system(loc,system) $env($system(systemU))
  if { $system(verbose) } {
    puts stdout "** Override system(loc,system): $system(loc,system)"
  }
}

lappend system(path) $system(loc,system) $system(loc,library)

proc _check(file) {path name {default ""}} {
  global system

  foreach dir $system(path) {
    set file "$dir/$path/$name"
    if { [file exists $file] == 1 } then {
      return $file
    }
  }

  if { $default == {} } then {
    error "** Can't find file $path/$name"
  }

  if { $default == "~" } {
    set default "[lindex $system(path) 0]/$path/$name"
  }

  return $default
}

#---------------------------------------------------------------------------

proc the(command) {name} {
  global system

  foreach sys [list $system(application) $system(system) Library] {
    set f [_check(file) Software/${sys}:Bin $name -]
    if { $f != "-" } {
      return $f
    }
  }
  error "** Can't find program $name"
}

proc the(module) {name file} {
  _check(file) Software/${name}:Tk ${file}.tcl
}

proc the(image) {file {type xbm}} {
  _check(file) Images/${type} ${file}
}

proc the(project) {file} {
  global system

  _check(file) Projects/$system(application):$system(project) ${file}
}

#---------------------------------------------------------------------------

proc load(file) {name file {catchp 0}} {
  global system

  set f [the(module) $name $file]
  if { $system(verbose) } {
    set liblength [expr [string length $system(loc,library)]-1]
    set libprefix [string range $f 0 $liblength]
    if { $libprefix == [string range $system(loc,library) 0 $liblength] } {
      set fname "LIBRARY[string range $f [expr $liblength+1] end]"
    } {
      set fname $f
    }
    puts stdout "** Load $fname"
    flush stdout
  }
  if { $catchp } {
    if { [catch "uplevel #0 source $f"] } {
      puts stdout "  ** Load of $file failed"
      flush stdout
    }
  } {
    uplevel #0 source $f
  }
  return $f
}

#---------------------------------------------------------------------------

proc load(lib) {args} {
  global system

  set dir $system(library)
  foreach file $args {
    load(file) $dir $file
  }
}

proc load(sys) {args} {
  global system

  set dir $system(system)
  foreach file $args {
    load(file) $dir $file
  }
}

proc load(app) {args} {
  global system

  set dir $system(application)
  foreach file $args {
    load(file) $dir $file
  }
}

proc load(project) {file} {
  global system

  set f [the(project) ${file}.tcl]
  if { $system(verbose) } {
    puts stdout "** Project file $f"
    flush stdout
  }
  uplevel #0 source $f
  return $f
}

proc load(toolkit) {} {
  load(lib) the-toolkit
}

#---------------------------------------------------------------------------
#
#	Load the basic requisites already here
#

load(lib) the-basics
load(lib) the-methods

#---------------------------------------------------------------------------

default system	application	{}
default system	project		{}
default system	auto		{}

#--	Set variable holding name of automatically-started application
#	and variable indicating demo mode.
#
if { [llength $argv] == 2 } then {
  set system(auto) "[lindex $argv 1]"
}

if { $system(auto) == "." } {
  set system(auto) Demo
  set system(demo) 1
  if { $system(verbose) } {
    puts stdout "** Automatic start of demonstration scenario"
  }
} {
  set system(demo) 0
  if { $system(verbose) } {
    puts stdout "** Manual operation mode"
  }
}

#---------------------------------------------------------------------------
#
# proc auto_mkindex {dir files}
#	Modified version of auto_mkindex which knows about "defobject" and
#	"defwidget".
#
proc auto_mkindex {dir files} {
  global errorCode errorInfo
  set oldDir [pwd]
  cd $dir
  set dir [pwd]
  append index "# Tcl autoload index file: each line identifies a Tcl\n"
  append index "# procedure and the file where that procedure is\n"
  append index "# defined.  Generated by the \"auto_mkindex\" command.\n"
  append index "\n"
  set procs(0) 0
  foreach file [glob $files] {
    set f ""
    set error [catch {
      set f [open $file]
      while {[gets $f line] >= 0} {
	set new 0
	if [regexp {^proc[ 	]+([^ 	]*)} $line ign proc] {
	  set new 1
	} elseif [regexp {^defwidget[ 	]+([^ 	]*)} $line ign proc] {
	  set new 1
	} elseif [regexp {^defobject[ 	]+([^ 	]*)} $line ign proc] {
	  set new 1
	} elseif [regexp {^defmethod[ 	]+([^ 	]*)} $line ign proc] {
	  set new 1
	}
	if { $new && ! [info exists procs($proc)] } {
	  set procs($proc) 1
	  append index "[list $proc $file]\n"
	}
      }
      close $f
    } msg]
    if $error {
      set code $errorCode
      set info $errorInfo
      catch [close $f]
      cd $oldDir
      error $msg $info $code
    }
  }
  set f [open tclIndex w]
  puts $f $index nonewline
  close $f
  cd $oldDir
}

if { $system(autoload) } {
  lappend auto_path $system(loc,library)/Software/Library:Tk
}
