#
#	Implementation of a simple bar chart
#
#	A bar chart is given by
#	o  a title
#	o  a range specification
#	o  two axis labels
#	o  a list of bar labels
#	o  a list of data elements
#
#	Each data element is a list of a label and values, where the
#	number of values must be the same as the number of bar labels
#	given.
#

defwidget Barchart

defmethod Barchart new {name args} {

  args	title text {step 10} {max 100} xlabel ylabel \
	legend data layout colors {factor 1} {embedded true}

  if { $embedded != "true" } {
    Toplevel new $name
  } {
    Frame new $name
  }
  defsuper $name Barchart

  set sizex 900
  set sizey 600
  set sep 4

  set margl [expr 6*[Font slot text,w]+10]
  set margr 10
  set margb [expr [Font slot textbold,h]+[Font slot text,h]+10]
  set margt [expr [Font slot largebold,h]+[Font slot textbold,h]+10]

  set n [llength $data]
  set m [llength $legend]

  set smallboldfont [Font slot smallbold]
  set smallboldh [Font slot smallbold,h]
  set textfont [Font slot text]
  set textboldfont [Font slot textbold]

  # Determine the colors for the bars
  if { $colors == {} } {
    set level 216
    if { $m == 1 } {
      set levelinc 0
    } {
      set levelinc [expr -1*($level-60)/$m]
      incr level [expr $levelinc/2]
    }
    foreach c $legend {
      lappend colors [format "#%02x%02x%02x" $level $level $level]
      incr level $levelinc
    }
  }

  foreach label $legend {
    set sum($label) 0
  }

  Canvas new $name.c -width $sizex -height $sizey -layout {top}
  set canvas [$name.c canvas]

  # Create the title and the axes
  $canvas! create text [expr $sizex/2] 10 \
	-text $text -anchor n -font [Font slot hugebold] -tags title
  set bbox [$canvas! bbox title]
  set bbox [list [expr [lindex $bbox 0]-5] \
		 [expr [lindex $bbox 1]-5] \
		 [expr [lindex $bbox 2]] \
		 [expr [lindex $bbox 3]+5]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill white -tags titlebackground]]
  $canvas! lower titlebackground title
  set bbox [list [expr [lindex $bbox 0]+4] \
		 [expr [lindex $bbox 1]+4] \
		 [expr [lindex $bbox 2]+4] \
		 [expr [lindex $bbox 3]+4]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill black -tags titlebackground2]]
  $canvas! lower titlebackground2 titlebackground

  $canvas! create text 10 [expr 10+[Font slot largebold,h]] \
	-text $xlabel -anchor nw -font $textboldfont
  $canvas! create text [expr $sizex-10] [expr $sizey-10] \
	-text $ylabel -anchor se -font $textboldfont

  $canvas! create line $margl $margt $margl [expr $sizey-$margb] \
	[expr $sizex-$margr] [expr $sizey-$margb] \
	-width 2 -fill black

  set stepy [expr ($sizey-$margt-$margb)*$step/$max]
  set maxx [expr $sizex-$margr]

  # Draw the y axis labels
  for {set c 0; set y [expr $sizey-$margb]; set i 0} \
	{$c <= $max} \
	{incr c $step; incr y -$stepy; incr i} {
    $canvas! create text [expr $margl-5] $y \
	-text $c -anchor e -font $textfont
    $canvas! create line $margl $y $maxx $y \
	-width 1 -fill black
  }

  set stepx [expr ($sizex-$margl-$margr)/$n]
  set maxy [expr $sizey-$margb]
  set texty [expr $maxy+3]
  set barx [expr ($stepx-2*$sep)/$m]
  set height [expr $max/$step*$stepy]

  # Draw the barchart itself
  for {set c 0; set x $margl} {$c < $n} {incr c; incr x $stepx} {

    set values [lindex $data $c]
    $canvas! create text [expr $x+$stepx/2] $texty \
	-text [lindex $values 0] -anchor n -font $textfont

    set pos [expr $x+$sep]
    set i 1
    foreach label $legend {
      set val [expr [lindex $values $i]*$factor]
      incr sum($label) $val
      if { $val != {} && $val > 0 } {
	set valuey [expr $maxy-$val*$height/$max]
	$canvas newRect3D $pos $valuey [expr $pos+$barx+1] $maxy \
	  -fill [lindex $colors [expr $i-1]]
	set bbox [textsize $smallboldfont $val]
	set tx [lindex $bbox 0]
	set ty [lindex $bbox 1]
	set xpos [expr $pos+$barx/2+2]
	$canvas! create rectangle \
		[expr $xpos-$tx/2-1] [expr $valuey-7] \
		[expr $xpos+$tx/2+1] [expr $valuey-9-$ty] \
		-fill white
	$canvas! create text $xpos [expr $valuey-8] \
		-text $val -anchor s -font $smallboldfont
      }
      incr pos $barx
      incr i
    }
  }

  set xpos [expr $sizex-$margr-10]
  set ypos $margt
  set x2pos [expr $xpos-$smallboldh*3]
  set y2pos [expr $ypos+$smallboldh*6/5]
  set steppos [expr $smallboldh*13/5]

  # Draw the legend
  set i 0
  foreach label $legend {
    $canvas! create rectangle $x2pos [expr $y2pos+$smallboldh] $xpos $ypos \
	-fill [lindex $colors $i] -tags legend
    $canvas! create text [expr $x2pos-6] [expr ($ypos+$y2pos)/2] \
	-text $label -font $smallboldfont -anchor e -tags legend
    $canvas! create text [expr $x2pos-6] [expr ($ypos+$y2pos)/2+$smallboldh] \
	-text "(Total: $sum($label))" -font [Font slot small] \
	-anchor e -tags legend
    incr ypos $steppos
    incr y2pos $steppos
    incr i
  }
  set bbox [$canvas! bbox legend]
  set bbox [list [expr [lindex $bbox 0]-5] \
		 [expr [lindex $bbox 1]-5] \
		 [expr [lindex $bbox 2]+5] \
		 [expr [lindex $bbox 3]+5]]
  eval [concat	[list $canvas! create rectangle] $bbox \
		[list -fill white -tags legendbackground]]
  $canvas! lower legendbackground legend

  $name layout $layout
}

defmethod Barchart postscript {args} {

  eval [concat [list $self.c postscript] $args]
}

Window addDemo Barchart

defmethod Barchart demo {} {

  set items {
	{Abt1 0 30} {Abt2 100 100} {Abt3 150 300} {Abt4 100 550}
	{Abt5 20 610} {Abt6 250 600} {Abt7 350 750} {Abt8 100 1000}
	{Abt9 100 420} {Abt10 10 30}
	}

  set b [anon]

  Barchart new $b \
	-colors {#e74f4f #ac1c1c} \
	-text "3.5\" Diskettenverbrauch in 1993" \
	-legend [list "3.5 DD" "3.5 HD"] \
	-embedded false \
	-max 1500 \
	-step 100 \
	-xlabel "Verbrauch" \
	-ylabel "Abteilungen" \
	-data $items

  Button new *$b \
	-text "Ende" \
	-textfont large \
	-action "$b Dismiss" \
	-layout {bottom padx 30 pady 30}

  Button new *$b \
	-text "Drucken" \
	-textfont large \
	-action [list $b postscript -colormode gray -file Stats.ps -rotate 1] \
	-layout {bottom padx 30 pady 30}

  $b layout +10+10
}
