/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/* otCreateHeadHistFile.c
 *
 * Create a file with the header and history lines in it.
 *
 * otCreateHeadHistFile  [-p<proj>] [-t<num>] [-b<num>] [-o<file>] [file]
 *
 * where:
 *	[-p<proj>]  - OT projectname
 *	[-t<num>]   - top CR number for the range of CRs to process
 *	[-b<num>]   - bottom CR number for the range of CRs to process
 *	[-o]	    - output file (default is /project/ot/<proj>/HeadAndHist)
 *	[file]	    - one template file to process
 *
 */

#define MAINPROGRAM

#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <pwd.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_ot[] = 
    "$RCSfile: otCreateHeadHistFile.c,v $ $Revision: 1.1.5.3 $ $Date: 1993/12/01 20:50:38 $";
#endif

/* command line options for getopt() */
#define optString "p:t:b:o:"

OTErr   hhfParseArgs();
OTErr	hhfAddHeadHistLines();
void    hhfOneTemplateFile();
void    hhfCRnumLoop();
void	hhfAddPlaceHolderLines();


/*
 *			m a i n
 *
 */

main(argc, argv, envp)
int argc;
char ** argv;
char ** envp;
{
    char    tmp[COMMAND];
    char    tmp1[COMMAND];
    struct  stat stat_buf;
    OTErr   err;
    OTPrivateCB *pvp;
    OTQueryCB   *qryp;
    OTProject   *prjp;

    /*
     * Initialize TCL, control blocks, read environment variables and
     * preparse command line for project and template file names 
     */
    if ( err = otInitialInit(&argc, argv) ) {
	fprintf(stderr, "otInitialInit: %s\n", otGetPostedMessage() );
	exit(1);
    }

    if ( err = hhfParseArgs(argc, argv) ) {
	fprintf(stderr, "hhfParseArgs: %s\n", otGetPostedMessage() );
	exit(1);
    }

    if ( err = otReadProject( otCB->cb_project ) ) {
	fprintf(stderr, "otReadProject: %s\n", otGetPostedMessage() );
	exit(1);
    }

    pvp = otCB->cb_pcb;
    qryp = otCB->cb_qcb;
    prjp = pvp->pcb_project;

    if (!qryp->qcb_outputFile) {
	sprintf(tmp, "%s/%s/HeadAndHist", prjp->proj_dir, prjp->name);
	if (stat(tmp, &stat_buf) == 0) {
	    sprintf(tmp1, "file '%s' already exist", tmp);
	    otPutPostedMessage(OT_GENERAL_ERROR, tmp1);
	    fprintf(stderr, "\n%s\n", otGetPostedMessage() );
	    exit(1);
	}

	if ( (qryp->qcb_outputFile = fopen(tmp, "w")) == NULL) {
	    otPutPostedMessage(OT_CREATE_FILE, "header and history", tmp);
	    fprintf(stderr, "\n%s\n", otGetPostedMessage() );
	    exit(1);
	}
    }

    if ( err = otReadTopNumber(prjp, &pvp->pcb_topNumber) ) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    if (otCB->cb_templFile)     /* process one template file */
	hhfOneTemplateFile();

    else { 			/* CR number loop */
	rcsCmdPath();		/* find RCS commands */
	hhfCRnumLoop(); 	
    }

    if (qryp->qcb_outputFile) 
	fclose(qryp->qcb_outputFile);

    exit(0);
}



void
hhfCRnumLoop()
{
    long  i, j;
    int   k, iloop, oloop, ind;
    char  crname[SHORTSTR];
    char  filename[PATHLEN];
    char  rcsfname[PATHLEN];
    char  path[PATHLEN];
    char  command[COMMAND];
    struct  stat stat_buf;
    OTPrivateCB *pvp;
    OTEnterCB   *entp;
    OTQueryCB   *qryp;
    OTProject   *prjp;
    OTErr err;

    pvp = otCB->cb_pcb;
    entp = otCB->cb_ecb;
    qryp = otCB->cb_qcb;
    prjp = pvp->pcb_project;
    entp->ecb_tStruct = 0;

    if (qryp->qcb_hiRange)
	i = qryp->qcb_hiRange;
    else
	i = pvp->pcb_topNumber - 1;

    j = 0;
    if (qryp->qcb_loRange)
	j = qryp->qcb_loRange - 1;

    for ( ; i > j ; i-- ) {

/*
	oloop = i / 10000;
	iloop = i / 100;
*/
	k = i / 100;
	oloop = k / 100;
	iloop = k - oloop*100;

	sprintf(path, "%s/%s/d%02d/d%02d", prjp->proj_dir, prjp->name, 
	    oloop, iloop);
	sprintf(crname, "c%06d", i);

	sprintf(filename, "%s/%s", path, crname);

	DBUG_MIN((stderr, "%s\n", filename));
	fprintf(stderr, "\n%s", filename);

	if (stat(filename, &stat_buf) != 0) {
	    
	    /* get a file from the RCS */

	    sprintf(rcsfname, "%s/RCS/%s,v", path, crname);

	    if (stat(rcsfname, &stat_buf) != 0) {
		otPutPostedMessage(OT_STAT, rcsfname);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		hhfAddPlaceHolderLines(i);
		continue;
	    }

	    if (!pvp->pcb_rcsCmd[0]) {
		sprintf(command, 
		    "no RCS commands found - can't check out %s", crname);
		otPutPostedMessage(OT_GENERAL_ERROR, command);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		hhfAddPlaceHolderLines(i);
		continue;
	    }

	    sprintf(command, "cd %s; %s/co -q %s\n", path, pvp->pcb_rcsCmd, 
		crname);
	    if (system(command)) {
		sprintf(command, "system() error whicle checking out file %s", 
		    crname);
		otPutPostedMessage(OT_GENERAL_ERROR, command);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		hhfAddPlaceHolderLines(i);
		continue;
	    }
	}

	if (entp->ecb_tStruct) {
	    otFreeTemplate(entp->ecb_tStruct);
	    entp->ecb_tStruct = 0;
	}

	if ( err = otReadTemplateFromFilename(&entp->ecb_tStruct, 
		filename, TRUE, TRUE) ) {
	    fprintf(stderr, "  otReadTemplateFromFilename: --> %s\n", 
		otGetPostedMessage() );
	    hhfAddPlaceHolderLines(i);
	    continue;
	}
	
	ind = FALSE;
	if (err = hhfAddHeadHistLines(entp->ecb_tStruct, &ind) ) {
	    fprintf(stderr, "  hhfAddHeadHistLines: --> %s\n", 
		otGetPostedMessage() );
	    if (ind)
	        hhfAddPlaceHolderLines(i);
	    continue;
	}
    }
    otFreeTemplate(entp->ecb_tStruct);

    return;
}



OTErr
hhfAddHeadHistLines(tp, ind)
OTTemplate *tp;		
int	*ind;
{
    char   num[SHORTSTR];
    char  *kwikStr;
    FILE  *fp;
    OTEnterCB *entp;
    OTProject *prjp;

    OTErr errCode = OT_SUCCESS;
    num[0] = 0;
    kwikStr = 0;

    entp = otCB->cb_ecb;
    prjp = otCB->cb_pcb->pcb_project;
    fp = otCB->cb_qcb->qcb_outputFile;

    /* Put header line into the headhist file */

    if ( errCode = otWriteHeaderFieldToKwikString(entp->ecb_tStruct,
    			num, &kwikStr) ) {
	*ind = TRUE;
	goto errdone;
    }

    if (!num[0]) {
	otPutPostedMessage(OT_ILLEGAL_ID_NUMBER, prjp->object_name, num);
	errCode = OT_ILLEGAL_ID_NUMBER;
	*ind = TRUE;
	goto errdone;
    }

    fprintf(fp, "%s ", num);

    if (kwikStr) {
        fputs(kwikStr, fp);
        free(kwikStr);
	kwikStr = 0;
    }

    fprintf(fp, "\n%s ", num);

    /* Put history line into the qitem */

    if (!(errCode = otWriteNotedataToKwikString(entp->ecb_tStruct, &kwikStr))) {
        if (kwikStr) {
	    fputs(kwikStr, fp);
	    free(kwikStr);
	    kwikStr = 0;
        }
    }

    fputc('\n', fp);

errdone:

    if (kwikStr)
	free(kwikStr);

    return errCode;
}



void
hhfAddPlaceHolderLines(num)
long	num;
{
    fprintf(otCB->cb_qcb->qcb_outputFile, "%ld \n%ld \n", num, num);
    return;
}



void
hhfOneTemplateFile()
{
    int	  ind;
    OTErr err;
    OTEnterCB   *entp;


    DBUG_MIN((stderr, "%s\n", otCB->cb_templFile));
    fprintf(stderr, "\n%s", otCB->cb_templFile);

    entp = otCB->cb_ecb;

    if ( err = otReadTemplateFromFilename(&entp->ecb_tStruct,
				otCB->cb_templFile, TRUE, TRUE) ) 	{
	fprintf(stderr, "  otReadTemplateFromFilename: --> %s\n",
	    otGetPostedMessage() );
	return;
    }

    if (err = hhfAddHeadHistLines(entp->ecb_tStruct, &ind) ) {
	fprintf(stderr, "  hhfAddHeadHistLines: --> %s\n",
	    otGetPostedMessage() );
    }

    otFreeTemplate(entp->ecb_tStruct);

    return;
}



OTErr
hhfParseArgs(argc, argv)
int argc;
char * argv[];
{
    register int  i, opt;
    char tmp[COMMAND];
    struct stat stat_buf;

    OTErr errCode = OT_SUCCESS;

    otCB->cb_qcb->qcb_hiRange = 0;
    otCB->cb_qcb->qcb_outputFile = 0;

    while ( (opt = getopt(argc, argv, optString)) != EOF )  {
	switch ( opt )  {

	case 'p':
	    otPutPostedMessage(OT_INTERNAL_ERROR,
			"otPreparseInput while extracting OT projectname");
	    errCode = OT_INTERNAL_ERROR;
	    break;

	case 't':
	    otCB->cb_qcb->qcb_hiRange = atol(optarg);
	    break;

	case 'b':
	    otCB->cb_qcb->qcb_loRange = atol(optarg);
	    break;

	case 'o':
	    if (stat(optarg, &stat_buf) == 0) {
		sprintf(tmp, "file '%s' already exist", optarg);
		otPutPostedMessage(OT_GENERAL_ERROR, tmp);
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if ( (otCB->cb_qcb->qcb_outputFile = fopen(optarg, "w")) 
		    	== NULL) {
		    otPutPostedMessage(OT_CREATE_FILE, "header and history",
			optarg);
		    errCode = OT_CREATE_FILE;
		}
	    }
	    break;

	case '?':
	    otPutPostedMessage(OT_GENERAL_ERROR, 
"usage:  otCreateHeadHist [-p<projectname>] [-t<topNum [-b<bottomNum>] [-o<outputfile>] [templatefile]");
	    errCode = OT_GENERAL_ERROR;
	    break;
	}
    }

    for (i=optind;  (i < argc) && !errCode;  i++)    {
	if (otCB->cb_templFile) {
	    errCode = OT_ONE_FILENAME_ONLY;
	    otPutPostedMessage(OT_ONE_FILENAME_ONLY, optarg);
	}
	else {

	    if ( !(errCode = otSetCBMember("templFile", argv[i])) &&
	          (strcmp( "-", otCB->cb_templFile)) )  {

	        if (stat(otCB->cb_templFile, &stat_buf) != 0) {
		    errCode = OT_STAT;
		    otPutPostedMessage(OT_STAT, otCB->cb_templFile);
		}
		else if (!(stat_buf.st_mode & S_IFREG)) {
		    errCode = OT_NOT_REGULAR_FILE;
		    otPutPostedMessage(OT_NOT_REGULAR_FILE, otCB->cb_templFile);
		}
	    }
	}
    }

    return errCode;
}


