# read.tcl - routines for reading and extracting files from a possibly
#	     compressed tar file or device
#
# Copyright 1994, Paul E Coad
# The author disclaims all warranties with regard to this software, including
# all implied warranties of merchantability and fitness.  In no event
# shall the author be liable for any special, indirect or consequential
# damages or any damages whatsoever resulting from loss of use, data or
# profits, whether in an action of contract, negligence or other
# tortuous action, arising out of or in connection with the use or
# performance of this software.
#
# This package is being released under the terms of Larry Wall's
# "Artistic license".

# read.tcl - 1.0

#-------------------------------------------------------------------
# read_archive - reads a possibly compressed archive and returns 
#		 a list of the contents of the archive or a list
#		 of the errors encountered.
#
# args:		device - the name of the file/device to read
#		cat_cmd - the command to use for cating the file/device
#		tar_cmd - the command to use for reading the file/device
#		tar_opts - the tar options to use for reading the file/device
#		redir - the redirector to use for redirecting output
#		tmp_file - the temporary file to use for temproary storage
#
# returns:	the output from read_archive is a list in the form:
#			number_of_files_found output_line_1 output_line_2 ...
#		number_of_files_found will be 0 (zero) if an error ocurred
#
#		list:	
#		index	type	what
#		0	integer	0 = ok !0 = error
#		1	integer	number of files extracted
#		2	list	file lines from tar
#		3	list	error output from tar
#			
#-------------------------------------------------------------------
proc read_archive {device cat_cmd tar_cmd tar_opts redir tmp_file args} {
    if { [file exists $device] } {
	set is_compressed [is_dotZ $device]
	set is_zipped [is_dotgz $device]
	if { !$is_compressed && !$is_zipped } {
	    set cat_c cat 
	} else { 
	    set cat_c $cat_cmd 
	}
	if {[llength $args] > 0} { 
	    set files [format {- %s} [join $args] ]
	    set tar_args [list $tar_opts $files]
	} else {
	    set tar_args [list $tar_opts -]
	}
	set cmd "exec $cat_c $device | $tar_cmd \
					[join $tar_args] $redir $tmp_file"
	set retcode [catch {eval $cmd} compcode]
	set count 0
	#
#if { [file exists $tmp_file] || $retcode == 0 } { }
	#
	 if { [file exists $tmp_file] } {
	    # The listing was generated, build a list for output
	    set listfile 0
	    set listfile [open $tmp_file]
	    set file_contents ""
	    if {$listfile != 0} {
		while {[gets $listfile line] >= 0} {
		    lappend file_contents $line
		    incr count
		}
	    }
	    exec rm $tmp_file
	} else {
	    set file_contents 0
	}
	# set the return code, file count, file list, and error messages
	lappend return_list $retcode
	lappend return_list $count
	lappend return_list $file_contents
	lappend return_list [split $compcode \x0a]
    } else {
	# set the return code, file count, file list, and error messages
	# in this case failure, 0 files, 0 contents, and the error message
	lappend return_list 1
	lappend return_list 0
	lappend return_list 0
	lappend return_list "Device $device not found."
    }
    return $return_list
}
#---------------------------------------------------------------------
# is_dotgz determines if a string ends in ".gz" or ".tgz" and is therefore
#          assumed to be zipped with the GNU zip program.
#
# Args: str     The string to examine
#---------------------------------------------------------------------
proc is_dotgz str {
    case $str in {
	*.z      { set gzipped 1 }
	*.gz     { set gzipped 1 }
	*.tgz    { set gzipped 1 }
	default  { set gzipped 0 }
    }
    return $gzipped
}

#---------------------------------------------------------------------
# is_dotZ determines if a string ends in ".Z" and is therefore
#         assumed to be compressed.  If the global variable have_gzip
#         is set to 1 files ending in ".taz" or ".taZ" will be recognized
#         as being compressed tar files.
#
# Args: str     The string to examine
#---------------------------------------------------------------------
proc is_dotZ str {
    case $str in {
	*.Z     { set compressed 1 }
	*.taz   { set compressed 1 }
	*.taZ   { set compressed 1 }
	default { set compressed 0 }
    }
    return $compressed
}

#---------------------------------------------------------------------
# clean_messages replaces some of the common nonintuitive messages issued
#		 by tar, gzcat, and zcat.  Known messages are replaced.
#		 Unknown messages are kept. A new list of messages is returned.
#
# Args: device		the name of the device or file from which the messages 
#			were generated.
#	return_vals	the list of errors returned by read_archive.
#---------------------------------------------------------------------
proc clean_messages {device errors} {
    set num_errors [llength $errors]
    for { set i 0 } { $i < $num_errors } { incr i } {
	set err [lindex $errors $i]
	if { [string match "*archive - EOF not on block boundary" $err]\
	     == 1} {
	    lappend new_errors \
		"The file: $device does not look like a tar archive."
	} elseif { [string match "* not in gzip format" $err] == 1} {
	    lappend new_errors \
		"The file: $device is not gzipped."
	} elseif { [string match "* not in compressed format" $err] == 1} {
	    lappend new_errors \
		"The file: $device is not compressed."
	} elseif { [string match "directory checksum error" $err] == 1} {
	    lappend new_errors \
		"The file: $device does not look like a tar archive."
	} elseif { [string match "*.Z: No such file or directory" $err] == 1} {
	    lappend new_errors \
		"The file: $device requires the GNU zip utility to uncompress."
	} elseif {[string match "* Could not create file*Error 78" $err] == 1} {
	    # The file could not be created, this could be because the
	    # file or directory has a name which is longer than the maximum
	    # possible on the system.
	    scan $err "%s %s %s %s %s %s" d1 d2 d3 d4 d5 file_name
	    lappend new_errors \
		[concat "The file: $file_name could not be created. " \
		        "The file name might be too long."]
	} elseif { [string match "* - cannot create" $err] == 1} {
	    # This error can occur under two conditions:
	    #   1.) the file name ends with "/" (it is a directory)
	    #	    These errors should be absorbed as a work around for the
	    #	    known bug in sysV tar.
	    #   2.) the file name is too long.
	    #	    This is a limitation of the system, report it.
	    # When the file name ends with a "/" absorb the message.

	    scan $err "%s %s" d1 file_name
	    set slash_index [string last / $file_name]
	    set name_length [string length $file_name]
	    if {$slash_index != -1 || $name_length > 14} {
		set last_char_index [expr $name_length - 1]
		if { $last_char_index != $slash_index } {
		    lappend new_errors \
		    	[concat "The file: $file_name could not be created. " \
			  	"The file name might be too long."]
		}
	    }
	} elseif { [string match "* write on pipe with no readers" $err] == 1} {
	    # Do nothing here - listing a file with sysV tar which includes
	    # files ending in "/"
	} else {
	    # Unexpected error - display it.
	    lappend new_errors $err
	}
    }
    if { [info exists new_errors] == 0} {
	set new_errors 0
    }
    return $new_errors
}
