/*
 * bltGrAxis.c --
 *
 *	This module implements coordinate axes for a graph
 *	widget in the Tk toolkit.
 *
 * Copyright 1991-1993 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 */

#include "blt.h"
#include "bltGraph.h"
#include "bltGrElem.h"
#include <ctype.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#define TRUE  1
#define FALSE 0
#define NTICK 5

#ifndef SHRT_MIN
#define SHRT_MIN                -0x8000
#endif /* SHRT_MIN */
#ifndef SHRT_MAX
#define SHRT_MAX                 0x7FFF
#endif /* SHRT_MAX */


/*
 * Round x in terms of units
 */
#define UROUND(x,u)	(ROUND((x)/(u))*(u))
#define UCEIL(x,u)	(ceil((x)/(u))*(u))
#define UFLOOR(x,u)	(floor((x)/(u))*(u))

#ifdef hpux
#define BLACK		"black"
#define WHITE		"white"
#define LIGHTGREY       "lightgrey"
#define BISQUE1		"bisque1"
#else
#define BLACK		"#000000"
#define WHITE		"#ffffff"
#define LIGHTGREY       "#d3d3d3"
#define BISQUE1		"#ffe4c4"
#endif

#define AXIS_MIN_LIMIT	0
#define AXIS_MAX_LIMIT  1

static char *axisNames[] = {"X", "Y", "Z"};
static char *axisCmds[] = {"xaxis", "yaxis", "zaxis"};

/* Map normalized coordinates to window coordinates */
#define MAPX(X,x)    	(ROUND((x)*(X)->scale)+(X)->offset)
#define MAPY(Y,y)    	((Y)->offset-ROUND((y)*(Y)->scale))

/* Map graph coordinates to normalized coordinates [0..1] */
#define NORM(a,x) (((x) - (a)->min) / (a)->range)

/*
 * Sun's bundled and unbundled C compilers choke on static function
 * typedefs (while it can handle extern declarations) like
 *
 * 	static Tk_OptionParseProc parseProc;
 *  	static Tk_OptionPrintProc printProc;
 *
 * As a workaround, provide forward declarations here:
 */
static int ParseAxisLimit _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset));
static char *PrintAxisLimit _ANSI_ARGS_((ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption MinLimitOption =
{
    ParseAxisLimit, PrintAxisLimit, (ClientData)AXIS_MIN_LIMIT,
};
static Tk_CustomOption MaxLimitOption =
{
    ParseAxisLimit, PrintAxisLimit, (ClientData)AXIS_MAX_LIMIT,
};

/*
 * -------------------------------------------------------------------
 *
 * TickLabel --
 *
 * 	Structure contains the formatted string and either the x or
 *	y window coordinate position of the tick label (its center).
 *	There is a limit on number of characters in a tick label of
 *	32 which seems to be more than enough.  The notion here was
 * 	to be able to quickly allocate and deallocate space for the
 *	labels.
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *text;			/* Label for tick on axis */
    short int pos;		/* Window position of tick on graph */
} TickLabel;

/*
 * -------------------------------------------------------------------
 *
 * Axis --
 *
 * 	Structure contains options controlling how the axis will be
 * 	displayed.
 *
 * -------------------------------------------------------------------
 */

typedef struct {
    AxisDisplayProc *displayProc;
    AxisPrintProc *printProc;
    AxisLayoutProc *layoutProc;
    AxisDestroyProc *destroyProc;

    Tcl_Interp *interp;
    enum AxisTypes type;	/* Type of axis: X_AXIS_TYPE or Y_AXIS_TYPE */
    int logScale;		/* If non-zero, scale values logarithmically */
    int showTicks;		/* If non-zero, display (both major and minor)
				 * ticks on the axis */
    int loose;			/* If non-zero, autoscale limits loosely */

    int flags;
    double limits[2];		/* Limits for scaling of axis */

    double reqStep;		/* Manually selected step size for major
				 * ticks: If zero or less, automatically
				 * calculate a "best" step size based on range
				 * of values. */
    int reqSubTicks;		/* Manually selected number of subticks: The
				 * default value is 2. */
    XFontStruct *fontPtr;	/* Font used to draw tick labels. */
    XColor *fgColorPtr;		/* Foreground color for ticks, labels, and
				 * axis */
    int lineWidth;		/* Line thickness of axis and ticks */
    double theta;		/* Rotation of tick labels in degrees. */
    char *formatCmd;		/* If non-NULL, indicates a Tcl proc to call
				 * when formatting tick labels. See the manual
				 * for its usage. */
    double elemMin, elemMax;	/* Autoscale axis limits (min/max element data
				 * values from the element display list) */
    double tickMin, tickMax;	/* Smallest, largest possible tick on plot */

    double min, max;		/* Actual (including padding) axis limits */
    double range;		/* Range of values (max-min) */
    double scale;		/* Scale factor to convert values to pixels */
    int offset;			/* Offset of plotting region from window
				 * origin */

    double subStep;		/* Step interval between minor ticks */
    int subTicks;		/* Number of minor ticks between major ticks */
    double step;		/* Step interval between major ticks */
    int numTicks;		/* Number of major ticks possible on axis:
				 * Calculated by tick layout routines. */

    XSegment *segArr;		/* Array of computed tick line segments. Also
				 * includes the axis line */
    int numSegments;		/* Size of the above segment array */

    TickLabel *labelArr;	/* Array of computed tick labels: See the
				 * description of the TickLabel structure. */
    int numLabels;		/* Size of the above label array */
    GC lineGC;			/* Graph context for axis lines and ticks */
    GC textGC;			/* Graphic context for tick labels: Must be
				 * private (can't use Tk_GetGC) because the
				 * FillStyle, TSOrigin, and Stipple fields may
				 * change when the graph is layout is
				 * calculated, due to rotation of text. Also,
				 * note that the background is reset when the
				 * background color of the graph changes. */


} Axis;

#define AXIS_STEP_SET_FLAG	0x02
#define AXIS_LIMIT_FLAG_BIT	0x04
#define AXIS_MIN_SET_FLAG	(AXIS_LIMIT_FLAG_BIT << AXIS_MIN_LIMIT)
#define AXIS_MAX_SET_FLAG	(AXIS_LIMIT_FLAG_BIT << AXIS_MAX_LIMIT)
#define AUTO_MIN_SCALE(a) 	(((a)->flags & AXIS_MIN_SET_FLAG)==0)
#define AUTO_MAX_SCALE(a) 	(((a)->flags & AXIS_MAX_SET_FLAG)==0)

#define DEF_AXIS_FG_COLOR	BLACK
#define DEF_AXIS_FG_MONO	BLACK
#define DEF_AXIS_FONT	 	"*-Courier-Bold-R-Normal-*-100-*"
#define DEF_AXIS_TICKS		"1"
#define DEF_AXIS_SUBTICKS_GRAPH		"2"
#define DEF_AXIS_SUBTICKS_BARCHART	"0"
#define DEF_AXIS_STEPSIZE_BARCHART	"1.0"

static Tk_ConfigSpec xAxisConfigSpecs[] =
{
    {TK_CONFIG_COLOR, "-color", "xAxisColor", "AxisColor",
	DEF_AXIS_FG_COLOR, Tk_Offset(Axis, fgColorPtr),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_COLOR, "-color", "xAxisColor", "AxisColor",
	DEF_AXIS_FG_MONO, Tk_Offset(Axis, fgColorPtr),
	TK_CONFIG_MONO_ONLY | ALL_MASK},
    {TK_CONFIG_STRING, "-command", "xAxisCommand", "AxisCommand",
	(char *)NULL, Tk_Offset(Axis, formatCmd),
	ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", "xAxisFont", "AxisFont",
	DEF_AXIS_FONT, Tk_Offset(Axis, fontPtr), ALL_MASK},
    {TK_CONFIG_PIXELS, "-linewidth", "xAxisLinewidth", "AxisLinewidth",
	(char *)NULL, Tk_Offset(Axis, lineWidth), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-logscale", "xAxisLogscale", "AxisLogscale",
	(char *)NULL, Tk_Offset(Axis, logScale), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-loose", "xAxisLoose", "AxisLoose",
	(char *)NULL, Tk_Offset(Axis, loose), ALL_MASK},
    {TK_CONFIG_CUSTOM, "-max", "xAxisMax", "AxisMax",
	(char *)NULL, 0, ALL_MASK | TK_CONFIG_NULL_OK, &MaxLimitOption},
    {TK_CONFIG_CUSTOM, "-min", "xAxisMin", "AxisMin",
	(char *)NULL, 0, ALL_MASK | TK_CONFIG_NULL_OK, &MinLimitOption},
    {TK_CONFIG_DOUBLE, "-rotate", "xAxisRotate", "AxisRotate",
	(char *)NULL, Tk_Offset(Axis, theta), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-showticks", "xAxisShowticks", "AxisShowticks",
	DEF_AXIS_TICKS, Tk_Offset(Axis, showTicks), ALL_MASK},
    {TK_CONFIG_DOUBLE, "-stepsize", "xAxisStepsize", "AxisStepsize",
	(char *)NULL, Tk_Offset(Axis, reqStep), XYGRAPH_MASK},
    {TK_CONFIG_DOUBLE, "-stepsize", "xAxisStepsize", "AxisStepsize",
	DEF_AXIS_STEPSIZE_BARCHART, Tk_Offset(Axis, reqStep),
	BARCHART_MASK},
    {TK_CONFIG_INT, "-subticks", "xAxisSubticks", "AxisSubticks",
	DEF_AXIS_SUBTICKS_GRAPH, Tk_Offset(Axis, reqSubTicks),
	XYGRAPH_MASK},
    {TK_CONFIG_INT, "-subticks", "xAxisSubticks", "AxisSubticks",
	DEF_AXIS_SUBTICKS_BARCHART, Tk_Offset(Axis, reqSubTicks),
	BARCHART_MASK},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Tk_ConfigSpec yAxisConfigSpecs[] =
{
    {TK_CONFIG_COLOR, "-color", "yAxisColor", "AxisColor",
	DEF_AXIS_FG_COLOR, Tk_Offset(Axis, fgColorPtr),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_COLOR, "-color", "yAxisColor", "AxisColor",
	DEF_AXIS_FG_MONO, Tk_Offset(Axis, fgColorPtr),
	TK_CONFIG_MONO_ONLY | ALL_MASK},
    {TK_CONFIG_STRING, "-command", "yAxisCommand", "AxisCommand",
	(char *)NULL, Tk_Offset(Axis, formatCmd),
	ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", "yAxisFont", "AxisFont",
	DEF_AXIS_FONT, Tk_Offset(Axis, fontPtr), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-loose", "yAxisLoose", "AxisLoose",
	(char *)NULL, Tk_Offset(Axis, loose), ALL_MASK},
    {TK_CONFIG_PIXELS, "-linewidth", "yAxisLinewidth", "AxisLinewidth",
	(char *)NULL, Tk_Offset(Axis, lineWidth), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-logscale", "yAxisLogscale", "AxisLogscale",
	(char *)NULL, Tk_Offset(Axis, logScale), ALL_MASK},
    {TK_CONFIG_CUSTOM, "-max", "yAxisMax", "AxisMax",
	(char *)NULL, 0, ALL_MASK | TK_CONFIG_NULL_OK, &MaxLimitOption},
    {TK_CONFIG_CUSTOM, "-min", "yAxisMin", "AxisMin",
	(char *)NULL, 0, ALL_MASK | TK_CONFIG_NULL_OK, &MinLimitOption},
    {TK_CONFIG_DOUBLE, "-rotate", "yAxisRotate", "AxisRotate",
	(char *)NULL, Tk_Offset(Axis, theta), ALL_MASK},
    {TK_CONFIG_BOOLEAN, "-showticks", "yAxisShowticks", "AxisShowticks",
	DEF_AXIS_TICKS, Tk_Offset(Axis, showTicks), ALL_MASK},
    {TK_CONFIG_DOUBLE, "-stepsize", "yAxisStepsize", "AxisStepsize",
	(char *)NULL, Tk_Offset(Axis, reqStep), ALL_MASK},
    {TK_CONFIG_INT, "-subticks", "yAxisSubticks", "AxisSubticks",
	DEF_AXIS_SUBTICKS_GRAPH, Tk_Offset(Axis, reqSubTicks), ALL_MASK},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static Tk_ConfigSpec *axisConfigSpecs[2] =
{
    xAxisConfigSpecs, yAxisConfigSpecs
};

#ifndef sun
extern char *strdup _ANSI_ARGS_((CONST char *s));

#endif

/* ----------------------------------------------------------------------
 * Custom option parse and print procedures
 * ----------------------------------------------------------------------
 */
/*
 *----------------------------------------------------------------------
 *
 * ParseAxisLimit --
 *
 *	Convert the string representation of an axis limit into its
 *	numeric form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The symbol type is
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ParseAxisLimit(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* Either AXIS_MIN_LIMIT or AXIS_MAX_LIMIT */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* */
    char *widgRec;		/* Axis structure */
    int offset;			/* Offset of limit */
{
    Axis *axisPtr = (Axis *)(widgRec);
    int whichLimit = (int)clientData;

    if (value == NULL) {
	axisPtr->flags &= ~(AXIS_LIMIT_FLAG_BIT << whichLimit);
    } else {
	double newLimit;

	if (Tcl_ExprDouble(interp, value, &newLimit) != TCL_OK) {
	    return TCL_ERROR;
	}
	axisPtr->limits[whichLimit] = newLimit;
	axisPtr->flags |= (AXIS_LIMIT_FLAG_BIT << whichLimit);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * PrintAxisLimit --
 *
 *	Convert the floating point axis limit into a string.
 *
 * Results:
 *	The string representation of the limit is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
PrintAxisLimit(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Either AXIS_MIN_LIMIT or AXIS_MAX_LIMIT */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* */
    int offset;
    Tcl_FreeProc **freeProcPtr;
{
    Axis *axisPtr = (Axis *)(widgRec);
    int whichLimit = (int)clientData;
    char *result;

    result = "";
    if (axisPtr->flags & (AXIS_LIMIT_FLAG_BIT << whichLimit)) {
	char string[TCL_DOUBLE_SPACE];

	Tcl_PrintDouble(axisPtr->interp, axisPtr->limits[whichLimit], string);
	result = strdup(string);
	if (result == NULL) {
	    return "";
	}
	*freeProcPtr = TCL_DYNAMIC;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateLabel --
 *
 *	Converts a floating point tick value to a string representation.
 *
 * Results:
 *	Returns a malloc'ed copy of the formatted string.
 *
 * Side Effects:
 *	Formatted tick label will be displayed on the graph.
 *
 *----------------------------------------------------------------------
 */
static char *
CreateLabel(graphPtr, axisPtr, value)
    Graph *graphPtr;		/* Graph widget */
    Axis *axisPtr;		/* Axis structure */
    double value;		/* */
{
    char string[TCL_DOUBLE_SPACE + 1];
    char *result;

    if (axisPtr->logScale) {
	sprintf(string, "1E%d", ROUND(value));
    } else {
	if (axisPtr->formatCmd == NULL) {
	    sprintf(string, "%.*g", NUMDIGITS, value);
	} else {
	    Tcl_PrintDouble(axisPtr->interp, value, string);
	}
    }
    if (axisPtr->formatCmd != NULL) {
	Tcl_ResetResult(axisPtr->interp);
	if (Tcl_VarEval(axisPtr->interp,
		axisPtr->formatCmd, " ", Tk_PathName(graphPtr->tkwin),
		" ", string, (char *)NULL) != TCL_OK) {
	    Tk_BackgroundError(axisPtr->interp);
	} else {
	    result = axisPtr->interp->result;
	    if (*result != '\0') {
		strncpy(string, result, TCL_DOUBLE_SPACE);
		string[TCL_DOUBLE_SPACE] = '\0';
		Tcl_ResetResult(axisPtr->interp);
	    }
	}
    }
    result = strdup(string);
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * FreeLabels --
 *
 *	Frees all the text strings allocated in the array of tick labels.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Tick label strings are deallocated.
 *
 *----------------------------------------------------------------------
 */
static void
FreeLabels(labelArr, numLabels)
    TickLabel labelArr[];
    int numLabels;
{
    register int i;
    char *text;

    for (i = 0; i < numLabels; i++) {
	text = labelArr[i].text;
	if (text != NULL) {
	    free((char *)text);
	}
    }
}

/* Map graph coordinate to normalized coordinates (consider log scale) */
static double
Scale(axisPtr, x)
    Axis *axisPtr;
    double x;
{
    if (x == Blt_posInfinity) {
	return (1.0);
    } else if (x == Blt_negInfinity) {
	return (0.0);
    }
    if (axisPtr->logScale) {
	if (x > 0.0) {
	    x = log10(x);
	} else if (x < 0.0) {
	    x = 0.0;
	}
    }
    return (NORM(axisPtr, x));
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WorldX --
 *
 *	Maps the given window x-coordinate back to a graph coordinate
 *	value.  Called by the graph locater routine.
 *
 * Results:
 *	Returns the graph coordinate value at the given window
 *	x-coordinate.
 *
 *----------------------------------------------------------------------
 */
double
Blt_WorldX(graphPtr, winX)
    Graph *graphPtr;
    int winX;
{
    double x;
    Axis *axisPtr = (Axis *)graphPtr->X;

    x = (winX - axisPtr->offset) / axisPtr->scale;
    x = (x * axisPtr->range) + axisPtr->min;
    if (axisPtr->logScale) {
	x = EXP10(x);
    }
    return (x);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WorldY --
 *
 *	Maps the given window y-coordinate back to a graph coordinate
 *	value. Called by the graph locater routine.
 *
 * Results:
 *	Returns the graph coordinate value at the given window
 *	y-coordinate.
 *
 *----------------------------------------------------------------------
 */
double
Blt_WorldY(graphPtr, winY)
    Graph *graphPtr;
    int winY;
{
    double y;
    Axis *axisPtr = (Axis *)graphPtr->Y;

    y = (axisPtr->offset - winY) / axisPtr->scale;
    y = (y * axisPtr->range) + axisPtr->min;
    if (axisPtr->logScale) {
	y = EXP10(y);
    }
    return (y);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WinX --
 *
 *	Map the given graph x-coordinate value to a window position.
 *
 * Results:
 *	Returns the window coordinate position at the given graph
 *	x-coordinate.
 *
 * Note:
 *	Since line and polygon clipping is performed by the X server,
 *	we must be careful about coordinates which are outside of the
 *      range of a signed short int.
 *
 *----------------------------------------------------------------------
 */
int
Blt_WinX(graphPtr, x)
    Graph *graphPtr;
    double x;
{
    Axis *axisPtr = (Axis *)graphPtr->X;
    double normX;
    int winX;

    normX = Scale(axisPtr, x);
    winX = MAPX(axisPtr, normX);
    if (winX >= SHRT_MAX) {
	winX = SHRT_MAX - 1000;
    } else if (winX <= SHRT_MIN) {
	winX = SHRT_MIN + 1000;
    }
    return (winX);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WinY --
 *
 *	Maps the given graph y-coordinate value to a window position.
 *
 * Results:
 *	Returns the window coordinate position at the given graph
 *	y-coordinate.
 *
 * Note:
 *	Since line and polygon clipping is performed by the X server,
 *	we must be careful about coordinates which are outside of the
 *      range of a signed short int.
 *
 *----------------------------------------------------------------------
 */
int
Blt_WinY(graphPtr, y)
    Graph *graphPtr;
    double y;
{
    Axis *axisPtr = (Axis *)graphPtr->Y;
    double normY;
    int winY;

    normY = Scale(axisPtr, y);
    winY = MAPY(axisPtr, normY);
    if (winY >= SHRT_MAX) {
	winY = SHRT_MAX - 1000;
    } else if (winY <= SHRT_MIN) {
	winY = SHRT_MIN + 1000;
    }
    return (winY);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WinXAbs --
 *
 *	Map the given graph x-coordinate value to a window position.
 *
 * Results:
 *	Returns the window coordinate position at the given graph
 *	x-coordinate.
 *
 * Note:
 *	Since line and polygon clipping is performed by the X server,
 *	we must be careful about coordinates which are outside of the
 *      range of a signed short int.
 *
 *----------------------------------------------------------------------
 */
int
Blt_WinXAbs(graphPtr, x)
    Graph *graphPtr;
    double x;
{
    Axis *axisPtr = (Axis *)graphPtr->X;
    double normX;
    int winX;

    normX = Scale(axisPtr, x);
    winX = MAPX(axisPtr, normX);
    return (winX);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WinYAbs --
 *
 *	Maps the given graph y-coordinate value to a window position.
 *
 * Results:
 *	Returns the window coordinate position at the given graph
 *	y-coordinate.
 *
 * Note:
 *	Since line and polygon clipping is performed by the X server,
 *	we must be careful about coordinates which are outside of the
 *      range of a signed short int.
 *
 *----------------------------------------------------------------------
 */
int
Blt_WinYAbs(graphPtr, y)
    Graph *graphPtr;
    double y;
{
    Axis *axisPtr = (Axis *)graphPtr->Y;
    double normY;
    int winY;

    normY = Scale(axisPtr, y);
    winY = MAPY(axisPtr, normY);
    return (winY);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WinPt --
 *
 *	Maps the given graph x,y coordinate values to a window position.
 *
 * Results:
 *	Returns a XPoint structure containing the window coordinates of
 *	the given graph x,y coordinate.
 *
 *----------------------------------------------------------------------
 */
XPoint
Blt_WinPt(graphPtr, x, y)
    Graph *graphPtr;
    double x, y;
{
    XPoint winPos;

    winPos.x = Blt_WinX(graphPtr, x);
    winPos.y = Blt_WinY(graphPtr, y);
    return (winPos);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PointOnGraph --
 *
 *	Determines if the window coordinates given represent a point
 *	on the graph (within the bounds of the graph axes).
 *
 * Results:
 *	Returns 1 is the point is within the bounds of the graph,
 *	0 otherwise.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PointOnGraph(graphPtr, pointPtr)
    Graph *graphPtr;
    XPoint *pointPtr;
{
    double normX, normY;
    Axis *axisPtr;

    axisPtr = (Axis *)graphPtr->X;

    if (axisPtr->scale == 0.0) {
	return FALSE;		/* Axis layout hasn't been calculated yet */
    }
    normX = (pointPtr->x - axisPtr->offset) / axisPtr->scale;
    if ((normX < 0.0) || (normX > 1.0)) {
	return FALSE;		/* x-coordinates are off the graph */
    }
    axisPtr = (Axis *)graphPtr->Y;
    normY = (axisPtr->offset - pointPtr->y) / axisPtr->scale;
    return ((normY >= 0.0) && (normY <= 1.0));
}

/*
 *----------------------------------------------------------------------
 *
 * SetAxisLimits --
 *
 *	Updates the min and max values for each axis as determined by
 *	the data elements currently to be displayed.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Minimum, maximum data limit fields for both the X and Y axes
 *	in the graph widget record are updated.
 *----------------------------------------------------------------------
 */
static int
SetAxisLimits(axisPtr, listPtr)
    Axis *axisPtr;
    Blt_LinkedList *listPtr;
{
    register Element *elemPtr;
    Blt_ListEntry *entryPtr;
    register double min, max;
    double elemMin, elemMax;
    double prevMin, prevMax;
    int result;

    /* Save the previous minimum and maximum values */

    prevMin = axisPtr->limits[AXIS_MIN_LIMIT];
    prevMax = axisPtr->limits[AXIS_MAX_LIMIT];

    /*
     * Find the minimum and maximum values for all the elements displayed
     */
    min = Blt_posInfinity, max = Blt_negInfinity;
    for (entryPtr = Blt_FirstListEntry(listPtr); entryPtr != NULL;
	entryPtr = Blt_NextListEntry(entryPtr)) {
	elemPtr = (Element *)Blt_GetListValue(entryPtr);
	if ((*elemPtr->limitsProc) (elemPtr, axisPtr->type, axisPtr->logScale,
		&elemMin, &elemMax) > 0) {
	    if (min > elemMin) {
		min = elemMin;
	    }
	    if (max < elemMax) {
		max = elemMax;
	    }
	}
    }

    /*
     * Set arbitrarily limits, if no data exists
     */

    if (max == Blt_negInfinity) {
	max = 10.0;
    }
    if (min == Blt_posInfinity) {
	min = (axisPtr->logScale) ? 1.0 : -10.0;
    }
    if (AUTO_MIN_SCALE(axisPtr)) {
	axisPtr->limits[AXIS_MIN_LIMIT] = min;
    }
    if (AUTO_MAX_SCALE(axisPtr)) {
	axisPtr->limits[AXIS_MAX_LIMIT] = max;
    }
    axisPtr->elemMin = min, axisPtr->elemMax = max;

    /* Indicate if the axis limits have changed */
    result = (axisPtr->limits[AXIS_MAX_LIMIT] != prevMax) ||
	(axisPtr->limits[AXIS_MIN_LIMIT] != prevMin);
    return (result);
}

/*
 *----------------------------------------------------------------------
 *
 * NiceNum --
 *
 * 	Taken from Paul Heckbert's "Nice Numbers for Graph Labels" in
 *	Graphics Gems (pp 61-63).  Finds a "nice" number approximately
 *	equal to x.  Round the number if round=1, take ceiling if round=0.
 *
 *----------------------------------------------------------------------
 */
static double
NiceNum(x, round)
    double x;
    int round;
{
    double exponX;		/* exponent of x */
    double fractX;		/* fractional part of x */
    double nf;			/* nice, rounded fraction */

    exponX = floor(log10(x));
    fractX = x / EXP10(exponX);	/* between 1 and 10 */
    if (round) {
	if (fractX < 1.5) {
	    nf = 1.;
	} else if (fractX < 3.0) {
	    nf = 2.;
	} else if (fractX < 7.0) {
	    nf = 5.;
	} else {
	    nf = 10.;
	}
    } else if (fractX <= 1.0) {
	nf = 1.;
    } else if (fractX <= 2.0) {
	nf = 2.;
    } else if (fractX <= 5.0) {
	nf = 5.0;
    } else {
	nf = 10.0;
    }
    return (nf * EXP10(exponX));
}

/*
 *----------------------------------------------------------------------
 *
 * LogAxis --
 *
 *	Always calculates to
 *
 * 	If the number of decades is greater than ten, it is assumed
 *	that the full set of log-style ticks can't be drawn properly.
 *
 * Results:
 *
 *
 *----------------------------------------------------------------------
 */
static void
LogAxis(axisPtr, min, max)
    Axis *axisPtr;
    double min, max;
{
    double range;

    if (min > 0.0) {
	min = floor(log10(min));
    } else {
	min = 0.0;
    }
    if (max > 0.0) {
	max = ceil(log10(max));
    } else {
	max = 1.0;
    }
    range = max - min;
    if (range > 10) {

	range = NiceNum(range, 0);
	axisPtr->step = NiceNum(range / (NTICK - 1), 1);

	/*
	 * Find the outer limits in terms of the step.
	 */
	min = UFLOOR(min, axisPtr->step);
	max = UCEIL(max, axisPtr->step);
	axisPtr->numTicks = (int)((max - min) / axisPtr->step) + 1;
	axisPtr->subStep = EXP10(floor(log10(axisPtr->step)));
	if (axisPtr->step == axisPtr->subStep) {
	    axisPtr->subTicks = 5;
	    axisPtr->subStep = axisPtr->step * 0.2;
	} else {
	    axisPtr->subTicks = ROUND(axisPtr->step / axisPtr->subStep);
	}
    } else {
	if (min == max) {
	    max++;
	}
	axisPtr->numTicks = (int)((max - min) + 1);
	axisPtr->step = 1.0;
	axisPtr->subTicks = 10;
    }
    axisPtr->min = axisPtr->tickMin = min;
    axisPtr->max = axisPtr->tickMax = max;
    axisPtr->range = (max - min);
#ifdef notdef
    fprintf(stderr, "Major: %s\nRegion min=%g,max=%g\nTick min=%g,max=%g\n\
numTicks=%d, range=%g, step=%.15g\n", axisNames[axisPtr->type], min, max,
	axisPtr->tickMin, axisPtr->tickMax, axisPtr->numTicks,
	axisPtr->range, axisPtr->step);
    fprintf(stderr, "Minor numTicks=%d, step=%.15g\n\n",
	axisPtr->subTicks, axisPtr->subStep);
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * LinearAxis --
 *
 *	Always calculates to
 *
 * Results:
 *
 *
 *----------------------------------------------------------------------
 */
static void
LinearAxis(axisPtr, min, max)
    Axis *axisPtr;
    double min, max;
{
    double range, pad;

    /*
     * Calculate the major step.
     */
    range = max - min;
    if ((axisPtr->reqStep > 0.0) && (axisPtr->reqStep < range)) {
	axisPtr->step = axisPtr->reqStep;
    } else {
	range = NiceNum(range, 0);
	axisPtr->step = NiceNum(range / (NTICK - 1), 1);
    }

    /*
     * Find outer tick values in terms of the major step interval
     * Add +0.0 to preclude the possibility of an IEEE -0.0.
     */

    axisPtr->tickMin = UFLOOR(min, axisPtr->step) + 0.0;
    axisPtr->tickMax = UCEIL(max, axisPtr->step) + 0.0;
    range = axisPtr->tickMax - axisPtr->tickMin;
    axisPtr->numTicks = ROUND(range / axisPtr->step) + 1;

    /*
     * If the axis is "loose", the min and max are the outermost ticks.
     * Otherwise if it's "tight", the axis min and max are the min and
     * max of the data points.
     */

    if (axisPtr->loose) {
	axisPtr->min = axisPtr->tickMin, axisPtr->max = axisPtr->tickMax;
    } else {
	axisPtr->min = min, axisPtr->max = max;
    }

    /*
     * If either the min or max limits is auto-scaled, add some padding so
     * that symbols of data points at the extremes aren't clipped in half.
     * Two percent is an arbitrary guess.
     */

    pad = (axisPtr->max - axisPtr->min) * 0.02;
    if (AUTO_MIN_SCALE(axisPtr)) {
	axisPtr->min -= pad;
    }
    if (AUTO_MAX_SCALE(axisPtr)) {
	axisPtr->max += pad;
    }
    axisPtr->range = axisPtr->max - axisPtr->min;

#ifdef notdef
    fprintf(stderr, "Major: %s\nRegion min=%g,max=%g\nTick min=%g,max=%g\n\
numTicks=%d, range=%g, step=%.15g\n", axisNames[axisPtr->type], min, max,
	axisPtr->tickMin, axisPtr->tickMax, axisPtr->numTicks,
	axisPtr->range, axisPtr->step);
#endif
    /*
     * Now calculate the minor tick step and number.
     */
    axisPtr->subTicks = axisPtr->reqSubTicks;
    if (axisPtr->subTicks < 0) {
	axisPtr->subTicks = 0;
    }
    if (axisPtr->subTicks > 0) {
	axisPtr->subStep = axisPtr->step / axisPtr->subTicks;
    } else {
	axisPtr->subStep = axisPtr->step * 0.2;	/* Need this for layout */
    }
#ifdef notdef
    fprintf(stderr, "Minor numTicks=%d, step=%.15g\n\n",
	axisPtr->subTicks, axisPtr->subStep);
#endif
}

/*
 * -----------------------------------------------------------------
 *
 * ComputeTicks --
 *
 * Fill the Axis structure with the necessary information to
 * draw the axes.  Create a phony range when min equals max.
 *
 * Unless the axis limits are specified, the axis is scaled
 * automatically, where the smallest and largest major ticks
 * encompass the range of actual data values.  When an axis
 * limit is specified, that value represents the
 * smallest(min)/largest(max) value in the displayed range of
 * values.
 *
 * Both manual and automatic scaling are affected by the
 * step used.  By default, the step is the largest
 * power of ten to divide the range in more than one piece.
 *
 * Automatic scaling:
 *   Find the smallest number of units which contain the range of
 *   values.  The minimum and maximum major tick values will be
 *   represent the range of values for the axis. This greatest
 *   number of major ticks possible is 10.
 *
 * Manual scaling:
 *   Make the minimum and maximum data values the represent the
 *   range of the values for the axis.  The minimum and maximum
 *   major ticks will be inclusive of this range.  This provides
 *   the largest area for plotting and the expected results when
 *   the axis min and max values have be set by the user (.e.g zooming).
 *   The maximum number of major ticks is 20.
 *
 *   For log scale, there is always the possibility that the minimum
 *   and maximum data values are the same magnitude.  To represent
 *   the points properly, at least one full decade should be shown.
 *   However, if you zoom a log scale plot, the results should be
 *   predictable. Therefore, in that case, show only minor ticks.
 *   Lastly, there should be an appropriate way to handle numbers <=0.
 *
 *          maxY
 *            |    units = magnitude (of least significant digit)
 *            |    high  = largest unit tick < max axis value
 *      high _|    low   = smallest unit tick > min axis value
 *            |
 *            |    range = high - low
 *            |    # ticks = greatest factor of range/units
 *           _|
 *        U   |
 *        n   |
 *        i   |
 *        t  _|
 *            |
 *            |
 *            |
 *       low _|
 *            |
 *            |_minX________________maxX__
 *            |   |       |      |       |
 *     minY  low                        high
 *           minY
 *
 *
 * numTicks = Number of ticks
 * min = Minimum value of axis
 * max = Maximum value of axis
 * range    = Range of values (max - min)
 * -----------------------------------------------------------------
 */
static void
ComputeTicks(axisPtr)
    Axis *axisPtr;
{
    double min, max;

    min = axisPtr->limits[AXIS_MIN_LIMIT];
    max = axisPtr->limits[AXIS_MAX_LIMIT];

    if (max < min) {
	double temp;

	/* Swap min and max */
	temp = max, max = min, min = temp;
    } else if (min == max) {
	/* Set artificial min and max */
	if (min == 0.0) {
	    min = -0.1, max = 0.1;
	} else {
	    double x;

	    x = FABS(min) * 0.1;
	    min -= x;
	    max += x;
	}
    }
    if (axisPtr->logScale) {
	LogAxis(axisPtr, min, max);
    } else {
	LinearAxis(axisPtr, min, max);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * ComputeXAxis --
 *
 * -----------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ComputeXAxis(graphPtr, axisPtr)
    Graph *graphPtr;		/* Not used. */
    Axis *axisPtr;
{
    ComputeTicks(axisPtr);
}

/*
 * -----------------------------------------------------------------
 *
 * ComputeYAxis --
 *
 * -----------------------------------------------------------------
 */
static void
ComputeYAxis(graphPtr, axisPtr)
    Graph *graphPtr;
    Axis *axisPtr;
{
    /*
     * For barcharts, adjust the minimum or maximum values to include 0.0.
     */
    if (graphPtr->type == BARCHART_TYPE) {
	if (AUTO_MIN_SCALE(axisPtr) && (axisPtr->elemMin > 0.0)) {
	    axisPtr->elemMin = 0.0;
	}
	if (AUTO_MAX_SCALE(axisPtr) && (axisPtr->elemMax < 0.0)) {
	    axisPtr->elemMax = 0.0;
	}
    }
    ComputeTicks(axisPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ComputeAxes --
 *
 * Results:
 *	None.
 *----------------------------------------------------------------------
 */
void
Blt_ComputeAxes(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr;

    /*
     * If either of the X or Y axis limits change, we need to re-layout the
     * entire graph.
     */
    axisPtr = (Axis *)graphPtr->X;
    if (SetAxisLimits(axisPtr, &(graphPtr->elemList))) {
	ComputeXAxis(graphPtr, axisPtr);
	graphPtr->flags |= (LAYOUT_ALL | REDRAW_ALL);
    }
    axisPtr = (Axis *)graphPtr->Y;
    if (SetAxisLimits(axisPtr, &(graphPtr->elemList))) {
	ComputeYAxis(graphPtr, axisPtr);
	graphPtr->flags |= (LAYOUT_ALL | REDRAW_ALL);
    }
    graphPtr->flags |= LAYOUT_PENDING;
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureAxis --
 *
 *	Configures axis attributes (font, line width, label, etc) and
 *	allocates a new (possibly shared) graphics context.  Line cap
 *	style is projecting.  This is for the problem of when a tick
 *	sits directly at the end point of the axis.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Axis resources are allocated (GC, font). Axis layout is
 *	deferred until the height and width of the window are known.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureAxis(graphPtr, axisPtr, argc, argv, flags)
    Graph *graphPtr;
    Axis *axisPtr;
    int argc;
    char *argv[];
    int flags;
{
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;
    Tk_ConfigSpec *configSpecs;

    configSpecs = axisConfigSpecs[(int)axisPtr->type];
    if (flags & TK_CONFIG_ARGV_ONLY) {
	if (argc == 0) {
	    return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		    configSpecs, (char *)axisPtr, (char *)NULL, flags));
	} else if (argc == 1) {
	    return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		    configSpecs, (char *)axisPtr, argv[0], flags));
	}
    }
    if (Tk_ConfigureWidget(graphPtr->interp, graphPtr->tkwin, configSpecs,
	    argc, argv, (char *)axisPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Check requested X and Y axis limits. Can't allow min to be greater than
     * max, or undefined log scale limits.
     */
    if ((axisPtr->flags & (AXIS_MIN_SET_FLAG | AXIS_MAX_SET_FLAG)) &&
	(axisPtr->limits[AXIS_MIN_LIMIT] >= axisPtr->limits[AXIS_MAX_LIMIT])) {
	sprintf(graphPtr->interp->result,
	    "impossible %s axis limits (min %g >= max %g)",
	    axisNames[(int)axisPtr->type], axisPtr->limits[AXIS_MIN_LIMIT],
	    axisPtr->limits[AXIS_MAX_LIMIT]);
	return TCL_ERROR;
    }
    if ((axisPtr->logScale) && (axisPtr->flags & AXIS_MIN_SET_FLAG) &&
	(axisPtr->limits[AXIS_MIN_LIMIT] <= 0.0)) {
	sprintf(graphPtr->interp->result,
	    "invalid %s axis limits (min=%g,max=%g) for log scale",
	    axisNames[(int)axisPtr->type], axisPtr->limits[AXIS_MIN_LIMIT],
	    axisPtr->limits[AXIS_MAX_LIMIT]);
	return TCL_ERROR;
    }
    /*
     * Reset bogus line widths to zero. Can't allow bad line widths because
     * the layout routines compute axis and tick positions with them.
     */
    if (axisPtr->lineWidth < 1) {
	axisPtr->lineWidth = 0;
    }
    /*
     * Create an unshared GC for the tick labels. The GC is private because
     * the labels may be rotated, requiring the GCStipple and GCTSOffset
     * fields to change.
     */
    gcMask = GCForeground | GCFont;
    gcValues.font = axisPtr->fontPtr->fid;
    gcValues.foreground = axisPtr->fgColorPtr->pixel;
    if (graphPtr->border != NULL) {
	gcValues.background = Tk_3DBorderColor(graphPtr->border)->pixel;
	gcMask |= GCBackground;
    }
    newGC = XCreateGC(graphPtr->display, Tk_WindowId(graphPtr->tkwin), gcMask,
	&gcValues);
    if (axisPtr->textGC != NULL) {
	XFreeGC(graphPtr->display, axisPtr->textGC);
    }
    axisPtr->textGC = newGC;

    /*
     * Create GC for axis line and ticks.
     */
    gcMask = GCForeground | GCLineWidth | GCCapStyle;
    gcValues.line_width = axisPtr->lineWidth;
    gcValues.cap_style = CapProjecting;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (axisPtr->lineGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->lineGC);
    }
    axisPtr->lineGC = newGC;

    if (AUTO_MIN_SCALE(axisPtr)) {
	axisPtr->limits[AXIS_MIN_LIMIT] = axisPtr->elemMin;
    }
    if (AUTO_MAX_SCALE(axisPtr)) {
	axisPtr->limits[AXIS_MAX_LIMIT] = axisPtr->elemMax;
    }
    if (axisPtr->type == X_AXIS_TYPE) {
	ComputeXAxis(graphPtr, axisPtr);
    } else if (axisPtr->type == Y_AXIS_TYPE) {
	ComputeYAxis(graphPtr, axisPtr);
    }
    graphPtr->flags |= (REDRAW_ALL | LAYOUT_PENDING | LAYOUT_ALL);
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyAxis --
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources (font, color, gc, labels, etc.) associated with the
 *	axis are deallocated.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyAxis(graphPtr, axis)
    Graph *graphPtr;
    GraphAxis *axis;
{
    Axis *axisPtr = (Axis *)axis;

    if (axisPtr->fontPtr != NULL) {
	Tk_FreeFontStruct(axisPtr->fontPtr);
    }
    if (axisPtr->fgColorPtr != NULL) {
	Tk_FreeColor(axisPtr->fgColorPtr);
    }
    if (axisPtr->formatCmd != NULL) {
	free((char *)axisPtr->formatCmd);
    }
    if (axisPtr->lineGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->lineGC);
    }
    if (axisPtr->textGC != NULL) {
	XFreeGC(graphPtr->display, axisPtr->textGC);
    }
    if (axisPtr->labelArr != NULL) {
	FreeLabels(axisPtr->labelArr, axisPtr->numLabels);
	free((char *)axisPtr->labelArr);
    }
    if (axisPtr->segArr != NULL) {
	free((char *)axisPtr->segArr);
    }
    free((char *)axisPtr);
}


static float logTable[] =	/* Precomputed log10 values [1..10] */
{
    0.0, 0.301, 0.477, 0.602, 0.699, 0.778, 0.845, 0.903, 0.954, 1.0
};

/*
 * -----------------------------------------------------------------
 *
 * LayoutXAxis --
 *
 *	Pre-calculate the x-coordinate positions of the axis baseline,
 *	ticks and labels to be used later when displaying the X axis.
 *	Ticks (minor and major) will be saved in an array of XSegments
 *	so that they can be drawn in one XDrawSegments call. The
 *	strings representing the tick labels and the corresponding
 *	window positions are saved in an array of TickLabels.
 *
 *      Calculates the values for each major and minor tick and checks to
 *	see if they are in range (the outer ticks may be outside of the
 *	range of plotted values).
 *
 * Results:
 *	None.
 *
 * SideEffects:
 *	Line segments and tick labels saved will be used to draw
 *	the X axis.
 * -----------------------------------------------------------------
 */
static void
LayoutXAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->X;
    double normX;
    XSegment *segArr;
    register int count, tick;
    unsigned int need;
    int padY;			/* Offset of axis from interior region. This
				 * includes a possible border and the axis
				 * line width. */
    int baseLine;
    int majorTickPos, minorTickPos;
    double minX, maxX;
    double value, subValue;
    double minLimit, maxLimit;
    register int i, j;
    int halfLW;

    halfLW = axisPtr->lineWidth / 2;
    padY = graphPtr->intBWidth + halfLW + 1;
    if (graphPtr->intBWidth > 0) {
	padY++;
    }
    majorTickPos = MAPY((Axis *)graphPtr->Y, -MAJOR_TICK) + padY + halfLW;
    minorTickPos = MAPY((Axis *)graphPtr->Y, -MINOR_TICK) + padY + halfLW;
    baseLine = graphPtr->origin.y + padY;

    /*
     * Save all line coordinates in an array of line segments.
     */
    need = (1 + (axisPtr->numTicks * (axisPtr->subTicks + 1)));
    segArr = (XSegment *)malloc(need * sizeof(XSegment));

    if (segArr == NULL) {
	return;			/* Can't allocate array of segments */
    }
    if ((axisPtr->logScale) || (axisPtr->loose) ||
	(axisPtr->limits[AXIS_MIN_LIMIT] == axisPtr->limits[AXIS_MAX_LIMIT])) {
	minX = axisPtr->tickMin, maxX = axisPtr->tickMax;
    } else {
	minX = axisPtr->limits[AXIS_MIN_LIMIT];
	maxX = axisPtr->limits[AXIS_MAX_LIMIT];
    }

    segArr[0].y1 = segArr[0].y2 = baseLine;	/* Axis baseline */
    normX = NORM(axisPtr, minX);
    segArr[0].x1 = MAPX(axisPtr, normX);
    normX = NORM(axisPtr, maxX);
    segArr[0].x2 = MAPX(axisPtr, normX);

    count = 1, tick = 0;
    if (!axisPtr->showTicks) {
	goto done;		/* Only display axis line */
    }
    /*
     * Need to use numbers just beyond the limits when testing for equality
     */
    minLimit = minX - (axisPtr->subStep * 0.001);
    maxLimit = maxX + (axisPtr->subStep * 0.001);

    value = axisPtr->tickMin;	/* Start from smallest axis tick */
    for (i = 0; i < axisPtr->numTicks; i++) {
	subValue = value = UROUND(value, axisPtr->step);
	for (j = 1; j < axisPtr->subTicks; j++) {
	    if ((axisPtr->logScale) && (axisPtr->step == 1.0)) {
		subValue = value + logTable[j];
	    } else {
		subValue += axisPtr->subStep;
	    }
	    if ((subValue >= minLimit) && (subValue <= maxLimit)) {
		normX = NORM(axisPtr, subValue);

		segArr[count].y1 = baseLine;	/* Minor tick */
		segArr[count].y2 = minorTickPos;
		segArr[count].x1 = segArr[count].x2 = MAPX(axisPtr, normX);
		count++;
	    }
	}
	if ((value >= minLimit) && (value <= maxLimit)) {
	    normX = NORM(axisPtr, value);

	    segArr[count].y1 = baseLine;	/* Major tick */
	    segArr[count].y2 = majorTickPos;
	    segArr[count].x1 = segArr[count].x2 = MAPX(axisPtr, normX);

	    /* Save tick position */
	    axisPtr->labelArr[tick++].pos = segArr[count].x1;
	    count++;
	}
	value += axisPtr->step;
    }
  done:
#ifdef notdef
    if (count > need) {
	fprintf(stderr, "Number allocated = %d, used = %d\n", need, count);
    }
#endif
    if (axisPtr->segArr != NULL) {
	free((char *)axisPtr->segArr);
    }
    axisPtr->segArr = segArr;
    axisPtr->numSegments = count;
}

/*
 * -----------------------------------------------------------------
 *
 * DisplayXAxis --
 *
 * -----------------------------------------------------------------
 */
static void
DisplayXAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->X;
    int labelY;
    register int i;
    TextAttr textAttr;

    if (axisPtr->numTicks < 1) {
	return;
    }
    labelY = MAPY((Axis *)graphPtr->Y, -LABEL_TICK) +
	axisPtr->lineWidth + graphPtr->intBWidth + PADY;

    textAttr.theta = axisPtr->theta;
    textAttr.anchor = TK_ANCHOR_N;
    textAttr.fontPtr = axisPtr->fontPtr;
    textAttr.bgColorPtr = Tk_3DBorderColor(graphPtr->border);
    textAttr.fgColorPtr = axisPtr->fgColorPtr;
    textAttr.gc = axisPtr->textGC;

    for (i = 0; i < axisPtr->numLabels; i++) {
	Blt_DrawText(graphPtr->display, graphPtr->canvas,
	    axisPtr->labelArr[i].text, &textAttr, axisPtr->labelArr[i].pos,
	    labelY);
    }
    if (axisPtr->numSegments > 0) {
	XDrawSegments(graphPtr->display, graphPtr->canvas, axisPtr->lineGC,
	    axisPtr->segArr, axisPtr->numSegments);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * PrintXAxis --
 *
 * -----------------------------------------------------------------
 */
static void
PrintXAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->X;
    int labelY;
    register int i;
    TextAttr textAttr;

    if (axisPtr->numTicks <= 0) {
	return;
    }
    labelY = MAPY((Axis *)graphPtr->Y, -LABEL_TICK);
    labelY += graphPtr->intBWidth + PADY;

    textAttr.theta = axisPtr->theta;
    textAttr.fgColorPtr = axisPtr->fgColorPtr;
    textAttr.bgColorPtr = (XColor *)NULL;
    textAttr.fontPtr = axisPtr->fontPtr;
    textAttr.anchor = TK_ANCHOR_N;

    Blt_FontToPostScript(graphPtr, axisPtr->fontPtr);
    Blt_ForegroundToPostScript(graphPtr, axisPtr->fgColorPtr);
    for (i = 0; i < axisPtr->numLabels; i++) {
	Blt_TextToPostScript(graphPtr, axisPtr->labelArr[i].text, &textAttr,
	    axisPtr->labelArr[i].pos, labelY);
    }
    if (axisPtr->numSegments > 0) {
	/* Set the axis color and line width */
	Blt_SetLineAttributes(graphPtr, axisPtr->fgColorPtr,
	    axisPtr->lineWidth, 0);
	Blt_SegmentsToPostScript(graphPtr, axisPtr->segArr,
	    axisPtr->numSegments);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * LayoutYAxis --
 *
 *	Pre-calculate the y-coordinate positions of the axis baseline,
 *	ticks and labels to be used later when displaying the Y axis.
 *	Ticks (minor and major) will be saved in an array of XSegments
 *	so that they can be drawn in one XDrawSegments call. The
 *	strings representing the tick labels and the corresponding
 *	window positions are saved in an array of TickLabels.
 *
 *      Calculates the values for each major and minor tick and checks to
 *	see if they are in range (the outer ticks may be outside of the
 *	range of plotted values).
 *
 * Results:
 *	None.
 *
 * SideEffects:
 *	Line segments and tick labels saved will be used to draw
 *	the Y axis.
 * -----------------------------------------------------------------
 */
static void
LayoutYAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->Y;
    XSegment *segArr;
    unsigned int need;
    int padX, baseLine;
    int majorTickPos, minorTickPos;
    double normY;
    double minY, maxY;
    double minLimit, maxLimit;
    double value, subValue;
    register int count, tick;
    register int i, j;
    int halfLW;

    /* Adjust offset for the interior border width and the line width */
    halfLW = axisPtr->lineWidth / 2;
    padX = graphPtr->intBWidth + halfLW + 1;
    if (graphPtr->intBWidth > 0) {
	padX++;
    }
    /*
     * Pre-calculate the x-coordinate positions of the baseline, tick labels,
     * and the individual major and minor ticks.
     */
    majorTickPos = MAPX((Axis *)graphPtr->X, -MAJOR_TICK) - padX + halfLW;
    minorTickPos = MAPX((Axis *)graphPtr->X, -MINOR_TICK) - padX + halfLW;
    baseLine = graphPtr->origin.x - padX;

    /*
     * We want to draw all the ticks and subticks in one XDrawSegments call.
     * Allocate an array to hold all the tick and axis line segments.
     */
    need = (1 + (axisPtr->numTicks * (axisPtr->subTicks + 1)));
    segArr = (XSegment *)malloc(need * sizeof(XSegment));

    if (segArr == NULL) {
	return;
    }
    if ((axisPtr->logScale) || (axisPtr->loose) ||
	(axisPtr->limits[AXIS_MIN_LIMIT] == axisPtr->limits[AXIS_MAX_LIMIT])) {
	minY = axisPtr->tickMin, maxY = axisPtr->tickMax;
    } else {
	minY = axisPtr->limits[AXIS_MIN_LIMIT];
	maxY = axisPtr->limits[AXIS_MAX_LIMIT];
    }
    segArr[0].x1 = segArr[0].x2 = baseLine;	/* Axis baseline */
    normY = NORM(axisPtr, minY);
    segArr[0].y1 = MAPY(axisPtr, normY);
    normY = NORM(axisPtr, maxY);
    segArr[0].y2 = MAPY(axisPtr, normY);

    count = 1, tick = 0;
    if (!axisPtr->showTicks) {
	goto done;
    }
    minLimit = minY - (axisPtr->subStep * 0.001);
    maxLimit = maxY + (axisPtr->subStep * 0.001);

    value = axisPtr->tickMin;	/* Start from axis low value */
    for (i = 0; i < axisPtr->numTicks; i++) {
	subValue = value = UROUND(value, axisPtr->step);
	for (j = 1; j < axisPtr->subTicks; j++) {

	    /*
	     * Use graduated subticks for logscale if major units are decades
	     */
	    if ((axisPtr->logScale) && (axisPtr->step == 1.0)) {
		subValue = value + logTable[j];
	    } else {
		subValue += axisPtr->subStep;
	    }
	    if ((subValue >= minLimit) && (subValue <= maxLimit)) {
		normY = NORM(axisPtr, subValue);
		segArr[count].x1 = baseLine;
		segArr[count].x2 = minorTickPos;
		segArr[count].y1 = segArr[count].y2 = MAPY(axisPtr, normY);
		count++;
	    }
	}
	if ((value >= minLimit) && (value <= maxLimit)) {
	    normY = NORM(axisPtr, value);
	    segArr[count].x1 = baseLine;	/* Major tick */
	    segArr[count].x2 = majorTickPos;
	    segArr[count].y1 = segArr[count].y2 = MAPY(axisPtr, normY);

	    axisPtr->labelArr[tick++].pos = segArr[count].y1;
	    count++;
	}
	value += axisPtr->step;
    }
  done:
#ifdef notdef
    if (count > need) {
	fprintf(stderr, "Number allocated = %d, used = %d\n", need, count);
    }
#endif
    if (axisPtr->segArr != NULL) {
	free((char *)axisPtr->segArr);
    }
    axisPtr->segArr = segArr;
    axisPtr->numSegments = count;
}

/*
 * -----------------------------------------------------------------
 *
 * DisplayYAxis --
 *
 * -----------------------------------------------------------------
 */
static void
DisplayYAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->Y;
    register int i;
    int labelX;
    TextAttr textAttr;

    if (axisPtr->numTicks < 1) {
	return;
    }
    labelX = MAPX((Axis *)graphPtr->X, -LABEL_TICK) -
	(axisPtr->lineWidth + graphPtr->intBWidth + PADX);

    textAttr.theta = axisPtr->theta;
    textAttr.anchor = TK_ANCHOR_E;
    textAttr.fontPtr = axisPtr->fontPtr;
    textAttr.fgColorPtr = axisPtr->fgColorPtr;
    textAttr.bgColorPtr = Tk_3DBorderColor(graphPtr->border);
    textAttr.gc = axisPtr->textGC;

    for (i = 0; i < axisPtr->numLabels; i++) {
	Blt_DrawText(graphPtr->display, graphPtr->canvas,
	    axisPtr->labelArr[i].text, &textAttr, labelX,
	    axisPtr->labelArr[i].pos);
    }
    if (axisPtr->numSegments > 0) {
	XDrawSegments(graphPtr->display, graphPtr->canvas, axisPtr->lineGC,
	    axisPtr->segArr, axisPtr->numSegments);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * PrintYAxis --
 *
 * -----------------------------------------------------------------
 */
static void
PrintYAxis(graphPtr)
    Graph *graphPtr;
{
    Axis *axisPtr = (Axis *)graphPtr->Y;
    register int i;
    int labelX;
    TextAttr textAttr;

    if (axisPtr->numTicks <= 0) {
	return;
    }
    labelX = MAPX((Axis *)graphPtr->X, -LABEL_TICK) -
	(graphPtr->intBWidth + PADX);

    textAttr.theta = axisPtr->theta;
    textAttr.anchor = TK_ANCHOR_E;
    textAttr.fontPtr = axisPtr->fontPtr;
    textAttr.fgColorPtr = axisPtr->fgColorPtr;
    textAttr.bgColorPtr = (XColor *)NULL;

    Blt_FontToPostScript(graphPtr, axisPtr->fontPtr);
    Blt_ForegroundToPostScript(graphPtr, axisPtr->fgColorPtr);
    for (i = 0; i < axisPtr->numLabels; i++) {
	Blt_TextToPostScript(graphPtr, axisPtr->labelArr[i].text, &textAttr,
	    labelX, axisPtr->labelArr[i].pos);
    }
    /* Set the axis color and line width */
    if (axisPtr->numSegments > 0) {
	Blt_SetLineAttributes(graphPtr, axisPtr->fgColorPtr,
	    axisPtr->lineWidth, 0);
	Blt_SegmentsToPostScript(graphPtr, axisPtr->segArr,
	    axisPtr->numSegments);
    }
}

static void
GetAxisGeometry(graphPtr, axisPtr, widthPtr, heightPtr)
    Graph *graphPtr;
    Axis *axisPtr;
    int *widthPtr, *heightPtr;
{
    register int i;
    register int count;
    char *string;
    int textWidth, textHeight;
    int boxWidth, boxHeight;
    int maxWidth, maxHeight;
    double value;
    double minAxis, maxAxis;
    TickLabel *labelArr;

    if ((axisPtr->logScale) || (axisPtr->loose) ||
	(axisPtr->limits[AXIS_MIN_LIMIT] == axisPtr->limits[AXIS_MAX_LIMIT])) {
	minAxis = axisPtr->tickMin, maxAxis = axisPtr->tickMax;
    } else {
	minAxis = axisPtr->limits[AXIS_MIN_LIMIT];
	maxAxis = axisPtr->limits[AXIS_MAX_LIMIT];
    }
    /*
     * Need to use numbers just beyond the limits when testing for equality
     */
    minAxis -= (axisPtr->subStep * 0.001);
    maxAxis += (axisPtr->subStep * 0.001);

    labelArr = (TickLabel *)malloc(axisPtr->numTicks * sizeof(TickLabel));

    textHeight = TEXTHEIGHT(axisPtr->fontPtr);

    maxHeight = maxWidth = 0;
    count = 0;
    value = axisPtr->tickMin;
    for (i = 0; i < axisPtr->numTicks; i++, value += axisPtr->step) {
	value = UROUND(value, axisPtr->step);
	if ((value < minAxis) || (value > maxAxis)) {
	    continue;		/* out of range */
	}
	string = CreateLabel(graphPtr, axisPtr, value);
	textWidth = Blt_TextStringWidth(axisPtr->fontPtr, string);
	labelArr[count++].text = string;

	if (axisPtr->theta == 0.0) {
	    boxWidth = textWidth, boxHeight = textHeight;
	} else {
	    Blt_GetBoundingBox(textWidth, textHeight, axisPtr->theta,
		&boxWidth, &boxHeight, (XPoint *)NULL);
	}
	if (boxWidth > maxWidth) {
	    maxWidth = boxWidth;
	}
	if (boxHeight > maxHeight) {
	    maxHeight = boxHeight;
	}
    }
    *widthPtr = maxWidth, *heightPtr = maxHeight;
    if (axisPtr->labelArr != NULL) {
	FreeLabels(axisPtr->labelArr, axisPtr->numLabels);
	free((char *)axisPtr->labelArr);
    }
    axisPtr->labelArr = labelArr;
    axisPtr->numLabels = count;
}

void
Blt_UpdateAxisBackgrounds(graphPtr, colorPtr)
    Graph *graphPtr;
    XColor *colorPtr;
{
    Axis *axisPtr;

    axisPtr = (Axis *)graphPtr->Y;
    XSetBackground(Tk_Display(graphPtr->tkwin), axisPtr->textGC,
	colorPtr->pixel);
    axisPtr = (Axis *)graphPtr->X;
    XSetBackground(Tk_Display(graphPtr->tkwin), axisPtr->textGC,
	colorPtr->pixel);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_ComputeLayout --
 *
 * 	Calculate the layout of the graph.  Based upon the data,
 *	axis limits, X and Y titles, and title height, determine
 *	the cavity left which is the plotting surface.  The first
 *	step get the data and axis limits for calculating the space
 *	needed for the top, bottom, left, and right margins.
 *
 * 	1) The LEFT margin is the area from the left border to the
 *	   Y axis (not including ticks). It composes the border
 *	   width, the width an optional Y axis label and its padding,
 *	   and the tick numeric labels. The Y axis label is rotated
 *	   90 degrees so that the width is the font height.
 *
 * 	2) The RIGHT margin is the area from the end of the graph
 *	   to the right window border. It composes the border width,
 *	   some padding, the font height (this may be dubious. It
 *	   appears to provide a more even border), the max of the
 *	   legend width and 1/2 max X tick number. This last part is
 *	   so that the last tick label is not clipped.
 *
 *           Area Width
 *      _____________________________________________________
 *      _______________________________________________________
 *      |         |                               |           |
 *      |         |   TOP  height of title        |           |  A
 *      |_________|_______________________________|___________|  r
 *      |         |                               |           |  e
 *      |  Left   |                               | Right     |  a
 *      |         |                               |           |
 *      | Y       |                               |           |  H
 *      |         |     PLOTTING SURFACE  102%    |           |  e
 *      | l       |     100 + 2% tick length      |           |  i
 *      | a       |                               |           |  g
 *      | b       |                               | legend    |  h
 *      | e       |                               | width     |  t
 *      | l       |                               |           |
 *      |         |                               |           |
 *      | width of|                               |           |
 *      | widest  |                               |           |
 *      | number  |                               |           |
 *      |         |                               |           |
 *      |         |                               |           |
 *      |         |                               |           |
 *      |         | origin (xoffset, yoffset)     |           |
 *      |_________|_______________________________|___________|
 *      |         |   height of number       1/2 width of     |
 *      |         |                         max tick number   |
 *      |         |   BOTTOM   height of X label  |           |
 *      |_________|_______________________________|___________|
 *
 * 3) The TOP margin is the area from the top window border to the top
 *    of the graph. It composes the border width, twice the height of
 *    the title font (if one is given) and some padding between the
 *    title.
 *
 * 4) The BOTTOM margin is area from the bottom window border to the
 *    X axis (not including ticks). It composes the border width, the height
 *    an optional X axis label and its padding, the height of the font
 *    of the tick labels.
 *
 * The plotting area is between the margins which includes the X and Y axes
 * including the ticks but not the tick numeric labels. The length of
 * the ticks and its padding is 5% of the entire plotting area.  Hence the
 * entire plotting area is scaled as 105% of the width and height of the
 * area.
 *
 * The axis labels, ticks labels, title, and legend may or may not be
 * displayed which must be taken into account.
 *
 *
 * -----------------------------------------------------------------
 */
int
Blt_ComputeLayout(graphPtr)
    Graph *graphPtr;
{
    int left, right, top, bottom;
    int maxTickWidth = 0;
    int leftOver;
    int borderWidths;
    int lineHeight = TEXTHEIGHT(graphPtr->fontPtr);
    Axis *X, *Y;
    int twiceHeight = (2 * lineHeight);
    int width, height;
    int halfHeight = (lineHeight / 2);

    /*  Blt_ComputeAxes(graphPtr); */
    X = (Axis *)graphPtr->X;
    Y = (Axis *)graphPtr->Y;

    top = (graphPtr->title != NULL) ? twiceHeight : halfHeight;
    left = (graphPtr->yTitle != NULL) ? twiceHeight : halfHeight;

    GetAxisGeometry(graphPtr, Y, &width, &height);
    if (Y->showTicks) {		/* Get the width of the widest Y tick label */
	left += width + PADX;
    }
    bottom = (graphPtr->xTitle != NULL) ? twiceHeight : halfHeight;
    (*graphPtr->legendPtr->geomProc) (graphPtr);

    GetAxisGeometry(graphPtr, X, &width, &height);
    if (X->showTicks) {
	bottom += height;
	maxTickWidth = width / 2;
    }
    /* Override calculated values if user specified margins */
    if (graphPtr->leftMargin > 0) {
	left = graphPtr->leftMargin;
    }
    if (graphPtr->topMargin > 0) {
	top = graphPtr->topMargin;
    }
    if (graphPtr->bottomMargin > 0) {
	bottom = graphPtr->bottomMargin;
    }
    right = lineHeight;		/* dubious using font height here */
    if ((graphPtr->legendPtr->mapped) &&
	(graphPtr->legendPtr->position.x == DEF_POSITION)) {
	right += MAX(maxTickWidth, graphPtr->legendPtr->width);
    }
    if (graphPtr->rightMargin > 0) {
	right = graphPtr->rightMargin;
    }
    /*
     * Based upon the margins, calculate the space left for the graph.
     */
    borderWidths = graphPtr->extBWidth + graphPtr->intBWidth;
    X->offset = left + borderWidths;
    Y->offset = graphPtr->height - (bottom + borderWidths);
    leftOver = graphPtr->width - (left + right + (2 * borderWidths));
    if (leftOver < 0) {
	return TCL_ERROR;
    }
    X->scale = leftOver / (1.0 + LABEL_TICK);	/* Pixels per X unit */
    leftOver = graphPtr->height - (top + bottom + (2 * borderWidths));
    if (leftOver < 0) {
	return TCL_ERROR;
    }
    Y->scale = leftOver / (1.0 + LABEL_TICK);	/* Pixels per Y unit */
    /*
     * Add tick distance to center graph
     */
    X->offset += ROUND(LABEL_TICK * X->scale);
    Y->offset -= ROUND(LABEL_TICK * Y->scale);

    /* Calculate the average symbol (formula is arbitrary) */
    graphPtr->avgSymSize = ROUND(log(((double)X->scale) * Y->scale) * 0.8);

    graphPtr->origin.x = MAPX(X, 0.0);
    graphPtr->origin.y = MAPY(Y, 0.0);
    graphPtr->extreme.x = MAPX(X, 1.0);
    graphPtr->extreme.y = MAPY(Y, 1.0);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GetAxisLimits --
 *
 *	This procedure returns a string representing the axis limits
 *	of the graph.  The format of the string is { xmin ymin xmax ymax}.
 *
 * Results:
 *	Always returns TCL_OK.  The interp->result field is
 *	a list of the graph axis limits.
 *
 *--------------------------------------------------------------
 */
static int
GetAxisLimits(axisPtr)
    Axis *axisPtr;
{
    char string[200];

    Tcl_PrintDouble(axisPtr->interp, axisPtr->min, string);
    Tcl_AppendElement(axisPtr->interp, string);
    Tcl_PrintDouble(axisPtr->interp, axisPtr->max, string);
    Tcl_AppendElement(axisPtr->interp, string);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_CreateAxis --
 *
 *	Create and initialize a structure containing information to
 * 	display a graph axis.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
int
Blt_CreateAxis(graphPtr, type, flags)
    Graph *graphPtr;
    enum AxisTypes type;
    int flags;			/* Configuration flags */
{
    Axis *axisPtr;

    axisPtr = (Axis *)calloc(1, sizeof(Axis));

    if (axisPtr == NULL) {
	Tcl_AppendResult(graphPtr->interp, "can't allocate axis structure",
	    (char *)NULL);
	return TCL_ERROR;
    }
    axisPtr->type = type;
    axisPtr->interp = graphPtr->interp;
    axisPtr->flags = 0;
    axisPtr->elemMin = 1.0;	/* Arbitrary positive values */
    axisPtr->elemMax = 5.0;
    axisPtr->step = 1.0;
    axisPtr->theta = 0.0;

    if (ConfigureAxis(graphPtr, axisPtr, 0, (char **)NULL, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    axisPtr->destroyProc = DestroyAxis;
    if (axisPtr->type == X_AXIS_TYPE) {
	axisPtr->displayProc = DisplayXAxis;
	axisPtr->printProc = PrintXAxis;
	axisPtr->layoutProc = LayoutXAxis;
	graphPtr->X = (GraphAxis *)axisPtr;
    } else if (axisPtr->type == Y_AXIS_TYPE) {
	axisPtr->displayProc = DisplayYAxis;
	axisPtr->printProc = PrintYAxis;
	axisPtr->layoutProc = LayoutYAxis;
	graphPtr->Y = (GraphAxis *)axisPtr;
    }
    return TCL_OK;
}

int
Blt_AxisCmd(graphPtr, axis, argc, argv, flags)
    Graph *graphPtr;
    GraphAxis *axis;
    int argc;
    char **argv;
    int flags;
{
    int result = TCL_ERROR;
    Axis *axisPtr = (Axis *)axis;
    Tcl_Interp *interp = graphPtr->interp;
    char c;
    int length;
    char *name;

    name = axisCmds[(int)axisPtr->type];
    if (argc < 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " ", name, " option ?args?\"", NULL);
	return TCL_ERROR;
    }
    c = argv[2][0];
    length = strlen(argv[2]);

    if ((c == 'c') && (strncmp(argv[2], "configure", length) == 0)) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" ", name, " configure ?args?\"", NULL);
	    return TCL_ERROR;
	}
	result = ConfigureAxis(graphPtr, axisPtr, argc - 3, argv + 3, flags);
    } else if ((c == 'l') && (strncmp(argv[2], "limits", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" ", name, " limits\"", NULL);
	    return TCL_ERROR;
	}
	result = GetAxisLimits(axisPtr);
    } else {
	Tcl_AppendResult(interp, "bad ", name, " option \"", argv[2], "\":\
 should be configure or limits", (char *)NULL);
	return TCL_ERROR;
    }
    return result;
}
