/* isis-tcl.c - ISIS/Tcl Interface
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * isis-tcl.c,v 1.1 1992/07/31 20:12:07 snl Exp
 *
 * isis-tcl.c,v
 * Revision 1.1  1992/07/31  20:12:07  snl
 * Massive checkin
 *
 */

static char rcsID[] = "isis-tcl.c,v 1.1 1992/07/31 20:12:07 snl Exp";

typedef struct JoinRec {
  List_Links links;
} JoinRec;

#include <sys/types.h>
#include <sys/time.h>
#include <isis.h>
#include <tcl/tcl.h>
#include <list.h>

#define TclProc(name) \
int name/**/_cmd (clientData, interp, argc, argv) \
ClientData clientData; \
Tcl_Interp *interp; \
int argc; \
char **argv;

#define UNIMPLEMENTED()\
Tcl_AppendResult(interp,argv[0],": Tcl interface not yet implemented",0);\
return TCL_ERROR

#define USAGE(e,u)\
Tcl_AppendResult(interp,argv[0],": ",e," -- usage: \"",argv[0]," ",u,"\"",0);\
return TCL_ERROR

struct _QueuedJoin {
  List_Links links;
  char *pgname;
  Tcl_Interp *interp;
  char *cmd;
  int done;
} QueuedJoin;

struct _JoinedGroups {
  List_Links links;
  char *pgname;
  address *gaddr;
} JoinedGroups;

struct _MonitorClosure {
  char *cmd;
  Tcl_Interp *interp;
} MonitorClosure;

/*
 * Exported interface
 */

void
ISIS_InitInterp(interp)
     Tcl_Interp *interp;
{
#ifndef USE_ISIS
  Tcl_SetVar(interp, "HAVE_ISIS", "0", TCL_GLOBAL_ONLY);
#else
  int broadcast_cmd(), lookup_process_group_cmd(),
      join_cmd(), leave_cmd(), entry_cmd();

  Tcl_CreateCommand(interp, "isis_pg_join", join_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_pg_cleanup", join_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_pg_lookup", lookup_process_group_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_pg_leave", leave_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_entry", entry_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_bcast", broadcast_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_abcast", broadcast_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_cbcast", broadcast_cmd, 0, 0);
  Tcl_CreateCommand(interp, "isis_gbcast", broadcast_cmd, 0, 0);
  Tcl_SetVar(interp, "HAVE_ISIS", "1", TCL_GLOBAL_ONLY);
  isis_remote_init(0, 0, 0, ISIS_NOCOPYRIGHTNOTICE | ISIS_PANIC);
  isis_start_done();
  THREAD_ISIS_ENTER();
#endif /* USE_ISIS */
}

void
ISIS_TkInit();
{
#ifdef USE_ISIS
#ifdef TK_INTERFACE

  extern int isis_socket;
  extern int intercl_socket;
  void handle_ISIS_input();
  void add_next_ISIS_timeout();

  add_next_ISIS_timeout();
  Tk_CreateFileHandler(isis_socket, TK_READABLE|TK_EXCEPTION,
		       handle_ISIS_input, 0);
  Tk_CreateFileHandler(intercl_socket, TK_READABLE|TK_EXCEPTION,
		       handle_ISIS_input, 0);
#endif /* TK_INTERFACE */
#endif /* USE_ISIS */
}

/*
 * Tcl command procedures (exported to Tcl)
 */

#ifdef USE_ISIS

static List_Links *_join_queue = (List_Links *)0;

static TclProc(join)
{
  void clean_join_queue(), queue_join(), do_joins();

  if (!strcmp(argv[0], "pg_cleanup")) {
    clean_join_queue();
    return TCL_OK;
  }
  if (argc < 2) {
    USAGE("wrong # args","pgname ?monitor-cmd ?args??");
  }

  queue_join(interp, argc - 1, &argv[1]);
  do_joins();

  return TCL_OK;
}

static TclProc(lookup_process_group)
{
  address *result;

  if (argc != 2) {
    Tcl_AppendResult(interp, "wrong # args: must be \"", argv[0],
		     " gname\"", 0);
    return TCL_ERROR;
  }
  result = pg_lookup(argv[1]);
  if (!addr_isnull(result))
    sprintf(interp->result, "%d %d %d %d %d",
	    (int)result->addr_process,
	    (int)result->addr_portno,
	    (int)result->addr_site,
	    (int)result->addr_incarn,
	    (int)result->addr_entry);
  return TCL_OK;
}

static TclProc(leave)
{
}

static TclProc(entry)
{
}

static TclProc(broadcast)
{
}

/*
 * Support routines
 */

static void
clean_join_queue()
{
  List_Links *jl;

  LIST_FORALL(_join_queue, jl) {
    QueuedJoin *j = (QueuedJoin *)jl;

    if (j->done) {
      List_Remove(jl);
      ckfree(j->pgname);
      if (j->cmd != (char *)0)
	ckfree(j->cmd)
      ckfree(jl);
    }
  }
}

static void
queue_join(interp, argc, argv)
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  QueuedJoin *j;
  List_Links *jl;

  j = (QueuedJoin *)ckalloc(sizeof(QueuedJoin));
  List_InitElement((List_Links *)j);
  j->pgname = strdup(argv[1]);
  if (argc == 2)
    j->cmd = (char *)0;
  else
    j->cmd = Tcl_Merge(argc - 2, &argv[2]);
  j->interp = interp;
  j->done = 0;

  if (_join_queue == (List_Links *)0) {
    _join_queue = (List_Links *)ckalloc(sizeof(List_Links));
    List_Init(_join_queue);
  }
  jl = (List_Links *)j;
  List_Insert(jl, LIST_ATREAR(_join_queue));
}

static void
remember_group(gname, gaddr)
     char *gname;
     address *gaddr;
{
}

static void
do_joins()
{
  List_Links *jl;

  isis_task(do_joins, "tcl:do_joins");

  LIST_FORALL(_join_queue, jl) {
    QueuedJoin *j = (QueuedJoin *)jl;
    address *gaddr;
    void change_monitor();

    if (j->done)
      continue;
    if (j->cmd == (char *)0)
      gaddr = pg_join(j->pgname, 0);
    else {
      MonitorClosure *monitor_info;
      void change_monitor();
      
      monitor_info = (MonitorClosure *)
	ckalloc(sizeof(MonitorClosure));
      monitor_info->cmd = j->cmd;
      monitor_info->interp = j->interp;
      gaddr = pg_join(pgname,
		      PG_MONITOR, change_monitor, monitor_info,
		      0);
    }
    if (addr_isnull(gaddr))
      Tcl_VarEval(interp, "isiserror join_failed ", j->pgname, " ", j->cmd, 0);
    j->done = 1;
  }
}


static void
change_monitor(pg, arg)
     groupview *pg;
     MonitorClosure *arg;
{
  char n_mem_str[20], my_index_str[20];

  gaddr = &pg->gv_gaddr;
  n_memb = pg->gv_nmemb;
  my_index = pg_rank(gaddr, &my_address);
  sprintf(n_mem_str, "%d", n_memb);
  sprintf(my_index_str, "%d", my_index);
  Tcl_VarEval(arg->interp, arg->cmd, " ", n_mem_str, " ", my_index_str, 0);
}

void
recieve(mp)
  message *mp;
{
  int op, status;

#ifdef TALKY
  fprintf(stderr, "ISIS recieve entry called\n");
#endif
  if (msg_get(mp, "%l", &op) != 1) {
#ifdef TALKY
    fprintf(stderr, "bad remote message recieved - ignored\n");
#endif
    status = REVAL_BAD_OP;
  } else {
    status = TCL_OK;
    switch (op) {
    case REVAL:
      {
	int argc;
	
	if (msg_get(mp, "%d", &argc) != 1) {
#ifdef TALKY
	  fprintf(stderr, "bad remote eval recieved - ignored\n");
#endif
	  status = REVAL_BAD_ARGC;
	} else {
	  int i;
	  char **argv, *cmd;
	  
	  argv = (char **)ckalloc(argc * sizeof(char *));
	  if (argv == (char **)0) {
	    status = REVAL_NO_MEM;
	    break;
	  }
	  for (i = 0; i < argc; i++) {
	    if (msg_get(mp, "%-s", &argv[i]) != 1) {
	      status = REVAL_BAD_ARGV;
	      ckfree(argv);
	      break;
	    }
	  }
	  cmd = Tcl_Merge(argc, argv);
	  if (cmd == (char *)0)
	    status = REVAL_NO_MEM;
	  else
	    status = Tcl_Eval(interp, cmd, 0, 0);
	}
      }
      break;
#ifdef NDIM_SPECIAL_REQUESTS
    case SHIP:
    case MSG:
#endif
    default:
#ifdef TALKY
      fprintf(stderr, "bad remote message op %d recieved - ignored\n", op);
#endif
      break;
    }
  }
  if (status >= 0)
    reply(mp, "%d%s%s", status, ndimws_username, interp->result);
  else
    reply(mp, "%d%s", status, ndimws_username);
}

int
broadcast_eval_cmd(clientData, interp, argc, argv)
  ClientData clientData;
  Tcl_Interp *interp;
  int argc;
  char **argv;
{
  char opts[10];
  int nwant, nreply, i;
  message *msg, *replies[MAX_PROCS];
  address *addr;

  if (argc < 5) {
    Tcl_AppendResult(interp, "wrong # args: must be \"", argv[0],
		     " options address nwant cmd ?args?\"", 0);
    return TCL_ERROR;
  }
  opts[0] = '\0';
  if (index(argv[1], 'm') == (char *)0)
    strcat(opts, "m");
  strcat(opts, argv[1]);
  addr = pg_lookup(argv[2]);
  if (addr == (address *)0) {
    Tcl_AppendResult(interp, argv[0], ": bad address -- \"", argv[1], "\"", 0);
    return TCL_ERROR;
  } else if (addr_isnull(addr)) {
    Tcl_AppendResult(interp, argv[0], ": bad address -- \"", argv[1], "\"", 0);
    return TCL_ERROR;
  }
  if (!strcasecmp(argv[3], "all"))
    nwant = ALL;
  else if (!strcasecmp(argv[3], "majority"))
    nwant = MAJORITY;
  else if (sscanf(argv[3], "%d", &nwant) != 1) {
    Tcl_AppendResult(interp, "\"",argv[3],"\" must be an integer or one of: ",
		     "all, majority", 0);
    return TCL_ERROR;
  }
  msg = msg_gen("%d%d%s", REVAL, argc-4, argv[4]);
  for (i = 5; i < argc; i++)
    msg_put(msg, "%s", argv[i]);
  if (!strcmp(argv[0], "bcast") || !strcmp(argv[0], "abcast"))
    nreply = abcast_l(opts, addr, NDIM_RECV, msg, nwant, &replies);
  else if (!strcmp(argv[0], "abcast"))
    nreply = abcast_l(opts, addr, NDIM_RECV, msg, nwant, &replies);
  else if (!strcmp(argv[0], "cbcast"))
    nreply = cbcast_l(opts, addr, NDIM_RECV, msg, nwant, &replies);
  else {
    Tcl_AppendResult(interp, "\"", argv[0], "\" must be one of: ",
		     "bcast, abcast, cbcast, gbcast", 0);
    return TCL_ERROR;
  }
  if (nreply < 0) {
    extern int isis_errno;

    sprintf(interp->result, "ISIS Error code %d", isis_errno);
    return TCL_ERROR;
  }
  for (i = 0; i < nreply; i++) {
    char sstring[30], *result, *rem_user;
    int rem_stat;

    Tcl_AppendResult(interp, "{", 0);
    if (msg_get(replies[i], "%d%-s", &rem_stat, &rem_user) != 2)
      Tcl_AppendResult(interp, "-100 \"Badly formed reply\"", 0);
    else {
      address *from;
      long id;
      char *stat_string;

      from = msg_getsender(replies[i]);
      id = msg_getid(replies[i]);
      if (rem_stat >= 0)
	stat_string = "OK";
      else {
	switch (rem_stat) {
	case REVAL_BAD_OP:
	  stat_string = "\"Bad operator\"";
	  break;
	case REVAL_BAD_ARGC:
	  stat_string = "\"Bad arg count\"";
	  break;
	case REVAL_BAD_ARGV:
	  stat_string = "\"Bad arg vector\"";
	  break;
	case REVAL_NO_MEM:
	  stat_string = "\"No memory\"";
	  break;
	default:
	  stat_string = "UNKNOWN STATUS CODE";
	  break;
	}
      }

#define CatNumber(n,fmt)\
 sprintf(sstring,fmt,n); Tcl_AppendResult(interp, sstring, " ", 0);

      CatNumber(rem_stat, "%d");
      Tcl_AppendResult(interp, stat_string, " ", rem_user, " ", 0);
      CatNumber(id, "%l");
      CatNumber((int)from->addr_process, "%d");
      CatNumber((int)from->addr_site, "%d");
      CatNumber((int)from->addr_incarn, "%d");

#undef CatNumber

      if (rem_stat >= 0) {
	if (msg_get(replies[i], "%-s", &result) != 1)
	  Tcl_AppendResult(interp, "{No result}", 0);
	else
	  Tcl_AppendResult(interp, "{", result, "}", 0);
      }
    }
    Tcl_AppendResult(interp, "} ", 0);
  }
  return TCL_OK;
}


#ifdef TK_INTERFACE
static Tk_TimerToken _ISIS_Tk_timeout = (Tk_TimerToken)0;
int _calling_ISIS = 0;

static void
add_next_ISIS_timeout()
{
  void handle_ISIS_timeout();
  unsigned long next_timeout, timeout;

  next_timeout = isis_next_timeout();
  if (_ISIS_Tk_timeout != (Tk_TimerToken)0)
    Tk_DeleteTimerHandler(_ISIS_Tk_timeout);
  timeout = (unsigned long)(next_timeout + 1000);
  if (timeout > 1000)
    timeout = 1000;
  _ISIS_Tk_timeout = Tk_CreateTimerHandler(timeout,handle_ISIS_timeout,0);
}

static void
handle_ISIS_timeout(clientData)
  ClientData clientData;
{
  void add_next_ISIS_timeout();

  _ISIS_Tk_timeout = (Tk_TimerToken)0;
  if (!_calling_ISIS) {
    _calling_ISIS = 1;
    isis_accept_events(ISIS_ASYNC);
    _calling_ISIS = 0;
  }
  add_next_ISIS_timeout();
}

static void
handle_ISIS_input(clientData, mask)
  ClientData clientData;
  int mask;
{
  void add_next_ISIS_timeout();

  if (!_calling_ISIS) {
    _calling_ISIS = 1;
    isis_accept_events(ISIS_ASYNC);
    _calling_ISIS = 0;
  }
  add_next_ISIS_timeout();
}
#endif /* TK_INTERFACE */

#endif /* USE_ISIS */
