/* bos.h - Public Interface to BOS C-callable library */

/* Copyright (c) 1992 Engineering Design Research Center
 */

#ifndef __bos_H__
#define __bos_H__

#ifndef __bosInt_H__
#include <tcl.h>
#include "bosMacros.h"
#endif /* __bosInt_H__ */
#include <gdbm.h>

typedef void ANYTHING;			/* Universal type */

/* Slot types */

typedef short Bos_Slot_Type;

#define Bos_SLOT_NORMAL 0	/* Normal (value) slot */
#define Bos_SLOT_METHOD 1	/* Tcl method */
#define Bos_SLOT_CMETHOD 2	/* C method */
#define Bos_SLOT_OBJECT 3	/* pointer to another object */
#define Bos_SLOT_REFERENCE 4	/* reference to another object */
#define Bos_SLOT_FOREIGN 5	/* Defined in C (a foreign slot type) */
#define Bos_SLOT_ILLEGAL 999	/* special undefined type */
#define Bos_SLOT_EVANESCENT_MASK (1<<15)

#define Bos_PlainSlotType(st) ((st) & ~Bos_SLOT_EVANESCENT_MASK)

/* Priorities */

typedef short Bos_Slot_Pri;

#define Bos_PRI_HIGHEST -1
#define Bos_PRI_LOWEST -2
#define Bos_PRI_ILLEGAL -3

typedef short Bos_Slot_Subtype;

#ifndef __bosInt_H__
typedef struct _Bos_Object {
  char *name;			/* Name of the object. */
} Bos_Object;
#else
typedef struct _Bos_Object Bos_Object;
#endif /* __bosInt_H__ */

#ifndef __bosInt_H__
typedef ANYTHING Bos_World;
#else
typedef struct _Bos_World Bos_World;
#endif /* __bosInt_H__ */

typedef struct _Bos_CMethod_ClientData {
  struct _Bos_World *world;
  struct _Bos_Object *object;
  struct _Bos_Object *self;
} Bos_CMethod_ClientData;

/* Return status codes from routines that don't return Bos_Object *
 */

#define BOS_OK 0		/* No error. */
#define BOS_NOT_FOUND 1		/* Object or slot not found. */
#define BOS_TCL_ERROR 2		/* Tcl error during method invocation. */
#define BOS_ALREADY 3		/* Object or slot already exists. */
#define BOS_ERROR 4             /* Unknown error. */

/*
 * Public function declarations
 */

void
Bos_InitUtilities _ARGS_((
	Tcl_Interp *interp
));

void
Bos_InitializeBuiltins _NO_ARGS_;

void
Bos_CreateBuiltinObjects _ARGS_((
	Bos_World *world,
	Tcl_Interp *interp
));

int
Bos_HookCMethod _ARGS_((
	char *name,
	ANYTHING *hook
));

ANYTHING *
Bos_GetCMethodPointer _ARGS_((
	char *name
));

char *
Bos_GetCMethodName _ARGS_((
	ANYTHING *ptr
));

void
Bos_UnhookCMethod _ARGS_((
	char *name
));

Bos_Slot_Subtype
Bos_DefineCSlotType _ARGS_((
	char *name,
	ANYTHING *print,
	ANYTHING *print_arg,
	ANYTHING *free,
	ANYTHING *free_arg,
	ANYTHING *parse,
	ANYTHING *parse_arg,
	ANYTHING *copy,
	ANYTHING *copy_arg
));

Bos_Slot_Subtype
Bos_GetCSlotType _ARGS_((
	char *name
));

/* Bos_GetCSlotType returns Bos_INVALID_SLOT_SUBTYPE if there is no
 * such named FOREIGN slot type.
 */

#define Bos_INVALID_SLOT_SUBTYPE ((Bos_Slot_Subtype)0)

char *
Bos_GetCSlotTypeName _ARGS_((
	Bos_Slot_Subtype subtype
));

void
Bos_UndefineCSlotType _ARGS_((
	Bos_Slot_Subtype subtype
));

char *
Bos_GetCSlotString _ARGS_((
	ANYTHING *value,
	Bos_Slot_Subtype subtype
));

ANYTHING *
Bos_ParseCSlotString _ARGS_((
	char *string,
	Bos_Slot_Subtype subtype
));

void
Bos_FreeCSlotValue _ARGS_((
	ANYTHING *value,
	Bos_Slot_Subtype subtype
));

ANYTHING *
Bos_CopyCSlotValue _ARGS_((
	ANYTHING *value,
	Bos_Slot_Subtype subtype
));

Bos_World *
Bos_InitInterp _ARGS_((
	Tcl_Interp *interp,
	int local_only_p
));

_Tcl_Proc_Decl(Bos_ObjectCmd);
_Tcl_Proc_Decl(Bos_CreateObjectCmd);
_Tcl_Proc_Decl(Bos_SendCmd);
_Tcl_Proc_Decl(Bos_CopyCmd);
_Tcl_Proc_Decl(Bos_DestroyCmd);
_Tcl_Proc_Decl(Bos_AddSlotCmd);
_Tcl_Proc_Decl(Bos_RemoveSlotCmd);
_Tcl_Proc_Decl(Bos_SetSlotCmd);
_Tcl_Proc_Decl(Bos_GetSlotsCmd);
_Tcl_Proc_Decl(Bos_SearchCmd);
_Tcl_Proc_Decl(Bos_SaveCmd);
_Tcl_Proc_Decl(Bos_LoadCmd);
_Tcl_Proc_Decl(Bos_RemoveCmd);
_Tcl_Proc_Decl(Bos_InStoreCmd);
_Tcl_Proc_Decl(Bos_ShutdownStoreCmd);
_Tcl_Proc_Decl(Bos_CompactCmd);
_Tcl_Proc_Decl(Bos_DumpStorageStatusCmd);
_Tcl_Proc_Decl(Bos_WhoCmd);
_Tcl_Proc_Decl(Bos_LocalObjectsCmd);
_Tcl_Proc_Decl(Bos_SearchCmd);
_Tcl_Proc_Decl(Bos_ListStoredObjectsCmd);

void
Bos_FatalError _ARGS_((
	char *str
));

Bos_Object *
Bos_Find _ARGS_((
	Bos_World *world,
	char *obj_name
));

Bos_Object *
Bos_CreateNewObject _ARGS_((
	Bos_World *world,
	char *obj_name
));

Bos_Object *
Bos_Copy _ARGS_((
	Bos_World *world,
	char *obj_name,
	char *new_name
));

Bos_Object *
Bos_Destroy _ARGS_((
	Bos_World *world,
	char *obj_name
));

int
Bos_AddSlot _ARGS_((
	Bos_Object *obj,
	char *slot_name,
	Bos_Slot_Type slot_type,
	Bos_Slot_Pri slot_pri,
	ANYTHING *slot_value
));

int
Bos_RemoveSlot _ARGS_((
	Bos_Object *obj,
	char *slot_name,
	Bos_Slot_Pri slot_pri
));

int
Bos_SetSlot _ARGS_((
	Bos_Object *obj,
	short op_mask,
	char *slot_name,
	Bos_Slot_Type slot_type,
	Bos_Slot_Pri slot_pri,
	ANYTHING *slot_value
));

/* Flag bits for the op_mask parameter to Bos_SetSlot() */

#define Bos_SET_NAME 0x0001	/* Set the name */
#define Bos_SET_TYPE 0x0002	/* Set the type */
#define Bos_SET_PRI 0x0004	/* Set the priority */
#define Bos_SET_VALUE 0x0008	/* Set the value */
#define Bos_SET_EVANESCENT_ON 0x0010 /* Turn on the E bit */
#define Bos_SET_EVANESCENT_OFF 0x0020 /* Turn off the E bit */

int
Bos_Send _ARGS_((
	Bos_World *world,
	Tcl_Interp *interp,
	char *obj_name,
	char *message
));

int
Bos_Sendv _ARGS_((
	Bos_World *world,
	Tcl_Interp *interp,
	char *obj_name,
	int argc,
	char **argv
));

char *
Bos_Object2ByteStream _ARGS_((
        Bos_Object *obj,
        char *buf,
        int *idx_p,
        int *size_p,
        char *errmsg
));

Bos_Object *
Bos_ByteStream2Object _ARGS_((
        Bos_World *world,
        char *buf,
        int *idx_p,
        char *errmsg
));

char *
Bos_ObjectSlots2ByteStream _ARGS_((
        Bos_Object *obj,
        char *buf,
        int *idx_p,
        int *size_p,
        char *errmsg
));

int
Bos_ByteStream2ObjectSlots _ARGS_((
        Bos_Object *obj,
        char *buf,
        int *idx_p,
        char *errmsg
));

int
Bos_ObjectSlotsSize _ARGS_((
        Bos_Object *obj,
        int *nslots_p
));

GDBM_FILE
Bos_InitializeStorage _ARGS_((
        char *dir,
        int lock,
        int mode
));

int
Bos_ShutdownStorage _ARGS_((
        char *dir
));

int
Bos_CompactStorage _ARGS_((
        char *dir
));

int
Bos_StoreObject _ARGS_((
        Bos_World *world,
        char *obj_name,
        char *dir,
        int lock
));

Bos_Object *
Bos_FetchObject _ARGS_((
        Bos_World *world,
        char *obj_name,
        char *dir,
        int lock
));

int
Bos_RemoveObject _ARGS_((
        char *obj_name,
        char *dir,
        int lock
));

int
Bos_IsObjectStored _ARGS_((
        char *obj_name,
        char *dir,
        int lock
));

void
Bos_DumpStorageStatus _ARGS_((
        FILE *f
));

#endif /* __BOS_h_ */
