/* 
 * tkCanv3DLine.c --
 *
 *	This file implements 3D line items (wireframes) for canvas widgets.
 *
 * Adapted from tkCanvLine.c by Paul Mackerras (paulus@cs.anu.edu.au).
 *
 * Copyright 1992 The Australian National University.
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, provided that the above copyright
 * notice appears in all copies.  This software is provided without any
 * warranty, express or implied. The Australian National University
 * makes no representations about the suitability of this software for
 * any purpose.
 *
 * Copyright 1991-1992 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef lint
static char rcsid[] = "$Header: /localdisks/kaffa/cap/CaVis/src/tk2.3/RCS/tkCanv3DLine.c,v 1.2 1992/09/17 01:34:58 paulus Exp paulus $";
#endif

#include <stdio.h>
#include <math.h>
#include "tkInt.h"
#include "tkCanvas.h"
#include "tkConfig.h"
#include "tkCamera.h"

/*
 * The structure below defines the record for each line item.
 */

typedef struct LineItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_3DItem hdr3d;		/* Generic stuff for 3D items.
				 * Must follow `header'. */
    int viewportId;		/* ID of viewport item */
    int numPoints;		/* Number of points in line (always >= 2). */
    double *worldCoords;	/* Pointer to malloc-ed array containing
				 * x-, y- and z-coords of points in line. */
    int numClippedPoints;	/* total # endpoints after clipping */
    int numSegs;		/* # segments the clipped points form */
    int *segLengthPtr;		/* # points in each segment (malloced) */
    XPoint *coordPtr;		/* Pointer to malloc-ed array containing
				 * transformed & clipped coords. */
    int	camera_uid;		/* Camera generation used to transform. */
    int width;			/* Width of line. */
    XColor *fg;			/* Foreground color for line. */
    Pixmap fillStipple;		/* Stipple bitmap for filling line. */
    int capStyle;		/* Cap style for line. */
    int joinStyle;		/* Join style for line. */
    GC gc;			/* Graphics context for filling line. */
} LineItem;

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeLineBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    LineItem *linePtr));
static int		ConfigureLine _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateLine _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteLine _ANSI_ARGS_((Tk_Item *itemPtr));
static void		DisplayLine _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static int		LineCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		LineToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double x1, double y1,
			    double x2, double y2));
static double		LineToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double x, double y));
static void		ScaleLine _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static int		TransformCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    LineItem *linePtr));
static void		TranslateLine _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateLine.
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CAP_STYLE, "-capstyle", (char *) NULL, (char *) NULL,
	"butt", Tk_Offset(LineItem, capStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(LineItem, fg), 0},
    {TK_CONFIG_JOIN_STYLE, "-joinstyle", (char *) NULL, (char *) NULL,
	"round", Tk_Offset(LineItem, joinStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(LineItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_INT, "-viewport", (char *) NULL, (char *) NULL,
	 "-1", Tk_Offset(LineItem, viewportId), 0},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(LineItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the line item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType Tk3DLineType = {
    "3Dline",				/* name */
    sizeof(LineItem),			/* itemSize */
    CreateLine,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureLine,			/* configureProc */
    LineCoords,				/* coordProc */
    DeleteLine,				/* deleteProc */
    DisplayLine,			/* displayProc */
    0,					/* alwaysRedraw */
    LineToPoint,			/* pointProc */
    LineToArea,				/* areaProc */
    (Tk_ItemPostscriptProc *) NULL,	/* postscriptProc */
    ScaleLine,				/* scaleProc */
    TranslateLine,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* cursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};


/*
 *--------------------------------------------------------------
 *
 * CreateLine --
 *
 *	This procedure is invoked to create a new 3-D line item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new 3-D line item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateLine(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing line. */
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    int i;

    if (argc < 2) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin),
		"\" create {x1 y1 z1} {x2 y2 z2} ?{x3 y3 z3}...? ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    linePtr->hdr3d.next = NULL;
    linePtr->hdr3d.camera = NULL;
    linePtr->numPoints = 0;
    linePtr->worldCoords = NULL;
    linePtr->segLengthPtr = NULL;
    linePtr->coordPtr = NULL;
    linePtr->camera_uid = 0;
    linePtr->width = 1;
    linePtr->fg = None;
    linePtr->fillStipple = None;
    linePtr->capStyle = CapButt;
    linePtr->joinStyle = JoinRound;
    linePtr->gc = None;

    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 2; i < argc; ++i) {
	if ((!isdigit(argv[i][0])) &&
		((argv[i][0] != '-') || (!isdigit(argv[i][1])))) {
	    break;
	}
    }

    /*
     * Process the coordinates and the initial configuration options given.
     */
    if( LineCoords(canvasPtr, itemPtr, i, argv) != TCL_OK
       || ConfigureLine(canvasPtr, itemPtr, argc-i, argv+i, 0) != TCL_OK ){
	DeleteLine(itemPtr);
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * LineCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on lines.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
LineCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    int i, numPoints;
    double *coordPtr;
    char buffer[128];

    if( argc == 0 ){
	coordPtr = linePtr->worldCoords;
	for( i = linePtr->numPoints; i > 0; --i ){
	    sprintf(buffer, "%g %g %g", coordPtr[0], coordPtr[1], coordPtr[2]);
	    Tcl_AppendElement(canvasPtr->interp, buffer, 0);
	    coordPtr += 3;
	}

    } else if( argc < 2 ){
	Tcl_AppendResult(canvasPtr->interp, "too few world coords for",
			 " 3D line: must have at least 2", (char *) NULL);
	return TCL_ERROR;

    } else {
	numPoints = argc;
	if( linePtr->numPoints != numPoints ){
	    if( linePtr->worldCoords != NULL ){
		ckfree((char *) linePtr->worldCoords);
		ckfree((char *) linePtr->segLengthPtr);
		ckfree((char *) linePtr->coordPtr);
	    }
	    linePtr->numPoints = numPoints;
	    linePtr->worldCoords = (double *) ckalloc((unsigned)
					(sizeof(vector) * numPoints));
	    /*
	     * Allocate space for the transformed points.  If each line
	     * segment gets clipped on both ends, we will end up with
	     * n-1 separate segments and 2(n-1) points, for n points
	     * in world space.
	     */
	    i = numPoints - 1;
	    linePtr->segLengthPtr = (int *) ckalloc(i * sizeof(int));
	    linePtr->coordPtr = (XPoint *) ckalloc(2 * i * sizeof(XPoint));
	}
	for (i = argc-1; i >= 0; i--) {
	    if (ParseVector(NULL, canvasPtr->interp, canvasPtr->tkwin,
			    argv[i], (char *) &linePtr->worldCoords[3*i], 0)
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}

	ComputeLineBbox(canvasPtr, itemPtr);

    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureLine --
 *
 *	This procedure is invoked to configure various aspects
 *	of a line item such as its background color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureLine(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Line item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) linePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if( Tk_AttachToViewport(canvasPtr->interp, canvasPtr, linePtr->viewportId,
			    itemPtr) != TCL_OK )
	return TCL_ERROR;

    if (linePtr->fg == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = linePtr->fg->pixel;
	gcValues.join_style = linePtr->joinStyle;
	if (linePtr->width < 0) {
	    linePtr->width = 1;
	}
	gcValues.line_width = linePtr->width;
	mask = GCForeground|GCJoinStyle|GCLineWidth;
	if (linePtr->fillStipple != None) {
	    gcValues.stipple = linePtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask |= GCStipple|GCFillStyle;
	}
	gcValues.cap_style = linePtr->capStyle;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (linePtr->gc != None) {
	Tk_FreeGC(linePtr->gc);
    }
    linePtr->gc = newGC;

    /*
     * Recompute bounding box for line.
     */

    ComputeLineBbox(canvasPtr, itemPtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteLine --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteLine(itemPtr)
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register LineItem *linePtr = (LineItem *) itemPtr;

    if (linePtr->worldCoords != NULL) {
	ckfree((char *) linePtr->worldCoords);
    }
    if (linePtr->segLengthPtr != NULL) {
	ckfree((char *) linePtr->segLengthPtr);
    }
    if (linePtr->coordPtr != NULL) {
	ckfree((char *) linePtr->coordPtr);
    }
    if (linePtr->fg != NULL) {
	Tk_FreeColor(linePtr->fg);
    }
    if (linePtr->fillStipple != None) {
	Tk_FreeBitmap(linePtr->fillStipple);
    }
    if (linePtr->gc != None) {
	Tk_FreeGC(linePtr->gc);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeLineBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a line.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeLineBbox(canvasPtr, linePtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    LineItem *linePtr;			/* Item to be displayed. */
{
    register XPoint *coordPtr;
    int i;
    double coords[2];

    if( TransformCoords(canvasPtr, linePtr) != 1 ){
	/* no viewport, so not visible; or the points were
	   recalculated and we came here recursively. */
	linePtr->header.x1 = linePtr->header.x2 = 0;
	linePtr->header.y1 = linePtr->header.y2 = 0;
	return;
    }

    coordPtr = linePtr->coordPtr;
    linePtr->header.x1 = linePtr->header.x2 = coordPtr->x;
    linePtr->header.y1 = linePtr->header.y2 = coordPtr->y;

    /*
     * Compute the bounding box of all the points in the line,
     * then expand in all directions by the line's width to take
     * care of butting or rounded corners and projecting or
     * rounded caps.  This expansion is an overestimate (worst-case
     * is square root of two over two) but it's simple.
     */

    for (i = 1, coordPtr = linePtr->coordPtr+1; i < linePtr->numClippedPoints;
	    i++, coordPtr++) {
	coords[0] = coordPtr->x;
	coords[1] = coordPtr->y;
	TkIncludePoint(canvasPtr, (Tk_Item *) linePtr, coords);
    }
    linePtr->header.x1 -= linePtr->width;
    linePtr->header.x2 += linePtr->width;
    linePtr->header.y1 -= linePtr->width;
    linePtr->header.y2 += linePtr->width;

    /*
     * For mitered lines, make a second pass through all the points.
     * Compute the locations of the two miter vertex points and add
     * those into the bounding box.
     */

    if (linePtr->joinStyle == JoinMiter) {
	coordPtr = linePtr->coordPtr;
	for (i = linePtr->numClippedPoints; i >= 3; i--, coordPtr++) {
	    double miter[4];
	    int j;
    
	    if (TkGetMiterPoints(coordPtr, coordPtr+1, coordPtr+2,
		    (double) linePtr->width, miter, miter+2)) {
		for (j = 0; j < 4; j += 2) {
		    TkIncludePoint(canvasPtr, (Tk_Item *) linePtr, miter+j);
		}
	    }
	}
    }

    /*
     * Add one more pixel of fudge factor just to be safe (e.g.
     * X may round differently than we do).
     */

    linePtr->header.x1 -= 1;
    linePtr->header.x2 += 1;
    linePtr->header.y1 -= 1;
    linePtr->header.y2 += 1;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayLine --
 *
 *	This procedure is invoked to draw a line item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayLine(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    XPoint *pointPtr;
    int i;

    if (linePtr->gc == None || !TransformCoords(canvasPtr, linePtr)) {
	return;
    }

    /*
     * Subtract (drawableXOrigin, drawableYOrigin) from the coordinates
     * to allow for the window being scrolled or for only a part of the
     * window being redrawn.
     */
    if( canvasPtr->drawableXOrigin != 0 || canvasPtr->drawableYOrigin != 0 ){
	pointPtr = linePtr->coordPtr;
	for( i = 0; i < linePtr->numClippedPoints; ++i, ++pointPtr ){
	    pointPtr->x -= canvasPtr->drawableXOrigin;
	    pointPtr->y -= canvasPtr->drawableYOrigin;
	}
    }

    /*
     * Display the line segments.
     */
    pointPtr = linePtr->coordPtr;
    for( i = 0; i < linePtr->numSegs; ++i ){
	XDrawLines(Tk_Display(canvasPtr->tkwin), drawable, linePtr->gc,
		   pointPtr, linePtr->segLengthPtr[i], CoordModeOrigin);
	pointPtr += linePtr->segLengthPtr[i];
    }

    /*
     * Restore the coordinate values.
     */
    if( canvasPtr->drawableXOrigin != 0 || canvasPtr->drawableYOrigin != 0 ){
	pointPtr = linePtr->coordPtr;
	for( i = 0; i < linePtr->numClippedPoints; ++i, ++pointPtr ){
	    pointPtr->x += canvasPtr->drawableXOrigin;
	    pointPtr->y += canvasPtr->drawableYOrigin;
	}
    }

}

/*
 *--------------------------------------------------------------
 *
 * LineToPoint --
 *
 *	Computes the distance from a given point to a given
 *	line, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the line.  If the
 *	point isn't inside the line then the return value is the
 *	distance from the point to the line.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
LineToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    register XPoint *coordPtr;
    double poly[10];
    double bestDist, dist;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */
    double pt0[2], pt1[2], pt2[2];

    bestDist = 1.0e40;
    numPoints = linePtr->numClippedPoints;

    if( !TransformCoords(canvasPtr, linePtr) ){
	/* no viewport, so not visible */
	return bestDist;
    }

    /*
     * The overall idea is to iterate through all of the edges of
     * the line, computing a polygon for each edge and testing the
     * point against that polygon.  In addition, there are additional
     * tests to deal with rounded joints and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints, coordPtr = linePtr->coordPtr;
	    count >= 2; count--, coordPtr++) {

	/*
	 * If rounding is done around the first point then compute
	 * the distance between the point and the point.
	 */

	if (((linePtr->capStyle == CapRound) && (count == numPoints))
		|| ((linePtr->joinStyle == JoinRound)
			&& (count != numPoints))) {
	    dist = hypot(coordPtr->x - pointPtr[0], coordPtr->y - pointPtr[1])
		    - linePtr->width/2.0;
	    if (dist <= 0.0) {
		return 0.0;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}

	/*
	 * Compute the polygonal shape corresponding to this edge,
	 * consisting of two points for the first point of the edge
	 * and two points for the last point of the edge.
	 */

	pt0[0] = coordPtr[0].x;
	pt0[1] = coordPtr[0].y;
	pt1[0] = coordPtr[1].x;
	pt1[1] = coordPtr[1].y;

	if (count == numPoints) {
	    TkGetButtPoints(pt1, pt0, (double) linePtr->width,
		    linePtr->capStyle == CapProjecting, poly, poly+2);
	} else if ((linePtr->joinStyle == JoinMiter) && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    TkGetButtPoints(pt1, pt0, (double) linePtr->width, 0,
		    poly, poly+2);

	    /*
	     * If this line uses beveled joints, then check the distance
	     * to a polygon comprising the last two points of the previous
	     * polygon and the first two from this polygon;  this checks
	     * the wedges that fill the mitered joint.
	     */

	    if ((linePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		dist = TkPolygonToPoint(poly, 5, pointPtr);
		if (dist <= 0.0) {
		    return 0.0;
		} else if (dist < bestDist) {
		    bestDist = dist;
		}
	    }
	}
	if (count == 2) {
	    TkGetButtPoints(pt0, pt1, (double) linePtr->width,
		    linePtr->capStyle == CapProjecting, poly+4, poly+6);
	} else if (linePtr->joinStyle == JoinMiter) {
	    pt2[0] = coordPtr[2].x;
	    pt2[1] = coordPtr[2].y;
	    if (TkGetMiterPoints(pt0, pt1, pt2,
		    (double) linePtr->width, poly+4, poly+6) == 0) {
		changedMiterToBevel = 1;
		TkGetButtPoints(pt0, pt1, (double) linePtr->width,
			0, poly+4, poly+6);
	    }
	} else {
	    TkGetButtPoints(pt0, pt1, (double) linePtr->width, 0,
		    poly+4, poly+6);
	}
	poly[8] = poly[0];
	poly[9] = poly[1];
	dist = TkPolygonToPoint(poly, 5, pointPtr);
	if (dist <= 0.0) {
	    return 0.0;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    /*
     * If caps are rounded, check the distance to the cap around the
     * final end point of the line.
     */

    if (linePtr->capStyle == CapRound) {
	dist = hypot(coordPtr->x - pointPtr[0], coordPtr->y - pointPtr[1])
		- linePtr->width/2.0;
	if (dist <= 0.0) {
	    return 0.0;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    return bestDist;
}

/*
 *--------------------------------------------------------------
 *
 * LineToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	insde the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
LineToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against line. */
    double *rectPtr;
{
    register LineItem *linePtr = (LineItem *) itemPtr;
    register XPoint *coordPtr;
    double poly[10];
    double radius;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */
    int inside;			/* Tentative guess about what to return,
				 * based on all points seen so far:  one
				 * means everything seen so far was
				 * inside the area;  -1 means everything
				 * was outside the area.  0 means overlap
				 * has been found. */ 
    double pt0[2], pt1[2], pt2[2];

    if( !TransformCoords(canvasPtr, linePtr) ){
	/* no viewport, so not visible */
	return -1;
    }

    radius = linePtr->width/2.0;
    inside = -1;
    numPoints = linePtr->numClippedPoints;

    coordPtr = linePtr->coordPtr;
    if ((coordPtr->x >= rectPtr[0]) && (coordPtr->x <= rectPtr[2])
	    && (coordPtr->y >= rectPtr[1]) && (coordPtr->y <= rectPtr[3])) {
	inside = 1;
    }

    /*
     * Iterate through all of the edges of the line, computing a polygon
     * for each edge and testing the area against that polygon.  In
     * addition, there are additional tests to deal with rounded joints
     * and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints; count >= 2; count--, coordPtr++) {

	/*
	 * If rounding is done around the first point of the edge
	 * then test a circular region around the point with the
	 * area.
	 */

	if (((linePtr->capStyle == CapRound) && (count == numPoints))
		|| ((linePtr->joinStyle == JoinRound)
		&& (count != numPoints))) {
	    poly[0] = coordPtr->x - radius;
	    poly[1] = coordPtr->y - radius;
	    poly[2] = coordPtr->x + radius;
	    poly[3] = coordPtr->y + radius;
	    if (TkOvalToArea(poly, rectPtr) != inside) {
		return 0;
	    }
	}

	/*
	 * Compute the polygonal shape corresponding to this edge,
	 * consisting of two points for the first point of the edge
	 * and two points for the last point of the edge.
	 */

	pt0[0] = coordPtr[0].x;
	pt0[1] = coordPtr[0].y;
	pt1[0] = coordPtr[1].x;
	pt1[1] = coordPtr[1].y;

	if (count == numPoints) {
	    TkGetButtPoints(pt1, pt0, (double) linePtr->width,
		    linePtr->capStyle == CapProjecting, poly, poly+2);
	} else if ((linePtr->joinStyle) == JoinMiter && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    TkGetButtPoints(pt1, pt0, (double) linePtr->width, 0,
		    poly, poly+2);

	    /*
	     * If the last joint was beveled, then also check a
	     * polygon comprising the last two points of the previous
	     * polygon and the first two from this polygon;  this checks
	     * the wedges that fill the mitered joint.
	     */

	    if ((linePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		if (TkPolygonToArea(poly, 5, rectPtr) != inside) {
		    return 0;
		}
		changedMiterToBevel = 0;
	    }
	}
	if (count == 2) {
	    TkGetButtPoints(pt0, pt1, (double) linePtr->width,
		    linePtr->capStyle == CapProjecting, poly+4, poly+6);
	} else if (linePtr->joinStyle == JoinMiter) {
	    pt2[0] = coordPtr[2].x;
	    pt2[1] = coordPtr[2].y;
	    if (TkGetMiterPoints(pt0, pt1, pt2,
		    (double) linePtr->width, poly+4, poly+6) == 0) {
		changedMiterToBevel = 1;
		TkGetButtPoints(pt0, pt1, (double) linePtr->width,
			0, poly+4, poly+6);
	    }
	} else {
	    TkGetButtPoints(pt0, pt1, (double) linePtr->width, 0,
		    poly+4, poly+6);
	}
	poly[8] = poly[0];
	poly[9] = poly[1];
	if (TkPolygonToArea(poly, 5, rectPtr) != inside) {
	    return 0;
	}
    }

    /*
     * If caps are rounded, check the cap around the final point
     * of the line.
     */

    if (linePtr->capStyle == CapRound) {
	poly[0] = coordPtr->x - radius;
	poly[1] = coordPtr->y - radius;
	poly[2] = coordPtr->x + radius;
	poly[3] = coordPtr->y + radius;
	if (TkOvalToArea(poly, rectPtr) != inside) {
	    return 0;
	}
    }

    return inside;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleLine --
 *
 *	This procedure is invoked to rescale a 3-D line item.
 *	In fact it does nothing: if you want to rescale a
 *	3-D line, rescale its viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ScaleLine(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing line. */
    Tk_Item *itemPtr;			/* Line to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
}

/*
 *--------------------------------------------------------------
 *
 * TranslateLine --
 *
 *	This procedure is called to move a 3-D line by a given amount.
 *	In fact it does nothing - if you want to move the line, move
 *	the viewport it's displayed in.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
TranslateLine(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
}

/*
 *--------------------------------------------------------------
 *
 * TransformCoords --
 *
 *	This procedure transforms the world coordinates in
 *	linePtr->worldCoords into canvas coordinates and clips
 *	each line segment to the edges of the line's viewport.
 *	This may result in several disconnected line segments.
 *
 * Results:
 *	Zero is returned if the line is not visible because
 *	it is not associated with a viewport, 1 if the canvas
 *	coordinates were correct, or 2 if they (and the bounding
 *	box) were recalculated.
 *
 * Side effects:
 *	The numClippedPoints and numSegs fields of *linePtr are
 *	updated, as are the values in *linePtr->segLengthPtr
 *	and *linePtr->coordPtr.
 *
 *--------------------------------------------------------------
 */

#define VecCopy(d, s)	((d)[0] = (s)[0], (d)[1] = (s)[1], (d)[2] = (s)[2])
#define Dot(a, b)	((a)[0] * (b)[0] + (a)[1] * (b)[1] + (a)[2] * (b)[2])

static int
TransformCoords(canvasPtr, linePtr)
    Tk_Canvas *canvasPtr;
    LineItem *linePtr;
{
    Camera *cam;
    int i, j, end_segment, mod;
    XPoint *coordPtr;
    double *worldPtr, z0, z1, x, y;
    vector p0, p1, p1_orig;

    if( (cam = linePtr->hdr3d.camera) == NULL )
	return 0;		/* we have no viewport */

    if( cam->uid == linePtr->camera_uid )
	return 1;		/* values are up-to-date */

    linePtr->numClippedPoints = 0;	/* # visible points so far */
    linePtr->numSegs = 0;		/* # visible polylines so far */
    coordPtr = linePtr->coordPtr;
    worldPtr = linePtr->worldCoords;
    end_segment = -1;

    if( !cam->perspective )
	z0 = z1 = cam->view_distance;	/* orthogonal projection */

    for( i = 0; i < linePtr->numPoints; ++i, worldPtr += 3 ){
	VecCopy(p0, p1_orig);

	/*
	 * Transform this point to screen coordinates
	 */
	for( j = 0; j < 3; ++j )
	    p1[j] = cam->xmat[j][3] + Dot(cam->xmat[j], worldPtr);
	VecCopy(p1_orig, p1);	/* need unclipped value for next time */

	/*
	 * Need 2 points to make a line.
	 */
	if( i == 0 )
	    continue;

	/*
	 * Clip to the planes z = zmin, z = zmax, x/z = xmax,
	 * x/z = -xmax, y/z = ymax, y/z = -ymax.
	 */
	if( cam->zmin < 0 )
	    cam->zmin = 0;
	if( (mod = ClipLineToPlane(p0, p1, 0.0, 0.0, cam->zmin,
				   0.0, 0.0, 1.0)) < 0 )
	    continue;		/* neither endpoint visible */
	if( cam->zmax > cam->zmin
	   && (mod |= ClipLineToPlane(p0, p1, 0.0, 0.0, cam->zmax,
				      0.0, 0.0, -1.0)) < 0 )
	    continue;
	if( cam->perspective ){
	    /* perspective projection */
	    if( (mod |= ClipLineToPlane(p0, p1, 0.0, 0.0, 0.0,
					-1.0, 0.0, cam->xmax)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, 0.0, 0.0, 0.0,
					  1.0, 0.0, cam->xmax)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, 0.0, 0.0, 0.0,
					  0.0, -1.0, cam->ymax)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, 0.0, 0.0, 0.0,
					  0.0, 1.0, cam->ymax)) < 0 )
		continue;
	    z0 = p0[2];
	    z1 = p1[2];
	    /*
	     * We could conceivably have either or both z values = 0.
	     * If so, we don't draw this segment.
	     */
	    if( z0 <= 0.0 || z1 <= 0.0 )
		continue;

	} else {
	    /* orthogonal projection */
	    x = cam->xmax * cam->view_distance;
	    y = cam->ymax * cam->view_distance;
	    if( (mod |= ClipLineToPlane(p0, p1, x, 0.0, 0.0,
					-1.0, 0.0, 0.0)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, -x, 0.0, 0.0,
					  1.0, 0.0, 0)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, 0.0, y, 0.0,
					  0.0, -1.0, 0.0)) < 0
	       || (mod |= ClipLineToPlane(p0, p1, 0.0, -y, 0.0,
					  0.0, 1.0, 0.0)) < 0 )
		continue;
	}

	/*
	 * Calculate screen coordinates, and start a new segment
	 * if necessary.
	 */
	if( (mod & 1) != 0 || i - 1 != end_segment ){
	    /*
	     * Point p0 is different from the last point
	     * of the previous segment: start a new segment.
	     */
	    ++linePtr->numSegs;
	    linePtr->segLengthPtr[linePtr->numSegs - 1] = 1;
	    ++linePtr->numClippedPoints;
	    coordPtr->x = (int)(p0[0] / z0 + cam->xcenter + 0.5);
	    coordPtr->y = (int)(p0[1] / z0 + cam->ycenter + 0.5);
	    ++coordPtr;
	}
	++linePtr->segLengthPtr[linePtr->numSegs - 1];
	++linePtr->numClippedPoints;
	coordPtr->x = (int)(p1[0] / z1 + cam->xcenter + 0.5);
	coordPtr->y = (int)(p1[1] / z1 + cam->ycenter + 0.5);
	++coordPtr;

	/*
	 * Terminate the current segment if p1 got clipped.
	 */
	if( (mod & 2) != 0 )
	    end_segment = -1;
	else
	    end_segment = i;
    }

    linePtr->camera_uid = cam->uid;

    ComputeLineBbox(canvasPtr, linePtr);

    return 2;			/* points were recalculated */
}

/*
 *--------------------------------------------------------------
 *
 * ClipLineToPlane --
 *
 *	This procedure clips a 3-D line segment to a 3-D plane,
 *	specified by a point on the plane (px, py, pz) and
 *	a normal vector (nx, ny, nz).  The vectors `a' and `b'
 *	are modified as necessary to ensure that are both on
 *	the positive side of the plane (dot products (a - p).n
 *	and (b - p).n are >= 0).
 *
 * Results:
 *	This procedure returns -1 if both `a' and `b' are on the
 *	negative side of the plane.  Otherwise, if neither `a'
 *	nor `b' were modified, it returns 0; if `a' was modified,
 *	it returns 1; if `b' was modified, it returns 2.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ClipLineToPlane(a, b, px, py, pz, nx, ny, nz)
    vector a, b;
    double px, py, pz;
    double nx, ny, nz;
{
    register double t, t0, t1;

    t0 = (a[0] - px) * nx + (a[1] - py) * ny + (a[2] - pz) * nz;
    t1 = (b[0] - px) * nx + (b[1] - py) * ny + (b[2] - pz) * nz;
    if( t0 < 0 && t1 < 0 )
	return -1;
    if( t0 < 0 ){
	t = t0 / (t0 - t1);
	a[0] += t * (b[0] - a[0]);
	a[1] += t * (b[1] - a[1]);
	a[2] += t * (b[2] - a[2]);
	return 1;
    }
    if( t1 < 0 ){
	t = t1 / (t1 - t0);
	b[0] += t * (a[0] - b[0]);
	b[1] += t * (a[1] - b[1]);
	b[2] += t * (a[2] - b[2]);
	return 2;
    }
    return 0;
}
