/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef MIDI_H
#define MIDI_H
/*
 * MIDI MPU401 ioctls
 */
#define DSPIOCTL ('m' << 8)
#define MRESET	(DSPIOCTL | 0x01)	/* no read, no write */
#define MDRAIN	(DSPIOCTL | 0x02)	/* no read, no write */
#define MFLUSH	(DSPIOCTL | 0x03)	/* no read, no write */
#define MGPLAYQ	(DSPIOCTL | 0x04)	/* read int */
#define MGRECQ	(DSPIOCTL | 0x05)	/* read int */
#define MSDIVISION	(DSPIOCTL | 0x06)	/* write int */
#define MGDIVISION	(DSPIOCTL | 0x07)	/* read int */
#define MGQAVAIL	(DSPIOCTL | 0x08)	/* read int */
/*
 * I'd argue that this ioctl is only necessary because of a
 * bug in the Linux kernel.  They ioctl handler processes the
 * FIOASYNC ioctl and does not check the underlying driver to
 * see if other actions should be performed.  The MASYNC ioctl
 * command is really just FIOASYNC.
 */
#define MASYNC	(DSPIOCTL | 0x09)	/* write int */
#define MTHRU		(DSPIOCTL | 0x0a)	/* write int */
#define MRECONPLAY	(DSPIOCTL | 0x0b)	/* write int */



#ifdef __KERNEL__
/*
 * midi port offsets
 */
#define	MIDI_DATA			0
#define	MIDI_STATUS			1
#define	MIDI_COMMAND			1


/*
 * midi data transfer status bits
 */
#define	MIDI_RDY_RCV			(1 << 6)
#define	MIDI_DATA_AVL			(1 << 7)

/*
 * midi status flags
 */
#define MIDI_UNAVAILABLE	(1 << 0) /* not in uart mode */
#define MIDI_READING		(1 << 1) /* open for reading */
#define MIDI_WRITING		(1 << 2) /* open for writing */
#define MIDI_CALLBACK_ISSUED	(1 << 3) /* waiting for a timer to expire */
#define MIDI_RD_BLOCK		(1 << 4) /* read will block */
#define MIDI_WR_BLOCK		(1 << 5) /* write will block */
#define MIDI_WR_ABORT		(1 << 6) /* write should abort */
#define MIDI_OPEN		(1 << 7) /* device is open */
#define MIDI_FLUSH_SLEEP	(1 << 8) /* blocking on flush */
#define MIDI_RD_SLEEP		(1 << 9) /* blocking on read */
#define MIDI_WR_SLEEP		(1 << 10) /* blocking on write */
#define MIDI_BUFFER_SLEEP	(1 << 11) /* blocking on buffer */
#define MIDI_NEED_ACK		(1 << 12) /* command needs and ack */
#define MIDI_ASYNC		(1 << 13) /* send sigios */
#define MIDI_SENDIO		(1 << 14) /* a sigio should be send at low h2o */
#define MIDI_NONBLOCK		(1 << 15) /* non blocking mode */
#define MIDI_SELOUT		(1 << 16) /* wait for a select output */
#define MIDI_SELIN		(1 << 17) /* wait for a select input */
#define MIDI_THRU		(1 << 18) /* copy in port to out port? */
#define MIDI_RECONPLAY		(1 << 19) /* start record timer on play */

/*
 * These are the various input data states
 */
typedef enum {START, NEEDDATA1, NEEDDATA2, SYSEX, SYSTEM1, SYSTEM2} InputState;

/*
 * midi command values
 */
#define MIDI_RESET			0xff
#define MIDI_UART			0x3f
#define MIDI_ACK			0xfe
#define MIDI_TRIES			20000

/*
 * most events are short, so we use the static array,
 * but occationally we get a long event (sysex) and
 * we dynamically allocate that
 */
#define STYNAMIC_SIZE 4
#define STYNAMIC_ALLOC 256

struct stynamic {
	short	allocated;
	short	len;
	u_char	datas[4];
	u_char	*datad;
};

/*
 * data from the board that hasn't formed a complete event yet
 */
struct partial_event {
	struct	stynamic event;		/* the data */
	u_long	jiffs;			/* try out using jiffies */
	long	tempo;			/* tempo setting when event arrived */
	InputState	state;		/* what we are expecting next */
	u_char	rs;			/* the midi running state */
};

/*
 * the internal representation of an event
 */
typedef enum {NORMAL, TEMPO, TIMESIG, SYSX} EventType;

struct event {
	u_long	time;		/* time until event in kernel clicks */
        long    tempo;		/*
				 * not used in play events, but contains
				 * the tempo setting current when the
				 * incoming event arrived.  Used for
				 * convert kernel ticks to smf ticks
				 */
	EventType	type;
	struct	stynamic data;
};

/*
 * A event queue, used for both incoming and outgoing
 */
#define MIDI_Q_SIZE 150
#define MIDI_LOW_WATER 40

struct event_queue {
	int	count;
	struct	event events[MIDI_Q_SIZE];
	struct	event *end;
	struct	event *head;
	struct	event *tail;
};

struct midi_softc {      /* Driver status information */
	/*
	 * various wait queues - why can't linux use addresses
	 * like everybody else
	 */
	struct	wait_queue *flush_waitq;
	struct	wait_queue *read_waitq;
	struct	wait_queue *write_waitq;
	struct	wait_queue *selin_waitq;
	struct	wait_queue *selout_waitq;
	struct	wait_queue *forever_waitq;
	/* who's running us */
	struct	task_struct *owner;
	/* midi specific stuff */
	struct	event_queue *rqueue;
	struct	event_queue *wqueue;
	struct	stynamic rpartial;
	struct	stynamic wpartial;
	struct	partial_event partial_event;
	u_long	prev_incoming;	/* try using jiffies */
	struct	timer_list timer;
	volatile	long status;
 	long long	premainder;
 	long long	rremainder;
 	long	ptempo;			/* usec / beat */
 	long	rtempo;			/* usec / beat */
	long	prev_rtempo;
	int	pgid;
	int	addr;
	int	intr;
	int	division;
	int	wpartialpos;
	long	write_noop_time;
	short	noteon[0x80];		/*
					 * each element is a different pitch
					 * each bit is a different channel
					 */
	u_char	readrs;
	u_char	writers;
	u_char	noteonrs;
};

struct midi_config {
	int	addr;
	int	intr;
};

/* for mem.c */
extern long midiinit(long);

#endif
#endif
