## edit.tcl main file for tkWWW  user interface
## ==============
## Copyright (C) 1992-1993
## Global Network Academy
## Macvicar Institute for Educational Software Development
##
## See the file COPYRIGHT for conditions

## The procedures in this file are responsible for editing inside of tkWWW.

## Conventions:
##   All procedures in this file begin with tkWWWEdit

## Bugs and Todo:
##   Some abstraction questions

# This is the style that the editor is current writing in

set tkW3EditParaStyle ""
set tkW3EditCharStyle ""

# This is the cut buffer
set tkW3EditSelection {}

set tkW3EditModified 0

## ********************
## Initialization procedures
## ********************

proc tkW3EditInitialize {} {
    global tkW3EditConvert

    set convert_list {
	{"" "default"}
	{H1 heading1}
	{H2 heading2}
	{H3 heading3}
	{H4 heading4}
	{H5 heading5}
	{H6 heading6}
	{ADDRESS address}
	{PRE preformatted}
	{TT typewriter}
	{B bold}
	{I italics}
	{U underline}
	{EM emphasis}
	{STRONG "strong emphasis"}
	{SAMP sample}
	{KBD keyboard}
	{VAR variable}
	{DFN definition}
	{CITE citation}
    }

    foreach item $convert_list {
	set tkW3EditConvert([lindex $item 0]) [lindex $item 1]
    }
    tkW3EditBindText .f.msg
    tkW3EditDisplayStatus
}

set tkW3ConfigEditTk {
    .edit.menu.file.m delete "*Quit*"
    .edit.menu.file.m entryconfigure "Save" -command "tkW3EditSave"
    .edit.menu.edit.m add command -label "Spell Check" -command \
	{ispell_text .edit.main.t}
    set button_list {
	{generate "Generate Source" "tkW3EditGenerate"}
	{save "Save" "tkW3EditSave"}
	{close "Close Window" "tkW3EditClose"}
    }
    pack append .edit \
	[ tkW3OutputMakeButtons .edit.button_box $button_list] {bottom fillx}
}

proc tkW3EditBindText {w} {
    bind $w <Any-KeyPress> {tkW3EditInsert %W %A}
    bind $w <Control-y> {tkW3EditInsert [selection get]}

    bind $w <1> {
	set tk_priv(selectMode) char
	%W mark set insert @%x,%y
	%W mark set anchor insert
	if {[lindex [%W config -state] 4] == "normal"} {focus %W}
	tkW3EditSetStyle %W
    }

    bind $w <Return> {
	case $tkW3EditParaStyle {
	    {H1 H2 H3 H4 H5 H6} {%W insert insert \n
set tkW3EditParaStyle ""
	%W yview -pickplace insert
	tkW3EditDisplayStatus
	return
}
	    {PRE XMP} {%W insert insert \n}
	    {default} {tkW3HtAddBul "P"}
	}
	%W yview -pickplace insert
	tkW3EditSetStyle %W
    }

    bind $w <BackSpace> {tkW3EditBackspace %W}
    bind $w <Delete> {tkW3EditBackspace %W}
    bind $w <Control-h> {tkW3EditBackspace %W}

#   need to check for case when a newline is deleted in case a tag is removed
    bind $w <Control-d> {
	%W delete insert
	%W yview -pickplace insert
	tkW3EditSetStyle %W
    }  
    
    bind $w <Control-f> {tkW3EditMoveInsertTo %W "insert + 1 c"}
    bind $w <Right>     {tkW3EditMoveInsertTo %W "insert + 1 c"}

    bind $w <Control-b> {tkW3EditMoveInsertTo %W "insert - 1 c"}
    bind $w <Left>      {tkW3EditMoveInsertTo %W "insert - 1 c"}

    bind $w <Control-a> {tkW3EditMoveInsertTo %W "linestart"}
    
    bind $w <Control-e> {tkW3EditMoveInsertTo %W "lineend"}

    bind $w <Control-n> {tkW3EditMoveInsertTo %W "insert + 1 lines"}
    bind $w <Down>      {tkW3EditMoveInsertTo %W "insert + 1 lines"}

    bind $w <Control-p> {tkW3EditMoveInsertTo %W "insert - 1 lines"}
    bind $w <Up>        {tkW3EditMoveInsertTo %W "insert - 1 lines"}

    bind $w <Meta-f>    {tkW3EditMoveInsertTo %W "insert + 1 c wordend"}
    bind $w <Meta-b>    {tkW3EditMoveInsertTo %W "insert - 1 c wordstart"}
}

## ********************
## Keybinding actions
## ********************

proc tkW3EditInsert {w a} {
    global tkW3EditParaStyle tkW3EditCharStyle
    set start [$w index insert]
    set tags [$w tag names "insert - 1 c"]
    set add_tag ""
    
    set anchor_number [lsearch $tags "h*"]
    if {$anchor_number != -1} {
	lappend add_tag Anchor
	lappend add_tag [lindex $tags $anchor_number]
    }
    
    if {$tkW3EditParaStyle != ""} {
	lappend add_tag $tkW3EditParaStyle
    }

    if {$tkW3EditCharStyle != ""} {
	lappend add_tag $tkW3EditCharStyle
    }

    $w insert insert $a
    foreach tag [.f.msg tag names $start] {
        .f.msg tag remove $tag $start insert
    }

    foreach tag $add_tag {
	.f.msg tag add $tag $start insert
    }
}

proc tkW3EditBackspace {w} {
    tk_textBackspace $w
    $w yview -pickplace insert
    tkW3EditSetStyle $w
}

proc tkW3EditMoveInsertTo {w loc} {
    $w mark set insert $loc
    $w yview -pickplace insert
    tkW3EditSetStyle $w
}

proc tkW3EditSetStyle {w} {
    global tkW3EditParaStyle tkW3EditCharStyle
    set tags [$w tag names "insert - 1 c"]

    set para_tags {"H1" "H2" "H3" "H4" "H5" "H6" "PRE" "ADDRESS"}

    set tkW3EditParaStyle ""

    foreach tag $tags {
	if {[lsearch $para_tags $tag] != -1} {
	    set tkW3EditParaStyle $tag
	}
    }
    tkW3EditDisplayStatus
}

proc tkW3EditDisplayStatus {} {
    global tkW3EditParaStyle tkW3EditCharStyle tkW3EditConvert
 
    tkW3OutputEntryPrint .style.char $tkW3EditConvert($tkW3EditCharStyle)
    tkW3OutputEntryPrint .style.para $tkW3EditConvert($tkW3EditParaStyle)
}

proc tkW3EditSetParaStyle {style} {
    global tkW3EditParaStyle
    
    tkW3EditTagSelected .f.msg  $style
    set tkW3EditParaStyle $style
    tkW3EditDisplayStatus
}

proc tkW3EditSetCharStyle {style} {
    global tkW3EditCharStyle
    tkW3EditTagSelected .f.msg  $style
    set tkW3EditCharStyle $style
    tkW3EditDisplayStatus
}

## ********************
## Procedures to show or hide the edit window
## ********************

proc tkW3EditOpen {} {
    global tkW3ConfigEditTk
    if {[winfo exists ".edit"]} {
	wm deiconify .edit
    } {
	edittkmain "" "" toplevel "" $tkW3ConfigEditTk
    }
}

proc tkW3EditClose {} {
    wm withdraw .edit
}


## ********************
## Save Generated HTML to a file
## ********************

proc tkW3EditSave {} {
    global fsBox
    set list [split [tkW3NavigateGetAddress] ":"]
    set type [lindex $list 0]
    set name [lindex $list 1]
    set file "file.html"
    set fsBox(path) ""

    if {"$type" == "file" && ![string match $name "//*"]} {
	set fsBox(path) [file dirname $name]
	set file [file tail $name]
    }

    tkW3EditClose
    FSBox {Save HTML} $file \
	{tkW3EditOutputHTML $fsBox(path)/$fsBox(name)}
    DLG:show . .fsBox
}

proc tkW3EditOutputHTML {filename} {
    set file [open $filename "w"]
    puts $file [.edit.main.t get 1.0 end]
    close $file
}

## ********************
## Procedures to generate HTML in the edit window from 
## stuff displayed in the display window
## ********************

proc tkW3EditGenerate {} {
    tkW3EditOpen
    tkW3EditCopyText .f.msg .edit.main.t
    tkW3EditCopyTags .f.msg .edit.main.t
    tkW3EditConvertToHTML .edit.main.t
}


proc tkW3EditCopyText {from to} {
    $to delete 1.0 end
    $to insert 1.0 [$from get 1.0 end]
}

proc tkW3EditCopyTags {from to} {
    foreach item [$from tag names] {
	set range [$from tag ranges $item] 
	set length [llength $range]
	for {set i 0} {$i < $length} {incr i 2} {
	    set j $i
	    incr j 1
	    $to tag add $item [lindex $range $i] [lindex $range $j] 
	}
    }
}

proc tkW3EditConvertToHTML {w} {
    global tkW3NavigateIsIndex
    $w mark set tmp 1.0
    $w insert tmp "<HEADER>\n"
    $w insert tmp "<TITLE>[tkW3NavigateGetTitle]</TITLE>\n"
    if $tkW3NavigateIsIndex {
	$w insert tmp "<ISINDEX>\n"
    }
    $w insert tmp "</HEADER>\n"
    $w insert tmp "<BODY>\n"

    tkW3EditDeleteTextWithTag $w blank
    tkW3EditConvertBullets $w LI "\n<LI>"
    tkW3EditConvertBullets $w DD "\n<DD>"
    tkW3EditConvertBullets $w DT "\n<DT>"
    tkW3EditConvertBullets $w P "<P>\n"
    tkW3EditConvertBullets $w DL.mark "\n<DL>"
    tkW3EditConvertBullets $w /DL.mark "\n</DL>\n"
    tkW3EditConvertBullets $w UL.mark "\n<UL>"
    tkW3EditConvertBullets $w /UL.mark "\n</UL>\n"
    tkW3EditConvertBullets $w MENU.mark "\n<MENU>"
    tkW3EditConvertBullets $w /MENU.mark "\n</MENU>\n"


    tkW3EditBreakLines $w
    
    for {set n 1} {$n < 7} {incr n} {
	tkW3EditDelimitTag $w H$n "<H$n>" "</H$n>\n"
    }

    set delimit_tags {
	{ADDRESS "<ADDRESS>\n" "</ADDRESS>\n"}
	{PRE "<PRE>\n" "</PRE>\n"}
	{CODE "<CODE>" "</CODE>"}
	{TT "<TT>" "</TT>"}
	{B "<B>" "</B>"}
	{I "<I>" "</I>"}
	{U "<U>" "</U>"}
	{EM "<EM>" "</EM>"}
	{STRONG "<STRONG>" "</STRONG>"}
	{SAMP "<SAMP>" "</SAMP>"}
	{KBD "<KBD>" "</KBD>"}
	{VAR "<VAR>" "</VAR>"}
	{DFN "<DFN>" "</DFN>"}
	{CITE "<CITE>" "</CITE>"}
    }

    foreach tag $delimit_tags {
	tkW3EditDelimitTag $w [lindex $tag 0] [lindex $tag 1] [lindex $tag 2]
    }

    tkW3EditConvertAnchors $w
    tkW3EditConvertImages $w
    $w insert end "</BODY>"
}

proc tkW3EditConvertAnchors {w} {
    global tkW3HtAnchors
    $w mark set tmp 1.0
    while {[set range [$w tag nextrange "Anchor" tmp]] != ""} {
	$w mark set tmp [lindex $range 1]
	foreach t [$w tag names [lindex $range 0]] {
	    if [regexp {h([0-9]+)} $t {} n] {
		set string "<A"
		if {$tkW3HtAnchors(name.$n) != ""} {
		    append string  " NAME=\"$tkW3HtAnchors(name.$n)\""
		}
		if {$tkW3HtAnchors(href.$n) != ""} {
		    append string " HREF=\"$tkW3HtAnchors(href.$n)\""
		}
		append string ">"
		$w mark set insert [lindex $range 0]
		tkW3EditInsertClean $w $string
		$w mark set insert [$w index tmp]
		tkW3EditInsertClean $w "</A>"
	    }
	}
    }
}

proc tkW3EditBreakLines {w} {
    for {set i 1} {[$w compare $i.0 < end]} {incr i} {
	set chars [lindex [split [$w index "$i.0 lineend"] "."] 1]
	if {$chars > 80 && [$w compare "$i.80 wordstart" > "$i.0"]} {
	    $w mark set tmp "$i.80 wordstart"
	    if {[$w get tmp] == " "} {
		$w delete tmp
	    } {
		$w delete "tmp - 1 char"
	    }
	    $w insert tmp "\n"
	}
    }
}

proc tkW3EditConvertBullets {w text_tag sgml_tag} {
    $w mark set tmp 1.0
    while {[set range [$w tag nextrange $text_tag tmp]] != ""} {
	$w mark set tmp [lindex $range 0]
	$w delete tmp [lindex $range 1]
	$w mark set insert [$w index tmp]
	tkW3EditInsertClean $w $sgml_tag
    }
}

proc tkW3EditDeleteTextWithTag {w tag} {
    $w mark set tmp 1.0
    while {[set range [$w tag nextrange $tag tmp]] != ""} {
	$w mark set tmp [lindex $range 0]
	$w delete tmp [lindex $range 1]
    }
}

proc tkW3EditDelimitTag {w tag begin end} {
    $w mark set tmp 1.0
    while {[set range [$w tag nextrange $tag tmp]] != ""} {
	$w mark set tmp [lindex $range 1]
	
	$w mark set insert [lindex $range 0]
	tkW3EditInsertClean $w $begin

	$w mark set insert [$w index tmp]
	tkW3EditInsertClean $w $end
    }
}

proc tkW3EditInsertClean {w text} {
    set start [$w index insert]
    $w insert insert $text
    foreach tag [$w tag names $start] {
	$w tag remove $tag $start insert
    }
}

## ********************
## These procedure are used to edit the display window
## ********************


proc tkW3EditQueryChangeTitle {w} {
    DLG:entry . .change_title "Change Title Dialog" \
	{"Title:"} {Change title} "tkW3EditChangeTitle"
    DLG:show . .change_title
}    


proc tkW3EditChangeTitle {title} {
    tkW3OutputSetAddress [tkW3NavigateGetAddress] $title
}

proc tkW3EditClearSelection {w} {
    set range [$w tag ranges sel] 
    set length [llength $range]
    for {set i 0} {$i < $length} {incr i 2} {
	set j $i
	incr j 1
	$w tag remove sel [lindex $range $i] [lindex $range $j] 
    }
}

## tkW3EditAddAnchor
## -------------
## Turn currently highlighted area into an anchor

proc tkW3EditBeginAnchor {w {link_number ""}} {
    global tkW3HtAnchors
    if {[.f.msg tag ranges sel] != {} || $link_number != ""} {
        DLG:entry . .anchor_dialog "Anchor Dialog" \
            {"Name:" "Href:"} {Begin} "tkW3EditAddAnchor"
        if {$link_number != ""} {
	    tkW3EditClearSelection $w
            set name $tkW3HtAnchors(name.$link_number)
            set href $tkW3HtAnchors(href.$link_number)
            DLG:set_entry_value .anchor_dialog 1 $name
            DLG:set_entry_value .anchor_dialog 2 $href
            $w tag add sel \
                [$w index h$link_number.first] \
                [$w index h$link_number.last] 
        }
        DLG:show . .anchor_dialog
    }
}
 
proc tkW3EditAddAnchor {name href} {
    global tkW3HtAnchors
    tkW3HtBeginAnc $name $href
    set number $tkW3HtAnchors(index)
    tkW3HtEndAnc

    if {[.f.msg tag ranges sel] != {}} {
	set start [.f.msg index sel.first]
	set finish [.f.msg index sel.last]
	foreach tag [.f.msg tag names $start] {
	    if [regexp {h[0-9]+} $tag] {
		.f.msg tag remove $tag $start $finish
	    }
	}
	.f.msg tag add Anchor $start $finish 
	.f.msg tag add h$number $start $finish
	.f.msg tag remove sel $start $finish
    }
}

proc tkW3EditTagSelected {w tag {begin 0} {end 0}} {
    set list [tkW3EditClearTagsFromSelected $w]
    if {$list != {}} {
	if {$begin != 0} {
	    $w mark set insert [lindex $list 0]
	    
	}
	$w tag add $tag [lindex $list 0] [lindex $list 1]
	$w mark set insert [lindex $list 1]
	if {$end != 0} {
	    tkW3HtBlankLines $begin
	    $w mark set insert [lindex $list 1]
	}
    }
}

proc tkW3EditBeginListItem {w} {
    global tkW3EditCurrentStyle
    tkW3HtBuf List "\n"
    set start [$w index insert]
    tkW3HtInsert "* " ListItem
    set tkW3EditCurrentStyle List
    $w tag add $tkW3EditCurrentStyle $start insert
}

proc tkW3EditClearTagsFromSelected {w} {
    global tkW3EditCurrentStyle

    if {[$w tag ranges sel] != {}} {
	set start [$w index sel.first]
	set finish [$w index sel.last]
	foreach tag [.f.msg tag names $start] {
	    $w tag remove $tag $start $finish
	}
	return [list $start $finish]
    }
    set tkW3EditCurrentStyle Normal
    return {}
}

proc tkW3EditCut {w} {
    tkW3EditCopy $w
    tkW3EditDelete $w
}

proc tkW3EditCopy {w} {
    global tkW3EditSelection
    set tkW3EditSelection [$w get sel.first sel.last]
}

proc tkW3EditDelete {w} {
    $w delete sel.first sel.last
}

proc tkW3EditPaste {w} {
    global tkW3EditSelection
    if {"" != [$w tag ranges sel]} {
	$w delete sel.first sel.last
	$w set mark insert sel.first
    }
    $w insert insert "$tkW3EditSelection"
}

proc tkW3EditClear {w} {
    $w delete 1.0 end
}

proc tkW3EditInsertBullet {w type} {
    global tkW3EditCurrentStyle
    case $type {
	"LI" {
	    tkW3HtBuf offset " "
	    tkW3HtInsert "* " ListItem
	}
	"DD" {
	    tkW3HtInsert "* " GlossaryDefinition
	}
	"DT" {
	    tkW3HtInsert "* " GlossaryTerm
	}
	"P" {
	    tkW3HtInsert "\n\n" ParagraphBreak
	}
    }
}

proc tkW3EditSelectAll {w} {
    $w tag add sel 1.0 end
}

proc tkW3EditDeselectAll {w} {
    $w tag remove sel 1.0 end
}

proc tkW3EditCreateNewPage {page_name} {
    tkW3OutputSetAddress $page_name ""
    tkW3EditQueryChangeTitle .f.msg
    tkW3OutputClearBody
}

## ********************
## Procedures dealing with image tags
## ********************

proc tkW3EditConvertImages {w} {
    global tkW3HtImages
    $w mark set tmp 1.0
    while {[set range [$w tag nextrange "Image" tmp]] != ""} {
	$w mark set tmp [lindex $range 0]
	foreach t [$w tag names tmp] {
	    if [regexp {i([0-9]+)} $t {} number] {
		set src $tkW3HtImages($number)
	    }
	}
	$w delete tmp [lindex $range 1]
	$w mark set insert [$w index tmp]
	tkW3EditInsertClean $w "<IMG SRC=\"$src\">"
    }
}

proc tkW3EditAddImage {w} {
    DLG:entry . .image_dialog "Image Dialog" \
	{"Source:"} {Add} "tkW3HtSetImg "
    DLG:show . .image_dialog
}


proc tkW3EditChangeImage {i} {
    global tkW3HtImages
    DLG:entry . .image_dialog "Image Dialog" \
	{"Source:"} {Change} "set tkW3HtImages($i) "
    DLG:set_entry_value .image_dialog 1 $tkW3HtImages($i)
    DLG:show . .image_dialog
}

proc tkW3EditCheckModified {} {
    if {$tkW3EditModified == 0} {
	return 1
    }

    if {[DLG:question . .confirm_action_dialog \
	 "Are you sure you want to?" "Yes" "No"] == 1} {
	     return 1
	 }
    return 0
}
