# jlibrary.tcl - utility procedures
#
# these procedures are required by
#     help.tk
#     edit.tk
#     browser.tk
# they may be located in the file "~/.tk/jlibrary.tcl" (where they will
# be source'd by those applications on startup), or in the site-wide
# tk library directory, where they will be found (and loaded) by the
# default tk  unknown  procedure.
######################################################################

### TO DO
###   more error-checking in j:fs
###   fix focus on j:fs
###   option for load/save?
###   mkdir when saving?
###   reconcile prompt_file and j:fs (in applications)
###   `default' behaviour needs fixed (do we even need a default now?)
###   label to show cwd?

# j:selection_if_any - return selection if it exists, else {}
# j:no_selection - true if there is no selection
# j:source_config file ?dir? - read user preferences from a file
# j:write_prefs file ?dir? - write user preferences to a file
# j:dialogue w - arrange to position window w near ctr of screen
# j:longest_match l - longest common initial string in list l
# j:expand_filename f - expand filename prefix as much as possible
# j:alert msg - this displays an alert box
# j:confirm msg - this displays a Cancel/OK dialog box
# j:prompt prompt default ?files? - prompt the user for information
# j:prompt_file prompt default - prompt the user for a filename
# j:prompt_font prompt pattern - prompt for a font (via xfontsel)
# j:configure_font widget fontlist - use font from list, or default
# j:fs ?buttons? ?prompt? ?default? ?dir? ?fileprompt? - file selector box
# j:fs:fill_list lb - fill the listbox with files from CWD

######################################################################
# global variables:
#
global PREFS
if {! [info exists PREFS(autoposition)]} {set PREFS(autoposition) 0}
if {! [info exists PREFS(confirm)]} {set PREFS(confirm) 1}
#
######################################################################


######################################################################
# j:selection_if_any - return selection if it exists, else {}
#   this is from kjx@comp.vuw.ac.nz (R. James Noble)
######################################################################

proc j:selection_if_any {} {
  if {[catch {selection get} s]} {return ""} {return $s}
}

######################################################################
# j:no_selection - true if there is no selection
######################################################################

proc j:no_selection {} {
  if {[catch {selection get} s]} {return 1} {return 0}
}

######################################################################
# j:write_prefs file ?dir? - write user preferences to a file
#   file is assumed to be in env(HOME)/.tk unless dir is specified
######################################################################

proc j:write_prefs {file {dir {}}} {
  global PREFS env tk_strictMotif
  
  if {$dir == {}} then {
    set dir $env(HOME)/.tk		;# NOTE: dir created if necessary!
  }

  set PREFS(0) 1			;# dummy to make sure PREFS is array
  if {! [info exists env(PRINTER)]} {
    set env(PRINTER) lp			;# make sure $env(PRINTER) is valid
  }

  if {! [file isdirectory $dir]} {	;# make sure directory exists
    exec mkdir $dir
  }
  set preffile [open "$dir/$file" w]
  puts $preffile "global PREFS env tk_strictMotif"
  foreach pref [lsort [array names PREFS]] {
    puts $preffile "set PREFS($pref) $PREFS($pref)"
  }
  puts $preffile "set env(PRINTER) $env(PRINTER)"
  puts $preffile "set tk_strictMotif $tk_strictMotif"
  close $preffile
}

######################################################################
# j:source_config file ?dir? - read user preferences from a file
#   file is assumed to be in env(HOME)/.tk unless dir is specified
#   NOTE: this can also be used just to source an arbitrary Tcl file
######################################################################

proc j:source_config {file {dir {}}} {
  global PREFS env tk_strictMotif

  if {! [info exists env(PRINTER)]} {set env(PRINTER) lp}
  
  if {$dir == {}} then {
    set dir $env(HOME)/.tk
  }

  if {[file isfile "$dir/$file"]} then {
    source "$dir/$file"
  }
}

######################################################################
# j:dialogue w - arrange to position window w near ctr of screen
######################################################################

proc j:dialogue { w } {
  global PREFS

  if $PREFS(autoposition) {
    # first, display off-screen (wm withdraw should work, but doesn't):
    wm geometry $w +-1000+-1000
#   wm withdraw $w		;# hide the window

    wm transient $w .		;# tell wm not to draw a titlebar
    update idletasks		;# force geometry managers to run
    # calculate position:
    set wwidth [winfo reqwidth $w]
    set wheight [winfo reqheight $w]
    set swidth [winfo screenwidth $w]
    set sheight [winfo screenheight $w]
    set hpos [expr { ( $swidth - $wwidth ) / 2 }]
    set vpos [expr { ( $sheight - $wheight ) / 3 }]
    wm geometry $w +${hpos}+${vpos}

    update idletasks		;# force geometry managers to run
    wm deiconify $w		;# display window
    wm focus $w
  }
}

######################################################################
# j:longest_match l - longest common initial string in list l
#   used by tab-expansion in filename dialogue box
######################################################################
# this needs commenting desperately

proc j:longest_match { l } {
  case [llength $l] in {
    {0} { return {} }
    {1} { return [lindex $l 0] }
  }
  set first [lindex $l 0]
  set matchto [expr {[string length $first] - 1}]
  for {set i 1} {$i < [llength $l]} {incr i} {
    set current [lindex $l $i]
    # if they don't match up to matchto, find new matchto
    if { [string compare \
           [string range $first 0 $matchto] \
           [string range $current 0 $matchto]] } {
      # loop, decreasing matchto until the strings match that far
      for {} \
          {[string compare \
              [string range $first 0 $matchto] \
              [string range $current 0 $matchto]] } \
          {incr matchto -1 } \
          {}			;# don't need to do anything in body
    } ;# end if they didn't already match up to matchto
  } ;# end for each element in list
  if {$matchto < 0} then {
    return {}
  } else {
    return [string range $first 0 $matchto]
  }
}

######################################################################
# j:expand_filename f - expand filename prefix as much as possible
#       (for use in file dialogue boxes)
######################################################################
# note: if the filename has *, ?, or [...] in it, they will be used
#       as part of the globbing pattern.  i declare this a feature.

proc j:expand_filename { f } {
  set expansion [j:longest_match [glob -nocomplain "${f}*"]]
  if {$expansion == ""} {return $f}
  # make sure it doesn't already end in "/"
  set expansion [string trimright $expansion "/"]
  if [file isdirectory $expansion] {append expansion "/"}
  return $expansion
}

######################################################################
# j:alert msg - this displays an alert box
######################################################################

proc j:alert { msg } {
  set old_focus [focus]		;# so we can restore original focus
  toplevel .alert
  message .alert.msg -width 300 -anchor w -text $msg
  frame .alert.r -height 2 -width 200 -borderwidth 1 -relief sunken
  frame .alert.b
  button .alert.b.ok -text OK -bd 4 -width 8 -command {
    destroy .alert
  }
  pack append .alert.b .alert.b.ok {right padx 10 pady 10}
  pack append .alert \
    .alert.msg {top fill expand padx 10} \
    .alert.b {bottom fill} \
    .alert.r {bottom fillx}

  j:dialogue .alert		;# position in centre of screen

  focus .alert
  bind .alert <Key-Return> {.alert.b.ok invoke}
  grab .alert
  tkwait window .alert
  focus $old_focus
}

######################################################################
# j:prompt prompt default ?files? - prompt the user for information
#   if "files" is the third argument, <Tab> will expand filenames
######################################################################

proc j:prompt { {prompt "Enter a value:"} {default ""} {files ""}} {
  global prompt_result

  set old_focus [focus]		;# so we can restore original focus

  toplevel .pr
  message .pr.msg -width 300 -anchor w -text $prompt
  entry .pr.field -relief sunken -width 40
  frame .pr.b -relief sunken
  frame .pr.b.r -height 2 -width 200 -borderwidth 1 -relief sunken
  button .pr.b.ok -text OK -bd 4 -width 8 -command {
    set prompt_result [.pr.field get]
    destroy .pr
  }
  # return a sentinel value on cancel:
  button .pr.b.cancel -text Cancel -width 8 -command {
    set prompt_result {././/CANCEL//./.}
    destroy .pr
  }

  pack append .pr.b \
    .pr.b.r {top fillx} \
    .pr.b.ok {right padx 10 pady 10} \
    .pr.b.cancel {right pady 10}
  pack append .pr \
    .pr.msg {top fill expand padx 10} \
    .pr.field {top padx 10 pady 10} \
    .pr.b {bottom fillx}

  .pr.field delete 0 end
  .pr.field insert end $default

  j:dialogue .pr			;# position in centre of screen

  bind .pr.field <Return> {.pr.b.ok invoke}
  bind .pr.field <Control-c> {.pr.b.cancel invoke}
  bind .pr.field <Meta-c> {.pr.b.cancel invoke}
  bind .pr.field <Meta-period> {.pr.b.cancel invoke}
  if {$files == "files"} {
    bind .pr.field <Tab> {
      set f [%W get]
      %W delete 0 end
      %W insert end [j:expand_filename $f]
    }
  }
  focus .pr.field
  grab .pr
  tkwait window .pr
  focus $old_focus
  return $prompt_result
}

######################################################################
# j:prompt_file prompt default - prompt for a filename
#   <Tab> will expand filenames
######################################################################

proc j:prompt_file { {prompt "File name:"} {default ""} } {
  return [j:prompt $prompt $default {files}]
}

######################################################################
# j:prompt_font prompt pattern - prompt for a font (via xfontsel)
#   `prompt' argument currently ignored.
#   usage of xfontsel (`quit' button) not obvious!
######################################################################

proc j:prompt_font { {prompt "Font:"} {pattern "*"} } {
  return [exec xfontsel -pattern $pattern -print]
}

######################################################################
# j:configure_font widget fontlist - use font from list, or default
######################################################################

proc j:configure_font {widget fontlist} {
  foreach font $fontlist {
    # try to use each font, until one is successful:
    if {$font == {default}} {
      set font [option get $widget Tk Font]
      if {$font == {}} {set font {*-courier-medium-r-normal--12-120-*}}
    }
    if {! [catch {$widget configure -font $font}]} {return}
  }
}

######################################################################
# j:fs ?buttons? ?prompt? ?default? ?fileprompt? - file selector box
######################################################################

# PROBLEM - shouldn't rely on cd - could affect rest of app

proc j:fs {{buttons {ok cancel home}} \
           {prompt {Choose a file:}} \
           {default {}} \
           {fileprompt {File:}}} {
  global prompt_result env
  global fs_defaultbutton

  if {$buttons == {}} {set buttons {ok cancel home}}

  set file [file tail $default]		;# may end up {}
  set dir [file dirname $default]	;# may end up .

  if {![file isdirectory $dir]} {
    set dir .
  }

  set fs_defaultbutton [lindex $buttons 0]

  set prompt_result $file

  set old_focus [focus]		;# so we can restore original focus

  if [winfo exists .fs] {
    destroy .fs
  }

  cd $dir

  toplevel .fs
  wm minsize .fs 10 10

  label .fs.prompt -anchor w -text $prompt
  frame .fs.r1 -height 2 -width 200 -borderwidth 1 -relief sunken
  frame .fs.list
  listbox .fs.list.lb -yscroll ".fs.list.sb set" -geometry 30x20
  scrollbar .fs.list.sb -command ".fs.list.lb yview"
  frame .fs.r2 -height 2 -width 200 -borderwidth 1 -relief sunken
  frame .fs.file
  label .fs.file.l -text $fileprompt -anchor e
  entry .fs.file.e -relief sunken
  frame .fs.file.filler -width 10 -height 10 -relief flat

  frame .fs.b
  button .fs.b.ok -width 8 -text {OK} -command {
    set file [.fs.file.e get]
    if {[file isdirectory $file]} {
      cd $file			;# cd into directory, refresh list
      j:fs:fill_list .fs.list.lb
      .fs.file.e delete 0 end	;# clear filename space
    } else {
      set cwd [pwd]
      if {$cwd == "/"} {set cwd ""}
      set file [.fs.file.e get]
      case $file in {
        /*	{set prompt_result $file}
        default {set prompt_result $cwd/$file}
      }
      destroy .fs
    update
    }
  }
  button .fs.b.gointo -width 8 -text "Go Into" -command {
    set file [.fs.file.e get]
    if {[file isdirectory $file]} {
      cd $file			;# cd into directory, refresh list
      j:fs:fill_list .fs.list.lb
      .fs.file.e delete 0 end	;# clear filename space
    } else {
      j:alert "\"$file\" is not a directory."
    }
  }
  button .fs.b.home -width 8 -text {Home} -command {
    cd $env(HOME)
    j:fs:fill_list .fs.list.lb
  }
  button .fs.b.root -width 8 -text {Root} -command {
    cd /
    j:fs:fill_list .fs.list.lb
  }
  button .fs.b.here -width 8 -text {Here} -command {
    set prompt_result [pwd]
    destroy .fs
    update
  }
  button .fs.b.cancel -width 8 -text {Cancel} -command {
    set prompt_result {././/CANCEL//./.}
    destroy .fs
    update
  }
  frame .fs.b.filler -width 10 -height 10 -relief flat

  pack append .fs.list \
    .fs.list.lb {left expand fill} \
    .fs.list.sb {left filly}
  pack append .fs.file \
    .fs.file.l {left pady 10 padx 10} \
    .fs.file.e {left expand pady 10 fillx padx 10} \
    .fs.file.filler {left}

  # now create the buttons the caller requested:
  #    (NEEDS ERROR CHECKING!)
  pack append .fs.b \
    .fs.b.filler {bottom}
  foreach b $buttons {
    pack append .fs.b .fs.b.$b {bottom pady 10 padx 15}
  }
  # wider border on default button:
  .fs.b.$fs_defaultbutton configure -borderwidth 3

  pack append .fs \
    .fs.prompt {top fill} \
    .fs.r1 {top fillx} \
    .fs.file {bottom expand fillx} \
    .fs.r2 {bottom fillx} \
    .fs.b {right filly} \
    .fs.list {top expand fill}

  j:dialogue .fs		;# position in centre of screen

  .fs.file.e insert end $prompt_result

  focus .fs.file.e
  bind .fs.file.e <Key-Return> {
    set file [.fs.file.e get]
    if {$file != {} && [file isdirectory $file]} {
      .fs.b.gointo invoke
    } else {
      .fs.b.$fs_defaultbutton invoke
    }
  }
  bind .fs.file.e <Key-Tab> {	;# expand filename on <Tab>
    set f [%W get]
    %W delete 0 end
    %W insert end [j:expand_filename $f]
  }
  bind .fs.list.lb <Button-1> {	;# select, and insert filename into entry
    %W select from [%W nearest %y]
    set file [lindex [selection get] 0]
    .fs.file.e delete 0 end
    .fs.file.e insert end $file
  }

  bind .fs.list.lb <Double-Button-1> {	;# cd to dir or do default thing
    set file [lindex [j:selection_if_any] 0]
    if [file isdirectory $file] {
      .fs.b.gointo invoke
    } else {
      .fs.b.$fs_defaultbutton invoke
    }
  }

#  grab .fs			;# for some reason this screws up 
				;#   "bind .fs.list.lb <Double-Button-1> ..."

  j:fs:fill_list .fs.list.lb	;# fill the listbox for the first time
  tkwait window .fs
  focus $old_focus
  return $prompt_result
}

######################################################################
# j:fs:fill_list lb - fill the listbox with files from CWD
######################################################################

proc j:fs:fill_list {lb} {
  $lb delete 0 end

  # add ".." to go up a level:
  $lb insert end ".."

  update

  # add all normal (non-dot) files:
  foreach i [lsort [glob -nocomplain *]] {
    if {[file isdirectory $i]} {
      $lb insert end "$i/"
    } else {
      $lb insert end $i
    }
  }

  # add any dot-files:
  foreach i [lsort [glob -nocomplain .*]] {
    if {$i != "." && $i != ".."} {
      if {[file isdirectory $i]} {
        $lb insert end "$i/"
      } else {
        $lb insert end $i
      }
    }
  }
}
