#==============================================================================
#                                 bezier.tcl
#------------------------------------------------------------------------------
# SippBezierFile procedure.  Create an object for a Bezier curve or patch
# specified in a data file.
#------------------------------------------------------------------------------
# Copyright 1992-1993 Mark Diekhans
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  Mark Diekhans makes
# no representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.
#------------------------------------------------------------------------------
# $Id: bezier.tcl,v 5.0 1994/09/05 01:21:24 markd Rel $
#==============================================================================

#@package: TSIPP-BEZIERFILE SippBezierFile

#
# Fetch the next token out of the file. The variable passed in bufferVar
# should be initialized to an empty string and passed to the function on
# subsequent calls.
#
proc Sipp:BezierToken {fileid bufferVar} {
    upvar $bufferVar buffer

    while 1 {
        if [lempty $buffer] {
            if {[gets $fileid buffer] < 0} {
                error "premature EOF in Bezier description file"
            }
        }
        set token [ctoken buffer " \t"]
        if [cequal [cindex $token 0] "#"] {
            set buffer {}
        } elseif ![lempty $token] {
            return $token
        }
    }
}

#
# Fetch an integer token out of the file.
#
proc Sipp:BezierIntToken {fileid bufferVar} {
    upvar $bufferVar buffer
    
    set token [Sipp:BezierToken $fileid buffer]
    if [catch {
        format %d $token
    }] {
        error "expected integer, found \"$token\" in Bezier description file"
    }
    return $token
}

#
# Fetch an floating pointnumber token out of the file.
#
proc Sipp:BezierFloatToken {fileid bufferVar} {
    upvar $bufferVar buffer
    
    set token [Sipp:BezierToken $fileid buffer]
    if [catch {
        format %f $token
    }] {
        error "expected floating point number, found \"$token\" in Bezier description file"
    }
    return $token
}

#
# Issue an error about an unexpected token in the the file.
#
proc Sipp:BezierBadToken {token expectMsg} {
    error "unexpected token in Bezier description file \"$token\", expected $expectMsg"
}

#
# Read a vertex list.
#
proc Sipp:BezierVertexRead {fileid bufferVar} {
    upvar $bufferVar buffer
    
    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "vertices:"] {
        Sipp:BezierBadToken $token {"vertices:"}
    }
    set num [Sipp:BezierIntToken $fileid buffer]

    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "vertex_list:"] {
        Sipp:BezierBadToken $token {"vertex_list:"}
    }

    loop idx 0 $num {
        set vertex {}
        lappend vertex [Sipp:BezierFloatToken $fileid buffer]
        lappend vertex [Sipp:BezierFloatToken $fileid buffer]
        lappend vertex [Sipp:BezierFloatToken $fileid buffer]
        lappend vertexList $vertex
    }
    return $vertexList
}

#
# Read a curve list, converting to the SippBezierCurve format.
#
proc Sipp:BezierCurveRead {fileid bufferVar} {
    upvar $bufferVar buffer
    
    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "curves:"] {
        Sipp:BezierBadToken $token {"curves:"}
    }
    set num [Sipp:BezierIntToken $fileid buffer]

    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "curve_list:"] {
        Sipp:BezierBadToken $token {"curve_list:"}
    }

    loop idx 0 $num {
        set curve {}
        lvarpush curve [expr [Sipp:BezierIntToken $fileid buffer] -1]
        lvarpush curve [expr [Sipp:BezierIntToken $fileid buffer] -1]
        lvarpush curve [expr [Sipp:BezierIntToken $fileid buffer] -1]
        lvarpush curve [expr [Sipp:BezierIntToken $fileid buffer] -1]
        lappend curveList $curve
    }
    return $curveList
}

#
# Read a patch list, converting to the SippBezierPatch format.
#
proc Sipp:BezierPatchRead {fileid bufferVar} {
    upvar $bufferVar buffer
    
    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "patches:"] {
        Sipp:BezierBadToken $token {"patches:"}
    }
    set num [Sipp:BezierIntToken $fileid buffer]

    set token [Sipp:BezierToken $fileid buffer]
    if ![cequal $token "patch_list:"] {
        Sipp:BezierBadToken $token {"patch_list:"}
    }

    loop idx 0 $num {
        set patch {}
        loop idx2 0 4 {
            set line {}
            lappend line [expr [Sipp:BezierIntToken $fileid buffer] -1]
            lappend line [expr [Sipp:BezierIntToken $fileid buffer] -1]
            lappend line [expr [Sipp:BezierIntToken $fileid buffer] -1]
            lappend line [expr [Sipp:BezierIntToken $fileid buffer] -1]
            lappend patch $line
        }
        lappend patchList $patch
    }
    return $patchList
}


#
# Create an object for a Bezier curve or patch specified in a data file.
#
proc SippBezierFile {fileid resolution shaderhandle {texture NATURAL}} {
    set tokenBuffer {}

    set token [Sipp:BezierToken $fileid tokenBuffer]
    switch -- $token {
        "bezier_patches:" {
            set vertexList [Sipp:BezierVertexRead $fileid tokenBuffer]
            set patchList [Sipp:BezierPatchRead $fileid tokenBuffer]
            return [SippBezierPatch $resolution $vertexList $patchList \
                                    $shaderhandle $texture]
        }
        "bezier_curves:" {
            set vertexList [Sipp:BezierVertexRead $fileid tokenBuffer]
            set curveList [Sipp:BezierCurveRead $fileid tokenBuffer]
            return [SippBezierCurve $resolution $vertexList $curveList \
                                    $shaderhandle $texture]
        }
        default {
            Sipp:BezierBadToken $token {"bezier_patches:" or "bezier_curves:"}
        }
    }
}

