#------------------dv--Hilfroutinen fuer Darstellungroutinen------------------

#-----------------------------------------------------------------------------
# extremaget {} ermittelt das Minimum und Maximum einer Spalte ;
#               sowie minimale und Maximale Indizees;
#               beruecksichtigt eventuell undefinierte Werte 
#	        path     : Diagrammwidgetpfad
#               col      : Index der Spalte, deren Minimum zu berechnen ist
#               min      : ermitteltes Minimum
#               max      : ermitteltes Maximum
#               minindex : kleinster Index der definierten Werte
#               maxindex : groesster Index der definierten Werte
#-----------------------------------------------------------------------------
#
proc extremaget {path col min max minindex maxindex} {

  # Uebergabe "called by reference" realisieren
  #
  upvar $min mi
  upvar $max ma
  upvar $maxindex maindex
  upvar $minindex miindex

  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  # erstelle Liste definierter Teilstuecke
  #
  set deflist [eval $diagdata($path.partition) $path $col indexlist dummy]
  set partextr ""
  foreach part $deflist {
     set ll [llength $part]
     if {$ll == 0} {continue}
     if {$ll == 1} {
        # definiertes Teilstueck hat nur einen Wert =>
        # setze minimum=maximum= Element
        #
        lappend partextr $part 
      } {
        # definiertes Teilstueck hat mehr als einen Wert =>
        # berechne Minimum der Teilstuecke
        #
        set mi [eval min $part]
        set ma [eval max $part]
        lappend partextr $mi $ma
      }
  }

  set ll [llength $partextr]
  if {$ll == 0} {
     # Keine Extrema gefunden => mi,ma auf "undefined" setzen
     # 
     set mi "undefined"
     set ma "undefined"
     return
  }

  if {$ll == 1} {
     # nur ein Extremum gefunden => setze mi=ma=Extremum
     #
     set mi [lindex $partextr 0]
     set ma $mi
  } {
     # mehrere Extrema gefunden
     #
     set mi [eval min $partextr]
     set ma [eval max $partextr]
  } 

  # berechne minimalen Index der definierten Werte
  #
  set miindex [lindex $indexlist 0]

  # berechne maximalen Anfangsindex der definierten Spaltenstuecke 
  #
  set endindex [lvarpop indexlist end]
   
  # berechne maximalen Index der definierten Spaltenwerte
  # (letzter Anfangsindex + laenge des letzten Spaltenstueckes -1)
  #
  set endpart [lvarpop deflist end]
  set maindex [expr $endindex + [llength $endpart] -1]
} 

#-----------------------------------------------------------------------------
# autoscaleindex {} ermittelt Skalierungswerte fuer einen Diagrammdatensatz 
#              im Modus "domainindex", so dass alle gueltigen Kurven innerhalb 
#              der Zeichenflaeche liegen und gridx bzw gridy Skalierungen auf
#              der X bzw Y Achse gezeichnet werden 
#              path   : Diagrammwidgetpfad 
#              return : Liste {fromx tox stepx fromy toy stepy}
#-----------------------------------------------------------------------------
#
proc autoscaleindex {path} {
  
  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  if {$diagdata($path.status) != "defined" ||\
      $diagdata($path.curvelist) == ""} {

      # Diagrammdaten noch nicht definiert bzw keine darstellbaren Daten geladen
      #
      # in fromx,tox,fromy,toy die alten Werte setzen
      #
      set fromx $diagdata($path.fromx)
      set tox   $diagdata($path.tox)
      set fromy $diagdata($path.fromy)
      set toy   $diagdata($path.toy)
  } {
    # Daten definiert und darstellbare Werte vorhanden
    #
    set maxindex 1
    set extrema ""
    foreach col $diagdata($path.curvelist) {

         # Extrema berechnen 
         #
         extremaget $path $col min max dummy dummy
        
         if {$min != "undefined"} {
         
            # Minimum/Maximum gefunden =>
            # Liste von Minima, Maxima erstellen 
            #
            lappend extrema $min $max
         }

         # maximalen Index ermitteln
         #
         set ll [expr [llength $diagdata($path.col$col)]-1]
         if {$ll > $maxindex} {
            set maxindex $ll
         } 
    } ; #  foreach col 

     set ll [llength $extrema]

     if {$ll == 0 } {
         # kein  Extremum gefunden =>
         # in fromx,tox,fromy,toy die alten Werte setzen
         #
         set fromx $diagdata($path.fromx)
         set tox   $diagdata($path.tox)
         set fromy $diagdata($path.fromy)
         set toy   $diagdata($path.toy)        
     } {
        if {$ll == 1} {
           # nur ein Extremum gefunden =>
           #
           set fromy [lindex $extrema 0]
           set toy [expr $fromy+1]
        } {
           # mehr als ein Extremum gefunden
           # Minimum und Maximum aller aller ausgewahlten Spalten errechnen
           #
           set fromy [eval min $extrema]
           set toy [eval max $extrema]
           if {$fromy == $toy} {
              set toy [expr $fromy+1]
           }
        } ; # if $ll == 1

        # fromx setzen : immer bei Index 0 beginnen
        #
        set fromx 0
   
        # tox setzen : immer groessten vorkommenden Index waehlen
        #
        set tox $maxindex
     } ; # if $ll == 0
  } ; # Daten definiert

  # stepx berechnen : so, dass #gridx+1 Skalierungen gezeichnet werden
  #
  set stepx [expr ($tox-$fromx)/[double $diagdata($path.gridx)]]

  # stepy berechnen : so, dass #gridy+1 Skalierungen gezeichnet werden
  #
  set stepy  [expr ($toy-$fromy)/[double $diagdata($path.gridy)]] 

  return "[int $fromx] [int [ceil $tox]] [int [ceil $stepx]]\
          [double $fromy] [double $toy] [double $stepy]"
}

#-----------------------------------------------------------------------------
# autoscalereal {} ermittelt Skalierungswerte fuer einen Diagrammdatensatz 
#              im Modus "domainreal", so dass alle gueltigen Kurven innerhalb 
#              der Zeichenflaeche liegen und gridx bzw gridy Skalierungen auf
#              der X bzw Y Achse gezeichnet werden 
#              path   : Diagrammwidgetpfad 
#              return : Liste {fromx tox stepx fromy toy stepy}
#-----------------------------------------------------------------------------
#
proc autoscalereal {path} {
  
  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  if {$diagdata($path.status) != "defined" ||\
      [llength $diagdata($path.curvelist)] < 2} {
   
      # Daten sind nicht definiert oder keine Kurve gewaehlt=>
      # in fromx,tox,fromy,toy alte Werte setzen
      #
      set fromx $diagdata($path.fromx)
      set tox   $diagdata($path.tox)
      set fromy $diagdata($path.fromy)
      set toy   $diagdata($path.toy)
  } {
     # Daten definiert und Kurven gewaehlt
     #
     set extremadom ""
     set extremacod ""
     set i 0
     foreach col $diagdata($path.curvelist) {
         if {[expr $i%2] == 0} {
            # i gerade =>
            # Liste Minima;Maxima der Urbildmenge ermitteln
            #
            set ll [llength $diagdata($path.col$col)]
            if {$ll == 0} {continue}
            if {$ll == 1} {
               # nur ein Element in dieser Spalte
               #
               lappend extremadom $diagdata($path.col$col)
            } {
               # mehrere Elemente in dieser Spalte
               #
               set dmin [eval min $diagdata($path.col$col)]
               set dmax [eval max $diagdata($path.col$col)]
               lappend extremadom $dmax $dmin
            }
         } {
            # i ungerade =>

            # Minima;Maxima der Bildmenge ermitteln; dabei eventuelle
            # undefinierte Werte beruecksichtigen 
            #
            extremaget $path $col cmin cmax dummy dummy 

            if {$cmin != "undefined"} {
               
               if {$cmin == $cmax} {
                  # ein extremum gefunden
                  #
                  lappend extremacod $cmin
               } {
                  # verschiedene Minima und Maxima gefunden
                  #
                  lappend extremacod $cmin $cmax 
               }
            }
         }
         incr i
     } ; # foreach

     # Minimum und Maximum der Urbildmengen ermitteln 
     #
     set ll [llength $extremadom]
     if {$ll == 0} {
        # kein Extremum gefunden => fromx und tox beibehalten
        # 
        set fromx $diagdata($path.fromx)
        set tox   $diagdata($path.tox)
     } {
       if {$ll == 1} {
          # nur ein Extremum gefunden
          #
          set fromx [lindex $extremadom 0]
          set tox [expr $fromx+1]
       } {
          # mehrere Extrema gefunden
          #
          set fromx [eval min $extremadom]
          set tox [eval max $extremadom]
          if {$fromx == $tox} {
             set tox [expr $fromx+1]
          }
       }
     }

     # Minimum und Maximum der Bildmengen ermitteln
     #
     set ll [llength $extremacod]
     if {$ll == 0} {
        # kein Extremum gefunden => fromy, toy beibehalten
        #
        set fromy $diagdata($path.fromy)
        set toy $diagdata($path.toy)
     } {
       if {$ll == 1} {
          # nur ein Extremum gefunden
          #
          set fromy [lindex $extremacod 0]
          set toy   [expr $fromy+1]
       } { 
          # mehrere Extrema gefunden
          #
          set fromy [eval min $extremacod]
          set toy [eval max $extremacod]
          if {$fromy == $toy} {
             set toy [expr $fromy+1]
          }
       }
     }  
  } ; # Daten definiert

  # stepx berechnen : so, dass #gridx+1 Skalierungen gezeichnet werden
  #
  set stepx [expr ($tox-$fromx)/[double $diagdata($path.gridx)]]

  # stepy berechnen : so, dass #gridy+1 Skalierungen gezeichnet werden
  #
  set stepy  [expr ($toy-$fromy)/[double $diagdata($path.gridy)]] 

  return "[double $fromx] [double $tox] [double $stepx]\
          [double $fromy] [double $toy] [double $stepy]"
}

#----------------------------------------------------------------------------
# tab2index {} Routine, die die Daten von einem Diagrammdatensatz
#              in einen anderen Datensatz kopiert; Einstellungen zu den Daten
#              wie z.B. die gewaehlten Kurven werden nicht mitkopiert.
#              source : Name des Quelldatensatzes
#              dest   : Name des Datensatzes
#----------------------------------------------------------------------------
#
proc tab2index {source dest} {

   # Diagrammvariablen sichtbar machen
   #
   global diagdata

   # Alle  Kurvendaten kopieren
   #
   loop index 0 $diagdata($source.colnum) {
      set diagdata($dest.col$index) $diagdata($source.col$index) 
   }

   # Kopieraktion gelungen => Datensatz als definiert erklaeren
   #
   set diagdata($dest.status) "defined"
} 

#----------------------------------------------------------------------------
# tab2real {} Routine, die die aktuellen UrbildDaten von einem Diagrammdatensatz
#              in einen anderen Datensatz kopiert; Einstellungen zu den Daten
#              wie z.B. die gewaehlten Kurven werden nicht mitkopiert.
#              source : Name des Quelldatensatzes
#              dest   : Name des Datensatzes
#----------------------------------------------------------------------------
#
proc tab2real {source dest} {

   # Diagrammvariablen sichtbar machen
   #
   global diagdata

   # Alle derzeit ausgewaehlten Urbildspalten des Ziehldatensatzes kopieren
   #
   set i 0
   foreach index $diagdata($dest.curvelist) {

     if {[expr $i%2] == 0} {
     
        # Urbildspalte gefunden => Datenspalte kopieren
        #
        set diagdata($dest.col$index) $diagdata($source.col$index)
      }
      incr i
  }
  # Kopieraktion gelungen => Datensatz als definiert erklaeren
  #
  set diagdata($dest.status) "defined"
}
   
#----------------------------------------------------------------------------
# getdate {} testet, ob der uebergebene String ein gueltiges Zeitformat in
#            in der Form "YYMMDD" ist und berechnet gegebenenfalls
#            die zugehoerige Unixzeit (Zeitzone GMT, Uhrzeit 12.00 angenommen).
#            date     : String, der zu ueberpruefen ist
#            unixtime : zugehoerige Unixzeit, falls der String gueltig ist
#            return   : 1, falls der String ein gueltiges Datum enhaelt
#                       0, sonst
#----------------------------------------------------------------------------
#
proc getdate {date unixtime} {

    # unixtime als "called by reference" realisieren
    #
    upvar $unixtime utime

    if {[isint $date] && [clength $date] == 6} {

       # Datum in yy mm dd zerlegen
       #
       set yy [crange $date 0 1]
       set mm [crange $date 2 3]
       set dd [crange $date 4 5]

       # Unixzeit berechnen
       #
       set rflag [catch {set utime [convertclock "$mm/$dd/$yy 1200" GMT]}]

       return [expr 1-$rflag] ;# rflag ist 1, Falls Konvertierung mislingt
    }
    # Konvertierung nicht erfolgreich
    #
    return 0
}

#----------------------------------------------------------------------------
# coldate2real {} transformiert Datumseintraege einer Spalte in
#                 reele Spalten . Die Eintraege muessen die Form "YYMMDD" haben.
#                 source : Quellspalte
#                 dest   : Zielspalte, die transformierte Werte enthaelt
#                 return  0, falls die Transformation mislingt
#                         1, sonst 
#----------------------------------------------------------------------------
#
proc coldate2real {source dest} {

   # source und dest als "called by reference" realisieren
   #
   upvar $source s
   upvar $dest d

   # Alle Elemente der Urbildspalte $source transformieren
   # 
   set d ""
   foreach date $s {
   
      if {[getdate $date real]} {

         # Transformation geglueckt =>
         # Element an Liste anhaengen
         #
         lappend d $real
      } {
         # Transformation nicht geglueckt
         # 
         return 0
      }
    } ;# foreach date

  # Transformation der Spalte geglueckt
  #
  return 1
}

#----------------------------------------------------------------------------
# tabdate2real {} transformiert eine Tabelle, die Datumseintraege enthaelt auf
#           eine zugehoerige reelle Tabelle. Dabei wird erwartet, dass die
#           gewaehlten Urbildspalten nur Datumseintraege Form "YYMMDD"
#           enthalten. Tritt waehrend der Konvertierung ein Fehler auf,
#           so wird der Vorgang abgebrochen und eine entsprechende Fehler-
#           meldung ausgegeben. Die Einstellungen des Zieldatensatzes bleiben
#           erhalten. 
#           source : Quelldatensatz
#           dest   : Zieldatensatz 
#----------------------------------------------------------------------------
#
proc tabdate2real {source dest} {

   # Diagrammdatensatz sichtbar machen
   #
   global diagdata

   # Alle gewaehlten Urbildspalten transformieren
   #
   set i 0

   foreach index $diagdata($dest.curvelist) {
   
    if {[expr $i % 2] == 0} {
      
      # Urbildspale gefunden 
      #
      if {[coldate2real diagdata($source.col$index) col]} {

         # Transformation geglueckt =>
         # transformierte Spalte in den aktuellen Datensatz einfuegen 
         #
         set diagdata($dest.col$index) $col
   
       } {
         # Transformation abgebrochen
         # Fehlermeldung ausgeben
         #
         set diagdata($dest.status) "undefined"
         refreshwindows $dest .
         set coltitle $diagdata($dest.coltitle$index)
         error "The column \"$coltitle\" contains a value that is not a\
                valid date. Please use the option\
                \"Data: select curves...\" to define valid relations OR\
                  change domain."

      } ;# if Spalte transformierbar
    } ;# if Urbildspalte gefunden 
    incr i
  } ;# foreach index

  # transformation geglueckt => Datensatz als definiert erklaeren
  #
  set diagdata($dest.status) "defined"
}

#----------------------------------------------------------------------------
# gettime {} testet, ob der uebergebene String ein gueltiges Zeitformat in
#            der Form "HHMM" ist und berechnet gegebenenfalls die Unixzeit;
#            fehlt HH, so wird "00MM" angenommen.
#            (Zeitzone GMT, Datum 1.1.'70 angenommen).
#            time     : String, der zu ueberpruefen ist
#            unixtime : zugehoerige Unixzeit, falls der String gueltig ist
#            return   : 1, falls der String ein gueltiges Zeitformat enthaelt
#                       0, sonst
#----------------------------------------------------------------------------
#
proc gettime {time unixtime} {

     # unixtime als "called by reference" realisieren
     #
     upvar $unixtime utime

     if {[isint $time]} {

        # Zeitwert mit vorangestellten Nullen auf eine 4stellige Zahl
        # erweitern
        #
        set time [format "%04d" $time]

        # Zeit in hh und mm zerlegen
        #
        set hh [crange $time 0 1]
        set mm [crange $time 2 3]

        # 24.00 Uhr auf 00.00 Uhr setzen; dafuer 02/01/70 als Tag annehmen
        #
        if {$hh == 24} {
           set hh 00
           set day "01/02/70"
        } {
           set day "01/01/70"
        }

        # Unixzeit berechnen
        #
        set rflag [catch {set utime [convertclock "$day $hh:$mm" GMT]}]
        return [expr 1-$rflag] ;# rflag ist 1, falls Konvertierung mislingt
    }

    # Konvertierung nicht erfolgreich
    #
    return 0
}

#----------------------------------------------------------------------------
# coltime2real {} transformiert Zeiteintraege einer Spalte in
#                 reele Spalten . Die Eintraege muessen die Form "HHMM" haben.
#                 source : Quellspalte
#                 dest   : Zielspalte, die transformierte Werte enthaelt
#                 return  0, falls die Transformation mislingt
#                         1, sonst 
#----------------------------------------------------------------------------
#
proc coltime2real {source dest} {

   # source und dest als "called by reference" realisieren
   #
   upvar $source s
   upvar $dest d

   # Alle Elemente der Urbildspalte $source transformieren
   # 
   set d ""
   foreach time $s {
 
      if {[gettime $time real]} {

         # Transformation geglueckt => Element an Liste anhaengen
         #
         lappend d $real
      } {
         # Transformation nicht geglueckt
         # 
         return 0
      }
    } ;# foreach date

  # Transformation der Spalte geglueckt
  #
  return 1
}

#----------------------------------------------------------------------------
# tabtime2real {} transformiert eine Tabelle, die Datumseintraege enthaelt auf
#           eine zugehoerige reelle Tabelle. Dabei wird erwartet, dass die
#           gewaehlten Urbildspalten nur Datumseintraege Form "YYMMDD"
#           enthalten. Tritt waehrend der Konvertierung ein Fehler auf,
#           so wird der Vorgang abgebrochen und eine entsprechende Fehler-
#           meldung ausgegeben. Die Einstellungen des Zieldatensatzes bleiben
#           erhalten. 
#           source : Quelldatensatz
#           dest   : Zieldatensatz 
#----------------------------------------------------------------------------
#
proc tabtime2real {source dest} {

   # Diagrammdatensatz sichtbar machen
   #
   global diagdata

   # Alle gewaehlten Urbildspalten transformieren
   #
   set i 0
   foreach index $diagdata($dest.curvelist) {
   
    if {[expr $i % 2] == 0} {
      
      # Urbildspale gefunden 
      #
      if {[coltime2real diagdata($source.col$index) col]} {

         # Transformation geglueckt =>
         # transformierte Spalte in den aktuellen Datensatz einfuegen 
         #
         set diagdata($dest.col$index) $col
       } {
         # Transformation abgebrochen
         # Fehlermeldung ausgeben
         #
         set diagdata($dest.status) "undefined"
         refreshwindows $dest .
         set coltitle $diagdata($dest.coltitle$index)
         error "The column \"$coltitle\" contains a value that is not a\
                valid time. Please use the option\
                \"Data: select curves...\" to define valid relations OR\
                change domain."

      } ;# if Spalte transformierbar
    } ;# if Urbildspalte gefunden 
    incr i
  } ;# foreach index

  # transformation geglueckt => Datensatz als definiert erklaeren
  #
  set diagdata($dest.status) "defined"
}

#----------------------------------------------------------------------------
# randomcolor {} Funktion, die eine RGB - Farbe Pseudozufaellig generiert
#                curve : Index der Kurve, fuer die die Farbe generiert wird
#                        (wird als Startwert benutzt)
#                return  RGB-Farbe in der Form #RGB ,wobei R,G,B jeweils 
#                        zweistellige hexadezimale Zahlen sind
#---------------------------------------------------------------------------- 
#
proc randomcolor {curve} {

  # Hexadezimale RGB Farbwerte pseudozufaellig generieren
  #
  random seed [int $curve]
  append r [format %X [random 0xff]] 0
  append g [format %X [random 0xff]] 0
  append b [format %X [random 0xff]] 0
  set r [string range $r 0 1]
  set g [string range $g 0 1]
  set b [string range $b 0 1]
  return #$r$g$b  
}

#-----------------------------------------------------------------------------
# putcolor {} setzt die Farbe fuer Kurve mit Spaltenindex $curve in die
#             Farbliste, falls die Farbe fuer diese Kurve noch nicht gesetzt
#             ist
#             path     : widget Pfad des Diagramms
#             color    : zu setzende Farbe
#             curve    : Index der zugehoerigen Kurve
#-----------------------------------------------------------------------------
#
proc putcolor {path curve color} {

  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  set col [lsearch $diagdata($path.colors) "$curve*"]
  if {$col == -1} {
     # Farbe fuer die Kurve mit Index $curve ist noch nicht gespeichert
     # => Index und Farbe an Liste anhaengen
     #
     lappend diagdata($path.colors) "$curve $color"
  } {
    set colset [lindex $diagdata($path.colors) $col]
       # eine Farbe fuer Kurve mit Index $col ist gespeichert
       # => dieses Listenelement mit dem aktuellen Index/Farb-Paar austauschen
       #
       lvarpop diagdata($path.colors) $col "$curve $color" 
  }
}

#---------------------------------------------------------------------------- 
# getcolor {} liefert die Farbe einer Kurve zurueck 
#             path     : Widget Pfad des Diagramms
#             curve    : Index der zugehoerigen Kurve
#             return   : Farbstring
#-----------------------------------------------------------------------------
#
proc getcolor {path curve} {

   # Diagrammdaten sichtbar machen
   #
   global diagdata

   set col [lsearch $diagdata($path.colors) "$curve*"]
   if {$col == -1} {
      # fuer die Kurve mit Index $curve ist noch keine Farbe gespeichert
      # => Farbe pseudozufaellig generieren
      #
      return [randomcolor $curve]
   } {
      # die angeforderte Farbe ist in der Farbliste =>
      # curve'te Farbe zurueckgeben
      #
      set colset [lindex $diagdata($path.colors) $col]
      return [lindex $colset 1]
  }
}

#-----------------------------------------------------------------------------
# putcolor2 {} setzt die Farbe fuer die Kurve, die aus $dom $cod gebildet wird,
#              falls die Farbe fuer diese Kurve noch nicht gesetzt ist
#             path     : widget Pfad des Diagramms
#             dom      : Zeiger auf die Urbildspalte
#             cod      : Zeiger auf die Bildspalte
#             color    : zu setzende Farbe
#-----------------------------------------------------------------------------
#
proc putcolor2 {path dom cod color} {

  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  set col [lsearch $diagdata($path.colors) "$dom $cod*"]
  if {$col == -1} {
     # Farbe fuer die Kurve $com $cod ist noch nicht gespeichert
     # => Index und Farbe an Liste anhaengen
     #
     lappend diagdata($path.colors) "$dom $cod $color"
  } {
     # eine Farbe fuer Kurve $dom $cod ist gespeichert
     # => dieses Listenelement mit dem aktuellen Index/Index/Farb Tripple
     #     austauschen
     #
     lvarpop diagdata($path.colors) $col "$dom $cod $color" 
  }
}

#-----------------------------------------------------------------------------
# getcolor2 {} liefert die Farbe der Kurve, die aus $dom $cod gebildet wird
#              (wird z.B. fuer den Modus domainreal verwendet)
#              path   : Widget Pfad des Diagramms
#              dom    : Zeiger auf die Urbildspalte
#              cod    : Zeiger auf die Bildspalte
#              return : Zugehoerige Farbe
#-----------------------------------------------------------------------------
#
proc getcolor2 {path dom cod} {

   # Diagrammdaten sichtbar machen
   #
   global diagdata

   # Suche das Element der Index/Farbliste mit Praefix $dom $cod
   #
   set col [lsearch $diagdata($path.colors) "$dom $cod*"]
   if {$col == -1} {
      # fuer die Kurve $dom $cod ist noch keine Farbe gespeichert
      # => Farbe pseudozufaellig generieren 
      # eindeutigen Startwert in Abhaengigkeit von dom und cod (ohne die
      # aktuelle Spaltenanzahl der Tabelle mit einzubeziehen) fuer 
      # Zufallsfarbgenerierung ermitteln
      # benutze hierzu eine Diagonalisierungsfunktion:
      #
      set k [expr $dom+$cod]
      set seek [expr $k*($k+1)/2+$cod]

      # Zufallsfarbe mit Startwert seek zurueckgeben
      #
      return [randomcolor $seek]
   } {
      # die angeforderte Farbe ist in der Farbliste =>
      # Farbe fuer Kurve $dom $cod zurueckgeben 
      #
      set colset [lindex $diagdata($path.colors) $col]
      return [lindex $colset 2]
   }
}
 
#----------------------------------------------------------------------------
# partitionon {} partitioniert die Spalte i des Diagrammdatensatzes in
#                Listen mit definierten Werten;
#                $diagdata($dp.undefval) gilt als undefinierter Wert
#                dp     : Pfad des zugehoerigen Diagramms
#                col    : zu partitionierende Spalte
#                listi  : Liste von Anfangsindizees der Definitionsstuecke
#                listl  : Laenge der gekuerzten Liste
#                fromx  : startindex fuer Partitionierung
#                tox    : endindex fuer Partitionierung
#                return : Liste von Listen definierter Werte
#----------------------------------------------------------------------------
#
proc partitionon {dp col listi listl {fromx 0} {tox "end"}} {

  # Diagrammdaten sichtbar machen
  #
  global diagdata

  # Zugriff "called by reference" fuer ilist und ll
  #
  upvar $listi ilist
  upvar $listl llist

  # Liste der Anfangsindizees definierter Stuecke initialisieren
  #
  set ilist ""

  # Liste der definierten Werte initialisieren
  #
  set plist ""

  # Anfangsindex des ersten definierten Stuecks initialisieren
  #
  set il 0

  # Ausschnitt der zu partitionierenden Liste kopieren
  #
  set ylist [lrange $diagdata($dp.col$col) $fromx $tox]

  # Laenge der abgeschnittenen Liste ermitteln
  #
  set llist [llength $ylist]

  while {$ylist != ""} {

        # Index eines undefinierten Wertes finden
        #
        set index [lsearch $ylist "$diagdata($dp.undefval)"]

        # ylist bis index abschneiden und abgeschnittenen Elemente als 
        # Liste an die Ergebnisliste anhaengen;
        #
        if {$index == -1} {
           # keinen undefinierten Wert gefunden; Gesamtliste uebernehmen
           # 
           lappend plist $ylist
           
           # Anfangsindex des letzten definierten Stuecks in ilist
           # 
           lappend ilist $il  
 
           # schleife verlassen
           #
           break
        } {
           # gefundenen definierten Abschnitt auslesen 
           #
           set l [lrange $ylist 0 [expr $index-1]]
           
           if {$l != ""} {
               # definierter Abschnitt ist nicht leer =>

               # gefundenen Abschnitt in Ergebnisliste haengen
               #
               lappend plist [lrange $ylist 0 [expr $index-1]]
               
               # AnfangsIndex dieses definierten Bereiches in ilist
               #
               lappend ilist $il
 
               # AnfangsIndex erhoehen
               #
               set il [expr $il+$index+1]
           } { 
               # definierter Bereich ist leer => nur AnfangsIndex erhoehen
               #
               incr il
           } ; # if $l != ""
           # gefundenen Abschnitt aus der Gesamtliste loeschen
           #
           set ylist [lreplace $ylist 0 $index]
       } ; # if $index...
  } ; # while
return  $plist 
}

#-----------------------------------------------------------------------------
# partitionoff {} Routine, die die Spalte i des Diagrammdatensatzes nicht
#                partitioniert zurueckgibt
#                    dp     : Pfad des zugehoerigen Diagramms
#                    col    : zu partitionierende Spalte
#                    listi  : Liste von Anfangsindizees der Definitionsstuecke
#                    return : Liste von Listen definierter Werte
#-----------------------------------------------------------------------------
#
proc partitionoff {dp col listi listl {dummy ""} {dummy ""}} {

  # Diagrammdaten sichtbar machen
  #
  global diagdata
 
  # Zugriff "called by reference" fuer ilist und llist
  #
  upvar $listi ilist
  upvar $listl llist

  # ilist setzen
  #
  set ilist 0

  # llist setzen
  #
  set llist [llength $diagdata($dp.col$col)]

  # col`te Spaltenliste zurueckgeben
  #
  return [list $diagdata($dp.col$col)]
}


#-----------------------------------------------------------------------------
# scalecurveindex {} skaliert eine DatenKurve relativ zur gesetzten
#              Zeichenflaeche fuer den Modus "domainint"
#              path   : identifiziert das zugehoerige Diagramm
#              col    : Index der Kurve
#-----------------------------------------------------------------------------
#
proc scalecurveindex {path col} { 

   # Diagrammvariablen sichtbar machen und umschreiben
   #
   global diagdata
   set xmin        [double $diagdata($path.fromx)]
   set xmax        [double $diagdata($path.tox)]
   set ymin        [double $diagdata($path.fromy)]
   set ymax        [double $diagdata($path.toy)]
   set clipxaxis   [double $diagdata($path.clipxaxis)]
   set clipyaxis   [double $diagdata($path.clipyaxis)]
   set cliporiginx [double $diagdata($path.cliporiginx)]
   set cliporiginy [double $diagdata($path.cliporiginy)]
   set originx     [double $diagdata($path.originx)]
   set yaxis       [double $diagdata($path.yaxis)]
   set rownum      [double $diagdata($path.rownum)]

   # es wird linear umscaliert, 
   # wenn vorher gilt :  xmin    <= xw <= xmax  
   # dann gilt nachher:  originx <= xw <= xaxis
   # bzw vorher       :  ymin    <= yw <= ymax
   #     nachher      :  originy >= yw >= yaxis 

   # erstellen einer Liste von Listen definierter Werte von $xmin bis $xmax
   # 
   set ylists [eval $diagdata($path.partition) $path $col ilist llist\
                      [int $xmin] [int $xmax]]

   # Ausgabeliste initialisieren
   #
   set diagdata($path.colscaled$col) ""

   # Markenliste der undefinierten Punkte initialisieren 
   #
   set diagdata($path.undefmarks$col) "" 

   # Zeiger auf Beginn eines undefinierten Stueckes initialisieren
   #
   set undefind 0 

   # Liste definierter Kurvenstuecke um ein leeres Element erweitern, damit
   # die nachfolgende Schleife auch undefinierte Werte nach dem letzten
   # definierten Stueck bearbeitet, entsprechend Indexliste um Zeiger auf
   # letztes Element erweitern
   #
   lappend ylists {}
   lappend ilist $llist
   
   # X-laufindex initialisieren
   # 
   if {$xmin < 0} {
      set xi 0
   } {
      set xi $xmin
   }
 
   foreach ylist $ylists {

     # Liste eines definierten Stuecks initialisieren
     #
     set coords ""

     # Index des naechsen definierten Stuecks ermitteln 
     #
     set il [lvarpop ilist]

     # Berechne Markenpositionen fuer undefinierte Werte---------------------
     #
     loop i $undefind $il {

        # scaliere xw so, dass xw elem [0;1]
        #
        set xw [expr ($xi-$xmin)/($xmax-$xmin)]

        # scaliere, so dass xw elem [clipxaxis;cliporiginx] ist
        #
        set xw [expr $xw*($clipxaxis-$cliporiginx)+$cliporiginx]

        # Markenposition an Stelle (aktuellen Xwert, Oberkante des Diagramms)
        #
        lappend diagdata($path.undefmarks$col) [expr $xw+$originx] $yaxis  

        # aktuellen xwert erhoehen
        #
        set xi [expr $xi+1]
     }
     # Zeiger auf den Anfang des naechsten undefinierten Bereich
     #
     set undefind [expr $il+[llength $ylist]]

     # Skaliere Werte eines definierten Kurvenstuecks-----------------------
     #
     foreach yw $ylist {

        # scaliere, so dass xw,yw elem [0;1]
        #
        set xw [expr ($xi-$xmin)/($xmax-$xmin)]
        set yw [expr ($yw-$ymin)/($ymax-$ymin)]

        # scaliere, so dass xw elem [clipxaxis;cliporiginx] und
        #                   yw elem [cliporiginy;clipyaxis] ist 
        #                 
        set xw [expr $xw*($clipxaxis-$cliporiginx)+$cliporiginx]
        set yw [expr $yw*($clipyaxis-$cliporiginy)+$cliporiginy]

        # scalierte Werte in Aussgabeliste
        #
        lappend  coords $xw $yw

        # aktuellen xindex erhoehen
        #
        set xi [expr $xi+1]
     }
     # Ausgabeliste im Diagrammdatensatz sichern
     #
     lappend diagdata($path.colscaled$col) $coords 
   } ; # foreach ylists
}

#-----------------------------------------------------------------------------
# scalecurvereal {} skaliert eine DatenKurve relativ zur Zeichenflaeche fuer
#                den Modus "domainreal" 
#                path   : identifiziert das zugehoerige Diagramm
#                cold   : Index der Urbildspalte 
#                colc   : Index der Bildspalte
#-----------------------------------------------------------------------------
#
proc scalecurvereal {path cold colc} { 

   # Diagrammvariablen sichtbar machen und umschreiben
   #
   global diagdata
   set xmin        [double $diagdata($path.fromx)]
   set xmax        [double $diagdata($path.tox)]
   set ymin        [double $diagdata($path.fromy)]
   set ymax        [double $diagdata($path.toy)]
   set clipxaxis   [double $diagdata($path.clipxaxis)]
   set clipyaxis   [double $diagdata($path.clipyaxis)]
   set cliporiginx [double $diagdata($path.cliporiginx)]
   set cliporiginy [double $diagdata($path.cliporiginy)]
   set yaxis       [double $diagdata($path.yaxis)]
   set originx     [double $diagdata($path.originx)]

   # es wird linear umscaliert, 
   # wenn vorher gilt :  xmin    <= xw <= xmax  
   # dann gilt nachher:  originx <= xw <= xaxis
   # bzw vorher       :  ymin    <= yw <= ymax
   #     nachher      :  originy >= yw >= yaxis 

   # erstellen einer Liste von Listen definierter Bildwerte
   # 
   set ylists [eval $diagdata($path.partition) $path $colc ilist llist]

   # Ausgabeliste initialisieren
   #
   set diagdata($path.colscaled$cold) ""

   # Markenliste der undefinierten Punkte initialisieren 
   #
   set diagdata($path.undefmarks$cold) "" 

   # Zeiger auf Beginn eines undefinierten Stueckes initialisieren
   #
   set undefind 0
 
   # Liste definierter Kurvenstuecke um ein leeres Element erweitern, damit
   # die nachfolgende Schleife auch undefinierte Werte nach dem letzten
   # definierten Stueck bearbeitet, entsprechend Indexliste um Zeiger auf
   # letztes Element erweitern
   #
   lappend ylists {}
   lappend ilist $llist

   foreach ylist $ylists {
   
     # definiertes Stueck und zugehoeriges Stueck der Urbildspalte ermitteln
     #
     set il [lvarpop ilist]
     set yl [llength $ylist]
     set xlist [lrange $diagdata($path.col$cold) $il [expr $il+$yl-1]]

     if {[llength $xlist] <= $yl} {

        # in diesem definierten Kurvenstueck sind weniger Urbildelemente als
        # Bildelemente enthalten => Bildspalte entsprechend kuerzen     
        # 
        set xl [llength $xlist]
        set ylist [lrange $ylist 0 [expr $xl-1]]
     }

     # Berechne Markenpositionen fuer undefinierte Werte----------------------
     #
     set ul [lrange $diagdata($path.col$cold) $undefind [expr $il-1]]
     foreach xw $ul {

        # Berechne xw, so dass xw elem [0;1]
        #
        set xw [expr ($xw-$xmin)/($xmax-$xmin)]

        # Berechne xw, so dass xw elem [cliporiginx;clipxaxis]
        #
        set xw [expr $xw*($clipxaxis-$cliporiginx)+$cliporiginx]

        # Markenposition an Stelle (aktuellen Xwert, Oberkante des Diagramms)
        #
        lappend diagdata($path.undefmarks$cold) [expr $xw+$originx] $yaxis
     }

     # Zeiger auf den Anfang des naechsten undefinierten Bereich
     #
     set undefind [expr $il + [llength $ylist]]

     # Liste eines definierten Stuecks initialisieren
     #
     set coords ""

     # Skaliere Werte eines definierten Kurvenstuecks-----------------------
     #
     foreach yw $ylist {

        # zugehoerigen xw ermitteln 
        # 
        set xw [lvarpop xlist]

        # scaliere, so dass xw,yw elem [0;1]
        #
        set yw [expr ($yw-$ymin)/($ymax-$ymin)]
        set xw [expr ($xw-$xmin)/($xmax-$xmin)]

        # scaliere, so dass yw elem [clipyaxis;cliporiginy] ist
        #           und     xw elem [cliporiginx,clipxaxis] ist
        #
        set yw [expr $yw*($clipyaxis-$cliporiginy)+$cliporiginy]
        set xw [expr $xw*($clipxaxis-$cliporiginx)+$cliporiginx]

        # scalierte Werte in Aussgabeliste
        #
        lappend  coords $xw $yw
     }
 
     # Ausgabeliste im Diagrammdatensatz sichern
     #
     lappend diagdata($path.colscaled$cold) $coords 
   }
}
