# $Id: pixmap.tcl,v 1.1 1993/03/12 02:28:10 sls Exp $
#
# $Log: pixmap.tcl,v $
# Revision 1.1  1993/03/12  02:28:10  sls
# Initial revision
#
#
# pixmap -- a color pixmap editor in Tcl.
#
# This program is in the public domain.  Do what you want with this code.
# Of course, this code comes with absolutely no warranty.
#
# Author: Sam Shen (sls@aero.org)
#
if {$argc != 1} {
    puts stderr "Ooops! pixmap has not been installed correctly."
    exit 1
}
set pixmap_library $argv
if {$tkVersion < 3.1} {
    puts stderr "Ooops! pixmap requires Tk3.1 or later."
    puts stderr "You are running Tk[set tkVersion]"
}
source $pixmap_library/class.tcl

set fileName none
wm title . "Pixmap Editor:$fileName"
wm iconname . "Pixmap Editor"
wm minsize . 100 100

#
# FileChooser implements a simple file chooser.
#
class FileChooser {
    member w .filechooser
    member ok 0
    member read 1
    member write 0
    member file {}
    method run {prompt {read 1} {write 0} {default {}}} {
	set w [getmember w]
	setmember read $read
	setmember write $write
	catch {destroy $w}
	toplevel $w
	frame $w.e
	label $w.e.l -text $prompt -anchor w
	entry $w.e.e -relief sunken -width 40
	if {$default != {}} {
	    $w.e.e insert 0 $default
	}
	pack append $w.e \
	    $w.e.l "left frame w padx 2" \
	    $w.e.e "right fillx expand"
	bind $w.e.e <Return> "FileChooser:ok $this"
	frame $w.buts
	button $w.ok -text "Ok" -command "FileChooser:ok $this"
	button $w.cancel -text "Cancel" -command "FileChooser:cancel $this"
	pack append $w.buts \
	    $w.ok "left fillx expand" \
	    $w.cancel "left fillx expand"
	label $w.status -anchor w -bd 2 -relief sunken \
	    -font -*-Helvetica-Medium-R-*-*-14-*
	pack append $w \
	    $w.e "top fillx expand" \
	    $w.buts "fillx pady .5c" \
	    $w.status "fillx"
	grab set $w
	tkwait window $w
	if [getmember ok] {
	    return [getmember file]
	}
	return {}
    }
    method ok {} {
	set w [getmember w]
	set file [$w.e.e get]
	setmember file $file
	if [getmember read] {
	    if [file readable $file] {
		setmember ok 1
		destroy $w
	    } else {
		$w.status config -text "Can't read file $file"
	    }
	    return
	}
	if [getmember write] {
	    if ![catch {open $file w} f] {
		close $f
		setmember ok 1
		destroy $w
	    } else {
		$w.status config -text "Can't write to file $file"
	    }
	    return
	}
    }
    method cancel {} {
	setmember ok 0
	destroy [getmember w]
    }
	
}

#
# Buttons implements the buttons on the main window.
#
class Buttons {
    member w {}
    method create {w} {
	setmember w $w
	frame $w
	button $w.new -text "New" \
	    -command "Buttons:new $this"
	button $w.save -text "Save Pixmap" \
	    -command "Buttons:savePixmap $this"
	button $w.load -text "Load Pixmap" \
	    -command "Buttons:loadPixmap $this"
	button $w.savep -text "Save Palette" \
	    -command "Buttons:savePalette $this"
	button $w.loadp -text "Load Palette" \
	    -command "Buttons:loadPalette $this"
	button $w.quit -text "Quit" -command { destroy . }
	pack append $w \
	    $w.new "left fill expand" \
	    $w.save "left fill expand" \
	    $w.load "left fill expand" \
	    $w.savep "left fill expand" \
	    $w.loadp "left fill expand" \
	    $w.quit "left fill expand"
	setmember filechooser [FileChooser]
	return $w
    }
    method new {} {
	global canvas
	EditorCanvas:new $canvas
    }
    method savePixmap {} {
	global fileName
	set file [FileChooser:run [getmember filechooser] \
		  "Save pixmap to:" 0 1 $fileName]
	if {$file != {}} {
	    global canvas
	    set fileName $file
	    wm title . "Pixmap Editor:$fileName"
	    EditorCanvas:save $canvas $file
	}
    }
    method loadPixmap {} {
	global fileName
	set file [FileChooser:run [getmember filechooser] \
		  "Load pixmap from:" 1 0 $fileName]
	if {$file != {}} {
	    global canvas
	    set fileName $file
	    wm title . "Pixmap Editor:$fileName"
	    EditorCanvas:load $canvas $file
	}
    }
    method savePalette {} {
	set file [FileChooser:run [getmember filechooser] \
		  "Save current palette to:" 0 1]
	if {$file != {}} {
	    set f [open $file "w"]
	    global palette
	    for {set i 0} {$i < 16} {incr i} {
		set color [ColorPalette:getColor $palette $i]
		puts $f "set defaultColors($i) $color"
	    }
	    close $f
	}
    }
    method loadPalette {} {
	set file [FileChooser:run [getmember filechooser] \
		  "Load palette from:"]
	if {$file != {}} {
	    global defaultColors
	    source $file
	    global palette
	    ColorPalette:loadDefaultColors $palette
	}
    }
}

pack append . [Buttons:create [Buttons] .buttons] "top fillx"

#
# bindEntry does better bindings for an entry.
#
proc bindEntry args {
    foreach e $args {
	bind $e <Return> "focus none"
    }
}

#
# Parameters implements the various entry widgets below the buttons.
#
set _pixmapHeight 20
set _pixmapWidth 20
set _zoom 15
set gridOn 1
set savePPM 0
class Parameters {
    member w {}
    method create {w} {
	setmember w $w
	frame $w
	frame $w.1
	label $w.lw -text "Width:"
	entry $w.ew -bd 2 -relief sunken -width 10 -textvariable _pixmapHeight
	label $w.lh -text "Height:"
	entry $w.eh -bd 2 -relief sunken -width 10 -textvariable _pixmapWidth
	label $w.lz -text "Zoom:"
	entry $w.ez -bd 2 -relief sunken -width 10 -textvariable _zoom
	bindEntry $w.ew $w.eh $w.ez
	button $w.apply -text "Apply" \
	    -command "Parameters:apply $this"
	pack append $w.1 \
	    $w.lw "left" \
	    $w.ew "left fillx expand" \
	    $w.lh "left pady 5 frame e" \
	    $w.eh "left fillx expand" \
	    $w.lz "left pady 5 frame e" \
	    $w.ez "left fillx expand" \
	    $w.apply "right pady 5"
	frame $w.2
	checkbutton $w.gr -variable gridOn -text "Gridding" -relief flat \
	    -command "Parameters:toggleGridding $this"
	checkbutton $w.sppm -variable savePPM -text "Save in PPM format" \
	    -relief flat
	label $w.fl -text "File:"
	entry $w.fn -bd 2 -width 15 -textvariable fileName -relief sunken
	bind $w.fn <Return> "focus none"
	label $w.x -font *-Courier-Medium-R-Normal-*-120-* \
	    -textvariable currentX -width 7
	label $w.y -font *-Courier-Medium-R-Normal-*-120-* \
	    -textvariable currentY -width 7
	pack append $w.2 \
	    $w.gr "left pady 5 frame w" \
	    $w.sppm "left pady 5 frame w" \
	    $w.fl "left pady 5 frame e" \
	    $w.fn "left pady 5 frame w" \
	    $w.y "right pady 5" \
	    $w.x "right pady 5"
	pack append $w $w.1 "top fillx" $w.2 "top fillx"
	return $w
    }
    method apply {} {
	global canvas _pixmapHeight _pixmapWidth _zoom
	EditorCanvas:setParameters $canvas $_pixmapHeight $_pixmapWidth \
	    $_zoom
    }
    method updateParameters {} {
	global canvas _pixmapHeight _pixmapWidth _zoom
	in $canvas {
	    set _pixmapHeight [getHeight]
	    set _pixmapWidth [getWidth]
	    set _zoom [getZoom]
	}
    }
    method toggleGridding {} {
	global gridOn canvas
	EditorCanvas:setGridding $canvas $gridOn
    }
	
}

pack append . [Parameters:create [set params [Parameters]] .params] "top fillx"
	
#
# ColorEditor is ripped off from the Tk demo program tcolor.
#
set colorSpace hsb
class ColorEditor {
    member red 65535
    member green 0
    member blue 0
    member color #ffff00000000
    member updating 0
    member name ""
    member w .ceditor
    member ok 0
    method run {{color gray}} {
	set w [getmember w]
	catch {destroy $w}
	toplevel $w
	wm title $w "Color Editor"
	frame $w.buttons
	radiobutton $w.rgb -text "RGB color space" -variable colorSpace \
	    -value rgb  -relief flat \
	    -command "ColorEditor:changeColorSpace $this rgb"
	radiobutton $w.cmy -text "CMY color space" -variable colorSpace \
	    -value cmy  -relief flat \
	    -command "ColorEditor:changeColorSpace $this cmy"
	radiobutton $w.hsb -text "HSB color space" -variable colorSpace \
	    -value hsb  -relief flat \
	    -command "ColorEditor:changeColorSpace $this hsb"
	button $w.ok -text "Ok" -command "ColorEditor:ok $this"
	button $w.cancel -text "Cancel" -command "ColorEditor:cancel $this"
	pack append $w.buttons \
	    $w.rgb "left padx 4" \
	    $w.cmy "left padx 4" \
	    $w.hsb "left padx 4" \
	    $w.cancel "right padx 4 pady 2" \
	    $w.ok "right padx 4 pady 2"
	frame $w.left
	foreach i {1 2 3} {
	    frame $w.left$i
	    label $w.label$i
	    scale $w.scale$i -from 0 -to 1000 -length 10c -orient horizontal \
		-command "ColorEditor:scaleChanged $this"
	    button $w.up$i -width 2 -text + \
		-command "ColorEditor:inc $this $i 1"
	    button $w.down$i -width 2 -text - \
		-command "ColorEditor:inc $this $i -1"
	    pack append $w.left$i \
		$w.label$i {top frame w} \
		$w.down$i {left padx .5c} \
		$w.scale$i left \
		$w.up$i {left padx .5c}
	    pack append $w.left $w.left$i "top expand"
	}
	frame $w.right
	frame $w.swatch -width 2c -height 5c -background $color
	label $w.value -text $color -width 13 \
	    -font -Adobe-Courier-Medium-R-Normal-*-120-*
	pack append $w.right \
	    $w.swatch {top expand fill} \
	    $w.value {bottom pady .5c}
	pack append $w \
	    $w.buttons "top fillx" \
	    $w.left "left expand filly" \
	    $w.right "right padx .5c pady .5c frame s"
	loadNamedColor $color
	changeColorSpace hsb
	grab set $w
	tkwait window $w
	if [getmember ok] {
	    return [getmember color]
	} else {
	    return {}
	}
    }
    method cancel {} {
	setmember ok 0
	destroy [getmember w]
    }
    method ok {} {
	setmember ok 1
	destroy [getmember w]
    }
    method inc {i inc} {
	set w [getmember w]
	$w.scale$i set [expr [$w.scale$i get]+$inc]
    }
    method scaleChanged args {
	if [getmember updating] {
	    return
	}
	global colorSpace
	set w [getmember w]
	if {$colorSpace == "rgb"} {
	    set red   [format %.0f [expr [$w.scale1 get]*65.535]]
	    set green [format %.0f [expr [$w.scale2 get]*65.535]]
	    set blue  [format %.0f [expr [$w.scale3 get]*65.535]]
	} else {
	    if {$colorSpace == "cmy"} {
		set red   [format %.0f [expr {65535 - [$w.scale1 get]*65.535}]]
		set green [format %.0f [expr {65535 - [$w.scale2 get]*65.535}]]
		set blue  [format %.0f [expr {65535 - [$w.scale3 get]*65.535}]]
	    } else {
		set list [hsbToRgb [expr {[$w.scale1 get]/1000.0}] \
			  [expr {[$w.scale2 get]/1000.0}] \
			      [expr {[$w.scale3 get]/1000.0}]]
		set red [lindex $list 0]
		set green [lindex $list 1]
		set blue [lindex $list 2]
	    }
	}
	set color [format "#%04x%04x%04x" $red $green $blue]
	setmember color $color
	setmember red $red
	setmember green $green
	setmember blue $blue
	$w.swatch config -bg $color
	$w.value config -text $color
	update idletasks
    }
    method setScales {} {
	set red [getmember red]
	set blue [getmember blue]
	set green [getmember green]
	set w [getmember w]
	setmember updating 1
	global colorSpace
	if {$colorSpace == "rgb"} {
	    $w.scale1 set [format %.0f [expr $red/65.535]]
	    $w.scale2 set [format %.0f [expr $green/65.535]]
	    $w.scale3 set [format %.0f [expr $blue/65.535]]
	} else {
	    if {$colorSpace == "cmy"} {
		$w.scale1 set [format %.0f [expr (65535-$red)/65.535]]
		$w.scale2 set [format %.0f [expr (65535-$green)/65.535]]
		$w.scale3 set [format %.0f [expr (65535-$blue)/65.535]]
	    } else {
		set list [rgbToHsv $red $green $blue]
		$w.scale1 set [format %.0f [expr {[lindex $list 0] * 1000.0}]]
		$w.scale2 set [format %.0f [expr {[lindex $list 1] * 1000.0}]]
		$w.scale3 set [format %.0f [expr {[lindex $list 2] * 1000.0}]]
	    }
	}
	setmember updating 0
    }
    method loadNamedColor name {
	set w [getmember w]
	if {[string index $name 0] != "#"} {
	    set list [winfo rgb $w.swatch $name]
	    set red [lindex $list 0]
	    set green [lindex $list 1]
	    set blue [lindex $list 2]
	} else {
	    case [string length $name] {
		4 {set format "#%1x%1x%1x"; set shift 12}
		7 {set format "#%2x%2x%2x"; set shift 8}
		10 {set format "#%3x%3x%3x"; set shift 4}
		13 {set format "#%4x%4x%4x"; set shift 0}
		default {error "syntax error in color name \"$name\""}
	    }
	    if {[scan $name $format red green blue] != 3} {
		error "syntax error in color name \"$name\""
	    }
	    set red [expr $red<<$shift]
	    set green [expr $green<<$shift]
	    set blue [expr $blue<<$shift]
	}
	setmember red $red
	setmember green $green
	setmember blue $blue
	set color [format "#%04x%04x%04x" $red $green $blue]
	setmember color $color
	setScales
	$w.swatch config -bg $color
	$w.value config -text $name
    }
    method setLabels {l1 l2 l3} {
	set w [getmember w]
	$w.label1 config -text $l1
	$w.label2 config -text $l2
	$w.label3 config -text $l3
    }
    method changeColorSpace space {
	global colorSpace
	set colorSpace $space
	if {$space == "rgb"} {
	    setLabels Red Green Blue
	    setScales
	    return
	}
	if {$space == "cmy"} {
	    setLabels Cyan Magenta Yellow
	    setScales
	    return
	}
	if {$space == "hsb"} {
	    setLabels Hue Saturation Brightness
	    setScales
	    return
	}
    }
    method rgbToHsv {red green blue} {
	if {$red > $green} {
	    set max $red.0
	    set min $green.0
	} else {
	    set max $green.0
	    set min $red.0
	}
	if {$blue > $max} {
	    set max $blue.0
	} else {
	    if {$blue < $min} {
		set min $blue.0
	    }
	}
	set range [expr $max-$min]
	if {$max == 0} {
	    set sat 0
	} else {
	    set sat [expr {($max-$min)/$max}]
	}
	if {$sat == 0} {
	    set hue 0
	} else {
	    set rc [expr {($max - $red)/$range}]
	    set gc [expr {($max - $green)/$range}]
	    set bc [expr {($max - $blue)/$range}]
	    if {$red == $max} {
		set hue [expr {.166667*($bc - $gc)}]
	    } else {
		if {$green == $max} {
		    set hue [expr {.166667*(2 + $rc - $bc)}]
		} else {
		    set hue [expr {.166667*(4 + $gc - $rc)}]
		}
	    }
	}
	return [list $hue $sat [expr {$max/65535}]]
    }
    method hsbToRgb {hue sat value}  {
	set v [format %.0f [expr 65535.0*$value]]
	if {$sat == 0} {
	    return "$v $v $v"
	} else {
	    set hue [expr $hue*6.0]
	    if {$hue >= 6.0} {
		set hue 0.0
	    }
	    scan $hue. %d i
	    set f [expr $hue-$i]
	    set p [format %.0f [expr {65535.0*$value*(1 - $sat)}]]
	    set q [format %.0f [expr {65535.0*$value*(1 - ($sat*$f))}]]
	    set t [format %.0f [expr {65535.0*$value*(1 - ($sat*(1 - $f)))}]]
	    case $i \
		0 {return "$v $t $p"} \
		1 {return "$q $v $p"} \
		2 {return "$p $v $t"} \
		3 {return "$p $q $v"} \
		4 {return "$t $p $v"} \
		5 {return "$v $p $q"}
	    error "i value $i is out of range"
	}
    }
}

source $pixmap_library/default.pal

#
# ColorPalette implements the color palette.
#
class ColorPalette {
    member w {}
    member editor {}
    method create {w} {
	global defaultColors currentColor
	setmember w $w
	frame $w -bd 2 -relief raised
	for {set i 0} {$i < 16} {incr i} {
	    frame $w.$i
	    radiobutton $w.$i.button -width 13 \
		-variable currentColor \
		-font *-Courier-Medium-R-Normal-*-120-* \
		-anchor w
	    button $w.$i.patch -text Edit \
		-command "ColorPalette:edit $this $i"
	    pack append $w.$i \
		$w.$i.button "right frame e" \
		$w.$i.patch "left padx 4 pady 4 frame w"
	    pack append $w $w.$i "top fillx"
	}
	loadDefaultColors
	setmember editor [ColorEditor]
	return $w
    }
    method loadDefaultColors {} {
	set w [getmember w]
	global defaultColors currentColor
	for {set i 0} {$i < 16} {incr i} {
	    $w.$i.button config -text $defaultColors($i) \
		-value $defaultColors($i)
	    $w.$i.patch config -background $defaultColors($i)
	}
	set currentColor $defaultColors(0)
    }
    method edit {i} {
	set w [getmember w]
	set color [ColorEditor:run [getmember editor] \
		   [lindex [$w.$i.patch config -background] 4]]
	if {$color != {}} {
	    if {$i == 0} {
		global canvas
		EditorCanvas:setBackground $canvas $color
	    }
	    $w.$i.button config -text $color -value $color
	    $w.$i.patch config -background $color
	}
    }
    method getColor {i} {
	set w [getmember w]
	return [lindex [$w.$i.patch config -background] 4]
    }
}

pack append . \
    [ColorPalette:create [set palette [ColorPalette]] .palette] "left filly"


#
# EditorCanvas implements the canvas part of the editor.
#
class EditorCanvas {
    member c {}
    member background {}
    member gridding 1
    member zoom 15
    member width 20
    member height 20
    method create {c} {
	setmember c $c
	set Width [getmember width]
	set Height [getmember height]
	global zoom
	set zoom [getmember zoom]
	global palette
	setmember background [ColorPalette:getColor $palette 0]
	set bg [getmember background]
	canvas $c -width [expr {$zoom*$Width}] \
	    -height [expr {$zoom*$Height}] \
	    -background $bg
	drawGrid
	bind $c <1> "EditorCanvas:fillCell $this %x %y"
	bind $c <B1-Motion> "EditorCanvas:fillCell $this %x %y"
	bind $c <Motion> {
	    global zoom currentX currentY
	    set currentX "X: [format %%-3d [expr %x/$zoom]]"
	    set currentY "Y: [format %%-3d [expr %y/$zoom]]"
	}
	global palette
	bind $c <3> "EditorCanvas:fillCell $this %x %y background"
	bind $c <B3-Motion> "EditorCanvas:fillCell $this %x %y background"
	return $c
    }
    method new {} {
	set c [getmember c]
	$c delete all
	drawGrid
    }
    method setBackground {color} {
	set c [getmember c]
	setmember background $color
	$c delete $color
	$c config -background $color
    }
    method setParameters {new_width new_height new_zoom} {
	set Width [getmember width]
	set Height [getmember height]
	global zoom
	set zoom [getmember zoom]
	if {$new_width == $Width && $new_height == $Height} {
	    set c [getmember c]
	    set s [expr 1.1*$new_zoom/$zoom/1.1]
	    $c scale all 0 0 $s $s
	    setmember zoom $new_zoom
	    set zoom $new_zoom
	    return
	}
	setmember width $new_width
	setmember height $new_height
	setmember zoom $new_zoom
	set zoom $new_zoom
	new
    }
    method setWidth {w} {
	setmember width $w
    }
    method setHeight {h} {
	setmember height $h
    }
    method getWidth {} {
	getmember width
    }
    method getHeight {} {
	getmember height
    }
    method getZoom {} {
	getmember zoom
    }
    method fillCell {x y {color {}}} {
	set Width [getmember width]
	set Height [getmember height]
	set zoom [getmember zoom]
	global currentColor palette
	if {$color == {}} {
	    set color $currentColor
	} elseif {$color == "background"} {
	    set color [ColorPalette:getColor $palette 0]
	}
	set x [expr {$x/$zoom}]
	set zx [expr {$x*$zoom}]
	set y [expr {$y/$zoom}]
	set zy [expr {$y*$zoom}]
	if {$x >= $Width || $y >= $Height} return
	set c [getmember c]
	$c delete [list at $x $y]
	if {$color != [getmember background]} {
	    $c create rect [expr {$zx+1}] [expr {$zy+1}] \
		[expr {$zx+$zoom}] [expr {$zy+$zoom}] -fill $color \
		-outline $color \
		-tag [list cell $color [list at $x $y]]
	}
	update idletasks
    }
    method drawGrid {} {
	set c [getmember c]
	set Width [getmember width]
	set Height [getmember height]
	set zoom [getmember zoom]

	if ![getmember gridding] {
	    $c create rect 0 0 [expr $Height*$zoom] [expr $Width*$zoom] \
		-outline white -tag border
	    return
	}
	$c delete border
	global palette
	set zy [expr $Height*$zoom]
	for {set x 0} {$x <= $Width} {incr x} {
	    set zx [expr $x*$zoom]
	    $c create line $zx 0 $zx $zy -tag grid -fill white
	}
	set zx [expr $Width*$zoom]
	for {set y 0} {$y <= $Height} {incr y} {
	    set zy [expr $y*$zoom]
	    $c create line 0 $zy $zx $zy -tag grid -fill white
	}
    }
    method setGridding {on} {
	set gridding [getmember gridding]
	if {$on == $gridding} return
	setmember gridding $on
	if !$on {
	    [getmember c] delete grid
	}
	drawGrid
    }
    method save {file} {
	set f [open $file w]
	set Width [getmember width]
	set Height [getmember height]
	set zoom [getmember zoom]
	set c [getmember c]
	puts $f "PW $Width"
	puts $f "PH $Height"
	foreach cell [$c find withtag cell] {
	    set tags [$c gettags $cell]
	    set ndx [lsearch $tags "at *"]
	    if {$ndx == -1} {
		puts stdout "Hmm, there's a cell with no at tag: cell=$cell"
		continue
	    }
	    if {[scan [lindex $tags $ndx] "at %f %f" x y] != 2} {
		puts stdout "Can't scan cell=$cell, at=[lindex $tags $ndx]"
		continue
	    }
	    set color [lindex [$c item $cell -fill] 4]
	    puts $f "PC $x $y $color [winfo rgb $c $color]"
	}
	close $f
	global savePPM
	if $savePPM {
	    convertPixToPPM $file
	}
    }
    method load {file} {
	set c [getmember c]
	$c delete cell
	global zoom params
	set zoom [getmember zoom]
	uplevel #0 "source $file"
	Parameters:updateParameters $params
	drawGrid
    }
}

pack append . \
    [EditorCanvas:create [set canvas [EditorCanvas]] .canvas] \
    "expand fill right frame nw"

#
# PW, PH, and PC are used to load pixmaps.
#
proc definePixLoadProcs {} {
    proc PW {w} {
	global canvas
	EditorCanvas:setWidth $canvas $w
    }
    proc PH {h} {
	global canvas
	EditorCanvas:setHeight $canvas $h
    }
    proc PC {x y color r g b} {
	global canvas zoom
	EditorCanvas:fillCell $canvas [expr $x*$zoom] [expr $y*$zoom] $color
    }
}
definePixLoadProcs
    
#
# convertPixtoPPM converts our format to PPM format.
#
proc convertPixToPPM {file} {
    proc PW {w} {
	global ppm_width
	set ppm_width $w
    }
    proc PH {h} {
	global ppm_height
	set ppm_height $h
    }
    proc PC {x y color r g b} {
	global ppm_pixels
	set ppm_pixels($x/$y) "[expr $r>>8] [expr $g>>8] [expr $b>>8]"
    }
    global ppm_width ppm_height ppm_pixels
    source $file
    set f [open $file w]
    puts $f "P3"
    puts $f "$ppm_width $ppm_height"
    puts $f "255"
    for {set y 0} {$y < $ppm_height} {incr y} {
	puts $f "# row $y"
	for {set x 0} {$x < $ppm_width} {incr x} {
	    if [info exists ppm_pixels($x/$y)] {
		puts $f $ppm_pixels($x/$y)
	    } else {
		puts $f "0 0 0"
	    }
	}
    }
    close $f
    unset ppm_pixels
    definePixLoadProcs
}


