#
# $Id: args.tcl,v 1.9 1995/03/21 03:47:37 sls Exp $
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
# Command line argument handling
#

document_title args "command line argument handling" {
    These procedures parse command line arguments.
}

proc args_eval {__formals__ __argv__ __body__} {
    set __i__ 0
    foreach __arg__ $__formals__ {
	set [lindex $__formals__ $__i__] [lindex $__argv__ $__i__]
	incr __i__
    }
    eval $__body__
}

document_proc args_parse {
    parses the command line arguments in `argv'.  It works by eval-ing
    `body', then matching arguments with flags.  `body' should contain
    #args_arg#, #args_remaining#, and #args_error# commands.  Returns
    1 on success, 0 on failure.
}
proc args_parse {argv body} {
    global args_priv
    catch {unset args_priv}
    set args_priv(flags) ""
    set args_priv(error) ""
    uplevel $body
    while {[set len [llength $argv]]} {
	set flag [lindex $argv 0]
	if {[string index $flag 0] != "-"} {
	    if [info exists args_priv(remaining)] {
		args_eval argv [list $argv] $args_priv(remaining)
		return 1
	    } else {
		args_eval detail {"wrong number of arguments"} \
		    $args_priv(error)
		return 0
	    }
	}
	if ![info exists args_priv($flag,body)] {
	    args_eval detail [list "unknown flag $flag"] \
		$args_priv(error)
	    return 0
	}
	set flen [llength $args_priv($flag,formals)]
	if {$flen >= $len} {
	    args_eval detail [list "wrong number of arguments to $flag"] \
		$args_priv(error)
	    return 0
	}
	args_eval [concat flag $args_priv($flag,formals)] \
	    [concat $flag [lrange $argv 1 $flen]] \
	    $args_priv($flag,body)
	set argv [lrange $argv [expr $flen + 1] end]
    }
    return 1
}

document_proc args_arg {
    defines a flag.  It should be invoked in a #args_parse# body.
    `flag' names the flag, it should start with a \-.  `description'
    should describe the flag.  `formals' defines the arguments of the
    flag.  When a flag is matched, there should be an argument for each
    element of `formals'.  The arguments are then bound to `formals',
    the variable #flag# is set to the flag, and `body' is eval-ed.
    (`body' will not be executed at the same level as the one in
     which #args_parse# was invoked, so global variables will have
     to be declared as such.)
}
proc args_arg {flag formals description body} {
    global args_priv
    lappend args_priv(flags) $flag
    set args_priv($flag,description) $description
    set args_priv($flag,formals) $formals
    set args_priv($flag,body) $body
}

document_proc args_remaining {
    invokes `body' with any remaining command line arguments.  The variable
    #argv# will hold the remaining arguments.
}
proc args_remaining {description body} {
    global args_priv
    set args_priv(remaining_description) $description
    set args_priv(remaining) $body
}

document_proc args_usage {
    prints out on stderr usage information for the set of flags that
    #args_parse# is processing.
}
proc args_usage {} {
    global args_priv
    set max 0
    foreach flag $args_priv(flags) {
	if {[set len [string length "$flag $args_priv($flag,formals)"]] > $max} {
	    set max $len
	}
    }
    if {$max <= 50} {
	set spaces ""
	for {set i 0} {$i < [expr $max+3]} {incr i} {
	    append spaces " "
	}
    }
    foreach flag $args_priv(flags) {
	puts -nonewline stderr \
	    "$flag [string toupper $args_priv($flag,formals)]"
	if {$max > 50} {
	    puts stderr "\n[word_wrap $args_priv($flag,description)]\n"
	} else {
	    set len [string length "$flag $args_priv($flag,formals)"]
	    set text [word_wrap $args_priv($flag,description) [expr 75-$max]]
	    set prefix [string range $spaces 0 [expr $max-$len+2]]
	    foreach line [split $text \n] {
		puts stderr "$prefix$line"
		set prefix $spaces
	    }
	}
    }
    if [info exists args_priv(remaining)] {
	puts stderr "\n[word_wrap $args_priv(remaining_description)]"
    }
}

document_proc args_error {
    invokes `body' whenever #args_parse# detects an error.  The variable
    #detail# will contain a description of the error.
}
proc args_error body {
    global args_priv
    set args_priv(error) $body
}

document_proc args_gset {
    sets the global variable `var' to `value'.
}
proc args_gset {var value} {
    upvar #0 $var v
    set v $value
}

document_proc args_glappend {
    #lappend#s `value' to the global variable `var'.
}
proc args_glappend {var value} {
    upvar #0 $var v
    lappend v $value
}

document_example EXAMPLE {
    Here is the argument processing code from #mksh.tcl#:
} {
args_parse $argv {
    args_arg -nostdin {} "write code to connect stdin to /dev/null (avoids Tk's read/eval/print loop)" {
	args_gset no_stdin 1
    }
    args_arg -append {code} "write CODE after all initializations have been called" {
	args_gset extra_code $code
    }
    args_arg -out {file} "write C source code to FILE" {
	if [catch {args_gset fp [open $file "w"]}] {
	    puts stderr "couldn't open \"$file\""
	    exit 1
	}
    }
    args_remaining "Any remainings args are the names of Init functions." {
	args_gset fns $argv
    }
    args_error {
	puts stderr "$detail, usage is:"
	args_usage
	exit 1
    }
}
}
