#
# $Id: strip.tcl,v 1.4 1995/03/21 03:44:51 sls Exp $
#
# StripChart is a class that implements a strip chart.
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#

widget stripchart {

    param height 100
    param width 200
    param font -misc-fixed-medium-r-*-*-7-*-*-*-*-*-*-*
    param scale 1
    param tickevery 1
    param label "" Text
    param color black Color
    param background bisque Color
    param labelcolor black Color
    param tickcolor black Color
    member canvas
    member x 0
    member label_item ""
    member maxYLeft 0
    member maxYRight 0
    member old_height -1
    member old_width -1

    method create {} {
	set c [set slot(canvas) [canvas $self.c]]
	pack $c -in $self -fill both -expand yes
	bind $c <Configure> "$self config -width %w -height %h"
	set slot(label_item) [$c create text 2 2 -anchor nw -tag label] \
    }

    method reconfig {} {
	set c $slot(canvas)
	$c itemconfig $slot(label_item) -font $slot(font) \
	    -text $slot(label) -fill $slot(labelcolor)
	$c itemconfig ticks -fill $slot(tickcolor)
	$c itemconfig left -fill $slot(color)
	$c itemconfig right -fill $slot(color)
	$c config -background $slot(background)
	set w $slot(width)
	set h $slot(height)
	if {$slot(old_height) != $h || $slot(old_width) != $w} {
	    $c config -width $w -height $h
	    $self notifyNewSize
	    set slot(old_height) $h
	    set slot(old_width) $w
	}
    }

    # notifyNewSize is called when the size of the window has changed.
    # It retags things so that left and right are correct for the new
    # width and height.

    method notifyNewSize {} {
	set width $slot(old_width)
	set height $slot(old_height)
	set new_width $slot(width)
	set new_height $slot(height)
	set c $slot(canvas)
	set doshift 0
	if {[info exists c] && $height != 0} {
	    set sy [expr {1.0*$new_height/$height}]
	    $c scale left 0 0 1.0 $sy
	    $c scale right 0 0 1.0 $sy
	    $c scale ticks 0 0 1.0 $sy
	    foreach t [$c find withtag ticks] {
		scan [$c coords $t] "%f %f %f %f" tx0 ty0 tx1 ty1
		$c coords $t 0 $ty0 $new_width $ty1
	    }
	    set w2 [expr {$new_width/2}]
	    if {$new_width > $width} {
		foreach l [$c find withtag right] {
		    if {[lindex [$c coords $l] 0] <= $w2} {
			$c addtag switchtag withtag $l
		    }
		}
		$c addtag left withtag switchtag
		$c dtag switchtag right
		$c dtag switchtag
	    } elseif {$new_width < $width} {
		foreach l [$c find withtag left] {
		    if {[lindex [$c coords $l] 0] >= $w2} {
			$c addtag switchtag withtag $l
		    }
		}
		$c addtag right withtag switchtag
		$c dtag switchtag left
		$c dtag switchtag
		foreach l [$c find withtag right] {
		    if {[lindex [$c coords $l] 0] > $new_width} {
			$c delete $l
		    }
		}
		if {$slot(x) > $new_width} {
		    set doshift 1
		}
	    }
	}
	if {$doshift} {
	    $self shiftLeft
	}
    }

    method append {y} {
	set width $slot(width)
	set height $slot(height)
	set x $slot(x)
	set maxYLeft $slot(maxYLeft)
	set maxYRight $slot(maxYRight)
	set c $slot(canvas)
	set scale $slot(scale)

	if {$x >= [expr {$width/2}]} {
	    set tag right
	    set maxYRight [$self max $y $maxYRight]
	} else {
	    set tag left
	    set maxYLeft [$self max $y $maxYLeft]
	}
	$self checkMaxY [$self max $maxYLeft $maxYRight]
	set slot(maxYLeft) $maxYLeft
	set slot(maxYRight) $maxYRight
	set cy [expr {$height-$height*$y/$slot(scale)}]
	set id [$c create line $x $height $x $cy -tag $tag \
		-fill $slot(color)]
	$c lower $id
	incr x
	set slot(x) $x
	if {$x == $width} {
	    $self shiftLeft
	}
    }

    # max returns the maximum of its args.
    method max args {
	set max [lindex $args 0]
	foreach x $args {
	    if {$x > $max} {
		set max $x
	    }
	}
	return $max
    }

    # checkMaxY checks the max y value and changes the scale if neccessary.
    method  checkMaxY {y} {
	set scale $slot(scale)
	set tickevery $slot(tickevery)
	if {!($y > $scale || (($y+$tickevery) < $scale))} {
	    return
	}
	set c $slot(canvas)
	set width $slot(width)
	set height $slot(height)
	set new_scale [expr int($y+$tickevery)]
	$c delete ticks
	for {set i 1} {$i <= $new_scale/$tickevery} {incr i} {
	    set y [expr {$height-$i*$tickevery*$height/$new_scale}]
	    $c create line 0 $y $width $y -tag ticks \
		    -fill $slot(tickcolor)
	}
	set sy [expr {1.0*$scale/$new_scale}]
	$c scale left 0 $height 1.0 $sy
	$c scale right 0 $height 1.0 $sy
	set slot(scale) $new_scale
    }


    # shiftLeft deletes everything on the left, moves everything
    # on the right to the left, updates the tags, and updates the max
    # Y on the left and right.
    method shiftLeft {} {
	set width $slot(width)
	set maxYLeft $slot(maxYLeft)
	set maxYRight $slot(maxYRight)
	set scale $slot(scale)
	set c $slot(canvas)
	set x [expr {$width/2}]
	set dx $x
	if {($width % 2) == 1} {
	    incr dx
	}
	$c delete left
	$c addtag left withtag right
	$c dtag right
	$c move left -$dx 0
	$c raise ticks
	$c raise label
	set slot(maxYLeft) $maxYRight
	set slot(maxYRight) 0
	set slot(x) $x
	$self checkMaxY $maxYLeft
    }
}
