set tabList {}
#------------------------------------------------------------------------ 
#
#  mkParam w
#
#  Create a top-level window that displays some controls
#  to adjust ising model parameters.
#
#  Arguments:
#	w -	Name to use for the top-level window
#
#------------------------------------------------------------------------
proc mkParam {{w .param}} {
	global RemoteInterp
	
	catch {destroy $w}
	toplevel $w
	wm title $w "VIM \[Parameters\]"
	wm iconname $w "Param"


	set paramF   [mkPrParams $w]
	set buttonF  [mkPrBtns $w]

	pack $paramF -side top -fill both
	pack $buttonF -side bottom -fill x

	#----------------------------------------------------------------
	# startup slave's init procedure
	#---------------------------------------------------------------- 
	remote "initParam $w.paramF.sweepsF"

}

#------------------------------------------------------------------------ 
#
#  mkPrParam top
#
#  builds the parameter controls and the container frame
#
#  Arguments:
#	top -	Name of the top-level window
#
#------------------------------------------------------------------------
proc mkPrParams {top} {

	set w [frame $top.paramF -relief raised -bd 1]
	set sweepsF  [mkPrSweeps $w]
	set betasF   [mkPrBetas $w]
	set latticeF [mkPrLattic $w]

	pack $sweepsF -side top -fill x -expand yes -padx 10 -pady 10
	pack $betasF $latticeF -side left -fill x -expand yes -padx 10 -pady 10
	pack $latticeF -side right -fill x -expand yes -padx 15 -pady 10

	return $w	
}

#------------------------------------------------------------------------ 
#
#  mkPrSweeps top
#
#  builds the sweep controlling frame and slider
#
#  Arguments:
#	top -	Name of the top-level window
#
#------------------------------------------------------------------------
proc mkPrSweeps {top} {
	global sweeps

	set w [frame $top.sweepsF]

	label $w.slabel \
		-text Sweeps:
	scale $w.sscale \
		-from 1 -to 100 -orient horizontal \
		-sliderlength 20  \
		-showvalue yes

	bind $w.sscale <B1-ButtonRelease> "changeSweeps $w.sscale"
	bind $w.sscale <Enter> "focus $w.sscale"
	bind $w.sscale <Leave> "focus none"
	bind $w.sscale <Key-Left> "moveSlider $w.sscale 1 left"
	bind $w.sscale <Key-Right> "moveSlider $w.sscale 1 right"
	bind $w.sscale <Control-Key-Left> "moveSlider $w.sscale 10 left"
	bind $w.sscale <Control-Key-Right> "moveSlider $w.sscale 10 right"
	bind $w.sscale <Key-End> "$w.sscale set 100;setSweeps 100"
	bind $w.sscale <Key-Home> "$w.sscale set 1;setSweeps 1"

	pack $w.slabel -side left 
	pack $w.sscale -side left -fill x -expand yes -padx 10

	return $w	
}

proc changeSweeps {s} {
	setSweeps [$s get]
}

#------------------------------------------------------------------------ 
#
#  mkPrBetas
#
#  builds the controls for beta-parameters
#
#  Arguments:
#	top -	Name of the top-level window
#
#------------------------------------------------------------------------
proc mkPrBetas {top} {
	global tabList

	set w [frame $top.betasF ]
	foreach i {1 2 3 4 5} {
		global beta$i
		
		frame $w.$i  
		label $w.$i.blabel \
			-width 7 \
			-text "Beta $i:"  -anchor w
		entry $w.$i.bentry \
			-bd 2 -relief sunken -width 10 \
			-textvariable beta$i
			
		bind $w.$i.bentry <Return> "prApply"
		bind $w.$i.bentry <Tab> "tab \$tabList"			

		pack $w.$i.blabel -side left 
		pack $w.$i.bentry -side left -padx 10
		pack $w.$i
	}

	# set tab order for entry windows
	set tabList "$w.1.bentry $w.2.bentry $w.3.bentry \
			$w.4.bentry $w.5.bentry"

	return $w
}


#------------------------------------------------------------------------ 
#
#  mkPrLattic
#
#  Builds radiobuttons for the setting of sublattices
#
#  Arguments:
#	top -	Name of the top-level window
#
#------------------------------------------------------------------------
proc mkPrLattic {top} {		

	global sublat

	set w [frame $top.latticeF] 

	label $w.label -text "Sub-lattices:"
	radiobutton $w.r1 \
		-text "1" \
		-relief groove \
		-variable sublat -value 1 \
		-command "remote {.ising configure -lat 1}" \
		-width 7
	radiobutton $w.r2 \
		-text "2" \
		-relief groove \
		-variable sublat -value 2 \
		-command "remote {.ising configure -lat 2}" \
		-width 7
	radiobutton $w.r3 \
		-text "3" \
		-relief groove \
		-variable sublat -value 3 \
		-command "remote {.ising configure -lat 3}"\
		-width 7
		
	pack $w.label $w.r1 $w.r2 $w.r3   

	return $w
}


#------------------------------------------------------------------------ 
#
#  mkPrBtns
#
#  builds the a dialog button frame
#
#  Arguments:
#	top -	Name of the top-level window
#
#------------------------------------------------------------------------
proc mkPrBtns {top} {
	
	set w [tixDlgBtns $top.buttonF]

	$w add apply -text Apply -command prApply -width 7
	$w add close -text Close -command "destroy $top" -width 7
	$w add help -text Help -width 7 -state disabled

	return $w
}


#------------------------------------------------------------------------ 
#
#  setSweeps
#
#  Arguments:
#	sw -	new value for sweeps given by slider
#
#------------------------------------------------------------------------
proc setSweeps {sw} {
	remote "setSweeps $sw" 
}


#------------------------------------------------------------------------ 
#
#  moveSlider s n side
#
#  move the slider of scale <s> <n> positions to side <side>.
#
#  Arguments:
#	s -	pathname of scale
#	n -	number of positions
#	side -	side to which the slider must be moved
#
#------------------------------------------------------------------------
proc moveSlider {s n side} {
	set sliderpos [$s get]
	if {$side == "left"} {
		$s set [expr $sliderpos - $n]
		setSweeps [expr $sliderpos - 1]
		return
	}
	if {$side == "right"} {
		$s set [expr $sliderpos + $n]
		setSweeps [expr $sliderpos + 1]
		return
	}
}	


#------------------------------------------------------------------------ 
#
#  prApply 
#
#  Set global variables beta<n> to the current values of the
#  entry windows and reconfigure ising widget.
#
#------------------------------------------------------------------------
proc prApply {} {
#	set betalist {}
	
#	foreach i {1 2 3 4 5} {
#		global beta$i
#		lappend betalist ${beta$i}
#	}
#	remote {apply $betalist}

	global beta1 beta2 beta3 beta4 beta5
	remote "applyBeta $beta1 $beta2 $beta3 $beta4 $beta5"

}

#------------------------------------------------------------------------ 
#
#  tab list
#
#  This procedure is invoked in response to tabs in the entry
#  windows. It moves the focus to the next window in the tab list.
#
#  Arguments:
#
#  list -	Ordened list of windows to receive focus
#
#------------------------------------------------------------------------
proc tab {list} {
	set i [lsearch $list [focus]]
	if {$i < 0} {
		set i 0
	} else {
		incr i
		if {$i >= [llength $list]} {
			set i 0
		}
	}
	focus [lindex $list $i]
}
