#!./ising -f
#-----------------------------------------------------------------------------
#
#	the Visual Ising Machine (VIM)
#
#	versie 0.6
#
#	ising widget source script (TCL)
#
#	M.Figee en M.Hufken
#
#-----------------------------------------------------------------------------
source settings.tcl

set auto_path "$VIM_PATH $auto_path"
set RemoteInterp "vimmain.tcl"

#-----------------------------------------------------------------------------
# global variables
#-----------------------------------------------------------------------------
set scrMaxX [winfo screenwidth .]
set scrMaxY [winfo screenheight .]
set xres 32		;# xresolution
set yres 32		;# yresolution
set xzoom $xres 	;# display x res.
set yzoom $yres 	;# display y res. 
set dragx 0		;# first dragging point x
set dragy 0		;# first dragging point y
set scrollx 0		;# x scroll distance
set scrolly 0		;# y scroll distance
set sweeps 1		;# number of sweeps per frame
set cursweep  0		;# current sweep
set beta1 1.333334	;#
set beta2 -30.0		;# 
set beta3 -30.0		;# model parameters
set beta4 0.0		;#
set beta5 0.0		;#
set sublat 2		;# number of sub-lattices to show
set appTitle "VIM: the Visual Ising Machine"
set state STOPPED

#-----------------------------------------------------------------------------
# information about toplevel window
#-----------------------------------------------------------------------------
wm title . $appTitle 
wm minsize . 450 450
wm geometry . -0+0
wm iconname . "VIM"
wm title . $appTitle
wm protocol . WM_DELETE_WINDOW {remote "mkQuit"}
. configure -background "SGIgrey80"

bind . <Configure> {resizeIsing .ising}

#-----------------------------------------------------------------------------
#  Ising widget and frame
#-----------------------------------------------------------------------------
frame .isingF -bd 1 -relief sunken -bg "grey80"
ising .ising -xres ${xres} -yres ${yres} \
		-xzoomres ${xres} -yzoomres ${yres}
pack .isingF -padx 10 -pady 10 
pack .ising -in .isingF -padx 10 -pady 10

#bind .ising <1> "markZoomRect .ising %x %y"
#bind .ising <B1-Motion> "drawRubberBand .ising %x %y"
#bind .ising <B1-ButtonRelease> "zoomIn .ising"
bind .ising <2> "markDragging .ising %x %y"
bind .ising <B2-Motion> "drag .ising %x %y"


#-----------------------------------------------------------------------------
#  markDragging w x y
#
#  Mark the first point of dragging, measured in isingmodel resolution
#  If ising has already been scrolled, the marking point will be relative
#
#  Arguments:
#	w -	ising widget
#	x, y -	x and y mouse cursor position
#-----------------------------------------------------------------------------
proc markDragging {w x y} {
	global xres yres dragx dragy scrollx scrolly
	set sWidth [expr [winfo width $w] / $xres]
	set sHeight [expr [winfo height $w] / $yres]
	set dragx [expr ($x / $sWidth) + $scrollx] 
	set dragy [expr ($y / $sHeight) + $scrolly]
}

#-----------------------------------------------------------------------------
#  drag w x y
#
#  Scroll isingwidget in any direction
#
#  Arguments:
#	w -	ising widget
#	x, y -	x and y mouse cursor position
#-----------------------------------------------------------------------------
proc drag {w x y} {
	global xres yres dragx dragy scrollx scrolly
	set sWidth [expr [winfo width $w] / $xres]
	set sHeight [expr [winfo height $w] / $yres]
	set curx [expr $x / $sWidth]
	set cury [expr $y / $sHeight]
	set scrollx [expr -1 * [expr $curx - $dragx]]
	set scrolly [expr -1 * [expr $cury - $dragy]]
	$w configure -xscroll $scrollx -yscroll $scrolly
	
}


#-----------------------------------------------------------------------------
#
# changeSize xres yres
#
# Arguments:
#  xres -	new x resolution value
#  yres -	new y resolution value
#-----------------------------------------------------------------------------
proc changeSize {xr yr} {
	global xres yres
	
	set xres $xr
	set yres $yr
}


#-----------------------------------------------------------------------------
#
# displayFrame w
#
# Arguments:
#  w -	ising widget
#-----------------------------------------------------------------------------
proc displayFrame {w} {
	global state
	global cursweep
	global sweeps
	
	set cursweep [expr {$cursweep + $sweeps}]


	$w run -sweeps $sweeps
	if {$state == "PLAYING"} { after 30 displayFrame $w }
}


#-----------------------------------------------------------------------------
#
# remote
# 
# Send a script to another interpreter
#
#-----------------------------------------------------------------------------
proc Rremote {script} {
	global RemoteInterp
	send $RemoteInterp $script
}


#-----------------------------------------------------------------------------
#
# initParam w
# 
# Initialize the 'Parameter' window in the other application
#
# Arguments:
#   w -	window handle of Parameter-window 
#-----------------------------------------------------------------------------
proc initParam {w} {
	global sweeps
	remote "$w.sscale set $sweeps"
}


#------------------------------------------------------------------------ 
#
#  applyBeta args
#
#  Set global variables beta1..beta-n to the current values in the
#  args-list and reconfigure ising widget.
#
#  Arguments:
#    args -	list of beta values 
#------------------------------------------------------------------------
proc applyBeta {args} {
	set i 1
	foreach v $args {
		global beta$i

		set beta$i $v
		if { $v < -30.0} {
			set beta$i -30.0
		} else {if { $v > 30.0} {
			set beta$i 30.0 }
		} 
		
		set b [set beta$i]
		.ising configure -beta$i $b
		remote "set beta$i $b"
		incr i
	}		
}

#------------------------------------------------------------------------ 
#
#  playPause w
#
#  Keep displaying frames until state changes to PAUSED or STOPPED
#
#  Arguments:
#    w -	ising widget handle
#------------------------------------------------------------------------
proc playPause {{w .ising}} {
	global state
	global cursweep
	global xres yres

	switch $state {
	{STOPPED} { set state PLAYING
		    remote { 
			set state PLAYING
		    	$MAIN.cpPlayPauseB configure \
				-bitmap @$BITMAP_PATH/playpause1.xbm
		    	$MAIN.cpStopB configure \
				-bitmap @$BITMAP_PATH/stop.xbm
		    }
		    $w configure -xzoomres $xres -yzoomres $yres 
#		    $w configure -xoffset 10 -yoffset 10 \
#				-xzoomres 30 -yzoomres 20
		    $w init -xres $xres -yres $yres
		    displayFrame $w
		  }

	{PAUSED}  { set state PLAYING
		    remote {
			set state PLAYING
		    	$MAIN.cpPlayPauseB configure \
				-bitmap @$BITMAP_PATH/playpause1.xbm
#			$MAIN.cpLeftB configure -state disabled
			$MAIN.cpRightB configure -state disabled	
		    }
		    displayFrame $w
		  }

	{PLAYING} { set state PAUSED
		    remote {
			set state PAUSED
		    	$MAIN.cpPlayPauseB configure \
		    		-bitmap @$BITMAP_PATH/playpause2.xbm
#			$MAIN.cpLeftB configure -state normal
			$MAIN.cpRightB configure -state normal	
		    }
		  } 
	}
}

#------------------------------------------------------------------------ 
#
#  stop
#  
#  Stop displaying frames
#
#------------------------------------------------------------------------
proc stop {} {
	global state
	global cursweep

	set state STOPPED
	set cursweep 0

	remote {set state STOPPED
		$MAIN.cpStopB configure -bitmap @$BITMAP_PATH/fillstop.xbm
		$MAIN.cpPlayPauseB configure -bitmap @$BITMAP_PATH/playpause.xbm
		$MAIN.cpLeftB configure -state disabled
		$MAIN.cpRightB configure -state disabled	
		}
}

#------------------------------------------------------------------------ 
#
#  stepForw w
#
#  Display only one frame
#
#  Arguments:
#    w -	ising widget handle
#------------------------------------------------------------------------
proc stepForw {{w .ising}} {
	global state
	
	if {$state == "PAUSED"} {
		displayFrame $w
	}
}

#------------------------------------------------------------------------ 
#
#  resizeIsing
#  
#  Resize window w to make an optimal fit in toplevel window.
#  This procedure is invoked, everytime the ising window is 
#  moved, resized etc.
#
#  Arguments:
#    w -	ising widget handle
#------------------------------------------------------------------------
proc resizeIsing {w} {
	set iMaxY [expr [winfo height .] -42 ] ;# 42 = padsizes + bd
	set iMaxX [expr [winfo width .] -42 ]
	$w configure -xsize $iMaxX -ysize $iMaxY

	# send window positions to master
	set xpos [winfo x .]
	remote "set drawX $xpos"
}

#------------------------------------------------------------------------ 
#
#  setSweeps sw
#  
#  Set global var sweeps and reconfigure ising widget
#
#  Arguments:
#    sw -	new sweep number
#------------------------------------------------------------------------
proc setSweeps {sw} {
	global sweeps

	if {$sw < 1} {set sw 1}
	if {$sw > 100} {set sw 100}
	set sweeps $sw
	.ising configure -sweeps $sw
}
