/* 
 * Mach Operating System
 * Copyright (c) 1991 Carnegie Mellon University
 * Copyright (c) 1991 IBM Corporation 
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation,
 * and that the name IBM not be used in advertising or publicity 
 * pertaining to distribution of the software without specific, written
 * prior permission.
 * 
 * CARNEGIE MELLON AND IBM ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON AND IBM DISCLAIM ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */


/*
 * HISTORY 
 * $Log:	kd_mouse.c,v $
 * Revision 2.2  93/02/04  08:00:56  danner
 * 	Integrate PS2 code from IBM.
 * 	[93/01/18            prithvi]
 * 
 */

/*
 * PS2 mouse driver feeding keyboard/mouse queue.
 * Mach Microkernel only.
 */


#include <mach/boolean.h>
#include <sys/types.h>
#include <device/errno.h>
#include <device/io_req.h>

#include <kern/queue.h>

#include <i386/ipl.h>

#include <i386at/kd.h>
#include <i386at/kd_queue.h>

#include <i386ps2/abios.h>
#include <i386ps2/kd_mouse_abios.h>
#include <i386ps2/kd_mouse_io.h>

kd_event_queue  mouse_queue;            /* queue of mouse events */
boolean_t       mouse_in_use = FALSE;

queue_head_t    mouse_read_queue = { &mouse_read_queue, &mouse_read_queue };

int             kd_mouse = 0;           /* used by kd.c */

/*
 * Current mouse buttons
 */
int             last_buttons = 0;       /* previous state of mouse buttons */
#define MOUSE_UP        1
#define MOUSE_DOWN      0
#define MOUSE_ALL_UP    0x7

void mouse_enqueue();
int  mouseintr();

#define SPLMS   SPLKD

int
mouseopen(dev, flags)
        dev_t   dev;
        int     flags;
{
        if (mouse_in_use)
	    return EBUSY;
	mouse_in_use = TRUE;
	kdq_reset(&mouse_queue);
	last_buttons = MOUSE_ALL_UP;

	if (ps2_mouse_open())
	    return D_NO_SUCH_DEVICE;

	return D_SUCCESS;
}

int
mouseclose(dev, flags)
        dev_t   dev;
        int     flags;
{
        ps2_mouse_close();

	kdq_reset(&mouse_queue);
	mouse_in_use = FALSE;

	return 0;
}

boolean_t       mouse_read_done();      /* forward */

mouseread(dev, ior)
        dev_t   dev;
        io_req_t ior;
{
        int     err, s, count;

	err = device_read_alloc(ior, (vm_size_t)ior->io_count);
	if (err != KERN_SUCCESS)
	    return err;

	s = SPLMS();
	if (kdq_empty(&mouse_queue)) {
	    if (ior->io_mode & D_NOWAIT) {
	        splx(s);
		return D_WOULD_BLOCK;
	    }
	    ior->io_done = mouse_read_done;
	    enqueue_tail(&mouse_read_queue, (queue_entry_t) ior);
	    splx(s);
	    return D_IO_QUEUED;
	}

	count = 0;
	while (!kdq_empty(&mouse_queue) && count < ior->io_count) {
	    kd_event *ev;

	    ev = kdq_get(&mouse_queue);
	    *(kd_event *)(&ior->io_data[count]) = *ev;
	    count += sizeof(kd_event);
	}

	splx(s);
	ior->io_residual = ior->io_count = count;
	return D_SUCCESS;
}

boolean_t mouse_read_done(ior)
	io_req_t	ior;
{
	int	s, count;

	s = SPLMS();
	if (kdq_empty(&mouse_queue)) {
	    ior->io_done = mouse_read_done;
	    enqueue_tail(&mouse_read_queue, (queue_entry_t)ior);
	    splx(s);
	    return FALSE;
	}

	count = 0;
	while (!kdq_empty(&mouse_queue) && count < ior->io_count) {
	    kd_event *ev;

	    ev = kdq_get(&mouse_queue);
	    *(kd_event *)(&ior->io_data[count]) = *ev;
	    count += sizeof(kd_event);
	}
	splx(s);

	ior->io_residual = ior->io_count - count;
	ds_read_done(ior);

	return TRUE;
}

/*
 * 3 byte ps2 format used
 *
 * 7  6  5  4  3  2  1  0
 * YO XO YS XS 1  0  R  L
 * X7 X6 X5 X4 X3 X3 X1 X0
 * Y7 Y6 Y5 Y4 Y3 Y2 Y1 Y0
 *
 */
#define BUTTON_MASK		0x3
#define LBUTTON_MASK		0x1
#define RBUTTON_MASK		0x2
#define MBUTTON_MASK		0x3
#define XNEG			0x00000010
#define YNEG			0x00000020
#define XOVER			0x00000040
#define YOVER			0x00000080
#define DELTA_MASK		0x000000ff
#define SIGNXTND		0xffffff00

#if     1   /* 3-button mouse */
void
mouse_packet_ps2_mouse(m_buttons, m_x, m_y)
        int     m_buttons, m_x, m_y;
{
	unsigned char buttons, buttonchanges;
	struct mouse_motion moved;

	buttons = m_buttons & BUTTON_MASK;  /* get current state of buttons */
	buttonchanges = buttons ^ last_buttons;
	moved.mm_deltaX = ((m_buttons & XNEG) ? SIGNXTND : 0 ) | m_x;
	moved.mm_deltaY = ((m_buttons & YNEG) ? SIGNXTND : 0 ) | m_y;

	if (moved.mm_deltaX != 0 || moved.mm_deltaY != 0)
	    mouse_moved(moved);

	if (buttonchanges != 0) {
	    last_buttons = buttons;
	    switch (buttonchanges) {
		case LBUTTON_MASK:
		    mouse_button(MOUSE_LEFT,  (buttons & LBUTTON_MASK) == 0);
		    break;
		case RBUTTON_MASK:
		    mouse_button(MOUSE_RIGHT, (buttons & RBUTTON_MASK) == 0);
		    break;
		case MBUTTON_MASK:
		    mouse_button(MOUSE_MIDDLE, (buttons != MBUTTON_MASK));
		    break;
		}
	}
}
#else
/*
 * Simulate a 3-button mouse with 2 buttons.
 */
/*
 * msddecode looks at the state of the buttons and handles the three button
 * simulation using two buttons.
 * states (single_but):
 * 0	no buttons pressed
 * 1	single button pressed left or right
 *
 * logic:
 *	if both buttons are pressed then pretend that the middle button was
 *	pressed.
 * states:
 * 1.	not in middle of a single button test, and both buttons were down and
 *	exactly one was let up --> we ignore this transition until both are
 *	let up.
 * 2.	not in middle of a single button test, and both buttons were up, and
 *	now is pressed --> arrange to check again in 5 ticks and now in single
 *	button test mode.
 * 3.	in the middle of the single button test, and the buttons are unchanged
 *	--> cancel the timeout.  If we have gone through this twice before then
 *	assume that the second button will not be pressed, otherwise arrange
 *	another poll.
 * 4.	otherwise, if in single button test ( and buttons are different)
 *	then cancel the timeout and the single button mode (e.g. both buttons
 *	are up or both down.
 *
 *	The x and y value from the mouse are in two's complement form.  If
 *	the negative flag is set, sign extend the value so that the fact that
 *	it is in two's complement form won't be lost.
 */

void
mouse_packet_ps2_mouse(m_buttons, m_x, m_y)
        int     m_buttons, m_x, m_y;
{
        switch (m_buttons & BUT_MASK) {
	    case MBUT_MASK:
	        flags = MIDDLE_BUT;
		break;
	    case LBUT_MASK:
	        flags = LEFT_BUT;
		break;
	    case RBUT_MASK:
	        flags = RIGHT_BUT;
		break;
	    default:
		flags = 0;
	}

	mouse_x = m_x & DELTA_MASK;
	if (m_buttons & XNEG) {
	    mouse_x |= SIGNXTND;
	    if (m_buttons & XOVER)
	        mouse_x -= 0x100;
	}
	else {
	    if (m_buttons & XOVER)
	        mouse_x += 0x100;
	}

	mouse_y = m_y & DELTA_MASK;
	if (m_buttons & YNEG) {
	    mouse_y |= SIGNXTND;
	    if (m_buttons & YOVER)
	        mouse_y -= 0x100;
	}
	else {
	    if (m_buttons & YOVER)
	        mouse_y += 0x100;
	}
	mouse_y = -mouse_y;

	if (!single_button && (last_buttons != buttons)) {
	    if (last_buttons == MIDDLE_BUT && buttons != NO_BUT) {
	        buttons = MIDDLE_BUT;
		single_button = 0;            /* only 1 button let up */
	    }
	    else if (last_buttons == NO_BUT && buttons != MIDDLE_BUT) {
	        set_timeout(mouse_timeout, (char *) 0, 4);
		single_button++;
	    }
	    /* otherwise single button let up */
	}
	else if (single_button && last_buttons == buttons) {
	    /*
	     * no change in buttons
	     */
	    untimeout(mouse_timeout, (char *) 0);
	    if (++single_button > 2)
	        single_button = 0;      /* third time through */
	    else
	        timeout(mouse_timeout, (char *)0, 4);
	}
	else if (single_button) {
	    /*
	     * change in buttons
	     */
	    untimeout(mouse_timeout, (char *) 0);
	    single_button = 0;
	    if (button_status == NO_BUT) {
	        button_status = last_buttons;
		/*
		 * button let up - note both down and up reports
		 */
		...report down...

		/* report the up motion */
		...
	    }
	}
	last_buttons = button_status;

	return (!single_button);
}

mouse_timeout()
{
        if (msddecode(0)) msdreport();
}
#endif


/*
 * Enqueue a mouse-motion event.  Called at splmouse.
 */
mouse_moved(where)
	struct mouse_motion where;
{
	kd_event ev;

	ev.type = MOUSE_MOTION;
	ev.time = time;
	ev.value.mmotion = where;
	mouse_enqueue(&ev);
}


/*
 * Enqueue an event for mouse button press or release.  Called at splmouse.
 */
mouse_button(which, direction)
	kev_type which;
	u_char direction;
{
	kd_event ev;

	ev.type = which;
	ev.time = time;
	ev.value.up = (direction == MOUSE_UP) ? TRUE : FALSE;
	mouse_enqueue(&ev);
}


/*
 * mouse_enqueue - enqueue an event and wake up selecting processes, if
 * any.  Called at splmouse.
 */

void
mouse_enqueue(ev)
	kd_event *ev;
{
	register io_req_t	ior;

	if (kdq_full(&mouse_queue))
		printf("mouse: queue full\n");
	else
		kdq_put(&mouse_queue, ev);

	while ((ior = (io_req_t)dequeue_head(&mouse_read_queue)) != 0)
	    iodone(ior);

}
	    
/*
 * PS2 mouse stuff
 */

struct Mouse_request    mouserb;
struct Mouse_request    mousecontrb;

struct Logical_id_params mouseparams = { 0 };
struct Logical_id_params mousecontparams = { 0 };

void   mouseabioswait();        /* forward */

boolean_t ps2_mouse_inited = FALSE;

ps2_mouse_init()
{
        int  lid_num;

	if (ps2_mouse_inited)
	    return 0;

        /*
	 * Init mouse hardware: get logical parameters
	 */
        mouserb.r_current_req_blck_len = sizeof(struct Mouse_request);
	mouserb.request_header.Request_Block_Flags = 0;
	mouserb.request_header.ELA_Offset = 0;

	lid_num = MS_ID;
	lid_num = abios_next_LID(MS_ID, lid_num);
	MOUSE_SET_RESERVED_ABIOS_LOGICAL_PARAMETER(mouserb);

	mouserb.r_logical_id = lid_num;
	mouserb.r_unit = 0;
	mouserb.r_flags = MS_NOERR;

	mouseabioswait(&mouserb,
		       ABIOS_LOGICAL_PARAMETER,
		       mouseparams.Logical_id_flags);

	if (mouserb.r_return_code != ABIOS_MOUSE_RC_DONE) {
	    if (mouserb.r_return_code == ABIOS_MOUSE_RC_INVALID_LOGICAL_ID)
	        return 1;  /* no mouse */
	}

	if (mouserb.r_request_block_length > mouserb.r_current_req_blck_len)
	    panic("mouse: rb len");

	mouserb.r_current_req_blck_len = mouserb.r_request_block_length;
	mouseparams = mouserb.un.logical_id_params;

	take_irq(mouserb.r_hardware_intr,
		 0,
		 mouseintr,
		 SPLTTY,
		 "ms");

	/*
	 * Continuous Read has its own Request Block, because it will
	 * always run in the background.
	 */
	mousecontrb.r_current_req_blck_len = mouserb.r_current_req_blck_len;
	mousecontrb.request_header.Request_Block_Flags =
	        mouserb.request_header.Request_Block_Flags;
	mousecontrb.request_header.ELA_Offset =
	        mouserb.request_header.ELA_Offset;
	mousecontrb.r_logical_id = mouserb.r_logical_id;
	mousecontrb.r_unit       = mouserb.r_unit;
	mousecontparams          = mouseparams;
	MOUSE_SET_RESERVED_ABIOS_POINTING_DEVICE_CONTINUOUS_READ(mousecontrb);
	mousecontrb.r_8_data_package_size = 3;
	mousecontrb.r_flags = MS_DEFLT;
	mouseabioswait(&mousecontrb,
		       ABIOS_MOUSE_POINTING_DEVICE_CONTINUOUS_READ,
		       mousecontparams.Logical_id_flags);

#if 0
	/*
	 * Reset keyboard after mouse screws it up
	 */
	kbd_tx_off();
#endif

	ps2_mouse_inited = TRUE;
	return 0;
}

ps2_mouse_open()
{
        if (ps2_mouse_init())
	    return 1;  /* no mouse */

        /*
	 * Enable Pointing Device
	 */
        MOUSE_SET_RESERVED_ABIOS_ENABLE_POINTING_DEVICE(mouserb);
	mouserb.r_flags = MS_DEFLT;
	mouseabioswait(&mouserb,
		       ABIOS_MOUSE_ENABLE_POINTING_DEVICE,
		       mouseparams.Logical_id_flags);

	/*
	 * Reset/Initialize Pointing Device
	 */
	MOUSE_SET_RESERVED_ABIOS_RESET_POINTING_DEVICE(mouserb);
	mouserb.r_flags = MS_DEFLT;
	mouseabioswait(&mouserb,
		       ABIOS_MOUSE_RESET_POINTING_DEVICE,
		       mouseparams.Logical_id_flags);

        return 0;
}

ps2_mouse_close()
{
        int  s;

        /*
	 * Disable Pointing Device
	 */
        s = SPLMS();

	MOUSE_SET_RESERVED_ABIOS_DISABLE_POINTING_DEVICE(mouserb);
	mouserb.r_flags = MS_DEFLT;
	mouseabioswait(&mouserb,
		       ABIOS_MOUSE_DISABLE_POINTING_DEVICE,
		       mouseparams.Logical_id_flags);

	mouse_in_use = FALSE;

	splx(s);

	return 0;
}

int
mouseintr()
{
        /*
	 * Check for continuous read interrupts
	 */
        abios_common_interrupt(&mousecontrb,
			       mousecontparams.Logical_id_flags);

	switch (mousecontrb.r_return_code) {
	    case ABIOS_MOUSE_RC_DONE:
	        panic("mouseintr: continuous read stopped!");
		break;
	    case ABIOS_MOUSE_RC_STAGE_ON_INT:
		break;
	    case ABIOS_MOUSE_RC_STAGE_ON_TIME:
		timeout(mouseintr,
			0,
			(MOUSE_TIME_TO_WAIT(mousecontrb)/1000000) * hz);
		break;
            case ABIOS_MOUSE_RC_NOT_MY_INT:
	    case ABIOS_UNDEFINED:
		/*
		 * Must belong to a different mouse function.
		 * Check the other request block.
		 */
		abios_common_interrupt(&mouserb,
				       mouseparams.Logical_id_flags);
		switch (mouserb.r_return_code) {
		    case ABIOS_MOUSE_RC_DONE:
		        wakeup((char *)&mouserb.r_return_code);
			break;
		    case ABIOS_MOUSE_RC_STAGE_ON_INT:
			break;
		    case ABIOS_MOUSE_RC_STAGE_ON_TIME:
			timeout(mouseintr,
				0,
				(MOUSE_TIME_TO_WAIT(mouserb)/1000000)*hz);
			break;
		    case ABIOS_MOUSE_RC_NOT_MY_INT:
		    case ABIOS_UNDEFINED:
			break;  /* ??? */
		    case ABIOS_MOUSE_RC_ATTENTION:
		    default:
			panic("mouseintr: non CR attention");
			break;
		}
		break;
	    case ABIOS_MOUSE_RC_ATTENTION:
		mouse_packet_ps2_mouse(
				mousecontrb.r_pointing_device_data_status,
				mousecontrb.r_pointing_device_data_deltax,
				mousecontrb.r_pointing_device_data_deltay);
		break;
	    default:
		break;
	}

	return 0;
}

void
mouseabioswait(request, abios_function, mouse_flag)
        struct Mouse_request *request;
        int     abios_function;
        int     mouse_flag;
{
        int     done;

	request->r_function = abios_function;
	request->r_return_code = ABIOS_UNDEFINED;

	abios_common_start(request, mouse_flag);

	done = 0;
	while ((request->r_return_code != ABIOS_MOUSE_RC_DONE) && !done) {
	    switch (request->r_return_code) {
	        case ABIOS_UNDEFINED:
	            break;
		case ABIOS_MOUSE_RC_STAGE_ON_INT:
		    if (request->r_function ==
			ABIOS_MOUSE_POINTING_DEVICE_CONTINUOUS_READ)
		    {
		        done = 1;
		    }
		    else {
		        sleep((char *) &request->r_return_code, 0);
		    }
		    break;
		case ABIOS_MOUSE_RC_STAGE_ON_TIME:
		    timeout(mouseintr,
			    0,
			    (MOUSE_TIME_TO_WAIT(*request)/1000000)*hz);
		    if (request->r_function ==
			ABIOS_MOUSE_POINTING_DEVICE_CONTINUOUS_READ)
		    {
		        done = 1;
		    }
		    else {
		        sleep((char *) &request->r_return_code, 0);
		    }
		    break;
		case ABIOS_MOUSE_RC_NOT_MY_INT:
		    /*
		     * Wait until we get an informative reply
		     */
		    if (request->r_function ==
			ABIOS_MOUSE_POINTING_DEVICE_CONTINUOUS_READ)
		    {
		        printf("mouseabioswait: NMI waiting...\n");
		    }
		    else {
		        sleep((char *) &request->r_return_code, 0);
		    }
		    break;
		case ABIOS_MOUSE_RC_ATTENTION:
		    /*
		     * We shouldn't get this so soon
		     */
		    panic("mouseabioswait: early attention");
		    break;
		default:
		    if (!(request->r_flags & MS_NOERR))
		        panic("mouseabioswait %x", request->r_return_code);
		    else
		        done = 1;
		    break;
	    }
	}
}

int
mousegetstat(dev, flavor, data, count)
        dev_t   dev;
        int     flavor;
        int *   data;
        unsigned int *count;
{
        int     s;

	s = SPLMS();
	switch (flavor) {
            /* For now let Abios deal with errors */
            case MSIC_STATUS:
	    {
	        struct mouse_status *status;

		if (*count < sizeof(struct mouse_status)/sizeof(int)) {
		    splx(s);
		    return KERN_INVALID_ARGUMENT;
		}

                /* 0x03 Read Device Parameters */
                status = (struct mouse_status *)data;
		MOUSE_SET_RESERVED_ABIOS_READ_PARAMETER(mouserb);
		mouserb.r_flags = MS_DEFLT;
		mouseabioswait(&mouserb,
			       ABIOS_READ_PARAMETER,
			       mouseparams.Logical_id_flags);
                /* Fill out data with mouse status */
		status->interface_status = mouserb.r_interface_status;
		status->data_package_size = mouserb.r_3_data_package_size;
		status->flag_word = mouserb.r_flag_word;
		status->current_resolution = mouserb.r_current_resolution;
                status->current_sample_rate = mouserb.r_current_sample_rate;
		break;
	    }

            case MSIC_PDIC:
		/* 0x0E Read Pointing Device Identication Code */
		MOUSE_SET_RESERVED_ABIOS_READ_POINTING_DEVICE_IDENTIFICATION_CODE(mouserb);
		mouserb.r_flags = MS_DEFLT;
		mouseabioswait(&mouserb,
			       ABIOS_MOUSE_READ_POINTING_DEVICE_IDENTIFICATION_CODE,
			       mouseparams.Logical_id_flags);
		*(long *)data = (long) mouserb.r_auxiliary_device_identification_code;
		/* IBM's value is 0x0 */
		break;

            default:
	        splx(s);
		return EINVAL;
        }
	splx(s);
	return D_SUCCESS;
}

int
mousesetstat(dev, flavor, data, count)
        dev_t   dev;
        int     flavor;
        int *   data;
        unsigned int count;
{
        int     s;

	s = SPLMS();
	switch (flavor) {
            /* For now let Abios deal with errors */
            case MSIC_DISABLE:
	        /* 0x07 Disable Pointing Device */
	        MOUSE_SET_RESERVED_ABIOS_DISABLE_POINTING_DEVICE(mouserb);
		mouserb.r_flags = MS_DEFLT;
	        mouseabioswait(&mouserb,
			       ABIOS_MOUSE_DISABLE_POINTING_DEVICE,
			       mouseparams.Logical_id_flags);
                break;

            case MSIC_ENABLE:
		/* 0x06 Enable Pointing Device */
	        MOUSE_SET_RESERVED_ABIOS_ENABLE_POINTING_DEVICE(mouserb);
		mouserb.r_flags = MS_DEFLT;
	        mouseabioswait(&mouserb,
			       ABIOS_MOUSE_ENABLE_POINTING_DEVICE,
			       mouseparams.Logical_id_flags);
                break;

            case MSIC_SCALE:
		/* 0x0D Set Scaling Factor */
		MOUSE_SET_RESERVED_ABIOS_SET_SCALING_FACTOR(mouserb);
		mouserb.r_scaling_factor = (u_char) *data;
		mouserb.r_flags = MS_DEFLT;
		mouseabioswait(&mouserb,
			       ABIOS_MOUSE_SET_SCALING_FACTOR,
		               mouseparams.Logical_id_flags);
                break;

            case MSIC_SAMPLE:
		/* 0x0B Set Sample Rate */
           /*   MOUSE_SET_RESERVED_ABIOS_SET_SAMPLE_RATE(mouserb); */
                mouserb.r_sample_rate = (u_short) *data;
		mouserb.r_flags = MS_DEFLT;
	        mouseabioswait(&mouserb,
			       ABIOS_MOUSE_SET_SAMPLE_RATE,
		               mouseparams.Logical_id_flags);
                break;

            case MSIC_RESL:
                /* 0x0C Set Resolution */
		MOUSE_SET_RESERVED_ABIOS_SET_RESOLUTION(mouserb);
                mouserb.r_resolution = (u_short) *data;
		mouserb.r_flags = MS_DEFLT;
	        mouseabioswait(&mouserb,
			       ABIOS_MOUSE_SET_RESOLUTION,
		               mouseparams.Logical_id_flags);
                break;
            default:
		splx(s);
		return EINVAL;
        }
	splx(s);

	return D_SUCCESS;
}

