/* Device input and output
   Copyright (C) 1992 Free Software Foundation

This file is part of the GNU Hurd.

The GNU Hurd is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

The GNU Hurd is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Hurd; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Michael I. Bushnell.  */

#include "ufs.h"
#include <device/device_reply.h>
#include <device/device_request.h>
#include <device/device.h>
#include <stdio.h>
#include <errno.h>

static struct ioreq *ioreq_cvt (mach_port_t);

/* Start a read on request REQ.  */
void
dev_read (struct ioreq *req)
{
  printf ("device read %d at %d\n", req->amt, req->diskaddr);
  device_read_request (ufs_device, (mach_port_t) req, 0, req->diskaddr,
		       req->amt * DEV_BSIZE);
}

/* Start a write on request REQ.  */
void
dev_write (struct ioreq *req)
{
  if (readonly)
    panic ("dev_write");
  device_write_request (ufs_device, (mach_port_t) req, 0, req->diskaddr,
			(io_buf_ptr_t) req->buffer, req->amt * DEV_BSIZE);
}

/* Write disk block ADDR with DATA of LEN bytes, waiting for completion.  */
error_t
dev_write_sync (daddr_t addr,
		void *data,
		long len)
{
  int foo;
  if (readonly)
    panic ("dev_write");
  if (device_write (ufs_device, 0, addr, data, len, &foo)
      || foo != len)
    return EIO;
  return 0;
}

/* Read disk block ADDR; put the address of the data in DATA; read LEN
   bytes. */
error_t
dev_read_sync (daddr_t addr,
	       void **data,
	       long len)
{
  int foo;
  if (device_read (ufs_device, 0, addr, len, (io_buf_ptr_t *) data, (u_int *)&foo)
      || foo != len)
    return EIO;
  return 0;
}

/* This is called by the MiG server when a read started with dev_read
   is done.  Call the callback routine after fixing everything up.  */
error_t
device_read_reply (mach_port_t reply,
		   error_t error,
		   void *data,
		   unsigned int datalen)
{
  struct ioreq *req;

  printf ("DRR\n");

  if (!(req = ioreq_cvt (reply)))
    return EOPNOTSUPP;
  
  if (datalen != req->amt * DEV_BSIZE)
    {
      error = 1;
      vm_deallocate (mach_task_self (), (int) data, datalen);
    }

  if (error)
    req->error = EIO;
  else
    {
      req->buffer = (vm_address_t) data;
      req->error = 0;
    }

  (*req->callback) (req);
  return 0;
}

/* This is called by the MiG server when a write started with dev_write
   is done.  Call the callback routine after fixing everything up.  */
error_t
device_write_reply (mach_port_t reply,
		    error_t error,
		    int len)
{
  struct ioreq *req;
  
  if (!(req = ioreq_cvt (reply)))
    return EOPNOTSUPP;

  if (len != req->amt * DEV_BSIZE || error)
    req->error = EIO;
  else
    req->error = 0;
  
  (*req->callback) (req);
  return 0;
}

/* Shouldn't happen; we never call device_write_request_inband.  */
error_t
device_write_reply_inband(mach_port_t a,
			  error_t b,
			  int c)
{
  printf ("device_write_reply_inband");
  return EOPNOTSUPP;
}

/* Shouln't happen; we never call device_open_request.  */
error_t 
device_open_reply(mach_port_t a,
		  error_t b,
		  mach_port_t c)
{
  printf ("device_open_reply");
  return EOPNOTSUPP;
}

/* Shouldn't happen; we never call device_read_request_inband.  */
error_t
device_read_reply_inband(mach_port_t a,
			 error_t b,
			 void *c,
			 int d)
{
  printf ("device_read_reply_inband");
  return EOPNOTSUPP;
}

static struct ioreq *
ioreq_cvt (mach_port_t port)
{
  if (*(int *)port == PT_DEVIO)
    return (struct ioreq *) port;
  else
    return 0;
}
