# wor.py: Write or reflect
# Spring Thing 2023 candidate

import os
import sys
import re
from collections import defaultdict
import random
import textwrap

import importlib

version_tuple = sys.version_info

use_asterisks = False

if version_tuple[0] < 3:
    sys.exit("Write or Reflect needs Python 3 to run and detect a module. While you could remove code (from importlib to wtxt) I can make no guarantees for its playability.")

try:
    if version_tuple[0] == 3 and version_tuple[1] < 4:
        colorama_loader = importlib.find_loader('colorama')
    else:
        colorama_loader = importlib.util.find_spec('colorama')
except:
    colorama_loader = False

if not colorama_loader:
    cmd = "pip install colorama"
    print("WoR uses minimal text coloring with the colorama package. It looks like you do not have it installed.")
    print("To try and install it, type i. To exit with installation instructions, type x. To continue and use asterisks instead, type a.")
    while 1:
        x = input("      i (install)/x (exit)/a (asterisks) >").lower()
        if x[0] == 'i':
            print("Calling", cmd)
            os.system(cmd)
            sys.exit("You'll need to restart wor.py for the changes to take effect.")
        elif x[0] == 'x':
            print("Run this command to install colorama:")
            sys.exit("    " + cmd)
        elif x[0] == 'a':
            use_asterisks = True
            break

if use_asterisks:
    wtxt = grn = red = yel = '**'
else:
    import colorama
    wtxt = colorama.Style.RESET_ALL
    grn = colorama.Fore.GREEN
    yel = colorama.Fore.YELLOW
    red = colorama.Fore.RED

class GameTracker:

    global_cfg_file = "wor-general.txt"

    # parser states for repeated commands
    NONE = 0
    QUIT = 1
    PROGRESS = 2
    ABOUT = 3
    CREDITS = 4

    # parser states for helpfulness
    NORMAL = 0
    HELPFUL = 1

    # these are deliberately not in the INIT function, as if a person specified they want help, we should let them keep it across instances.
    difficulty = NORMAL       # we want difficulty to fall through from 1st part to 2nd
    pp_noted = False          # first PP
    ppp_noted = False         # first PPP
    initial_choice = False    # choose normal or helpful
    
    constant_text_arrays = defaultdict(list)
    text_arrays = defaultdict(list)

    def __init__(self, string_cfg_file = "aaa.txt"):

        if len(self.constant_text_arrays) == 0:
            self.get_data(self.global_cfg_file, general_info = True)

        if self.initial_choice == False:
            self.choose_normal_or_helpful()

        self.invalid_sequences = defaultdict(str) # e.g. RR in first scenario, RRR/WWW in second

        self.level_sequence = defaultdict(list) # this is text for if you work through a level
        self.level_up_msgs = defaultdict(str) # what is shown at the start of the level?
        self.level_index = defaultdict(int)

        self.special_user_sequences = defaultdict(str) # for instance, WWWWW gives a response
        self.special_user_sequences_got = defaultdict(bool) # for instance, WWWWW gives a response
        self.substory_sequences = defaultdict(bool) # this is to make sure there are no duplicates for SEQ1-1, etc.

        self.integer_hashes = defaultdict(int)

        self.good_guesses = 0

        self.get_data(string_cfg_file)

        self.length_level = 1
        self.screen_width = 100
        self.parser_state = self.NONE

        self.this_cfg_file = string_cfg_file
        self.current_string = ''
        self.done_strings = []
        self.last_strings = [ '' ]
        self.second_last_strings = []
        self.these_strings = [ 'r', 'w' ]
        self.lengths_array = []
        self.total_in_past = []
        self.number_nudge = ''

        self.text_wrap_array(self.level_up_msgs[1])
        self.get_input()
        return

    def choose_normal_or_helpful(self):
        self.text_wrap_array(self.text_arrays['INTROCHOICE'])
        self.initial_choice = True
        while 1:
            x = input("(H)elpful or (N)ormal? (case insensitive, as all text inputs will be)").lower()
            if x.startswith('h'):
                self.difficulty = self.HELPFUL
                break
            elif x.startswith('h'):
                self.difficulty = self.NORMAL
                break
        return

    def player_help(self):
        self.text_wrap_array(self.text_arrays['HELP'])
        if self.length_level >= 5:
            self.text_wrap_array(self.text_arrays['HELPX'])
        return

    def get_data(self, string_cfg_file, general_info = False):
        with open(string_cfg_file) as file:
            for (line_count, line) in enumerate (file, 1):
                if line.startswith("#"):
                    continue
                if line.startswith(";"):
                    break
                ary = line.replace("\\n", "\n").strip().split("\t")
                if ary[0].isdigit():
                    self.level_up_msgs[int(ary[0])] = ary[1:]
                elif ary[0] == 'x':
                    self.invalid_sequences[ary[1]] = ary[2]
                elif ary[0].startswith("$SEQ"):
                    sequence_marker = ary[0][4:]
                    if not sequence_marker:
                        print("WARNING sequence", ary[0], "needs sub-value at line", line_count)
                    if sequence_marker in self.substory_sequences:
                        print("WARNING substory sequence duplicated at line", line_count)
                    temp_array = sequence_marker.split('-')
                    array_level = int(temp_array[0])
                    q = sorted(random.sample(range(0, len(self.level_sequence[array_level]) + len(ary[1:])), len(ary[1:])))
                    for a in range(0, len(q)):
                        self.level_sequence[array_level].insert(q[a], ary[1+a])
                elif ary[0].startswith("~"):
                    self.special_user_sequences[ary[0][1:]] = ary[1]
                elif ary[0] == 'TXT':
                    try:
                        self.text_arrays[ary[1]].append(ary[2])
                    except:
                        sys.exit("Bailing bad CFG value {} line {}.".format(string_cfg_file, line_count))
                elif ary[0] == 'INT':
                    try:
                        self.integer_hashes[ary[1]] = int(ary[2])
                    except:
                        sys.exit("Bailing bad CFG value {} line {}.".format(string_cfg_file, line_count))
                else:
                    print("BAD CFG BASE VALUE line {}, need to start with INT or TXT and not variable name: {}.".format(line_count, 'is', ary[0]))
        if general_info:
            return
        if 'MIN_GUESS' not in self.integer_hashes:
            print(red + "We need to set the min_guess_chapter with MIN_GUESS in {}.".format(string_cfg_file) + wtxt)
        if 'GUESS_TO_WIN' not in self.integer_hashes:
            print(red + "We need to set the guess_to_win with GUESS_TO_WIN in {}.".format(string_cfg_file) + wtxt)
        if 'ENDGAME_TEXT' not in self.text_arrays:
            print(red + "We need to set endgame text in {}.".format(string_cfg_file) + wtxt)
        return

    def print_joined(self, so_far):
        sf2 = sorted(so_far)
        per_line = 1 + (self.screen_width - 1) // (self.length_level + 2)
        for x in range(0, len(so_far), per_line):
            print(grn + ', '.join(so_far[x:x + per_line]).upper() + wtxt)

    def pp_basic(self):
        lsd = len(self.done_strings)
        if lsd:
            print("For chapter {}, you've had {} successful combination{} of writing and reflecting:".format(self.length_level, lsd, 's' if lsd > 1 else ''))
            self.print_joined(sorted(self.done_strings))
        else:
            print(yel + "You've completed no successful combinations of writing and reflecting for chapter {}.".format(self.length_level) + wtxt)
        if not self.ppp_noted:
            print(yel + "This is a one-time nag to note PPP gives additional detail about which writing sessions you've used for all chapters, {} you've gotten past chapter 1.".format('once' if self.length_level == 1 else 'now') + wtxt)
            self.ppp_noted = True

    def get_input(self):
        while 1:
            remaining = self.length_level - len(self.current_string)
            x = input("W (write) or R (reflect){}?{}".format( " ({} so far)".format(self.current_string.upper()) if self.current_string else '', self.number_nudge + ('?' if self.number_nudge else ''))).strip().lower()
            if in_test:
                print()
                if x.startswith("#"):
                    continue
                if x.startswith(";"):
                    sys.exit()
                print("TEST INPUT THIS TURN:", x)
            print()
            if ' ' in x:
                print("No need for spaces.")
                x = x.replace(' ', '')
            if not x or x == '?':
                self.player_help()
                continue
            if x == 'h':
                if self.difficulty == self.HELPFUL:
                    print("Already in HELPFUL mode.")
                    continue
                print("Switching to HELPFUL mode.")
                self.difficulty = self.HELPFUL
                if self.any_valid(self.current_string):
                    continue
                print("You think for a bit. You've been through all the patterns ahead of {}, you're pretty sure. So you start over.".format(self.current_string))
                self.current_string = ''
                continue
            if x == 'n':
                if self.difficulty == self.NORMAL:
                    print("Already in NORMAL mode.")
                else:
                    print("Switching to NORMAL mode. You will no longer get hints about paths you've already covered.")
                self.difficulty = self.NORMAL
                continue
            if x == 'q':
                if self.parser_state == self.QUIT:
                    print("Quitting.")
                    sys.exit()
                print("Just to make sure you want to do this, retype Q to quit.")
                self.parser_state = self.QUIT
                continue
            if x == 'a':
                if self.parser_state == self.ABOUT:
                    self.text_wrap_array(self.text_arrays['ABOUTX'])
                else:
                    self.text_wrap_array(self.text_arrays['ABOUT'])
                if self.parser_state != self.ABOUT:
                    self.parser_state = self.ABOUT
                else:
                    self.parser_state = self.NONE
                continue
            if x == 'c':
                if self.parser_state == self.CREDITS:
                    self.text_wrap_array(self.text_arrays['CREDITSX'])
                else:
                    self.text_wrap_array(self.text_arrays['CREDITS'])
                    self.text_wrap_array(self.text_arrays['CREDITSX'])
                if self.parser_state != self.CREDITS:
                    self.parser_state = self.CREDITS
                else:
                    self.parser_state = self.NONE
                continue
            if x == 'p':
                lsl = len(self.lengths_array)
                lsd = len(self.done_strings)
                if lsl == 0:
                    print(yel + "You haven't completed any strong writing sessions for chapter {} yet.".format(self.length_level) + wtxt)
                else:
                    print("Right now, you're on chapter {}, working on writing/reflecting sessions of length {}.".format(self.length_level, self.length_level))
                if not len(self.done_strings):
                    print(yel + "You haven't had any good reflecting or writing streaks for chapter {} yet.".format(self.length_level) + wtxt)
                if not self.pp_noted:
                    print(yel + "This is a one-time nag to note PP gives additional detail about which writing sessions you've used." + wtxt)
                    self.pp_noted = True
                continue
            if x == 'pp':
                self.pp_noted = True
                self.pp_basic()
                continue
            if x == 'ppp':
                self.ppp_noted = True
                self.pp_basic()
                if self.length_level == 1:
                    print("You will be able to look back on your study patterns lately, but there are no chapters to look back on, yet.")
                    continue
                print("Here is your process that has gotten writing in the books:")
                for idx in range(0, len(self.total_in_past)):
                    print("Chapter {}:".format(idx + 1), ", ".join(self.total_in_past[idx]))
                continue
            self.parser_state = self.NONE
            if x.isdigit():
                ix = int(x)
                if self.length_level < self.integer_hashes['MIN_GUESS']:
                    print("Sorry, but I'm not going to let you guess how much to do this early. Get to chapter {}.".format(self.integer_hashes['MIN_GUESS']))
                    continue
                if ix != len(self.these_strings):
                    print("You plan ... but no, that's not quite how many different sessions you need to work something out.")
                    continue
                self.good_guesses += 1
                if self.good_guesses == self.integer_hashes['GUESS_TO_WIN']:
                    print("Yes. You have it now. It's a Fibonacci sequence.")
                    print()
                    self.text_wrap_array(self.text_arrays['ENDGAME_TEXT'])
                    if in_test:
                        for x in self.level_sequence:
                            if len(self.level_sequence[x]) == self.level_index[x]:
                                continue
                            if self.level_index[x] == 0:
                                continue
                            print("Chapter/level {} showed only {} of {} sentences in data file.".format(x, self.level_index[x], len(self.level_sequence[x])))
                    break
                else:
                    print("You do some major-scale planning. You know what's left, and you're pretty sure how to achieve it.")
                    self.bump_up_level()
                    continue
            if x.replace('w', '').replace('r', ''):
                print("You only need to (W)rite or R(eflect), case-insensitively{}. ? or an empty command gives help.".format(self.number_nudge))
                continue
            if len(x) > self.length_level - len(self.current_string):
                print("That's looking too far ahead. Right now, you only need to plan {} more block{} of time.".format(remaining, 's' if remaining > 1 else ''))
                continue
            candidate = self.current_string + x
            if len(candidate) > self.length_level:
                delta = self.length_level - len(self.current_string)
                print("No, that's too much planning at once. You can look {} hour{} ahead.".format(delta, 's' if delta > 1 else ''))
                return
            if not self.valid_string(candidate, print_reject = True):
                continue
            if self.difficulty == self.HELPFUL:
                if not self.any_valid(candidate):
                    print("Hmm. Planning your time like that might leave you {}.".format('with no new paths to explore' if len(candidate) < self.length_level else 'down a creative path you\'ve already traversed'))
                    continue
            self.current_string = candidate
            remaining = len(candidate) - self.length_level
            if not remaining:
                if self.current_string in self.done_strings:
                    if len(candidate) == 1:
                        print("You'll want to start simple, here. Balance time writing and reflecting. You can combine the two later.")
                    else:
                        print("Something got written or thought. But ... nothing substantial. You already went down that path, this go-round. You need to find a new one.")
                    self.current_string = ''
                    continue
                if self.current_string in self.special_user_sequences and not self.special_user_sequences_got[self.current_string]:
                    self.text_wrap_array([self.special_user_sequences[self.current_string]])
                    self.special_user_sequences_got[self.current_string] = True
                elif self.level_index[self.length_level] < len(self.level_sequence[self.length_level]):
                    self.text_wrap_array(self.level_sequence[self.length_level][self.level_index[self.length_level]])
                    self.level_index[self.length_level] += 1
                else:
                    if self.length_level in self.level_sequence:
                        print("New ideas form. They should be more specific, but I forgot to fill them in! This is a bug that I should fill in, in {}.".format(self.this_cfg_file))
                    else:
                        print("You're almost on cruise control with ideas here. I can't help you with the details. Perhaps now is the time to take a bigger-picture approach and organize how many different patterns you'll need, to work through this chapter.")
                self.done_strings.append(self.current_string)
                self.current_string = ''
                if len(self.done_strings) == len(self.these_strings):
                    print(grn + "You take a step around after proofreading. Chapter {} is, more or less, completed. Time to take on a bit more!".format(self.length_level) + wtxt)
                    print()
                    self.bump_up_level()
                    continue
                print("That was a good, constructive session, even if you didn't break off anything major.")
                continue
            print("You wonder what else to do -- write or reflect.")

    def text_wrap_array(self, my_array, extra_cr = True):
        try:
            term_width = os.get_terminal_size().columns
        except:
            term_width = 5000
        if type(my_array) == str:
            if in_test:
                print(red + "WARNING! You need to convert code or structures from STR to LIST or DICT." + wtxt)
                print("Culprit = ", my_array)
            my_array = [ my_array ]
        for t in my_array:
            for tw in textwrap.wrap(t, term_width):
                print(tw)
            if extra_cr:
                print()

    def any_valid(self, this_string = ''):
        if not this_string:
            this_string = self.current_string
        for x in self.these_strings:
            if x in self.done_strings:
                continue
            if not x.startswith(this_string):
                continue
            return True
        return False

    def bump_up_level(self):
        self.lengths_array.append(len(self.these_strings))
        self.second_last_strings = list(self.last_strings)
        self.last_strings = list(self.these_strings)
        self.total_in_past.append(self.last_strings)
        self.these_strings = [ 'r' + x for x in self.last_strings if self.valid_string('r' + x)]
        self.these_strings.extend([ 'w' + x for x in self.last_strings if self.valid_string('w' + x)])
        self.length_level += 1
        self.done_strings = []
        self.text_wrap_array(self.level_up_msgs[self.length_level if self.length_level in self.level_up_msgs else 0])
        if self.length_level >= self.integer_hashes['MIN_GUESS']:
            self.number_nudge = ' or guess how many sessions of length {} are possible'.format(self.length_level)

    def valid_string(self, my_string, print_reject = False):
        for x in self.invalid_sequences:
            if x in my_string:
                if print_reject:
                    self.text_wrap_array(self.invalid_sequences[x])
                return False
        return True

count = 1

in_test = False

while count < len(sys.argv):
    arg = sys.argv[count]
    if arg == 't':
        in_test = True
    else:
        sys.exit("Unknown parameter. t is the only real parameter, for testing.")
    count += 1

game_1 = GameTracker("wor1.txt")
game_2 = GameTracker("wor2.txt")
