! ----------------------------------------------------------------------------
! FORM: Bureaucracy-style forms in Inform.
!
! The following code has been tested under Unix Inform 5.5 (v1502/a) and
! JZIP 2.0.1f. While Inform 5.5 is required to compile it, I hope other
! interpreters are able to handle the result as well.
!
! The program started out as a reply to a Usenet post to one of the
! int-fiction newsgroups. Despite a massive lack of outside input - until
! recently, at least - it kept growing as I learned more about Inform and the
! recent new edition (0.2) of the Z-machine specification.
!
! At the moment, it does violate the specification in assuming that backspace
! will cause @read_char to return either 8 or 127, but I am determined to get
! it to comply with the specification some day, even if it means having to
! wait for the next edition of it.
!
! I think the program has reached a reasonably stable change where I hope I
! won't have to make any further changes that break backwards compatibility.
! Still, there is probably room for improvements, so if you have any, I'd
! be very interested in seeing them. Or, even better, why not submit them to
! the IF archive?
!
! I wrote this program, partly to learn about Inform, but mostly just for fun.
! If you can find anything useful in it, then by all means use it. Fold,
! spindle and mutilate to your heart's content.
!      _
! Torbjorn Andersson, November 1995, No Rights Reserved
! ----------------------------------------------------------------------------

! ----------------------------------------------------------------------------
! CONSTANTS, PROPERTIES, ATTRIBUTES
! ----------------------------------------------------------------------------

! The following text is used for illegal characters (which at the moment is
! anything below 32 or above 127). And since I figured it might be useful in
! user-defined filters, I made it a constant, which should mean it is only
! stored once in memory.

Constant InvalidChar	"ERROR: Invalid character.";

Property form_width;		! Screen width of form. Default is 40.
Property form_text;		! Specification of the form.
Property form_parser;		! Parsing function for the form.
Property form_select;		! Function for selecting next question.

! I could have used a number of properties for the form's internal data, but
! instead I decided to put them all in an array. These are the indexes into
! said array.
!
! I could also have stored the boolean flags in one byte, but it's not like
! I'm running out of memory ...

Constant FormDataSize	8;

Constant FORM_LINES	0;	! Number of lines used by the form's questions
Constant FORM_CUR_X	1;	! Cursor's current X coordinate
Constant FORM_CUR_Y	2;	! Cursor's current Y coordinate
Constant FORM_LEFT	3;	! Leftmost position of form
Constant FORM_QLINE	4;	! Line on which the form's questions start
Constant FORM_NOISY	5;	! Should a form message produce noise?
Constant FORM_MSG_FLAG	6;	! There is a message in the message field.
Constant FORM_UPQ	7;	! The user used "^" to back up one step.

! The following constants have been added to - hopefully - simplify extensions
! to the way questions are stored in the 'form_text' property.

Constant FQSize		5;	! Number of elements per question

Constant FQLine		0;	! Question's line in question field
Constant FQCol		1;	! Question's column in question field
Constant FQInpLen	2;	! Question's max input length
Constant FQFilter	3;	! Question's filter function
Constant FQText		4;	! Question's text

! Constants for various ASCII codes.

Constant BS		$08;
Constant NL		$0a;
Constant CR		$0d;
Constant SO		$0e;
Constant DEL		$7f;

! ----------------------------------------------------------------------------
! GLOBAL VARIABLES
! ----------------------------------------------------------------------------

! The form-printing functions. Most of the code has been put into a specific
! form class, but some help functions are still needed.

Global form_buffer		-> 80;

! This used to be a property, but since only one form at a time needs the
! information, it might as well be a global array instead.

Global form_data 		-> FormDataSize;

! If the parsing function sets this variable to non-zero, input is rejected.
! After rejecting input, the variable is automatically re-set to 0.

Global form_parse_reject	= 0;

! This constant tells how many blank lines there should be above the top of
! the form.

Global form_gap			= 3;

! ----------------------------------------------------------------------------
! HELP FUNCTIONS
! ----------------------------------------------------------------------------

! This function is called to print a message in the form's message field.
! Perhaps it would be nicer to make this part of the class definition, but
! that would mean wasting a property. I can't use the <fake_action arg>
! construct, since the normal parser has been - temporarily - put on hold.

[ Form_Alert o str;

    ! A disquieting rumor has it that some interpreters print the character
    ! BEL [7] to get this sound effect. Since I don't know if this will show
    ! up as a blank on the screen, I'm testing 'str' twice; once before and
    ! once after moving the cursor.

    if (form_data->FORM_NOISY == 1 && str ~= NULL)
	@sound_effect 1;

    SetCursor(form_data->FORM_QLINE - 2, form_data->FORM_LEFT + 1);

    if (str ~= NULL) {
	form_data->FORM_MSG_FLAG = 1;
	style bold;
	print (string) str;
	style roman;
    } else {
	form_data->FORM_MSG_FLAG = 0;
	spaces o.form_width - 2;
    }

    SetCursor(form_data->FORM_CUR_Y, form_data->FORM_CUR_X);
];

! This might look like the most redundant function ever devised, but it seems
! the @set_cursor instruction wants constants or variables - not compound
! expressions - which makes sense, considering it's a low-level instruction.
		
[ SetCursor y x;
    @set_cursor y x;
];

! This function finds out the length of a string. The idea was taken from
! the answer to exercise 96 in the Inform Designer's Manual, 2nd Edition.

Global printed_text -> 80;

[ StringLength str;
    @output_stream 3 printed_text;
    print (string) str;
    @output_stream -3;
    return printed_text-->0;
];

! ----------------------------------------------------------------------------
! FORM CLASS DEFINITION
! ----------------------------------------------------------------------------
!
! The standard form class. Originally, I had conceived it as being a separate
! function instead, but it struck me that doing it in this slightly more
! object-oriented way meant I didn't have to pass so many seeminly arbitrary
! arguments to my help functions; all I had to do was pass the object.
!
! The code doesn't do much to take screen size into consideration. It does
! check the number of columns in order to center the form; everything else is
! left to the programmer.
!
! A rough map of the "before" rule for Complete is:
!
! (A) Initialisation; figure out the size of the form
! (B) Print the form
! (C) Wait for user input
! (D) Clean up afterwards

Class Form_Class
  with	form_width	40,
	form_text	NULL,
	form_select	NULL,
	form_parser	NULL,
	after [;
	    Complete:
		print "You have filled out ", (the) self, ".^";
	],
	before [ i j k m n p q addr key ret_val;
	    Complete:
		if (self.form_text == NULL)
		    "This form seems to be blank.";

		! This test should be able to determine if the 'form_text'
		! property was defined within the object definition itself or
		! outside, in a separate array.
		!
		! In the former case, the size should be the size of the array
		! itself, which is at least 3 words, i.e 6 bytes. In the latter
		! case, the size is the size of the address where the array is
		! stored, i.e. 2 bytes.
		!
		! ZRegion() seems to return 3 in the latter case and 2 in the
		! former case, but that signifies "strings" and "code area"
		! respectively, I find the result of this test too confusing
		! to use.

		if (self.#form_text > 2)
		    addr = self.&form_text;
		else
		    addr = self.form_text;

		@erase_window NULL;

		! **** (A) ****

		! First, find out where in the 'form_text' property the
		! questions begin and end.

		j = 3 + addr-->0;
		k = j + FQSize * addr-->(j - 1);

		! If the form's width is unspecified, calculate the smallest
		! possible form based on the header, the message field and
		! the questions.

		if (self.form_width == -1) {

		    ! Find the longest string in the header

		    if (addr-->0 > 0) {
			for (i = 0, n = 0 : i < addr-->0 : i++) {
			    m = StringLength(addr-->(i + 1));

			    if (m > n)
				n = m;
			}
		    }

		    ! Find the length of the message field message.

		    if (addr-->(1 + addr-->0) ~= NULL) {
			m = StringLength(addr-->(1 + addr-->0));

			if (m > n)
			    n = m;
		    }

		    ! Find the question with the largest space requirements.
		    ! Add space for the blank between the text and the input
		    ! field, and another one after the input field so the
		    ! cursor won't be put on the border of the form.

		    for (i = j : i < k : i = i + FQSize) {
			m = 2 + addr-->(i + FQCol) + addr-->(i + FQInpLen)
			  + StringLength(addr-->(i + FQText));

			if (m > n)
			    n = m;
		    }

		    ! Add two for the form's border and save the information
		    ! in the form_width property.

		    self.form_width = n + 2;
		}

		! To put the form in the middle of the screen we need to know
		! how wide the screen is. Checking the result of 0->33 should
		! be enough, but I notice Graham always add a check to see if
		! this value is 0; the result of a faulty Z-code interpreter,
		! I guess.

		n = 0->33;

		if (n == 0)
		    n = 80;

		form_data->FORM_LEFT = (n / 2) - (self.form_width / 2);

		! Traverse the questions to figure out how many screen lines
		! the questions need. The beginning and end of the questions
		! are still stored in 'j' and 'k' respectively.

		for (n = 0, i = j : i < k : i = i + FQSize) {
		    if (addr-->(i + FQLine) > n)
			n = addr-->(i + FQLine);
		}

		! Store the number of lines used by the questions in the
		! FORM_LINES data field, and make the tentative guess about on
		! which screen line the questions start.

		form_data->FORM_LINES = n + 1;
		form_data->FORM_QLINE = form_gap + 4;

		! The variable 'n' now holds the nethermost line used by any
		! question in the form. To find out how many lines the whole
		! form needs, add the 'form_gap' variable for the number of
		! blank lines at the top of the form.
		!
		! Add another one line because n + 1 is the number of lines
		! used by questions in the form.
		!
		! Add another three for the message field and the border
		! around it.
		!
		! Add another one for the last line of the border around the
		! questions.

		n = n + form_gap + 5;

		! If the form has a header, we have to count the space used
		! by that as well. Add the number of lines in the header, plus
		! one line for the extra border needed around the header.
		!
		! Of course, the same adjustment has to be done to the
		! FORM_QLINE data field.

		if (addr-->0 > 0) {
		    n = n + 1 + addr-->0;
		    form_data->FORM_QLINE =
			1 + form_data->FORM_QLINE + addr-->0;
		}

		! **** (B) ****

		! Split the screen into two windows. The form will use the
		! upper window (1), while the lower window probably won't be
		! used at all.
		!
		! This means a lot of the calculations above weren't really
		! necessary, as I could have just split off the largest window
		! the interpreter would allow, but I think that would have
		! been sloppy.

		@split_window n;
		@set_window   1;
		@buffer_mode  0;

		! Start the printing on the first line after the 'gap' above
		! the form.

		n = form_gap + 1;

		style reverse;

		! If there is a form header, draw it. Each line of it should be
		! centered.

		if (addr-->0 ~= 0) {
		    SetCursor(n, form_data->FORM_LEFT);
		    spaces self.form_width;

		    for (i = 0 : i < addr-->0 : i++) {
			SetCursor(++n, form_data->FORM_LEFT);

			j = StringLength(addr-->(i + 1));

			! Now, given the length of the string and the width of
			! the form, centering the string is fairly trivial.
			!
			! It is tempting to believe that the same expression
			! could be used for the number of spaces on both sides
			! of the string. This is not the case, however, as
			! their width might differ by 1.

			spaces (self.form_width - j) / 2;
			print (string) addr-->(i + 1);
			spaces self.form_width - (self.form_width - j) / 2 - j;
		    }

		    n++;
		}

		! Draw the frame around the form's message field.

		SetCursor(n, form_data->FORM_LEFT);
		spaces self.form_width;

		SetCursor(++n, form_data->FORM_LEFT);
		print_char ' ';

		SetCursor(n, form_data->FORM_LEFT + self.form_width - 1);
		print_char ' ';

		SetCursor(++n, form_data->FORM_LEFT);
		spaces self.form_width;

		! Draw the border around the form's questions field.

		for (i = 0 : i < form_data->FORM_LINES : i++) {
		    SetCursor(++n, form_data->FORM_LEFT);
		    print_char ' ';

		    SetCursor(n, form_data->FORM_LEFT + self.form_width - 1);
		    print_char ' ';
		}

		SetCursor(++n, form_data->FORM_LEFT);
		spaces self.form_width;

		! Print the questions

		style roman;
		style bold;

		! Again, figure out where in the array the questions start and
		! end. This is the exact same expressions as used earlier in
		! the code, but at this stage, I've already re-used the
		! variables for other things in between.

		j = 3 + addr-->0;
		k = j + FQSize * addr-->(j - 1);

		! Print all the questions. The coordinates are given relative
		! to the values of FORM_QLINE and FORM_LEFT plus 1 (since the
		! column given by FORM_LEFT is already occupied by the border
		! around the form).

		for (i = j : i < k : i = i + 5) {
		    SetCursor(form_data->FORM_QLINE + addr-->(i + FQLine),
			      form_data->FORM_LEFT + addr-->(i + FQCol) + 1);
		    print (string) addr-->(i + FQText);
		}

		! Is there a message to display in the message field? I'm
		! manipulating form_data here only to avoid trying to set the
		! cursor to position 0 0 which, while not illegal, isn't very
		! nice, as the upper left corner is supposed to be 1 1.

		if (addr-->(j - 2) ~= NULL) {
		    form_data->FORM_NOISY = 0;
		    form_data->FORM_CUR_X = 1;
		    form_data->FORM_CUR_Y = 1;
		    Form_Alert(self, addr-->(j - 2));
		}

		! Subsequent messages in the message field should be
		! accompanied by an annoying beep.

		form_data->FORM_NOISY = 1;

		! **** (C) ****

		! The form is on the screen. It's time to read the input.

		style roman;

		! Throughout the loop, the variable 'k' will keep track of the
		! number of questions on the form.

		k = addr-->(2 + addr-->0);

		for (i = 0 : : i++) {
		    .FNewQuestion;

		    ! Select a new question from the form. The default
		    ! behaviour is to simply select the questions in the order
		    ! they appear in the array.

		    if (self.form_select == NULL) {
			if (i >= k)
			    q = -1;
			else
			    q = i;
		    } else
			q = indirect(self.form_select, self, i, k);

		    ! When all questions have been answered, break out of the
		    ! loop.

		    if (q == -1)
			break;

		    ! Figure out where in the array the question is stored, and
		    ! figure out where to put the cursor. Two is added to the
		    ! column to compensate for the space used by the border,
		    ! and to leave a blank space after the text of the
		    ! question.

                    j = 3 + addr-->0 + q * FQSize;

		    m = form_data->FORM_QLINE + addr-->(j + FQLine);
		    n = form_data->FORM_LEFT + addr-->(j + FQCol)
		      + StringLength(addr-->(j + FQText)) + 2;

		    ! The cursor position is always saved so that it can be
		    ! restored after printing messages to the message field.

		    form_data->FORM_CUR_Y = m;
		    form_data->FORM_CUR_X = n;

		    SetCursor(m, n);

		    ! If the form "backed up" to a previous question it has to
		    ! clear (visually) any old input from the form.

		    if (form_data->FORM_UPQ ~= 0) {
			form_data->FORM_UPQ = 0;
			spaces addr-->(j + FQInpLen);
			SetCursor(m, n);
		    }

		    ! Read a line from the keyboard. Although there are
		    ! built-in functions for doing this, read the input one
		    ! character at a time. This allows the form to reject
		    ! individual characters and to convert accepted characters
		    ! to upper case.
		    !
		    ! The variable 'p' will keep track of the number of read
		    ! characters.

		    for (p = 0::) {
			.FNewInput;
			@read_char 1 key;

			! If there is a message in the message field, clear it
			! away.

			if (form_data->FORM_MSG_FLAG ~= 0)
			    Form_Alert(self, NULL);

			! Handle the pressed key.

			switch (key) {

			    ! End of line. The current version of the
			    ! Z-machine specification suggests that both NL
			    ! [10] and CR [13] should work, but that NL is
			    ! preferable. Bureaucracy only seems to check for
			    ! CR, though. For safety, I'm checking both.
			    !
			    ! NOTE: This is especially wise because Bureaucracy
			    !       only seems to check for CR, and not for NL.
			    !
			    ! End of line is only accepted if the user has
			    ! actually typed something before that.

			    NL, CR:
				if (p > 0) {
				    ! Store the length of the input and break
				    ! out of the loop.

				    form_buffer-->0 = p;
				    jump FQuestionDone;
				} else
				    Form_Alert(self,
					 "ERROR: Incomplete field entry.");

			    ! Backspace could be either BS [8] or DEL [127],
			    ! and I believe Bureaucracy checks both.
			    !
			    ! Backspace is only accepted if there is anything
			    ! to erase. No change is made to the contents of
			    ! the buffer, since erased characters will either
			    ! be overwritten by subsequent characters, or not
			    ! included by the buffer length.
			    !
			    ! NOTE: This is a direct violation of paragraph
			    !       10.7 of version 0.2 of the Z-machine
			    !       specification, which states that none of
			    !       these character codes are read by the
			    !       interpreter. I have reason to believe that
			    !       future specifications will allow for it,
			    !       though.

			    BS, DEL:
				if (p > 0) {
				    ! Decrease the number of read characters.

				    p--;

				    ! Adjust cursor position information.

				    form_data->FORM_CUR_X =
					form_data->FORM_CUR_X - 1;

				    ! Erase character from screen and
				    ! reposition the cursor.

				    SetCursor(form_data->FORM_CUR_Y,
					      form_data->FORM_CUR_X);
				    print_char ' ';
				    SetCursor(form_data->FORM_CUR_Y,
					      form_data->FORM_CUR_X);
				} else
				    Form_Alert(self,
					"ERROR: 1st char in field.");

			    ! Any other character.

			    default:

				! If it's the first character, check if it is
				! "^", to allow the user to back up one
				! question.
				!
				! NOTE: Jumping back to FNewQuestion means that
				!       the form gets to pick another question.
				!       It is probably a good idea to make sure
				!       it will pick the same question, but
				!       that is left to the user.
				!
				! NOTE: Bureaucracy seems to accept SO [ASCII
				!       14] as synonym for "^", so I'm doing
				!       that, too, even if I don't know why.

				if (p == 0 && key == 94 or SO) {
				    if (i > 0) {
					form_data->FORM_UPQ = 1;
					i--;
				    } else
					Form_Alert(self,
					    "ERROR: Top of form.");

				    jump FNewQuestion;
				}

				! Make sure we have not exceeded the max input
				! length for the question.

				if (p >= addr-->(j + FQInpLen)) {
				    Form_Alert(self, "ERROR: End of field.");
				    jump FNewInput;
				}

				! Only accept printable ASCII characters.
				!
				! The characters 32-126 (Standard ASCII) and
				! 155-251 (Accented letter codes) are
				! are considered to be printable (even if
				! only 155-219 actually have suggested
				! equivalences in version 0.2 of the Z-machine
				! specification)

				if (key < ' ' ||
				   (key > 126 && key < 155) ||
				   key > 251) {
				    Form_Alert(self, InvalidChar);
				    jump FNewInput;
				}

				! Convert lower-case to upper-case before
				! giving it to the character filter.

				if (key >= 'a' && key <= 'z')
				    key = key - ('a' - 'A');

				! Is there a character filter for this
				! question? If so, call it.

				if (addr-->(j + FQFilter) ~= NULL) {
				    ret_val = indirect(addr-->(j + FQFilter),
					key, p);

				    ! If the filter rejects the character,
				    ! print the error message supplied by the
				    ! filter and jump back to the beginning of
				    ! the loop to get a new character.

				    if (ret_val ~= NULL) {
					Form_Alert(self, ret_val);
					jump FNewInput;
				    }
				}

				! Print the character and update the cursor
				! position info.

				print_char key;
				form_data->FORM_CUR_X =
				    form_data->FORM_CUR_X + 1;

				! Store the character in the buffer and
				! increase the character counter.

				form_buffer->(p + 2) = key;
				p++;
			}
		    }

		    ! The question has been answered. Pass the object and the
		    ! question number to the parser. The input is in the
		    ! global variable form_buffer.

		    .FQuestionDone;

		    if (self.form_parser ~= NULL) {
			ret_val = indirect(self.form_parser, self, q);

			if (ret_val ~= NULL)
			    Form_Alert(self, ret_val);

			! Allow the form to reject the input based on the
			! whole input. I originally meant to set the
			! FORM_UPQ data field here, but that will fail
			! spectacularly if the form, for some reason, selects
			! a different question next time.

			if (form_parse_reject ~= 0) {
			    SetCursor(
				form_data->FORM_QLINE + addr-->(j + FQLine),
				form_data->FORM_LEFT + addr-->(j + FQCol)
			      + StringLength(addr-->(j + FQText)) + 2);
			    spaces(addr-->(j + FQInpLen));
			    form_parse_reject = 0;
			    i--;
			}
		    }
		}

		! **** (D) ****

		! All questions have been answered. Restore the screen to
		! (hopefully) some semblance of normality. I don't know if
		! this conflicts in any way with customized status lines, but
		! I hope not. Another thing left as an exercise for the
		! reader ...

		@erase_window NULL;
		@split_window 1;
		@set_window   0;
		@buffer_mode  1;

		! I hope this will take care of running the 'after' property
		! correctly. This should be the last of the 'before' rules
		! for Complete in this object, so it has to return true to
		! avoid falling into the CompleteSub() function.

		RunRoutines(self, after);
		rtrue;
	];

! ----------------------------------------------------------------------------
! GRAMMAR ADDITIONS
! ----------------------------------------------------------------------------

! This is the default "when-all-else-fails" rule.

[ CompleteSub; "That is easier said than done."; ];

Verb "complete"
	* noun				-> Complete;

Extend "fill"
	* "out" noun			-> Complete;

! ----------------------------------------------------------------------------
