/*
 * stream.c
 *
 * IO stream implementation
 *
 */

#include "frotz.h"
#include "s5api.h"

extern short handle_hot_key (struct sg *g, zchar);
extern short validate_click (struct sg *g);
extern void replay_open (struct sg *g);
extern void replay_close (struct sg *g);
extern void memory_open (struct sg *g, zword, zword, short);
extern void memory_close (struct sg *g);
extern void record_open (struct sg *g);
extern void record_close (struct sg *g);
extern void script_open (struct sg *g);
extern void script_close (struct sg *g);
extern void memory_word (struct sg *g, const zchar *);
extern void memory_new_line (struct sg *g);
extern void record_write_key (struct sg *g, short);
extern void record_write_input (struct sg *g, const zchar *, zchar);
extern void script_char (struct sg *g, zchar);
extern void script_word (struct sg *g, const zchar *);
extern void script_new_line (struct sg *g);
extern void script_write_input (struct sg *g, const zchar *, zchar);
extern void script_erase_input (struct sg *g, const zchar *);
extern void script_mssg_on (struct sg *g);
extern void script_mssg_off (struct sg *g);
extern void screen_char (struct sg *g, zchar);
extern void screen_word (struct sg *g, const zchar *);
extern void screen_new_line (struct sg *g);
extern void screen_write_input (struct sg *g, const zchar *, zchar);
extern void screen_erase_input (struct sg *g, const zchar *);
extern void screen_mssg_on (struct sg *g);
extern void screen_mssg_off (struct sg *g);
extern zchar replay_read_key (struct sg *g);
extern zchar replay_read_input (struct sg *g, zchar *);
extern zchar console_read_key (struct sg *g, zword);
extern zchar console_read_input (struct sg *g, short, zchar *, zword, short);
extern short direct_call (struct sg *g, zword);

/*
 * stream_mssg_on
 *
 * Start printing a "debugging" message.
 *
 */

void stream_mssg_on (struct sg *g)
{

    flush_buffer (g);

    if (g->ostream_screen)
	screen_mssg_on (g);
    if (g->ostream_script && g->enable_scripting)
	script_mssg_on (g);

    g->message = TRUE;

}/* stream_mssg_on */

/*
 * stream_mssg_off
 *
 * Stop printing a "debugging" message.
 *
 */

void stream_mssg_off (struct sg *g)
{

    flush_buffer (g);

    if (g->ostream_screen)
	screen_mssg_off (g);
    if (g->ostream_script && g->enable_scripting)
	script_mssg_off (g);

    g->message = FALSE;

}/* stream_mssg_off */

/*
 * z_output_stream, open or close an output stream.
 *
 *	zargs[0] = stream to open (positive) or close (negative)
 *	zargs[1] = address to redirect output to (stream 3 only)
 *	zargs[2] = width of redirected output (stream 3 only, optional)
 *
 */

void z_output_stream (struct sg *g)
{

    flush_buffer (g);

    switch ((short) (g->zargs[0])) {

    case  1: g->ostream_screen = TRUE;
	     break;
    case -1: g->ostream_screen = FALSE;
	     break;
    case  2: if (!g->ostream_script) script_open (g);
	     break;
    case -2: if (g->ostream_script) script_close (g);
	     break;
    case  3: memory_open (g, g->zargs[1], g->zargs[2], g->zargc >= 3);
	     break;
    case -3: memory_close (g);
	     break;
    case  4: if (!g->ostream_record) record_open (g);
	     break;
    case -4: if (g->ostream_record) record_close (g);
	     break;

    }

}/* z_output_stream */

/*
 * stream_char
 *
 * Send a single character to the output stream.
 *
 */

void stream_char (struct sg *g, zchar c)
{

    if (g->ostream_screen)
	screen_char (g,c);
    if (g->ostream_script && g->enable_scripting)
	script_char (g,c);

}/* stream_char */

/*
 * stream_word
 *
 * Send a string of characters to the output streams.
 *
 */

void stream_word (struct sg *g, const zchar *s)
{

    if (g->ostream_memory && !g->message)

	memory_word (g,s);

    else {

	if (g->ostream_screen)
	    screen_word (g,s);
	if (g->ostream_script && g->enable_scripting)
	    script_word (g,s);

    }

}/* stream_word */

/*
 * stream_new_line
 *
 * Send a newline to the output streams.
 *
 */

void stream_new_line (struct sg *g)
{

    if (g->ostream_memory && !g->message)

	memory_new_line (g);

    else {

	if (g->ostream_screen)
	    screen_new_line (g);
	if (g->ostream_script && g->enable_scripting)
	    script_new_line (g);

    }

}/* stream_new_line */

/*
 * z_input_stream, select an input stream.
 *
 *	zargs[0] = input stream to be selected
 *
 */

void z_input_stream (struct sg *g)
{

    flush_buffer (g);

    if (g->zargs[0] == 0 && g->istream_replay)
	replay_close (g);
    if (g->zargs[0] == 1 && !g->istream_replay)
	replay_open (g);

}/* z_input_stream */

/*
 * stream_read_key
 *
 * Read a single keystroke from the current input stream.
 *
 */

zchar stream_read_key ( struct sg *g, zword timeout, zword routine,
			short hot_keys )
{
    zchar key = ZC_BAD;

    flush_buffer (g);

    /* Read key from current input stream */

continue_input:

    do {

	if (g->istream_replay)
	    key = replay_read_key (g);
	else
	    key = console_read_key (g,timeout);

    } while (key == ZC_BAD);

    /* Verify mouse clicks */

    if (key == ZC_SINGLE_CLICK || key == ZC_DOUBLE_CLICK)
	if (!validate_click (g))
	    goto continue_input;

    /* Copy key to the command file */

    if (g->ostream_record && !g->istream_replay)
	record_write_key (g,key);

    /* Handle timeouts */

    if (key == ZC_TIME_OUT)
	if (direct_call (g,routine) == 0)
	    goto continue_input;

    /* Handle hot keys */
    if(hot_keys)
		  {
		  switch(key)
		  {
		  case 18: case 16:
          case 19: case 21:
          case 14: case 24:
          case 4:
		  	{
			if (g->h_version == V4 && key == ZC_HKEY_UNDO)
				goto continue_input;
			if (!handle_hot_key (g,key))
				goto continue_input;

			return ZC_BAD;
			}
		  default:break;
		  }
		  }

    /* Return key */

    return key;

}/* stream_read_key */

/*
 * stream_read_input
 *
 * Read a line of input from the current input stream.
 *
 */

zchar stream_read_input (struct sg *g, short max, zchar *buf,
			  zword timeout, zword routine,
			  short hot_keys,
			  short no_scripting )
{
    zchar key = ZC_BAD;

    flush_buffer (g);

    /* Remove initial input from the transscript file or from the screen */

    if (g->ostream_script && g->enable_scripting && !no_scripting)
	script_erase_input (g,buf);
    if (g->istream_replay)
	screen_erase_input (g,buf);

    /* Read input line from current input stream */

continue_input:

    do {
	if (g->istream_replay)
	    key = replay_read_input (g,buf);
	else
	    key = console_read_input (g,max, buf, timeout, key != ZC_BAD);

    } while (key == ZC_BAD);

    /* Verify mouse clicks */

    if (key == ZC_SINGLE_CLICK || key == ZC_DOUBLE_CLICK)
	if (!validate_click (g))
	    goto continue_input;

    /* Copy input line to the command file */

    if (g->ostream_record && !g->istream_replay)
	record_write_input (g,buf, key);

    /* Handle timeouts */

    if (key == ZC_TIME_OUT)
	if (direct_call (g,routine) == 0)
	    goto continue_input;

    /* Handle hot keys */

    if (hot_keys)
	    {
		switch(key)
		  {
		  case 18: case 16:
          case 19: case 21:
          case 14: case 24:
          case 4:  case 8:
		    {
        	if (!handle_hot_key (g,key))
	          goto continue_input;
			return ZC_BAD;
			}
		  default: break;
    	  }
        }

    /* Copy input line to transscript file or to the screen */

    if (g->ostream_script && g->enable_scripting && !no_scripting)
	script_write_input (g,buf, key);
    if (g->istream_replay)
	screen_write_input (g,buf, key);

    /* Return terminating key */

    return key;

}/* stream_read_input */
