/*
 * variable.c
 *
 * Variable and stack related opcodes
 *
 */

#include "frotz.h"
#include "s5api.h"

/*
 * z_dec, decrement a variable.
 *
 * 	zargs[0] = variable to decrement
 *
 */

void z_dec (struct sg *g)
{
    zword value;

    if (g->zargs[0] == 0)
	(*(g->sp))--;
    else if (g->zargs[0] < 16)
	(*(g->fp - g->zargs[0]))--;
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	LOW_WORD (addr, value)
	value--;
	SET_WORD (addr, value)
    }

}/* z_dec */

/*
 * z_dec_chk, decrement a variable and branch if now less than value.
 *
 * 	zargs[0] = variable to decrement
 * 	zargs[1] = value to check variable against
 *
 */

void z_dec_chk (struct sg *g)
{
    zword value;

    if (g->zargs[0] == 0)
	value = --(*(g->sp));
    else if (g->zargs[0] < 16)
	value = --(*(g->fp - g->zargs[0]));
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	LOW_WORD (addr, value)
	value--;
	SET_WORD (addr, value)
    }

    branch (g,(short) value < (short) (g->zargs[1]));

}/* z_dec_chk */

/*
 * z_inc, increment a variable.
 *
 * 	zargs[0] = variable to increment
 *
 */

void z_inc (struct sg *g)
{
    zword value;

    if (g->zargs[0] == 0)
	(*(g->sp))++;
    else if (g->zargs[0] < 16)
	(*(g->fp - g->zargs[0]))++;
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	LOW_WORD (addr, value)
	value++;
	SET_WORD (addr, value)
    }

}/* z_inc */

/*
 * z_inc_chk, increment a variable and branch if now greater than value.
 *
 * 	zargs[0] = variable to increment
 * 	zargs[1] = value to check variable against
 *
 */

void z_inc_chk (struct sg *g)
{
    zword value;

    if (g->zargs[0] == 0)
	value = ++(*(g->sp));
    else if (g->zargs[0] < 16)
	value = ++(*(g->fp - g->zargs[0]));
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	LOW_WORD (addr, value)
	value++;
	SET_WORD (addr, value)
    }

    branch (g,(short) value > (short) (g->zargs[1]));

}/* z_inc_chk */

/*
 * z_load, store the value of a variable.
 *
 *	zargs[0] = variable to store
 *
 */

void z_load (struct sg *g)
{
    zword value;

    if (g->zargs[0] == 0)
	value = *(g->sp);
    else if (g->zargs[0] < 16)
	value = *(g->fp - g->zargs[0]);
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	LOW_WORD (addr, value)
    }

    store (g,value);

}/* z_load */

/*
 * z_pop, pop a value off the game stack and discard it.
 *
 *	no zargs used
 *
 */

void z_pop (struct sg *g)
{

    (g->sp)++;

}/* z_pop */

/*
 * z_pop_stack, pop n values off the game or user stack and discard them.
 *
 *	zargs[0] = number of values to discard
 *	zargs[1] = address of user stack (optional)
 *
 */

void z_pop_stack (struct sg *g)
{

    if (g->zargc == 2) {		/* it's a user stack */

	zword size;
	zword addr = g->zargs[1];

	LOW_WORD (addr, size)

	size += g->zargs[0];
	storew (g,addr, size);

    } else g->sp += g->zargs[0];	/* it's the game stack */

}/* z_pop_stack */

/*
 * z_pull, pop a value off...
 *
 * a) ...the game or a user stack and store it (V6)
 *
 *	zargs[0] = address of user stack (optional)
 *
 * b) ...the game stack and write it to a variable (other than V6)
 *
 *	zargs[0] = variable to write value to
 *
 */

void z_pull (struct sg *g)
{
    zword value;

    if (g->h_version != V6) {	/* not a V6 game, pop stack and write */

	value = *(g->sp)++;

	if (g->zargs[0] == 0)
	    *(g->sp) = value;
	else if (g->zargs[0] < 16)
	    *(g->fp - g->zargs[0]) = value;
	else {
	    zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	    SET_WORD (addr, value)
	}

    } else {			/* it's V6, but is there a user stack? */

	if (g->zargc == 1) {	/* it's a user stack */

	    zword size;
	    zword addr = g->zargs[0];

	    LOW_WORD (addr, size)

	    size++;
	    storew (g,addr, size);

	    addr += 2 * size;
	    LOW_WORD (addr, value)

	} else value = *(g->sp)++;	/* it's the game stack */

	store (g,value);

    }

}/* z_pull */

/*
 * z_push, push a value onto the game stack.
 *
 *	zargs[0] = value to push onto the stack
 *
 */

void z_push (struct sg *g)
{
    *--(g->sp) = g->zargs[0];

}/* z_push */

/*
 * z_push_stack, push a value onto a user stack then branch if successful.
 *
 *	zargs[0] = value to push onto the stack
 *	zargs[1] = address of user stack
 *
 */

void z_push_stack (struct sg *g)
{
    zword size;
    zword addr = g->zargs[1];

    LOW_WORD (addr, size)

    if (size != 0) {

	storew (g,(zword) (addr + 2 * size), g->zargs[0]);

	size--;
	storew (g,addr, size);

    }

    branch (g,size);

}/* z_push_stack */

/*
 * z_store, write a value to a variable.
 *
 * 	zargs[0] = variable to be written to
 *      zargs[1] = value to write
 *
 */

void z_store (struct sg *g)
{
    zword value = g->zargs[1];

    if (g->zargs[0] == 0)
	*(g->sp) = value;
    else if (g->zargs[0] < 16)
	*(g->fp - g->zargs[0]) = value;
    else {
	zword addr = g->h_globals + 2 * (g->zargs[0] - 16);
	SET_WORD (addr, value)
    }

}/* z_store */
