
/*! 
*
* Rogue is a runner for Interactive fiction games. It supports games
* written in many different toolkits. These include Adrift, AGT/AGX
* and TADS.
*
* \file rg_game.c
* 	Functions for opening/stopping games. This also handles the
*	all kinds of games supported by Rogue.
* 	
* 	
* \author Pallav Nawani
* Copyright Pallav Nawani (c) 2005
* 
*/

#include <string.h>
#include "interface.h"
#include "rg_game.h"
#include "rg_basic.h"

#ifdef __cplusplus
extern "C" {
#endif

// Defined in agility
void set_default_options(void);

#ifdef __cplusplus
}
#endif

//==///////////////////////////////////////////////////////////////////
//
///	Identifies which game type is being loaded by looking at
///	the file extension. Of course, this is not always right,
///	but then people who don't give their games proper names
///	should face the music.
///
/// \param
///	fname:	Filename of the game to be opened.
///
/// \return
///	The type of the game if it can be identified, RG_NOGAME
///	if not.
///
//==///////////////////////////////////////////////////////////////////
int identify_game_type(char *fname)
{
   char *cptr;
   int tads_version;
   static const char *defexts[] = { "gam", "t3" };
   char pfname[OSFNMAX];

   cptr = strrchr(fname, '.');
   if(NULL == cptr)
      return RG_NOGAME;

   cptr++;
   if(*cptr == '\n' || *cptr == '\r' || *cptr == '\0')
      return RG_NOGAME;

   // Adrift game
   if(!(strcasecmp(cptr, "taf")))
      return RG_ADRIFT;

   // Agility game
   if(!(strcasecmp(cptr, "agx")))
      return RG_AGX;

   // AGT game
   if(!(strcasecmp(cptr, "D$$")) || !(strcasecmp(cptr, "DA1")))
      return RG_AGX;

   // TADS game
   if(!(strcasecmp(cptr, "gam")) || !(strcasecmp(cptr, "t3")))
   {

      // Init pfname
      strcpy(pfname, "NONAME");

      tads_version = vm_get_game_type(fname,
      	    pfname, sizeof(pfname),
	    defexts, sizeof(defexts)/sizeof(defexts[0]));
      
      // If we found some file, get the real filename if necessary
      if(tads_version == VM_GGT_TADS2 || tads_version == VM_GGT_TADS3)
      {
      	 if(strcmp(fname, pfname) != 0)
	    strcpy(fname, pfname);
      }

      if(tads_version == VM_GGT_TADS2)
      	 return RG_TADS2;
      
      if(tads_version == VM_GGT_TADS3)
      	 return RG_TADS3;

   }

   return RG_NOGAME;
}



//==///////////////////////////////////////////////////////////////////
//
/// Opens a game recognized by rogue, and loads it, ready to play.
///
/// \param
///	rdata:	The resources for Rogue
///	fname:	Filename of the game to be opened.
/// \return
///	1 on success, 0 on failure.
//==///////////////////////////////////////////////////////////////////
int rg_open_game(Gtk_sdata *rdata, char *fname)
{

   int tflags = 0;
   GtkWidget *info;
   
   
   // Open the game dependant on the type
   rdata->rgame.type = identify_game_type(fname);
   switch(rdata->rgame.type) {

      case RG_ADRIFT:
	 rdata->rgame.scare.sgame = 
	    sc_game_from_filename(fname, tflags);
	 if(rdata->rgame.scare.sgame == NULL)
	 {
	    info = create_about_dlg("Game opening failure!",
		  "Error loading game. Is this really a Adrift game?");
	    gtk_widget_show(info);
	    rdata->rgame.type = RG_NOGAME;
	    return 0;
	 }
	 return 1;
	 break;

      case RG_AGX:
	 set_default_options();
	 init_interface(1, NULL);
	 return 1;
	 break;

      case RG_TADS2:
      case RG_TADS3:
	 return 1;
	 break;

      default:
	 info = create_about_dlg("Unable to recognize game",
	       "I don't understand how to run the game!");
	 gtk_widget_show(info);
	 rdata->rgame.type = RG_NOGAME;
	 break;
   }

   return 0;
}
      



//==///////////////////////////////////////////////////////////////////
//
///	Get the name of the game currently being played.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Pointer to the name of the game if a game is running, else
///	returns NULL
//
//==///////////////////////////////////////////////////////////////////
char *rg_get_game_name(Gtk_sdata *rdata)
{
   char *cptr;

   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 return sc_get_game_name(rdata->rgame.scare.sgame);
	 break;

      case RG_AGX:
      case RG_TADS2:
      case RG_TADS3:
	 cptr = strrchr(rdata->name, '/');
	 if(NULL == cptr)
	    cptr = rdata->name;
	 else {
	    cptr++;
	    if(*cptr == '\0')
	       cptr = rdata->name;
	 }
	 return cptr;
	 break;

      default:
	 break;
   }

   return NULL;
}




//==///////////////////////////////////////////////////////////////////
//
///	Returns the name of the author of the game currently
///	being played.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Pointer to the name of the author if a game is running, else
///	returns NULL
//
//==///////////////////////////////////////////////////////////////////
char *rg_get_author_name(Gtk_sdata *rdata)
{
   char *cptr;

   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 return sc_get_game_author(rdata->rgame.scare.sgame);
	 break;

      case RG_AGX:
      case RG_TADS2:
      case RG_TADS3:
	 cptr = strrchr(rdata->name, '/');
	 if(NULL == cptr)
	    cptr = rdata->name;
	 else {
	    cptr++;
	    if(*cptr == '\0')
	       cptr = rdata->name;
	 }
	 return cptr;
	 break;

      default:
	 break;
   }

   return NULL;
}





//==///////////////////////////////////////////////////////////////////
//
///	Get the default font requested by the current game.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Pointer to the name of the font if a game is running, else
///	returns NULL
//
//==///////////////////////////////////////////////////////////////////
char *rg_get_game_font(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 return sc_get_game_preferred_font(rdata->rgame.scare.sgame);
	 break;

      default:
	 break;
   }

   return NULL;
}






//==///////////////////////////////////////////////////////////////////
//
///	Do some game specific setup and initialization before
///	starting to play.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Nothing.
//
//==///////////////////////////////////////////////////////////////////
void rg_game_setup(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 /* disable score change notification by default */
	 sc_set_game_notify_score_change(rdata->rgame.scare.sgame, FALSE);
	 sc_set_game_verbose(rdata->rgame.scare.sgame,
	       rdata->cdata.verbosity);
	 break;

      case RG_AGX:
	 start_interface(0);
	 break;

      case RG_TADS2:
      case RG_TADS3:
	 os_init(NULL, NULL, NULL, NULL, 0);
	 break;

      default:
	 break;
   }

}






//==///////////////////////////////////////////////////////////////////
//
/// Is the game running?
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	1 if the game is running, 0 if not.
//
//==///////////////////////////////////////////////////////////////////
int rg_game_running(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 return sc_is_game_running (rdata->rgame.scare.sgame);
	 break;

      case RG_AGX:
      case RG_TADS2:
      case RG_TADS3:
	 return 1;
	 break;

      default:
	 break;
   }

   return 0;
}





//==///////////////////////////////////////////////////////////////////
//
/// Is the game finished? That is, has the player won/lost?
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	1 if the game finished, 0 if the game is unfinished.
//
//==///////////////////////////////////////////////////////////////////
int rg_game_finished(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 return sc_has_game_completed(rdata->rgame.scare.sgame);
	 break;

      case RG_AGX:
      case RG_TADS2:
      case RG_TADS3:
	 return 0;
	 break;

      default:
	 break;
   }

   return 1;
}




//==///////////////////////////////////////////////////////////////////
//
///	Free the game data & do other cleanup.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Nothing
//
//==///////////////////////////////////////////////////////////////////
void rg_free_game(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 sc_free_game (rdata->rgame.scare.sgame);
	 rdata->rgame.scare.sgame = NULL;
	 break;

      case RG_AGX:
	 close_interface();
	 break;

      case RG_TADS2:
      case RG_TADS3:
	 if(rdata->tads.sstr != NULL)
	    sprintf(rdata->tads.sstr, "\t|\t0/0");
	 break;

      default:
	 break;
   }

   gtk_window_set_title(GTK_WINDOW(rdata->mwin), "R O G U E");
   rdata->rgame.type = RG_NOGAME;

}






//==///////////////////////////////////////////////////////////////////
//
/// Restarts a game if it is already running.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_restart_game(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running (rdata->rgame.scare.sgame)) 
      	    rg_send_command(rdata, "restart");
	 break;

      case RG_AGX:
	 agt_restart();
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "restart");
	 break;

      default:
	 break;
   }

}






//==///////////////////////////////////////////////////////////////////
//
/// Loads a save game if the game is already running.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_load_save_game(Gtk_sdata *rdata)
{

   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running (rdata->rgame.scare.sgame))
      	    rg_send_command(rdata, "restore");
	 break;

      case RG_AGX:
	 agt_restore();
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "restore");
	 break;

      default:
	 break;
   }

}







//==///////////////////////////////////////////////////////////////////
//
/// Saves the game if a game is currently running.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_save_the_game(Gtk_sdata *rdata)
{

   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running (rdata->rgame.scare.sgame))
	    sc_save_game (rdata->rgame.scare.sgame);
	 break;

      case RG_AGX:
	 agt_save();
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "save");
	 break;

      default:
	 break;
   }

}





//==///////////////////////////////////////////////////////////////////
//
/// Quits a running game.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
int rg_quit_game(Gtk_sdata *rdata)
{

   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running(rdata->rgame.scare.sgame)) {
	    sc_quit_game(rdata->rgame.scare.sgame);
	    return 1;
	 }
	 break;

      case RG_AGX:
	 agt_quit();
	 return 1;
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "quit");
	 break;

      default:
	 break;
   }

   return 0;

}






//==///////////////////////////////////////////////////////////////////
//
/// Sets the verbosity mode for the game.
///
/// \param
///	rdata: All the data used by Rogue.
///	mode:  The verbosity mode for the game.	
/// \return
///	None
//
//==///////////////////////////////////////////////////////////////////
void rg_set_game_verbosity(Gtk_sdata *rdata, int mode)
{

   rdata->cdata.verbosity = mode;
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running(rdata->rgame.scare.sgame))
	    sc_set_game_verbose(rdata->rgame.scare.sgame, mode);
	 break;

      case RG_AGX:
      	 rg_send_command(rdata, "verbose");
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "verbose");
	 break;

      default:
	 break;
   }

}





//==///////////////////////////////////////////////////////////////////
//
/// Undo a turn if a game is running.
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	Returns 1 if undo was successful, 0 if not.
//
//==///////////////////////////////////////////////////////////////////
int rg_undo_game_turn(Gtk_sdata *rdata)
{


   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 if(sc_is_game_running(rdata->rgame.scare.sgame))
	 {
	    if(sc_is_game_undo_available(rdata->rgame.scare.sgame)) {
	       sc_undo_game_turn(rdata->rgame.scare.sgame);
	       return 1;
	    }
	 }
	 break;

      case RG_AGX:
      	 rg_send_command(rdata, "undo");
	 return 1;
	 break;

      case RG_TADS2:
      case RG_TADS3:
      	 rg_send_command(rdata, "undo");
	 return 1;
	 break;

      default:
	 break;
   }

   return 0;
}



//==///////////////////////////////////////////////////////////////////
//
/// Make a string for the Title bar,
///
/// \param
///	rdata: All the data used by Rogue.
///	title: String to hold the title.
/// \return
///	Nothing
//
//==///////////////////////////////////////////////////////////////////
void rg_make_title_string(Gtk_sdata *rdata, char *title, int size)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 snprintf(title, size, "R O G U E: %s by %s",
	       rg_get_game_name(rdata), rg_get_author_name(rdata));
	 break;

      case RG_AGX:
      case RG_TADS2:
      case RG_TADS3:
	 snprintf(title, size, "R O G U E: %s", rg_get_game_name(rdata));
	 break;

      default:
	 snprintf(title, size, "R O G U E");
	 break;
   }

}





//==///////////////////////////////////////////////////////////////////
//
///	
///
/// \param
///	rdata: All the data used by Rogue.
/// \return
///	
//
//==///////////////////////////////////////////////////////////////////
void rg_template(Gtk_sdata *rdata)
{
   switch(rdata->rgame.type)
   {
      case RG_ADRIFT:
	 break;

      case RG_AGX:
	 break;

      case RG_TADS2:
      case RG_TADS3:
	 break;

      default:
	 break;
   }

}


//==///////////////////////////////////////////////////////////////////
//
///
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////


/*  End of file rg_game.c  */

