
/*! 
*
* Rogue is a runner for Interactive fiction games. It supports games
* written in many different toolkits. These include Adrift, AGT/AGX
* and TADS.
*
* \file rg_tads2.c
* 	Routines for Interface to TADS
* 	
* 	
* \author Pallav Nawani
* Copyright Pallav Nawani (c) 2005
* 
*/



#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "support.h"
#include "rogue.h"
#include "interface.h"
#include "rg_basic.h"


extern Gtk_sdata *idata;
extern char fstring[256];

int status_mode;

/* Stores starting time for the timing functions. */
static long time_zero = 0;

/* Used to check for CTRL-Break interrupt */
static int break_set = 0;



//==///////////////////////////////////////////////////////////////////
//
/// Tads needs memicmp.
/// Provide memicmp since it's not a standard libc routine.
/// Taken from osunixt.c
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int memicmp(const char *s1, const char *s2, int len)
{
   char    *x1, *x2;   
   int result;
   int i;

   x1 = (char *)malloc(len);
   x2 = (char *)malloc(len);

   if (!x1 || !x2) {
      printf("Out of memory!\n");
      exit(-1);
   }

   for (i = 0; i < len; i++) {
      if (isupper(s1[i]))
	 x1[i] = tolower(s1[i]);
      else
	 x1[i] = s1[i];

      if (isupper(s2[i]))
	 x2[i] = tolower(s2[i]);
      else
	 x2[i] = s2[i];
   }

   result = memcmp(x1, x2, len);
   free(x1);
   free(x2);
   return result;
}





//==///////////////////////////////////////////////////////////////////
//
/// Tads has it own memory copy function.
/// our_memcpy - copy bytes (handles overlap, so we can equivalence
/// memmove() to it
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void * our_memcpy(void *dst, const void *src, size_t size)
{
   register char *d;
   register const char *s;
   register size_t n;

   if (size == 0)
      return(dst);

   s =(const char *) src;
   d =(char *) dst;
   if (s <= d && s + (size-1) >= d) {
      /* Overlap, must copy right-to-left. */
      s += size-1;
      d += size-1;
      for (n = size; n > 0; n--)
	 *d-- = *s--;
   } else
      for (n = size; n > 0; n--)
	 *d++ = *s++;

   return(dst);
}



//==///////////////////////////////////////////////////////////////////
//
/// Lowercase a string.
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
char * os_strlwr(char *s)
{
   char *start;
   start = s;
   while (*s) {
      if (isupper(*s))
	 *s = tolower(*s);

      s++;
   }
   return start;
}





//==///////////////////////////////////////////////////////////////////
//
/// Get the temporary file path.  This should fill in the buffer with a
/// path prefix (suitable for strcat'ing a filename onto) for a good
/// directory for a temporary file, such as the swap file.  
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
   void
os_get_tmp_path(char *s)
{
   strcpy(s, "/tmp");
}





//==///////////////////////////////////////////////////////////////////
//
/// os_defext(fn, ext) should append the default extension ext to the filename
/// in fn.  It is assumed that the buffer at fn is big enough to hold the added
/// characters in the extension.  The result should be null-terminated.  When
/// an extension is already present in the filename at fn, no action should be
/// taken.  On systems without an analogue of extensions, this routine should
/// do nothing.
///
/// For Unix, we extend this to also prepend the default saved game or game
/// file path name.
///
///   - The TADSSAVE environment variable holds the name of the save file
/// directory
///   - The TADSGAME envirnoment variable holds the name of the game file
/// directory
///
/// We only prepend  if there are no slashes in the filename already.
/// We don't prepand paths when running as the compiler, because we don't want
/// the output files to go in weird places.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_defext( char *fn, const char *ext )
{
   char *p, *n, tmp[1024];
   char *defpath;

   /*
    * Prepend default path
    */
   if (!memicmp(ext, "sav", strlen(ext)))
      defpath = getenv("TADSSAVE");
   else if (!memicmp(ext, "gam", strlen(ext)))
      defpath = getenv("TADSGAME");
   else
      defpath = NULL;

   if (defpath) {
      /*
       * Look for slashes.  If there are any, don't mess with name.
       */
      n = fn;
      while (*n) {
	 if (*n == '/')
	    break;
	 n++;
      }
      if (!*n) {
	 strcpy(tmp, defpath);
	 if (defpath[strlen(defpath)] != '/')
	    strcat(tmp, "/");
	 strcat(tmp, fn);
	 strcpy(fn, tmp);
      }  
   }

   p = fn+strlen(fn);
   while ( p>fn )
   {
      p--;
      if ( *p=='.' ) return;      /* already has an extension */
      if ( *p=='/' || *p=='\\' || *p==':'
	 ) break;    /* found a path */
   }
   strcat( fn, "." );          /* add a dot */
   strcat( fn, ext );          /* add the extension */
}


//==///////////////////////////////////////////////////////////////////
//
/// os_remext(fn) removes the extension from fn, if present. The buffer at
/// fn should be modified in place. If no extension is present, no action
/// should be taken. For systems without an analogue of extensions, this
/// routine should do nothing.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_remext( char *fn )
{
   char *p = fn+strlen(fn);
   while ( p>fn )
   {
      p--;
      if ( *p=='.' )
      {
	 *p = '\0';
	 return;
      }
      if ( *p=='/' || *p=='\\' || *p==':'
	 ) return;
   }
}



//==///////////////////////////////////////////////////////////////////
//
/// Add an extension, even if the filename currently has one 
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_addext(char *fn, const char *ext)
{
   strcat(fn, ".");
   strcat(fn, ext);
}



//==///////////////////////////////////////////////////////////////////
//
/// Returns a pointer to the root portion of the filename.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
char *os_get_root_name(char *buf)
{
   char *p = buf;

   p += strlen(buf) - 1;
   while (*p != '/' && p > buf)
      p--;
   if (p != buf) p++;

   return p;
}




//==///////////////////////////////////////////////////////////////////
//
/// Open a file for reading and writing in binary mode;
/// do not truncate 
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
osfildef *osfoprwb(const char *fname, os_filetype_t typ)
{
   osfildef *fp;

   /* try opening an existing file in read/write mode */
   fp = fopen(fname, "r+b");

   /* if that failed, create a new file in read/write mode */
   if (fp == 0)
      fp = fopen(fname, "w+b");

   /* return the file */
   return fp;
}




//==///////////////////////////////////////////////////////////////////
//
/// Get a filename from a startup parameter, if possible.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_paramfile(char *buf)
{
   return 0;
}


//==///////////////////////////////////////////////////////////////////
//
/// Return current status mode. Status mode shows where the next
/// os_print(z) output should go.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_get_status()
{
   return status_mode;
}



//==///////////////////////////////////////////////////////////////////
//
/// Set score to a string value provided by the caller, and
/// Display it
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_strsc(const char *p)
{
   if (NULL == p)
   {
   }
   else
   {
      if(idata->tads.sstr == NULL) {
	 idata->tads.sstr = (char *)malloc(256);
      }
      sprintf(idata->tads.sstr, "\t|\t%s", p);
   }
}




//==///////////////////////////////////////////////////////////////////
//
///
/// Set the score.  If cur == -1, the LAST score set with a non-(-1)
/// cur is displayed; this is used to refresh the status line without
/// providing a new score (for example, after exiting scrollback mode).
/// Otherwise, the given current score (cur) and turncount are
/// displayed, and saved in case cur==-1 on the next call.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_score(int cur, int turncount)
{

   /* check for the special -1 turn count */
   if (turncount == -1)
   {
   }
   else
   {
      if(idata->tads.sstr == NULL) {
	 idata->tads.sstr = (char *)malloc(256);
      }
      sprintf(idata->tads.sstr,
	    "\t|\tScore: %d Turns: %d",
	    cur, turncount);
   }
}




//==///////////////////////////////////////////////////////////////////
//
/// Get time elapsed in milliseconds from the time game
/// was initialized.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
long os_get_sys_clock_ms(void)
{
   if (time_zero == 0)
      time_zero = time(0);

   return ((time(0) - time_zero) * 1000);
}





//==///////////////////////////////////////////////////////////////////
//
///
/// Read in a line of text from the keyboard
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
unsigned char *os_gets(unsigned char *buf, size_t bufl)
{

   rg_read_line(idata);
   strncpy((char *)buf, idata->userin, bufl);
   return buf;
}



//==///////////////////////////////////////////////////////////////////
//
///
/// Get a character from the keyboard. For extended characters, return 0,
/// then return the extended key at the next call to this function.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_getc(void)
{
   idata->got_input = 0;
   idata->command   = WAIT_FOR_KEY;
   while(!idata->got_input) {
      gtk_main_iteration_do(1);
   }

   return idata->userin[0];
}



//==///////////////////////////////////////////////////////////////////
//
///
/// Get a character from the keyboard, returning low-level, untranslated
/// key codes.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_getc_raw(void)
{
    return os_getc();
}




///==//////////////////////////////////////////////////////////////////
//
/// Wait for a key to be hit
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_waitc(void)
{
   printf("os_waitc called!\n");
   os_getc();
}




///==//////////////////////////////////////////////////////////////////
//
/// Read a line of text with timeout. This can be done by a modified
/// version of rg_read_line();
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_gets_timeout(unsigned char *buf, size_t bufl,
      unsigned long timeout, int resume_editing)
{
    /* tell the caller this operation is not supported */
    return OS_EVT_NOTIMEOUT;
}




///==//////////////////////////////////////////////////////////////////
//
/// Do something if the text os_gets_timeout. Its supposed to cancel
/// interrupted input. Means nothing in a GTK Gui Environment.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_gets_cancel(int reset)
{

}




///==//////////////////////////////////////////////////////////////////
//
/// Get an event. That event could be a keystroke, mouse click etc?
/// This version can only get an keystroke.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_get_event(unsigned long timeout, int use_timeout,
      os_event_info_t *info)
{

   /* if there's a timeout, return an error indicating we don't allow it */
   if (use_timeout)
      return OS_EVT_NOTIMEOUT;

   /* get a key the normal way */
   info->key[0] = os_getc();

   /* if it's an extended key, get the other key */
   if (info->key[0] == 0)
   {
      /* get the extended key code */
      info->key[1] = os_getc();

      /* if it's EOF, return an EOF event rather than a key event */
      if (info->key[1] == CMD_EOF)
	 return OS_EVT_EOF;
   }

   /* return the keyboard event */
   return OS_EVT_KEY;
}


//==///////////////////////////////////////////////////////////////////
//
/// This function tells tads whether we support a requested TADS
/// feature or not. So that TADS knows not to send us a MIDI file
/// for playing, for example.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_get_sysinfo(int code, void *parm, long *result)
{
   switch(code) {

      case SYSINFO_TEXT_HILITE:
	 /* we do support text highlighting */
	 *result = 1;
	 return TRUE;

      case SYSINFO_HTML:
      case SYSINFO_JPEG:
      case SYSINFO_PNG:
      case SYSINFO_WAV:
      case SYSINFO_MIDI:
      case SYSINFO_WAV_MIDI_OVL:
      case SYSINFO_WAV_OVL:
      case SYSINFO_PREF_IMAGES:
      case SYSINFO_PREF_SOUNDS:
      case SYSINFO_PREF_MUSIC:
      case SYSINFO_PREF_LINKS:
      case SYSINFO_MPEG:
      case SYSINFO_MPEG1:
      case SYSINFO_MPEG2:
      case SYSINFO_MPEG3:
      case SYSINFO_LINKS_HTTP:
      case SYSINFO_LINKS_FTP:
      case SYSINFO_LINKS_NEWS:
      case SYSINFO_LINKS_MAILTO:
      case SYSINFO_LINKS_TELNET:
      case SYSINFO_PNG_TRANS:
      case SYSINFO_PNG_ALPHA:
      case SYSINFO_OGG:
      case SYSINFO_BANNERS:

	 /* Since we support none of these, set result to 0 */
	 *result = 0;

	 /* We recognized the code */
	 return TRUE;

      case SYSINFO_INTERP_CLASS:
	 /* we're a text-only character-mode interpreter */
	 *result = SYSINFO_ICLASS_TEXTGUI;
	 return TRUE;

      default:
	 return FALSE;
   }

}




//==///////////////////////////////////////////////////////////////////
//
/// Translate Extended HTML codes to the Ascii equivalents. This is
/// used to display these codes in Text only interpreters.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_xlat_html4(unsigned int html4_char,
      char *result, size_t result_len)
{

   /* Return all standard Latin-1 characters as-is */
   if (html4_char <= 128 || (html4_char >= 160 && html4_char <= 255))
   {
      result[0] = (unsigned char)html4_char;
   }
   else {
      switch (html4_char) {
	 case 130:                                      /* single back quote */
	    result[0] = '`'; break;
	 case 132:                                      /* double back quote */
	    result[0] = '\"'; break;
	 case 153:                                             /* trade mark */
	    strcpy(result, "(tm)"); return;
	 case 140:                                            /* OE ligature */
	 case 338:                                            /* OE ligature */
	    strcpy(result, "OE"); return;
	 case 339:                                            /* oe ligature */
	    strcpy(result, "oe"); return;
	 case 159:                                                   /* Yuml */
	    result[0] = 255;
	 case 376:                                        /* Y with diaresis */
	    result[0] = 'Y'; break;
	 case 352:                                           /* S with caron */
	    result[0] = 'S'; break;
	 case 353:                                           /* s with caron */
	    result[0] = 's'; break;
	 case 150:                                                /* en dash */
	 case 8211:                                               /* en dash */
	    result[0] = '-'; break;
	 case 151:                                                /* em dash */
	 case 8212:                                               /* em dash */
	    strcpy(result, "--"); return;
	 case 145:                                      /* left single quote */
	 case 8216:                                     /* left single quote */
	    result[0] = '`'; break;
	 case 146:                                     /* right single quote */
	 case 8217:                                    /* right single quote */
	 case 8218:                                    /* single low-9 quote */
	    result[0] = '\''; break;
	 case 147:                                      /* left double quote */
	 case 148:                                     /* right double quote */
	 case 8220:                                     /* left double quote */
	 case 8221:                                    /* right double quote */
	 case 8222:                                    /* double low-9 quote */
	    result[0] = '\"'; break;
	 case 8224:                                                /* dagger */
	 case 8225:                                         /* double dagger */
	 case 8240:                                        /* per mille sign */
	    result[0] = ' '; break;
	 case 139:                       /* single left-pointing angle quote */
	 case 8249:                      /* single left-pointing angle quote */
	    result[0] = '<'; break;
	 case 155:                      /* single right-pointing angle quote */
	 case 8250:                     /* single right-pointing angle quote */
	    result[0] = '>'; break;
	 case 8482:                                           /* small tilde */
	    result[0] = '~'; break;

	 default:
	    /* unmappable character - return space */
	    result[0] = (unsigned char)' ';
	    break;
      }
   }
   result[1] = 0;
}




//==///////////////////////////////////////////////////////////////////
//
/// Clear the screen.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void oscls(void)
{
   rg_clear_screen(idata);
}



//==///////////////////////////////////////////////////////////////////
//
/// Sleep for the given number of Micro Seconds
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_sleep_ms(long delay_in_ms)
{
    usleep(delay_in_ms);
}


//==///////////////////////////////////////////////////////////////////
//
///
/// Character map loading. What's this supposed to do??
/// Why have a dozen useless functions??
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_advise_load_charmap(char *id, char *ldesc, char *sysinfo)
{
}



//==///////////////////////////////////////////////////////////////////
//
///
/// Why have a dozen useless functions?? Wastes the time of porters!!
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_gen_charmap_filename(char *filename, char *internal_id, char *argv0)
{
    filename[0] = '\0';
}



//==///////////////////////////////////////////////////////////////////
//
///
/// Build a full path name given a path and a filename
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_build_full_path(char *fullpathbuf, size_t fullpathbuflen,
      const char *path, const char *filename)
{
   size_t plen, flen;
   int add_sep;

   /*
    *   Note whether we need to add a separator.  If the path prefix ends
    *   in a separator, don't add another; otherwise, add the standard
    *   system separator character.
    *   
    *   Do not add a separator if the path is completely empty, since this
    *   simply means that we want to use the current directory.  
    */
   plen = strlen(path);
   add_sep = (plen != 0
	 && path[plen-1] != OSPATHCHAR
	 && strchr(OSPATHALT, path[plen-1]) == 0);

   /* copy the path to the full path buffer, limiting to the buffer length */
   if (plen > fullpathbuflen - 1)
      plen = fullpathbuflen - 1;
   memcpy(fullpathbuf, path, plen);

   /* add the path separator if necessary (and if there's room) */
   if (add_sep && plen + 2 < fullpathbuflen)
      fullpathbuf[plen++] = OSPATHCHAR;

   /* add the filename after the path, if there's room */
   flen = strlen(filename);
   if (flen > fullpathbuflen - plen - 1)
      flen = fullpathbuflen - plen - 1;
   memcpy(fullpathbuf + plen, filename, flen);

   /* add a null terminator */
   fullpathbuf[plen + flen] = '\0';
}



//==///////////////////////////////////////////////////////////////////
//
///
/// os_exeseek  - opens the given .EXE file and seeks to the end of the
/// executable part of it, on the presumption that a datafile is to be
/// found there.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
osfildef *os_exeseek(const char *exefile, const char *typ )
{
   return((osfildef *)0);
}


//==///////////////////////////////////////////////////////////////////
//
///
/// Why have a dozen useless functions?? Wastes the time of porters!!
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_nonstop_mode(int flag)
{
}


//==///////////////////////////////////////////////////////////////////
//
///
/// Set the saved-game extension.  Most platforms don't need to do
/// anything with this information, and in fact most platforms won't even
/// have a way of letting the game author set the saved game extension,
/// so this trivial implementation is suitable for most systems.
///
/// Why have a dozen useless functions?? Wastes the time of porters!!
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_set_save_ext(const char *ext)
{
    /* ignore the setting */
}




//==///////////////////////////////////////////////////////////////////
//
/// Sets the text attributes. For example, set the text to BOLD etc.
///
/// \param
///	attr:	The attribute to set the text to.
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_set_text_attr(int attr)
{

   if((attr & (OS_ATTR_BOLD | OS_ATTR_STRONG)))
   {
      if(idata->tsize < CMDBUF) {

	 if(strcmp(idata->tags[idata->tsize-1], "question") != 0)
	 {
	    strcpy(idata->tags[idata->tsize], "question");
	    idata->tsize++;
	 }
      }
   }
   else if((attr & (OS_ATTR_ITALIC|OS_ATTR_EM)))
   {
      if(idata->tsize < CMDBUF) {

	 if(strcmp(idata->tags[idata->tsize-1], "italic") != 0)
	 {
	    strcpy(idata->tags[idata->tsize], "italic");
	    idata->tsize++;
	 }
      }
   }
   else if(attr & OS_ATTR_HILITE)
   {
      if(idata->tsize < CMDBUF) {
	 if(strcmp(idata->tags[idata->tsize-1], "bold") != 0)
	 {
	    strcpy(idata->tags[idata->tsize], "bold");
	    idata->tsize++;
	 }
      }
   }
   else  {
      if(idata->tsize > 1)
	 idata->tsize--;
   }

}



//==///////////////////////////////////////////////////////////////////
//
/// It would be wonderful if Gtk2 provided a way to set the text
/// color using RGB values! Well, if there is a way, I don't know
/// about it.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_set_text_color(os_color_t fg, os_color_t bg)
{
}



//==///////////////////////////////////////////////////////////////////
//
/// Set the title of the window to the given string.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_set_title(const char *title)
{
   rg_update_status_bar(idata, title);
}


//==///////////////////////////////////////////////////////////////////
//
/// Output a NULL terminated string to the text widget.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_printz(const char *str)
{
   char *cp, *cp1, *cp2;
   int i, cplen;

   if(str[0] == '\0')
      return;

   //printf("os_printz() called!\n");
   switch(status_mode) {
      case 0:
	 rg_output_string(idata, str);
	 break;

      case 1:
	 cp1 = strdup(str);
	 cp  = cp1;
	 while(*cp == '\n')
	    cp++;

	 if(*cp != '\0' && *cp != '\r')
	 {
	    cplen = strlen(cp);
	    for(i = 0; i < cplen; i++)
	    {
	       if(cp[i] == '\n') {
		  status_mode = 2;
		  cp[i] = '\0';
		  break;
	       }
	    }
	    
	    if(NULL == idata->tads.sstr) {
	       cp2 = strdup(cp);
	    }
	    else {
	       cp2 = (char *)malloc(strlen(cp) + strlen(idata->tads.sstr) + 2);
	       sprintf(cp2, "%s%s", cp, idata->tads.sstr);
	    }
	    rg_update_status_bar(idata, cp2);
	    free(cp2);
	 }

	 free(cp1);
	 break;

      default:
	 break;

   }

   //printf("os_printz() Exited!\n");

}


//==///////////////////////////////////////////////////////////////////
//
/// Output a string of the given length to the text widget. The
/// string might not be NULL terminated.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_print(const char *str, size_t len)
{
   char *cp, *cp1, *cp2;
   int i, cplen;

   if(len == 0)
      return;

   //printf("os_print() called!\n");
   cp1 =(char *) malloc(len+1);
   cp  = cp1;
   memcpy(cp, str, len);
   cp[len] = 0;

   if(cp[0] == '\0')
      return;
   
   switch(status_mode) {
      case 0:
	 rg_output_string(idata, cp);
	 break;

      case 1:
	 while(*cp == '\n')
	    cp++;
	 if(*cp != '\0' && *cp != '\r')
	 {
	    
	    cplen = strlen(cp);
	    for(i = 0; i < cplen; i++)
	    {
	       if(cp[i] == '\n') {
		  status_mode = 2;
		  cp[i] = '\0';
		  break;
	       }
	    }

	    if(NULL == idata->tads.sstr) {
	       cp2 = strdup(cp);
	    }
	    else {
	       cp2 = (char *)malloc(strlen(cp) + strlen(idata->tads.sstr) + 2);
	       sprintf(cp2, "%s%s", cp, idata->tads.sstr);
	    }
	    rg_update_status_bar(idata, cp2);
	    free(cp2);

	 }
	 break;

      default:
	 break;
	 
   }

   free(cp1);

   //printf("os_print() exited!\n");
   //printf("%.*s", (int)len, str);

}


//==///////////////////////////////////////////////////////////////////
//
///
/// Set the output status mode.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_status(int stat)
{
    status_mode = stat;
}

//==///////////////////////////////////////////////////////////////////
//
///
/// Check for control-break.  Returns status of break flag, and clears
/// the flag. 
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_break(void)
{
    int ret;
    
    ret = break_set;
    break_set = 0;

    return ret;
}



//==///////////////////////////////////////////////////////////////////
//
///
/// Set the terminal into "plain" mode. Yet another useless TADS
/// routine.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
void os_plain(void)
{
}
void os_expause(void)
{
}
void os_term(int status)
{
}
void os_more_prompt(void)
{
}




//==///////////////////////////////////////////////////////////////////
//
/// Do some initialization
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_init(int *argc, char *argv[], const char *prompt,
        char *buf, int bufsiz)
{
   char *cptr;
   idata->tads.argc = 2;

   if(NULL == idata->tads.argv) {
      idata->tads.argv = (char **)malloc(sizeof(char **)*2);
   }
   else {
      free(idata->tads.argv[0]);
   }
   
   cptr = (char *)malloc(strlen(idata->argv[0])+ strlen(idata->name)+ 2);
   
   strcpy(cptr, idata->argv[0]);
   idata->tads.argv[0] = cptr;
   idata->tads.argv[1] = cptr + strlen(cptr) + 1;
   strcpy(idata->tads.argv[1], idata->name);

   //printf("Opening %s -> %s\n", idata->tads.argv[0], idata->tads.argv[1]);

   return 0;
}




//==///////////////////////////////////////////////////////////////////
//
/// Prompt user for a file to open.
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////
int os_askfile(const char *prompt, char *fname_buf, int fname_buf_len,
      int prompt_type, os_filetype_t file_type)
{

   int sflag = 0;

   if(prompt_type == OS_AFP_SAVE)
      sflag = 1;

   if(rg_get_filename(idata, sflag)) {
      strncpy(fname_buf, idata->name, fname_buf_len);
      return OS_AFE_SUCCESS;
   }

   return OS_AFE_CANCEL;

}

/*
 *   Get the name of the character mapping file.
 *   
 *   NOTE - we should use some means to find the actual character set that
 *   the user's terminal is using and return that; maybe some environment
 *   variable or some curses call will tell us what we need to know.  For
 *   now, we'll just return "asc7dflt", which is the 7-bit ASCII character
 *   mapping; this should be safe for any ASCII terminal, although it won't
 *   allow us to display accented characters that the user's terminal might
 *   actually be capable of displaying if we just knew the actual character
 *   set it was using.  
 */
void os_get_charmap(char *mapname, int charmap_id)
{
    strcpy(mapname, "asc7dflt");
}

/*
 *   Reallocate storage at a different size
 */
void *osrealloc(void *buf, size_t len)
{
    return realloc(buf, len);
}

/*
 *   Extract the path from a filename
 *   
 *   Copied over from osnoui.c by tril@igs.net, 2000/Aug/18 
 */
void os_get_path_name(char *pathbuf, size_t pathbuflen, const char *fname)
{
    const char *lastsep;
    const char *p;
    size_t len;

    /* find the last separator in the filename */
    for (p = fname, lastsep = fname ; *p != '\0' ; ++p)
    {
        /*
         *   if it's a path separator character, remember it as the last one
         *   we've found so far 
         */
        if (*p == OSPATHCHAR || strchr(OSPATHALT, *p)  != 0)
            lastsep = p;
    }

    /* get the length of the prefix, not including the separator */
    len = lastsep - fname;

    /* make sure it fits in our buffer (with a null terminator) */
    if (len > pathbuflen - 1)
        len = pathbuflen - 1;

    /* copy it and null-terminate it */
    memcpy(pathbuf, fname, len);
    pathbuf[len] = '\0';
}



void os_set_screen_color(os_color_t color)
{
    /* colors aren't supported in non-RUNTIME mode - ignore it */
}
void os_update_display(void)
{
}

//   open a file for reading and writing in text mode; do not truncate 
osfildef *osfoprwt(const char *fname, os_filetype_t typ)
{
    osfildef *fp;

    /* try opening an existing file in read/write mode */
    fp = fopen(fname, "r+");

    /* if that failed, create a new file in read/write mode */
    if (fp == 0)
        fp = fopen(fname, "w+");

    /* return the file */
    return fp;
}


/// Tads 3 banner rubbish
void *os_banner_create(void *parent, int where, void *other, int wintype,
                       int align, int siz, int siz_units, unsigned long style)
{
    return 0;
}

void os_banner_delete(void *banner_handle)
{
}

void os_banner_orphan(void *banner_handle)
{
}

int os_banner_getinfo(void *banner_handle, os_banner_info_t *info)
{
    return FALSE;
}

void os_banner_clear(void *banner_handle)
{
}

int os_banner_get_charwidth(void *banner_handle)
{
    return 0;
}

int os_banner_get_charheight(void *banner_handle)
{
    return 0;
}

void os_banner_disp(void *banner_handle, const char *txt, size_t len)
{
}

void os_banner_set_color(void *banner_handle, os_color_t fg, os_color_t bg)
{
}

void os_banner_set_screen_color(void *banner_handle, os_color_t color)
{
}

void os_banner_flush(void *banner_handle)
{
}

void os_banner_set_size(void *banner_handle, int siz, int siz_units,
                        int is_advisory)
{
}

void os_banner_size_to_contents(void *banner_handle)
{
}

void os_banner_start_html(void *banner_handle)
{
}

void os_banner_end_html(void *banner_handle)
{
}

void os_banner_goto(void *banner_handle, int row, int col)
{
}

void os_banner_set_attr(void *banner_handle, int attr)
{
}



//==///////////////////////////////////////////////////////////////////
//
///
///
/// \param
///	
/// \return
///	
//==///////////////////////////////////////////////////////////////////


/*  End of file rg_tads2.c  */

