/* Compiler.c - the compiler */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#include <string.h>
#include "Drool.h"
#include "Objects.h"
#include "Compiler.h"
#include "Execute.h"
#include "Parser.h"

/* fixup list terminator */
#define NIL	0

/* symbol tables */
ATABLE *arguments;

/* global variables */
static ObjectPtr nounsSymbol;
static ObjectPtr adjectivesSymbol;
static ObjectPtr verbsSymbol;
static ObjectPtr prepositionsSymbol;

/* default action variables */
static int def_flag;		/* flag value */
static int def_mask;		/* mask value */

/* buffer for building code strings */
static unsigned char *codebuf,*cptr,*ctop;

/* buffer for building the literal vector */
static ObjectPtr *literalbuf,*lptr,*ltop;

/* external variables */
extern int t_value;		/* token value */
extern char t_token[];		/* token string */
extern char *t_names[];		/* token names */

/* prototypes */
void do_word(int type);
void do_synonym(void);
void do_default(void);
void do_dflag(int flag);
void do_define(void);
void do_function(void);
void do_variable(char *name);
void do_defobject(void);
void AddClass(ObjectPtr obj,ObjectPtr class);
ObjectPtr AddItemToList(ObjectPtr item,ObjectPtr list);
void do_wordlist(ObjectPtr obj,ObjectPtr prop,int type);
void do_property(ObjectPtr obj);
void do_shared_property(ObjectPtr obj);
void do_defmethod(void);
void SetProperty(ObjectPtr obj,ObjectPtr tag,ObjectPtr value);
void SetSharedProperty(ObjectPtr obj,ObjectPtr tag,ObjectPtr value);
void CompileCode(ObjectPtr obj);
void do_expr(void);
int in_ntab(void);
int in_ftab(void);
void do_let(void);
void do_cond(void);
void do_and(void);
void do_or(void);
void do_if(void);
void do_while(void);
void do_begin(void);
void do_setq(void);
void do_call(void);
void do_nary(int op,int n);
void do_string(void);
void do_identifier(void);
void do_number(void);
ObjectPtr ConstValue(void);
void AddArgument(ATABLE *atable,char *name);
ATABLE *MakeArgFrame(void);
void PushArgFrame(ATABLE *atable);
void PushNewArgFrame(void);
void PopArgFrame(void);
int FindArgument(char *name,int *plev,int *poff);
ObjectPtr ofind(char *name);
ObjectPtr oenter(char *name);
ObjectPtr senter(char *name);
void frequire(int rtkn);
void require(int tkn,int rtkn);
char *save(char *str);
int match(char *str);
void code_argument(int lev,int off);
void code_setargument(int lev,int off);
void code_variable(ObjectPtr var);
void code_setvariable(ObjectPtr var);
void code_literal(ObjectPtr lit);
int codeaddr(void);
void putcbyte(int b);
void putcword(int w);
void fixup(int chn,int val);
int addliteral(ObjectPtr lit);

/* opcode tables */
static struct { char *nt_name; int nt_code,nt_args; } *nptr,ntab[] = {
	"not",			OP_NOT,		1,
	"+",			OP_ADD,		2,
	"-",			OP_SUB,		2,
	"*",			OP_MUL,		2,
	"/",			OP_DIV,		2,
	"rem",			OP_REM,		2,
	"<",			OP_LT,		2,
	"<=",			OP_LEQ,		2,
	"=",			OP_EQ,		2,
	"/=",			OP_NEQ,		2,
	">=",			OP_GEQ,		2,
	">",			OP_GT,		2,
	"getp",			OP_GETP,	2,
	"setp!",		OP_SETP,	3,
	"cons",			OP_CONS,	2,
	"car",			OP_CAR,		1,
	"set-car!",		OP_SETCAR,	2,
	"cdr",			OP_CDR,		1,
	"set-cdr!",		OP_SETCDR,	2,
	"vector-size",		OP_VSIZE,	1,
	"vector-ref",		OP_VREF,	2,
	"vector-set!",		OP_VSET,	3,
	"number?",		OP_NUMBERP,	1,
	"string?",		OP_STRINGP,	1,
	"symbol?",		OP_SYMBOLP,	1,
	"cons?",		OP_CONSP,	1,
	"vector?",		OP_VECTORP,	1,
	"object?",		OP_OBJECTP,	1,
	"method?",		OP_METHODP,	1,
	"cmethod?",		OP_CMETHODP,	1,
	"package?",		OP_PACKAGEP,	1,
	"null?",		OP_NULLP,	1,
	"call-next-method",	OP_CALLNEXT,	0,
	0
};
static struct { char *ft_name; void (*ft_fcn)(void); } *fptr,ftab[] = {
	"let",		do_let,
	"cond",		do_cond,
	"and",		do_and,
	"or",		do_or,
	"if",		do_if,
	"while",	do_while,
	"begin",	do_begin,
	"set!",		do_setq,
	0
};

/* InitCompiler - initialize the compiler */
void InitCompiler(long csize,long lsize)
{
    CheckNIL(codebuf = (unsigned char *)NewPtr(csize));
    CheckNIL(literalbuf = (ObjectPtr *)NewPtr(lsize * sizeof(ObjectPtr)));
    ctop = codebuf + csize;
    ltop = literalbuf + lsize;
    InitCompilerVariables();
}

/* InitCompilerVariables - initialize compiler variables */
void InitCompilerVariables(void)
{
    nounsSymbol = InternCString(symbolPackage,"nouns");
    adjectivesSymbol = InternCString(symbolPackage,"adjectives");
    verbsSymbol = InternCString(symbolPackage,"verbs");
    prepositionsSymbol = InternCString(symbolPackage,"prepositions");
    lptr = literalbuf;
 }

/* ProtectCompilerVariables - protect compiler variables from the garbage collector */
void ProtectCompilerVariables(void)
{
    ObjectPtr *p;
    nounsSymbol = CopyObject(nounsSymbol);
    adjectivesSymbol = CopyObject(adjectivesSymbol);
    verbsSymbol = CopyObject(verbsSymbol);
    prepositionsSymbol = CopyObject(prepositionsSymbol);
    for (p = literalbuf; p < lptr; ++p)
	*p = CopyObject(*p);
}

/* Compile - compile code */
void Compile(StreamHandle s)
{
    int tkn;

    /* initialize */
    def_flag = def_mask = 0;
    arguments = NULL;
    lptr = literalbuf;
    InitScan(s);

    /* process statements until end of file */
    while ((tkn = token()) == T_OPEN) {
	frequire(T_IDENTIFIER);

	/* vocabulary statements */
	if (match("adjective"))
	    do_word(WT_ADJECTIVE);
	else if (match("preposition"))
	    do_word(WT_PREPOSITION);
	else if (match("conjunction"))
	    do_word(WT_CONJUNCTION);
	else if (match("article"))
	    do_word(WT_ARTICLE);
	else if (match("synonym"))
	    do_synonym();
	else if (match("default"))
	    do_default();

	/* variable and function definition statement */
	else if (match("define"))
	    do_define();

	/* object and method definition statements */
        else if (match("defobject"))
	    do_defobject();
	else if (match("defmethod"))
	    do_defmethod();

	/* error, unknown statement */
	else
	    error("unknown statement type");
    }
    require(tkn,T_EOF);
}

/* do_word - enter words of a particular type */
void do_word(int type)
{
    ObjectPtr word;
    int tkn,flags;
    while ((tkn = token()) == T_IDENTIFIER) {
	word = InternCString(wordPackage,t_token);
	flags = NumberP(SymbolValue(word)) ? UnboxNumber(SymbolValue(word)) : 0;
	SetSymbolValue(word,BoxNumber(flags | type));
    }
    require(tkn,T_CLOSE);
}

/* do_synonym - handle the <SYNONYMS ... > statement */
void do_synonym(void)
{
    ObjectPtr word;
    int tkn;
    frequire(T_IDENTIFIER);
    cpush(InternSymbol(wordPackage,NewCStringObject(t_token)));
    while ((tkn = token()) == T_IDENTIFIER) {
	word = InternCString(wordPackage,t_token);
	SetSymbolValue(word,*sp);
    }
    drop(1);
    require(tkn,T_CLOSE);
}

/* do_default - handle the <DEFAULT ... > statement */
void do_default(void)
{
    int tkn;
    while ((tkn = token()) == T_OPEN) {
	frequire(T_IDENTIFIER);
	if (match("actor"))
	    do_dflag(A_ACTOR);
	else if (match("direct-object"))
	    do_dflag(A_DOBJECT);
	else if (match("indirect-object"))
	    do_dflag(A_IOBJECT);
	else
	    error("Unknown default definition statement type");
    }
    require(tkn,T_CLOSE);
}

/* do_dflag - handle ACTOR, DIRECT-OBJECT, and INDIRECT-OBJECT statements */
void do_dflag(int flag)
{
    int tkn;
    if ((tkn = token()) == T_IDENTIFIER) {
	if (match("required")) {
	    def_flag |= flag;
	    def_mask &= ~flag;
	}
	else if (match("forbidden")) {
	    def_flag &= ~flag;
	    def_mask &= ~flag;
	}
	else if (match("optional"))
	    def_mask |= flag;
	else
	    error("Expecting: REQUIRED, FORBIDDEN or OPTIONAL");
	tkn = token();
    }
    else {
	def_flag |= flag;
	def_mask &= ~flag;
    }
    require(tkn,T_CLOSE);
}

/* do_define - handle the (DEFINE ... ) statement */
void do_define(void)
{
    switch (token()) {
    case T_OPEN:
	do_function();
	break;
    case T_IDENTIFIER:
	do_variable(t_token);
	break;
    default:
	error("expecting a variable of function definition");
	break;
    }
}

/* do_function - handle (DEFINE (fun ...) ... ) statement */
void do_function(void)
{
    int tkn;

    /* get the function name */
    frequire(T_IDENTIFIER);
info("function: %s",t_token);

    /* create a new object */
    cpush(NewMethodObject(nilObject,nilObject));
    SetSymbolValue(senter(t_token),*sp);

    /* initialize the argument list */
    arguments = NULL;
    PushNewArgFrame();

    /* get the argument list */
    while ((tkn = token()) != T_CLOSE) {
	require(tkn,T_IDENTIFIER);
	AddArgument(arguments,t_token);
    }
    
    /* compile the code */
    CompileCode(pop());
}

/* do_variable - handle the (DEFINE var ... ) statement */
void do_variable(char *name)
{
    int tkn;

    /* create a new variable */
    cpush(senter(name));

    /* get the initial value */
    if ((tkn = token()) != T_CLOSE) {
	stoken(tkn);
	SetSymbolValue(*sp,ConstValue());
	frequire(T_CLOSE);
    }
    drop(1);
}

/* do_defobject - handle object definitions */
void do_defobject(void)
{
    int tkn;

    frequire(T_IDENTIFIER);
info("object: %s",t_token);
    cpush(oenter(t_token));

    /* copy the property list of each class object */
    frequire(T_OPEN);
    while ((tkn = token()) == T_IDENTIFIER) {
	cpush(ofind(t_token));
	AddClass(sp[1],*sp);
	for (*sp = ObjectProperties(*sp); *sp != nilObject; *sp = Cdr(*sp))
	    SetProperty(sp[1],PropertyTag(Car(*sp)),PropertyValue(Car(*sp)));
	drop(1);
    }
    require(tkn,T_CLOSE);

    /* process statements until end of file */
    while ((tkn = token()) == T_OPEN) {
	frequire(T_IDENTIFIER);
	if (match("noun"))
	    do_wordlist(*sp,nounsSymbol,WT_NOUN);
	else if (match("adjective"))
	    do_wordlist(*sp,adjectivesSymbol,WT_ADJECTIVE);
	else if (match("verb"))
	    do_wordlist(*sp,verbsSymbol,WT_VERB);
	else if (match("preposition"))
	    do_wordlist(*sp,prepositionsSymbol,WT_PREPOSITION);
	else if (match("property"))
	    do_property(*sp);
	else if (match("shared-property"))
	    do_shared_property(*sp);
	else
	    error("unknown object definition statement type");
    }
    require(tkn,T_CLOSE);
}

/* AddClass - add a class (and its superclasses) to an object */
void AddClass(ObjectPtr obj,ObjectPtr class)
{
    cpush(obj);
    cpush(class);
    SetObjectClassList(sp[1],AddItemToList(*sp,ObjectClassList(sp[1])));
    SetObjectSearchList(sp[1],AddItemToList(*sp,ObjectSearchList(sp[1])));
    for (*sp = ObjectSearchList(*sp); *sp != nilObject; *sp = Cdr(*sp))
	SetObjectSearchList(sp[1],AddItemToList(Car(*sp),ObjectSearchList(sp[1])));
    drop(2);
}

/* AddItemToList - add an item to a list without duplicates */
ObjectPtr AddItemToList(ObjectPtr item,ObjectPtr list)
{
    ObjectPtr last,p;
    for (p = list, last = nilObject; p != nilObject; last = p, p = Cdr(p))
	if (Car(p) == item)
	    return list;
    cpush(list);
    cpush(last);
    p = Cons(item,nilObject);
    if ((last = pop()) != nilObject)
	SetCdr(last,p);
    else
	*sp = p;
    return pop();
}
    
/* do_wordlist - handle the <NOUN/ADJECTIVE/VERB/PREPOSITION ... > statements */
void do_wordlist(ObjectPtr obj,ObjectPtr prop,int type)
{
    ObjectPtr word,listEntry,p;
    int tkn,flags;
    cpush(obj);
    cpush(prop);
    while ((tkn = token()) == T_IDENTIFIER) {
	word = InternCString(wordPackage,t_token);
	flags = NumberP(SymbolValue(word)) ? UnboxNumber(SymbolValue(word)) : 0;
	SetSymbolValue(word,BoxNumber(flags | type));
	listEntry = Cons(word,nilObject);
	if ((p = GetSharedProperty(sp[1],*sp)) == nil)
	    SetSharedProperty(sp[1],*sp,listEntry);
	else {
	    SetCdr(listEntry,PropertyValue(p));
	    SetPropertyValue(p,listEntry);
	}
    }
    drop(2);
    require(tkn,T_CLOSE);
}

/* do_property - handle the (PROPERTY ... ) statement */
void do_property(ObjectPtr obj)
{
    int tkn;
    cpush(obj);
    while ((tkn = token()) == T_IDENTIFIER) {
	cpush(senter(t_token));
	SetProperty(sp[1],*sp,ConstValue());
	drop(1);
    }
    drop(1);
    require(tkn,T_CLOSE);
}

/* do_shared_property - handle the (SHARED-PROPERTY ... ) statement */
void do_shared_property(ObjectPtr obj)
{
    int tkn;
    cpush(obj);
    while ((tkn = token()) == T_IDENTIFIER) {
	cpush(senter(t_token));
	SetSharedProperty(sp[1],*sp,ConstValue());
	drop(1);
    }
    drop(1);
    require(tkn,T_CLOSE);
}

void do_defmethod(void)
{
    char cname[TKNSIZE+1];
    int tkn;

    /* get the class name */
    frequire(T_OPEN);
    frequire(T_IDENTIFIER);
    strcpy(cname,t_token);
    
    /* get the method name */
    frequire(T_QUOTE);
    frequire(T_IDENTIFIER);
info("%s method: %s",cname,t_token);

    /* create a new method */
    cpush(NewMethodObject(nilObject,nilObject));	/* sp[2] - method */
    cpush(InternCString(symbolPackage,t_token));	/* sp[1] - method name */
    cpush(ofind(cname));				/* sp[0] - object */

    /* initialize the argument list */
    arguments = NULL;
    PushNewArgFrame();

    /* add the %selector argument */
    AddArgument(arguments,"%selector");
    
    /* get the argument list */
    while ((tkn = token()) != T_CLOSE) {
	require(tkn,T_IDENTIFIER);
	AddArgument(arguments,t_token);
    }
    
    /* add the instance and %more arguments */
    AddArgument(arguments,"self");
    AddArgument(arguments,"%next");

    /* store the object as the value of the property and in the method */
    SetSharedProperty(sp[0],sp[1],sp[2]);
    SetMethodClass(sp[2],sp[0]);
    drop(2);

    /* compile the code */
    CompileCode(pop());
}

void SetProperty(ObjectPtr obj,ObjectPtr tag,ObjectPtr value)
{
    ObjectPtr last = nilObject,p;
    for (p = ObjectProperties(obj); p != nilObject; last = p, p = Cdr(p))
	if (PropertyTag(Car(p)) == tag) {
	    SetPropertyValue(Car(p),value);
	    return;
	}
    cpush(obj);
    cpush(last);
    p = Cons(NewPropertyObject(tag,value),nilObject);
    if (*sp != nilObject) SetCdr(*sp,p);
    else SetObjectProperties(sp[1],p);
    drop(2);
}

void SetSharedProperty(ObjectPtr obj,ObjectPtr tag,ObjectPtr value)
{
    ObjectPtr last = nilObject,p;
    for (p = ObjectSharedProperties(obj); p != nilObject; last = p, p = Cdr(p))
	if (PropertyTag(Car(p)) == tag) {
	    SetPropertyValue(Car(p),value);
	    return;
	}
    cpush(obj);
    cpush(last);
    p = Cons(NewPropertyObject(tag,value),nilObject);
    if (*sp != nilObject) SetCdr(*sp,p);
    else SetObjectSharedProperties(sp[1],p);
    drop(2);
}

/* CompileCode - compile and build a code object */
void CompileCode(ObjectPtr code)
{
    unsigned char *src,*dst;
    int tkn;

    /* initialize */
    cptr = codebuf;
    lptr = literalbuf;
    cpush(code);

    /* compile the code */
    putcbyte(OP_PUSH);
    while ((tkn = token()) != T_CLOSE) {
	stoken(tkn);
	do_expr();
    }
    putcbyte(OP_RETURN);

    /* build the code object */
    SetMethodCode(*sp,NewStringObject(codebuf,cptr - codebuf));
    SetMethodLiterals(*sp,NewVectorObject(literalbuf,lptr - literalbuf));
    drop(1);

    /* pop the current argument frame */
    PopArgFrame();
}

/* do_expr - compile a subexpression */
void do_expr(void)
{
    int tkn;

    switch (token()) {
    case T_OPEN:
	switch (tkn = token()) {
	case T_IDENTIFIER:
	    if (in_ntab() || in_ftab())
		break;
	default:
	    stoken(tkn);
	    do_call();
	}
	break;
    case T_QUOTE:
	frequire(T_IDENTIFIER);
	code_literal(InternCString(symbolPackage,t_token));
	break;
    case T_STRING:
	do_string();
	break;
    case T_IDENTIFIER:
	do_identifier();
	break;
    case T_NUMBER:
	do_number();
	break;
    default:
	error("expecting expression");
    }
}

/* in_ntab - check for a function in ntab */
int in_ntab(void)
{
    for (nptr = ntab; nptr->nt_name; ++nptr)
	if (strcmp(t_token,nptr->nt_name) == 0) {
	    do_nary(nptr->nt_code,nptr->nt_args);
	    return (TRUE);
	}
    return (FALSE);
}

/* in_ftab - check for a function in ftab */
int in_ftab(void)
{
    for (fptr = ftab; fptr->ft_name; ++fptr)
	if (strcmp(t_token,fptr->ft_name) == 0) {
	    (*fptr->ft_fcn)();
	    return (TRUE);
	}
    return (FALSE);
}

/* do_let - compile the (LET ... ) expression */
void do_let(void)
{
    ATABLE *atable;
    int tkn,tcnt;

    /* make a new argument frame */
    atable = MakeArgFrame();

    /* compile each initialization expression */
    frequire(T_OPEN);
    for (tcnt = 0; (tkn = token()) != T_CLOSE; ++tcnt) {
	require(tkn,T_OPEN);
	frequire(T_IDENTIFIER);
	AddArgument(atable,t_token);
	putcbyte(OP_PUSH);
	do_expr();
	frequire(T_CLOSE);
    }
    PushArgFrame(atable);

    /* push a new argument frame */
    putcbyte(OP_TFRAME);
    putcbyte(tcnt);

    /* compile the body */
    putcbyte(OP_PUSH);
    do_begin();

    /* pop the argument frame */
    putcbyte(OP_TPOP);
    PopArgFrame();
}

/* do_cond - compile the (COND ... ) expression */
void do_cond(void)
{
    int tkn,nxt,end;

    /* initialize the fixup chain */
    end = NIL;

    /* compile each COND clause */
    while ((tkn = token()) != T_CLOSE) {
	require(tkn,T_OPEN);
	do_expr();
	putcbyte(OP_BRF);
	nxt = codeaddr();
	putcword(NIL);
	while ((tkn = token()) != T_CLOSE) {
	    stoken(tkn);
	    do_expr();
	}
	putcbyte(OP_BR);
	end = codeaddr();
	putcword(end);
	fixup(nxt,codeaddr());
    }

    /* fixup references to the end of statement */
    if (end)
	fixup(end,codeaddr());
    else
	putcbyte(OP_NIL);
}

/* do_and - compile the (AND ... ) expression */
void do_and(void)
{
    int tkn,end;

    /* initialize the fixup chain */
    end = NIL;

    /* compile each expression */
    while ((tkn = token()) != T_CLOSE) {
	stoken(tkn);
	do_expr();
	putcbyte(OP_BRF);
	end = codeaddr();
	putcword(end);
    }

    /* fixup references to the end of statement */
    if (end)
	fixup(end,codeaddr());
    else
	putcbyte(OP_NIL);
}

/* do_or - compile the (OR ... ) expression */
void do_or(void)
{
    int tkn,end;

    /* initialize the fixup chain */
    end = NIL;

    /* compile each expression */
    while ((tkn = token()) != T_CLOSE) {
	stoken(tkn);
	do_expr();
	putcbyte(OP_BRT);
	end = codeaddr();
	putcword(end);
    }

    /* fixup references to the end of statement */
    if (end)
	fixup(end,codeaddr());
    else
	putcbyte(OP_T);
}

/* do_if - compile the (IF ... ) expression */
void do_if(void)
{
    int tkn,nxt,end;

    /* compile the test expression */
    do_expr();

    /* skip around the 'then' clause if the expression is false */
    putcbyte(OP_BRF);
    nxt = codeaddr();
    putcword(NIL);

    /* compile the 'then' clause */
    do_expr();

    /* compile the 'else' clause */
    if ((tkn = token()) != T_CLOSE) {
	putcbyte(OP_BR);
	end = codeaddr();
	putcword(NIL);
	fixup(nxt,codeaddr());
	stoken(tkn);
	do_expr();
	frequire(T_CLOSE);
	nxt = end;
    }

    /* handle the end of the statement */
    fixup(nxt,codeaddr());
}

/* do_while - compile the (WHILE ... ) expression */
void do_while(void)
{
    int tkn,nxt,end;

    /* compile the test expression */
    nxt = codeaddr();
    do_expr();

    /* skip around the 'then' clause if the expression is false */
    putcbyte(OP_BRF);
    end = codeaddr();
    putcword(NIL);

    /* compile the loop body */
    while ((tkn = token()) != T_CLOSE) {
	stoken(tkn);
	do_expr();
    }

    /* branch back to the start of the loop */
    putcbyte(OP_BR);
    putcword(nxt);

    /* handle the end of the statement */
    fixup(end,codeaddr());
}

/* do_begin - compile the (BEGIN ... ) expression */
void do_begin(void)
{
    int tkn,n;

    /* compile each expression */
    for (n = 0; (tkn = token()) != T_CLOSE; ++n) {
	stoken(tkn);
	do_expr();
    }

    /* check for an empty statement list */
    if (n == 0)
	putcbyte(OP_NIL);
}

/* do_setq - compile the (SETQ v x) expression */
void do_setq(void)
{
    char name[TKNSIZE+1];
    int lev,off;

    /* get the symbol name */
    frequire(T_IDENTIFIER);
    strcpy(name,t_token);

    /* compile the value expression */
    do_expr();

    /* check for this being a local symbol */
    if (FindArgument(name,&lev,&off))
	code_setargument(lev,off);
    else
	code_setvariable(senter(name));
    frequire(T_CLOSE);
}

/* do_call - compile a function call */
void do_call(void)
{
    int tkn,n;
    
    /* compile the function itself */
    do_expr();

    /* compile each argument expression */
    for (n = 0; (tkn = token()) != T_CLOSE; ++n) {
	stoken(tkn);
	putcbyte(OP_PUSH);
	do_expr();
    }
    putcbyte(OP_CALL);
    putcbyte(n);
}

/* do_nary - compile nary operator expressions */
void do_nary(int op,int n)
{
    while (n--) {
	do_expr();
	if (n) putcbyte(OP_PUSH);
    }
    putcbyte(op);
    frequire(T_CLOSE);
}

/* do_string - compile a string */
void do_string(void)
{
    code_literal(NewCStringObject(t_token));
}

/* do_identifier - compile an identifier */
void do_identifier(void)
{
    ObjectPtr sym;
    int lev,off;
    if (match("t"))
	putcbyte(OP_T);
    else if (match("nil"))
	putcbyte(OP_NIL);
    else if (FindArgument(t_token,&lev,&off))
	code_argument(lev,off);
    else
	code_variable(senter(t_token));
}

/* do_number - compile a number */
void do_number(void)
{
    code_literal(BoxNumber(t_value));
}

/* ConstValue - get a constant value */
ObjectPtr ConstValue(void)
{
    ObjectPtr sym,obj;
    switch (token()) {
    case T_IDENTIFIER:	if (strcmp(t_token,"nil") == 0)
    			    return nilObject;
    			else if (strcmp(t_token,"t") == 0)
    			    return trueObject;
    			sym = senter(t_token);
			obj = SymbolValue(sym);
			if (!ObjectP(obj)) {
			    cpush(sym);
			    obj = NewObject();
			    SetSymbolValue(*sp,obj);
			    drop(1);
			}
			return obj;
    case T_NUMBER:	return BoxNumber(t_value);
    case T_STRING:	return NewCStringObject(t_token);
    default:		error("expecting identifier or number");
			return nilObject;
    }
}

/* AddArgument - add a formal argument */
void AddArgument(ATABLE *atable,char *name)
{
    ARGUMENT *arg;
    if ((arg = (ARGUMENT *)osalloc(sizeof(ARGUMENT))) == NULL)
	fatal("out of memory");
    arg->arg_name = save(name);
    arg->arg_next = atable->at_arguments;
    atable->at_arguments = arg;
}

/* MakeArgFrame - make a new argument frame */
ATABLE *MakeArgFrame(void)
{
    ATABLE *atable;
    if ((atable = (ATABLE *)osalloc(sizeof(ATABLE))) == NULL)
	fatal("out of memory");
    atable->at_arguments = NULL;
    atable->at_next = NULL;
    return (atable);
}

/* PushArgFrame - push an argument frame onto the stack */
void PushArgFrame(ATABLE *atable)
{
    atable->at_next = arguments;
    arguments = atable;
}

/* PushNewArgFrame - push a new argument frame onto the stack */
void PushNewArgFrame(void)
{
    PushArgFrame(MakeArgFrame());
}

/* PopArgFrame - push an argument frame off the stack */
void PopArgFrame(void)
{
    ARGUMENT *arg,*nxt;
    ATABLE *atable;
    for (arg = arguments->at_arguments; arg != NULL; arg = nxt) {
	nxt = arg->arg_next;
	osfree(arg->arg_name);
	osfree((char *)arg);
	arg = nxt;
    }
    atable = arguments->at_next;
    osfree((char *)arguments);
    arguments = atable;
}

/* FindArgument - find an argument offset */
int FindArgument(char *name,int *plev,int *poff)
{
    ATABLE *table;
    ARGUMENT *arg;
    int lev,off;
    lev = 0;
    for (table = arguments; table != NULL; table = table->at_next) {
	off = 0;
	for (arg = table->at_arguments; arg != NULL; arg = arg->arg_next) {
	    if (strcmp(name,arg->arg_name) == 0) {
		*plev = lev;
		*poff = off;
		return (TRUE);
	    }
	    ++off;
	}
	++lev;
    }
    return (FALSE);
}

/* ofind - find an object in the symbol table */
ObjectPtr ofind(char *name)
{
    ObjectPtr sym = senter(name);
    ObjectPtr obj = SymbolValue(sym);
    if (obj == nilObject) {
	cpush(sym);
	obj = NewObject();
	SetSymbolValue(*sp,obj);
	drop(1);
    }
    else if (!ObjectP(obj))
	error("not an object");
    return obj;
}

/* oenter - enter an object into the symbol table */
ObjectPtr oenter(char *name)
{
    ObjectPtr sym = senter(name);
    ObjectPtr obj = SymbolValue(sym);
    if (!ObjectP(obj)) {
	cpush(sym);
	obj = NewObject();
	SetSymbolValue(*sp,obj);
	drop(1);
    }
    SetObjectClassList(obj,nilObject);
    SetObjectSearchList(obj,nilObject);
    SetObjectProperties(obj,nilObject);
    SetObjectSharedProperties(obj,nilObject);
    return obj;
}

/* senter - find a symbol in the symbol table */
ObjectPtr senter(char *name)
{
    return InternCString(symbolPackage,name);
}

/* frequire - fetch a token and check it */
void frequire(int rtkn)
{
    require(token(),rtkn);
}

/* require - check for a required token */
void require(int tkn,int rtkn)
{
    extern int linenumber;
    if (tkn != rtkn)
	error("expecting %s on line %d, got %s",t_names[rtkn],linenumber,t_names[tkn]);
}

/* save - allocate memory for a string */
char *save(char *str)
{
    char *new;
    if ((new = osalloc(strlen(str)+1)) == NULL)
	fatal("out of memory");
    strcpy(new,str);
    return (new);
}

/* match - compare a string with the current token */
int match(char *str)
{
    return (strcmp(str,t_token) == 0);
}

/* code_argument - compile an argument reference */
void code_argument(int lev,int off)
{
    putcbyte(OP_ARG);
    putcbyte(lev);
    putcbyte(off);
}

/* code_setargument - compile a set argument reference */
void code_setargument(int lev,int off)
{
    putcbyte(OP_ASET);
    putcbyte(lev);
    putcbyte(off);
}

/* code_variable - compile a variable reference */
void code_variable(ObjectPtr var)
{
    putcbyte(OP_VAR);
    putcword(addliteral(var));
}

/* code_setvariable - compile a set variable reference */
void code_setvariable(ObjectPtr var)
{
    putcbyte(OP_SET);
    putcword(addliteral(var));
}

/* code_literal - compile a literal reference */
void code_literal(ObjectPtr lit)
{
    putcbyte(OP_LIT);
    putcword(addliteral(lit));
}

/* codeaddr - get the current code address (actually, offset) */
int codeaddr(void)
{
    return (cptr - codebuf);
}

/* putcbyte - put a code byte into the code buffer */
void putcbyte(int b)
{
    if (cptr >= ctop)
	error("insufficient code space");
    *cptr++ = b;
}

/* putcword - put a code word into the code buffer */
void putcword(int w)
{
    putcbyte(w);
    putcbyte(w >> 8);
}

/* fixup - fixup a reference chain */
void fixup(int chn,int val)
{
    int hval,nxt;
    for (hval = val >> 8; chn != NIL; chn = nxt) {
	nxt = (codebuf[chn] & 0xFF) | (codebuf[chn+1] << 8);
	codebuf[chn] = val;
	codebuf[chn+1] = hval;
    }
}

/* addliteral - add a literal to the literal vector */
int addliteral(ObjectPtr lit)
{
    ObjectPtr *p;
    int off;
    for (p = literalbuf; p < lptr; ++p)
	if (*p == lit)
	    return p - literalbuf;
    if (lptr >= ltop)
	error("insufficient literal space");
    off = lptr - literalbuf;
    *lptr++ = lit;
    return off;
}

          
